/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$ 
 *      
 * Authors:     
 *              Pedro Roque             <roque@di.fc.ul.pt>     
 *              Pascal Anelli           <anelli@masi.ibp.fr>
 *
 * Fixes:
 *
 *
 * Description:
 *
 *      IPv6 over IPv4 tunnel device - Simple Internet Transition (SIT)
 *      Linux INET6 implementation
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */
/*---------- Contents ----------------------------------------------------
|
|  sit_xmit, ipv6_add_tunnel, sit_err
|
-------------------------------------------------------------------------*/

#include <linux/errno.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/interrupt.h>

#include <linux/net.h>
#include <linux/netdevice.h>
#include <linux/in.h>
#include <linux/if_arp.h>
#include <linux/icmp.h>

#include <linux/sockios.h>
#include <linux/socket.h>

#include <net/protocol.h>
#include <net/snmp.h>
#include <net/sock.h>
#include <net/ip.h>

#include <linux/dret/ipv6_debug.h>
#include <net/dret/ipv6.h>
#include <net/dret/icmpv6.h>
#include <net/dret/nd.h>
#include <net/dret/ipv6_route.h>
#include <net/dret/addrconf.h>
#include <net/dret/sit.h>
#include <net/dret/route6.h>

#define SIT_TUNNELS
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[SIT]%X:%X:%X:%X \n",\
		      (a)->s6_addr32[0], (a)->s6_addr32[1], \
                      (a)->s6_addr32[2], (a)->s6_addr32[3]);

#ifndef SIT_TUNNELS
int sit_init_dev(struct device *dev);
struct device sit0_dev =
{
   "sit0",
   0, 0, 0, 0,
   0x0, 0,
   0, 0, 0, NULL, sit_init_dev
};
#endif

/* 	Functions Declaration  */
/*      =====================  */

static unsigned long		sit_gc_last_run;
static void			sit_mtu_cache_gc(void);

static int			sit_xmit(struct sk_buff *skb, 
					 struct device *dev);
static int			sit_rcv(struct sk_buff *skb, unsigned short len);
static void 			sit_err(struct sk_buff *skb, unsigned char *dp);

static int			sit_open(struct device *dev);
static int			sit_close(struct device *dev);

static struct enet_statistics *	sit_get_stats(struct device *dev);


static struct inet_protocol sit_protocol =
{
   sit_rcv,
   sit_err,
   0,
   IPPROTO_IPV6,
   0,
   NULL,
   "IPv6"
};

#define SIT_NUM_BUCKETS	16

struct sit_mtu_info *sit_mtu_cache[SIT_NUM_BUCKETS];

static struct sit_dev_info *sit_tunnel_base = NULL;

static __inline__ __u32 sit_addr_hash(__u32 addr)
{

   __u32 hash_val;

   hash_val = addr;

   hash_val ^= hash_val >> 16;
   hash_val ^= hash_val >> 8;

   return (hash_val & (SIT_NUM_BUCKETS - 1));
}

__u32 get_sit_end_point(struct device * dev)
{
   struct sit_dev_info *dinfo;

   for (dinfo = sit_tunnel_base; dinfo && dinfo->dev != dev;
	dinfo = dinfo->next);
   if (dinfo)
      return dinfo->addr;
   else
      return 0;
}

/*--START FUNCTION--(ipv6_add_tunnel)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 
|   (bb) 12/03/96
|
-----------------------------------------------------------------------------*/
int ipv6_add_tunnel(struct device *dev, struct in6_addr *a)
{
   struct sit_dev_info *dinfo;
   struct in6_rtmsg rtmsg;
   int err = 0;
   
#ifdef IPV6_DEBUG_SIT
   printk(KERN_DEBUG "[SIT]ipv6_add_tunnel: IN dev = %s %X\n", dev->name, dev->flags);
   PRINT_ADDR(a);
#endif

   if (!IS_IPV4ADDR6(*a))
      return -EINVAL;
   /* SIT device? */
   if (memcmp(dev->name, "ct", 2) == 0) {
      /* free? */
      cli();
      for (dinfo = sit_tunnel_base; 
	   (dinfo && dinfo->dev != dev && (dinfo->addr != a->s6_addr32[3]));
	   dinfo = dinfo->next);
      sti();
      if (dinfo == NULL) {
	 /* Put it in table */
	 dinfo = (struct sit_dev_info *) kmalloc(sizeof(*dinfo), GFP_KERNEL);
	 if (dinfo) {
	    memset(&dinfo->addr, 0, sizeof(struct in6_addr));
	    dinfo->addr = a->s6_addr32[3];
	    dinfo->dev = dev;
	    cli();
	    dinfo->next = sit_tunnel_base;
	    sit_tunnel_base = dinfo;
	    dev->flags |= (IFF_TUNNEL | IFF_MULTICAST);
            sti();
#ifdef IPV6_DEBUG_SIT	    
	    printk(KERN_DEBUG "[SIT]ipv6_add_tunnel: OUT OK %p %X\n", dev, dev->flags);
	    /* Add route for that destination */
	    printk(KERN_DEBUG "[ADDRCONF]sit_route_add: IN \n");
#endif
	    rtmsg.rtmsg_type = RTMSG_NEWROUTE;
	    COPY_ADDR6(rtmsg.rtmsg_dst, *a);
	    memset(&rtmsg.rtmsg_gateway, 0, sizeof(struct in6_addr));

	    /* prefix length - 96 bytes "::d.d.d.d" */
	    rtmsg.rtmsg_prefixlen = 128;
	    rtmsg.rtmsg_metric = 1;
	    rtmsg.rtmsg_flags = RTF_NEXTHOP | RTF_UP;
	    strcpy(rtmsg.rtmsg_device, dev->name);

	    err = ipv6_route_add(&rtmsg);

	    if (err) {
	       printk(KERN_DEBUG "sit_route_add: error in route_add\n");
	    }

#ifdef IPV6_DEBUG_SIT
	    printk(KERN_DEBUG "[ADDRCONF]sit_route_add: OUT \n");
#endif
	    return err;
	 } else
	    return -ENOMEM;
      } else
	 return -EEXIST;    
   }
   return -EINVAL;
}
/*--END FUNCTION-------------------------------------------------------------*/

/*--START FUNCTION--(ipv6_del_tunnel)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 
|   (bb) 12/03/96
|
-----------------------------------------------------------------------------*/
int ipv6_del_tunnel(struct device *dev, struct in6_addr *a)
{
   struct sit_dev_info *dinfo, *dinfo_prev = NULL;
#ifdef IPV6_DEBUG_SIT
   printk(KERN_DEBUG "[SIT]ipv6_del_tunnel: IN dev = %s", dev->name);
   PRINT_ADDR(a);
#endif

   /* SIT device? */
   if ((memcmp(dev->name, "ct", 2) == 0)) {
      /* free? */
      cli();
      for (dinfo = sit_tunnel_base; (dinfo && dinfo->dev != dev);
	   dinfo_prev = dinfo, dinfo = dinfo->next);
      if (dinfo) {
	 if (dinfo_prev)
	    dinfo_prev->next = dinfo->next;
	 else
	    sit_tunnel_base = dinfo->next;
	 dev->flags &= ~(IFF_TUNNEL | IFF_MULTICAST);
	 sti();
	 kfree(dinfo);
	 return 0;
	 /* Del corresponding routes */
	 /* FIXME */
      } else {
	 sti();
	 printk(KERN_DEBUG "No dinfo for this adress / interface\n");
	 return -EINVAL;
      }
      sti();
   }
   printk(KERN_DEBUG "Not a configured tunnel interface\n");
   return -EINVAL;
}
/*--END FUNCTION-------------------------------------------------------------*/

/*--START FUNCTION--(sit_cache_insert)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
static void sit_cache_insert(__u32 addr, int mtu)
{
   struct sit_mtu_info *minfo;
   int hash;

   minfo = kmalloc(sizeof(struct sit_mtu_info), GFP_ATOMIC);

   if (minfo == NULL)
      return;

   minfo->addr = addr;
   minfo->tstamp = jiffies;
   minfo->mtu = mtu;

   hash = sit_addr_hash(addr);

   minfo->next = sit_mtu_cache[hash];
   sit_mtu_cache[hash] = minfo;
}
/*--END FUNCTION--(sit_cache_insert)---------------------------------------*/

/*--START FUNCTION--(sit_mtu_lookup)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
static struct sit_mtu_info *sit_mtu_lookup(__u32 addr)
{
   struct sit_mtu_info *iter;
   int hash;

   hash = sit_addr_hash(addr);

   for (iter = sit_mtu_cache[hash]; iter; iter = iter->next) {
      if (iter->addr == addr) {
	 iter->tstamp = jiffies;
	 break;
      }
   }

   /*
    *    run garbage collector
    */

   if (jiffies - sit_gc_last_run > SIT_GC_FREQUENCY) {
      sit_mtu_cache_gc();
      sit_gc_last_run = jiffies;
   }
   return iter;
}
/*--END FUNCTION--(sit_mtu_lookup)---------------------------------------*/

/*--START FUNCTION--(sit_mtu_cache_gc)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
static void sit_mtu_cache_gc(void)
{
   struct sit_mtu_info *iter, *back;
   unsigned long now = jiffies;
   int i;

   for (i = 0; i < SIT_NUM_BUCKETS; i++) {
      back = NULL;
      for (iter = sit_mtu_cache[i]; iter;) {
	 if (now - iter->tstamp > SIT_GC_TIMEOUT) {
	    struct sit_mtu_info *old;

	    old = iter;
	    iter = iter->next;

	    if (back) {
	       back->next = iter;
	    } else {
	       sit_mtu_cache[i] = iter;
	    }

	    kfree(old);
	    continue;
	 }
	 back = iter;
	 iter = iter->next;
      }
   }
}
/*--END FUNCTION--(sit_mtu_cache_gc)---------------------------------------*/

/*--START FUNCTION--(sit_init_dev)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         nom
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
int sit_init_dev(struct device *dev)
{
   int i;

	dev->open	= sit_open;
	dev->stop	= sit_close;

	dev->hard_start_xmit	= sit_xmit;
	dev->get_stats		= sit_get_stats;

	dev->priv = kmalloc(sizeof(struct enet_statistics), GFP_KERNEL);

	if (dev->priv == NULL)
		return -ENOMEM;

	memset(dev->priv, 0, sizeof(struct enet_statistics));


	for (i = 0; i < DEV_NUMBUFFS; i++)
		skb_queue_head_init(&dev->buffs[i]);

	dev->hard_header	= NULL;
	dev->rebuild_header 	= NULL;
	dev->set_mac_address 	= NULL;
	dev->hard_header_cache 	= NULL;
	dev->header_cache_update= NULL;

	dev->type		= ARPHRD_SIT;

        dev->hard_header_len = MAX_HEADER + sizeof(struct iphdr);
	dev->mtu		= 1500 - sizeof(struct iphdr);
	dev->addr_len		= 0;
	dev->tx_queue_len	= 0;

	memset(dev->broadcast, 0, MAX_ADDR_LEN);
	memset(dev->dev_addr,  0, MAX_ADDR_LEN);

	dev->flags		= IFF_NOARP;	

	dev->family		= AF_INET6;
	dev->pa_addr		= 0;
	dev->pa_brdaddr 	= 0;
	dev->pa_dstaddr		= 0;
	dev->pa_mask		= 0;
	dev->pa_alen		= 4;
	
   return 0;
}
/*--END FUNCTION--(sit_init_dev)---------------------------------------*/

/*--START FUNCTION--(sit_open)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
int sit_open(struct device *dev)
{
   return 0;
}
/*--END FUNCTION--(sit_open)---------------------------------------*/

/*--START FUNCTION--(sit_close)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
int sit_close(struct device *dev)
{
   return 0;
}

/*--END FUNCTION--(sit_close)---------------------------------------*/

/*--START FUNCTION--(sit_init)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         nom
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
int sit_init(void)
{
   int i;
#ifdef IPV6_DEBUG_SIT   
   printk(KERN_DEBUG "[SIT]sit_init: IN \n");
#endif

   /* register device */
#ifndef SIT_TUNNELS
   printk(KERN_DEBUG "Registering sit device\n");
   if (register_netdev(&sit0_dev) != 0) {
      return -EIO;
   }
#endif
   
   inet_add_protocol(&sit_protocol);

   for (i = 0; i < SIT_NUM_BUCKETS; i++)
      sit_mtu_cache[i] = NULL;

   sit_gc_last_run = jiffies;

   return 0;
}

/*--END FUNCTION--(sit_init)---------------------------------------*/

/*--START FUNCTION--(sit_cleanup)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
void sit_cleanup(void)
{
#ifndef SIT_TUNNELS
   unregister_netdev(&sit0_dev);
#endif
   inet_del_protocol(&sit_protocol);
}

/*--END FUNCTION--(sit_cleanup)---------------------------------------*/

/*
 *    receive IPv4 ICMP messages
 */

/*--START FUNCTION--(sit_err)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
static void sit_err(struct sk_buff *skb, unsigned char *dp)
{
	struct iphdr *iph = (struct iphdr*)dp;
	int type = skb->h.icmph->type;
	int code = skb->h.icmph->code;

	if (type == ICMP_DEST_UNREACH && code == ICMP_FRAG_NEEDED)
	{
		struct sit_mtu_info *minfo;
		unsigned short new_mtu = skb->h.icmph->un.frag.mtu - sizeof(struct iphdr);

		minfo = sit_mtu_lookup(iph->daddr);

		if (minfo == NULL)
		{
			minfo = kmalloc(sizeof(struct sit_mtu_info),
					GFP_ATOMIC);

			if (minfo == NULL)
				return;

			start_bh_atomic();
			sit_cache_insert(iph->daddr, new_mtu);
			end_bh_atomic();
		}
		else
		{
			printk(KERN_DEBUG "sit: %08lx pmtu = %ul\n", ntohl(iph->saddr), minfo->mtu);
			minfo->mtu = new_mtu;
		}
	}
}
/*--END FUNCTION--(sit_err)---------------------------------------*/

/*--START FUNCTION--(sit_rcv)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
static int sit_rcv(struct sk_buff *skb, unsigned short len)
{
        char *name = "sit0";
   	struct enet_statistics *stats;
	struct device *dev;
  	struct sit_dev_info *dinfo = sit_tunnel_base;
	__u32  saddr = skb->nh.iph->saddr;
 
   printk(KERN_DEBUG "[SIT]sit_rcv IN\n");

   /*
    * This becomes an IPv6 incoming packet . . .  
    */
   skb->h.raw = skb->nh.raw = skb_pull(skb, skb->h.raw - skb->data);
   skb->protocol = __constant_htons(ETH_P_IPV6);

   dev = dev_get(name);

   while (dinfo) {
      if (dinfo->addr == saddr) {
	 dev = dinfo->dev;
	 break;
      }
      dinfo = dinfo->next;
   }

   if (!dev) 
      return -1;
   
   stats = (struct enet_statistics *) dev->priv;
   stats->rx_packets++;

   /*
    *    Force checksum for debug purposes
    */
   skb->ip_summed = CHECKSUM_NONE;
   skb->dev = dev;

   ipv6_rcv(skb, dev, NULL);
   printk(KERN_DEBUG "[SIT]sit_rcv OUT\n");
   return 0;
}

/*--END FUNCTION--(sit_rcv)---------------------------------------*/

/*--START FUNCTION--(sit_xmit)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
int sit_xmit(struct sk_buff *skb, struct device *dev)
{
   struct enet_statistics *stats;
   struct sit_mtu_info *minfo;
   struct neighbor *neigh;
   struct ipv6hdr *ip6h = skb->nh.ipv6h;
   /*unsigned long flags;*/
   struct rtable *rt;
   struct iphdr *iph;
   __u32 saddr;
   __u32 daddr;

   int mtu;
   int len;

   /* 
    *    Make sure we are not busy (check lock variable) 
    */
#ifdef IPV6_DEBUG_SIT   
   printk(KERN_DEBUG "[SIT]sit_xmit: skb = %p | dev = %p IN\n", skb, dev);
#endif
   stats = (struct enet_statistics *) dev->priv;

   neigh = skb->nexthop;

   if (IS_IPV4ADDR6(ip6h->daddr))
      daddr = ip6h->daddr.s6_addr32[3];
   else if (neigh) {
      if (IS_IPV4ADDR6(neigh->n_addr) || IS_LINKLADDR6(neigh->n_addr))
	 daddr = neigh->n_addr.s6_addr32[3];
      else {
	 PRINT_ADDR(&neigh->n_addr); 
	 printk(KERN_DEBUG "[SIT]on error 1\n");
	 goto on_error;
      }
   } else if (IS_MULTIADDR6(ip6h->daddr)) {
      daddr = get_sit_end_point(dev);
      if (daddr == 0) {
	 printk(KERN_DEBUG "[SIT]on error 2\n");
	 goto on_error;
      }
   } else {
      printk(KERN_DEBUG "[SIT]sit_xmit: non v4/v6 multicast address\n");
      goto on_error;
   }
   
   if (skb->sk) {
      atomic_sub(skb->truesize, &skb->sk->wmem_alloc);
   }
   
   skb->sk       = NULL;
   skb->protocol = htons(ETH_P_IP);

   /* get route */
   
#ifdef IPV6_DEBUG_SIT   
   printk(KERN_DEBUG "[SIT]Getting ipv4 route... \n");
#endif*

   if (ip_route_output(&rt, daddr, 0, 0, NULL)) {
      printk(KERN_DEBUG "sit: no route to host\n");
      goto on_error;
   }

   minfo = sit_mtu_lookup(daddr);
   if (minfo)
      mtu = minfo->mtu;
   else
      mtu = rt->u.dst.pmtu;

   len = skb->tail - (skb->data + sizeof(struct ipv6hdr));
   if ((mtu > 576) && (len > mtu)) {
      icmpv6_send_error(skb, ICMP6_PKTTOOBIG, 0, mtu, dev);
      ip_rt_put(rt);
      goto on_error;
   }
   
   iph = (struct iphdr *) skb_push(skb, sizeof(struct iphdr));
   
   skb->dev = rt->u.dst.dev;

   if (skb->dev->hard_header_len) {
      /* int mac; */

      if (skb->data - skb->head < skb->dev->hard_header_len) {	 
	 printk(KERN_DEBUG "sit: space at head (%d) < dev header (%d)\n",
		       skb->data - skb->head, skb->dev->hard_header_len);
	 ip_rt_put(rt);
	 goto on_error;
      }
     /* 
      if (skb->dev->hard_header) {
	 mac = skb->dev->hard_header(skb, skb->dev, ETH_P_IP,
				     NULL, NULL, len);

	 if (mac < 0)
	    skb->arp = 0;
      } */
   }
 
   /* IPv4 header */
   saddr       = rt->rt_src;
   dst_release(skb->dst);
   skb->dst    = &rt->u.dst;
   skb->nh.iph = iph;
   
   iph->version  = 4;
   iph->ihl      = 5;
   iph->tos      = 0;				/* tos set to 0... */
   
   if (mtu > 576)
      iph->frag_off = htons(IP_DF);
   else
      iph->frag_off = 0;
   
   iph->ttl      = 64;
   iph->saddr    = saddr;
   iph->daddr    = daddr;
   iph->protocol = IPPROTO_IPV6;
   iph->tot_len  = htons(skb->len); 
   iph->id	 = htons(ip_id_count++);   
   ip_send_check(iph);
 
 /*
   skb->sit_cookie = 0xFFFF;
 
 */
   
   printk(KERN_DEBUG "ip_send(skb) IN\n");
   ip_send(skb);   
   printk(KERN_DEBUG "ip_send(skb) OUT\n");
   stats->tx_packets++;

#ifdef IPV6_DEBUG_SIT
   printk(KERN_DEBUG "[SIT]sit_xmit: packet sent to %X\n", daddr);
#endif

   return 0;

 on_error:
   dev_kfree_skb(skb, FREE_WRITE);
   printk(KERN_DEBUG "[SIT]sit_xmit: exiting on error\n");
   stats->tx_errors++;
   return 0;
}
/*--END FUNCTION--(sit_xmit)---------------------------------------*/

/*--START FUNCTION--(enet_statistics)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
static struct enet_statistics *sit_get_stats(struct device *dev)
{
   return ((struct enet_statistics *) dev->priv);
}

/*--END FUNCTION--(enet_statistics)---------------------------------------*/

/*
 * Local variables:
 *  compile-command: "gcc -D__KERNEL__ -I/usr/src/linux/include -Wall -Wstrict-prototypes -O2 -fomit-frame-pointer -fno-strength-reduce -pipe -m486 -DCPU=486 -DMODULE -DMODVERSIONS -include /usr/src/linux/include/linux/modversions.h  -c -o sit.o sit.c"
 * c-file-style: "Linux"
 * End:
 */



