/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:     $Id$ 
 *      
 * Authors:     
 *              name    <E_mail>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *              RAW for IPv6.
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|  Function_name
|
-------------------------------------------------------------------------*/

#include <linux/config.h>
#include <asm/system.h>
#include <asm/segment.h>
#include <asm/uaccess.h>

#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/net.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>

#include <net/checksum.h>
#include <net/snmp.h>
#include <net/sock.h>

#include <net/dret/proc6.h>
#include <net/dret/protocol6.h>
#include <net/dret/raw6.h>
#include <net/dret/ipv6.h>

#define min(a,b)	((a)<(b)?(a):(b))

struct sock *raw6_sock_array[SOCK_ARRAY_SIZE];



/*
 * We deliver the packet to the second matching socket and the followings.
 * The first matching socket (in sk) will be delivered after, to avoid a copy.
 */
struct sock *raw6_deliver(struct sk_buff *skb, 
                          int nexthdr, 
                          struct in6_addr *saddr, 
                          struct in6_addr *daddr)
{
   struct sock *sk, *sk2;

   /* Get the first entry in the hash table */
   if (!(sk = raw6_prot.sock_array[(unsigned char) (nexthdr & (SOCK_ARRAY_SIZE - 1))]))
      return (NULL);

   if ((sk = get_sock_raw6(sk, nexthdr, daddr, saddr))) {
      sk2 = sk;

      /* warnig, but should be ok */
      while ((sk2 = get_sock_raw6(sk2->next, nexthdr,
				  daddr, saddr)) != 0) {
	 struct sk_buff *skb2;

	 skb2 = skb_clone(skb, GFP_ATOMIC);
	 skb2->sk = sk2;
	 raw6_rcv(sk2, skb2);
      }
   }
   return (sk);
}

/* 
 * Backlog
 */
int raw6_rcv_skb(struct sock *sk, struct sk_buff *skb)
{
   /* Charge it to the socket. */
   if (sock_queue_rcv_skb(sk, skb) < 0) {
      ip6_statistics.IpInDiscards++;
      skb->sk = NULL;
      kfree_skb(skb, FREE_READ);
      return 0;
   }
   ip6_statistics.IpInDelivers++;
   return 0;
}


/*
 * Deliver the packet to the socket. 
 * Parameter sk is not necessary (it is the same as skb->sk), 
 * but it avoids having a local variable.
 */
int raw6_rcv(struct sock *sk, struct sk_buff *skb)
{
   /* We transmit the full packet */
   skb->h.raw = (unsigned char *) skb->nh.ipv6h;

   /* Somebody is listening ? */
   if (sk->users) {
      __skb_queue_tail(&sk->back_log, skb);
      return (0);
   }
   raw6_rcv_skb(sk, skb);

   return (0);
}


/* 
 * Simply copies the fragment to the skbuff.
 * Used when protocol is not IPPROTO_RAW
 */
static void raw6_getfrag(const void *p, 
                         struct in6_addr *saddr, 
                         char *to, 
                         unsigned int offset, 
                         unsigned int fraglen)
{
   struct iovec *iov = (struct iovec *) p;

   memcpy_fromiovecend(to, iov, offset, fraglen);
}

/* 
 * Copies the fragment to the skbuff and computes a checksum
 * Used when protocol is IPPROTO_RAW 
 */
static void raw6_getfragcsum(const void *p, 
                             struct in6_addr *saddr, 
                             char *to, 
                             unsigned int offset, 
                             unsigned int fraglen)
{
   struct raw6_fake_header *hdr = (struct raw6_fake_header *) p;

   hdr->csum = csum_partial_copy_fromiovecend(to, hdr->iov,
					      offset, fraglen,
					      hdr->csum);
   if (offset == 0) {
      struct sock *sk;
      struct raw6_tpinfo *raw6_opt;

      sk = hdr->sk;
      hdr->csum = csum_ipv6_magic(saddr, &hdr->daddr, hdr->len,
				  hdr->protocol, hdr->csum);

      raw6_opt = &sk->tp_pinfo.raw6;

      /* Check if place to put checksum is not beyond the fragment */
      if ((raw6_opt->ck_offset) < fraglen) {
	 __u16 *csum;
	 csum = (__u16 *) (to + (raw6_opt->ck_offset));
	 *csum = hdr->csum;
      } else {
	 /* 
	  * FIXME 
	  * signal an error to user via sk->err
	  * The checksum is below or beyond the packet.
	  */
      }
   }
}


/* 
 * The old sendto() and sendmsg() are now merged in only one function.
 * That code is largely inspired by Pedro Roque's one.
 */
 
static int raw6_sendmsg(struct sock *sk, struct msghdr *msg, int len)
{
   struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *) msg->msg_name;
   struct raw6_tpinfo  *raw6_opt;
   struct in6_addr     *daddr, *saddr = NULL;
   struct cmsghdr      *cmsgptr;
   unsigned int        flags   = msg->msg_flags;
   unsigned int        noblock = msg->msg_flags & MSG_DONTWAIT;
   int tlport;			/* transport layer port */
   int err;
   int clen = msg->msg_controllen;

   /*
    *    Check a lot of things before processing the packet
    */

   if (flags & MSG_OOB)         /* Mirror BSD error message compatibility */
      return -EOPNOTSUPP;

   if (flags & ~(MSG_DONTROUTE|MSG_DONTWAIT)) /* Only MSG_DONTROUTE and MSG_DONTWAIT are permitted */
      return (-EINVAL);


   /*
    *    Get and verify the address. 
    */

   if (sin6) {
      /* Is it the right place to test that (maybe inet6_sendmsg) ? */
      if (msg->msg_namelen != sizeof(struct sockaddr_in6))
	  return (-EINVAL);

      if (sin6->sin6_family && sin6->sin6_family != AF_INET6)
	 return (-EINVAL);

      tlport = sin6->sin6_port;

      daddr = &sin6->sin6_addr;
   } else {
      if (sk->state != TCP_ESTABLISHED)
	 return (-EINVAL);

      /* No destination adress and no port were specified */
      daddr = &sk->net_pinfo.af_inet6.pi6_fl.fll_da;
      tlport = sk->num;
   }

   if (IS_ANYADDR6(*daddr))
      return (-EINVAL);		/* dunno where to send */

   /* FIXME: we don't carry Jumbo Payload yet */
   if (sk->ip_hdrincl) {
      if (len > 65535)
	 return (-EMSGSIZE);
      else if (len > (65535 - sizeof(struct ipv6hdr)))
	  return (-EMSGSIZE);
   }
   /* 
    * Any ancillary data ? 
    *
    * (bb) Should this be made more generic ?... and located at
    * the af_inet6 level for instance...
    * Same code must be written for UDP, TCP...
    * As of Nov 20, 96:
    *    source address is handled.
    */
   if (clen > 0) {
      for (cmsgptr = CMSG_FIRSTHDR(msg); (cmsgptr != NULL) && (clen > 0);
	   cmsgptr = CMSG_NXTHDR(msg, cmsgptr)) {
	 /* Source addr. info */
	 if (cmsgptr->cmsg_level == IPPROTO_IPV6 &&
	     cmsgptr->cmsg_type == IPV6_PKTINFO) {
	    saddr = &(((struct in6_pktinfo *) cmsgptr->cmsg_data)->ipi6_addr);
	 }
	 clen -= cmsgptr->cmsg_len;
      }
   }
   /* Now, things should be correct, so we can send.
      The packet to send can fit on 1 or more msg_iov.iov_base.
      If it is one, we give the adress, else we build a temporary 
      buffer */

   raw6_opt = &sk->tp_pinfo.raw6;
   if (raw6_opt->checksum) {

      struct raw6_fake_header fkhdr;

      COPY_ADDR6(fkhdr.daddr, *daddr);

      fkhdr.iov = msg->msg_iov;
      fkhdr.sk = sk;
      fkhdr.csum = 0;		/* checksum not computed yet, must be 0 */
      fkhdr.len = len;
      fkhdr.protocol = tlport;

      err = ipv6_build_xmit(sk, raw6_getfragcsum,
			    &fkhdr, len,
			    daddr, saddr, NULL, flags, tlport,
			    noblock);
   } else {
      err = ipv6_build_xmit(sk, raw6_getfrag,
			    msg->msg_iov, len,
			    daddr, saddr, 
			    NULL, flags, tlport,
			    noblock);
   }

   if (CMSG_FIRSTHDR(msg))
      kfree_s((void *) CMSG_FIRSTHDR(msg), msg->msg_controllen);

   if (err < 0)
      return (err);
   else
      return (len);
}

/* 
 * Close a socket. 
 */
 
static void raw6_close(struct sock *sk, unsigned long timeout)
{
   sk->state = TCP_CLOSE;
   destroy_sock(sk);
}

/*
 *    This should be easy, if there is something there
 *      we return it, otherwise we block.
 * Last Update:
 *  08/12/96: bb. IPV6_PKTINFO message control.
 *  See net/core/iovec.c, net/core/datagram.c, *recvmsg calls for details.
 */
int raw6_recvmsg(struct sock *sk, struct msghdr *msg, int len, int noblock,
		 int flags, int *addr_len)
{
   int copied = 0;
   struct sk_buff *skb;
   int err;
   int offset = 0;
   struct ipv6_pinfo *np = &sk->net_pinfo.af_inet6;


   struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *) msg->msg_name;

   if (flags & MSG_OOB)
      return -EOPNOTSUPP;

   if (sk->shutdown & RCV_SHUTDOWN)
      return (0);

   if (addr_len)
      *addr_len = sizeof(*sin6);

   if ((skb = skb_recv_datagram(sk, flags, noblock, &err)) == NULL)
      return err;

   copied = min(len, skb->len);

   /*if (sk->ip_hdrincl >= 2)
      offset = - sizeof(struct ipv6hdr); */

   
   /*
    * PKTINFO socket option
    */    
   if (np->pktinfo) {
      struct {
	 struct cmsghdr cmsg;                 /* Control msg header */
	 struct in6_pktinfo pktinfo;          /* Packet information */
      } message;

      if (msg->msg_controllen < sizeof(message))
	 return -ENOMEM;

      /* Control msg header */
      message.cmsg.cmsg_level = IPPROTO_IPV6;
      message.cmsg.cmsg_type = IPV6_PKTINFO;
      message.cmsg.cmsg_len = sizeof(message);
      
      /* Packet information */
      message.pktinfo.ipi6_ifindex = skb->dev->ifindex;
      COPY_ADDR6(message.pktinfo.ipi6_addr, skb->nh.ipv6h->saddr);

      /* Copy */
      err = copy_to_user(msg->msg_control,
			 (char *) &message, sizeof(message));
      if (err)
	 return -EFAULT;
      
      /* Update msg_control pointer (See net/socket.h) */
      msg->msg_control += sizeof(message);
   }
   skb_copy_datagram_iovec(skb, offset, msg->msg_iov, copied);
   sk->stamp = skb->stamp;

   /* Copy the address. */
   if (sin6) {
      sin6->sin6_family = AF_INET6;
      COPY_ADDR6(sin6->sin6_addr, skb->nh.ipv6h->daddr);
   }
   skb_free_datagram(sk, skb);
   return (copied);
}

int raw6_init_sk(struct sock *sk)
{
   struct raw6_tpinfo *raw6_opt;

   /* 
    * By default, packets will have a checksum at offset 2 
    */
   raw6_opt = &sk->tp_pinfo.raw6;
   raw6_opt->checksum = 1;
   raw6_opt->ck_offset = 2;

   if (sk->protocol == IPPROTO_RAW)
      sk->ip_hdrincl = 1;

   return (0);
}

struct proto raw6_prot =
{
   raw6_close,
   NULL,
   NULL,
   NULL,
   NULL,
   NULL,
   datagram_poll,
   NULL,
   raw6_init_sk,
   NULL,
   NULL,
   ipv6_setsockopt,
   ipv6_getsockopt,
   raw6_sendmsg,
   raw6_recvmsg,
   NULL,			/* No special bind */
   raw6_rcv_skb,
   128,
   0,
   "RAW6",
   0, 0,
   raw6_sock_array,
};

/* Initialisation of the RAW6 protocol */
void raw6_init(struct net_proto_family *ops)
{
   int i;

   /* Fill its sockarray */
   for (i = 0; i < SOCK_ARRAY_SIZE; i++)
      raw6_prot.sock_array[i] = NULL;

#ifdef CONFIG_PROC_FS
   /* Register in /proc/net/raw6 */
   proc_net6_register(&(struct proc_dir_entry) {
		      PROC_NET6_RAW6, 4, "raw6",
		      S_IFREG | S_IRUGO, 1, 0, 0,
		      0, &proc_net6_inode_operations,
		      raw6_get_info
		      });
#endif
}
