/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$ 
 *      
 * Authors:     
 *              Frank Zago      <Zago@masi.ibp.fr>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *              From protocol.c written by 
 *              Ross Biro, <bir7@leland.Stanford.Edu>
 *              Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|  Function_name
|
-------------------------------------------------------------------------*/



#include <asm/segment.h>
#include <asm/system.h>

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/timer.h>
#include <linux/skbuff.h>


#include <net/sock.h>

#include <linux/dret/ipv6.h>
#include <net/dret/protocol6.h>
#include <net/dret/raw6.h>
#include <net/dret/udp6.h>
#include <net/dret/icmpv6.h>
#include <net/dret/tcp6.h>
#include <net/dret/options6.h>

static struct inet6_protocol raw6_protocol =
{
   NULL,			/* RAW handler          */
   NULL,			/* RAW error control    */
   raw6_init,			/* RAW init function    */
   NULL,			/* next                 */
   IPPROTO_RAW,			/* protocol ID          */
   0,				/* copy                 */
   NULL,			/* data                 */
   "RAW6"			/* name                 */
};

static struct inet6_protocol icmp6_protocol =
{
   icmpv6_rcv,			/* RAW handler          */
   NULL,			/* RAW error control    */
   icmpv6_init,			/* RAW init function    */
   &raw6_protocol,		/* next                 */
   IPPROTO_ICMPV6,		/* protocol ID          */
   0,				/* copy                 */
   NULL,			/* data                 */
   "ICMPV6"			/* name                 */
};

static struct inet6_protocol udp6_protocol =
{
   udp6_rcv,			/* UDP handler          */
   udp6_err,			/* UDP error control    */
   udp6_init,			/* UDP init function    */
   &icmp6_protocol,		/* next                 */
   IPPROTO_UDP,			/* protocol ID          */
   0,				/* copy                 */
   NULL,			/* data                 */
   "UDP6"			/* name                 */
};

static struct inet6_protocol tcp6_protocol = 
{
	tcp6_rcv,		/* TCP handler		*/
	tcp6_err,		/* TCP error control	*/
	tcp6_init,		/* TCP init function    */
	&udp6_protocol,			/* next			*/
	IPPROTO_TCP,		/* protocol ID		*/
	0,			/* copy			*/
	NULL,			/* data			*/
	"TCP6"			/* name			*/
};

/* Hop-by-hop extension header - IPPROTO_HOPOPTS - 0 */
static struct inet6_protocol hbh_protocol =
{
   hopopts_handler,			/* hop-by-hop handler          */
   NULL,			/* hop-by-hop error control    */
   NULL /*hbh_init*/,			/* hop-by-hop init function    */
   &tcp6_protocol,			/* next                 */
   IPPROTO_HOPOPTS,			/* protocol ID          */
   0,				/* copy                 */
   NULL,			/* data                 */
   "HOPOPTS"			/* name                 */
};



/*  Routing extension header - IPPROTO_ROUTING -  43 */
static struct inet6_protocol routing_protocol =
{
   srcrt_handler,			/*  routing handler          */
   NULL,			/*  routing error control    */
   NULL /*routing_init*/,		/*  routing init function    */
   &hbh_protocol,			/* next                 */
   IPPROTO_ROUTING,		/* protocol ID          */
   0,				/* copy                 */
   NULL,			/* data                 */
   "ROUTING"			/* name                 */
};

/*  Fragmentation extension header - IPPROTO_FRAGMENT - 44 */
static struct inet6_protocol frag_protocol =
{
   fragm_handler,		/* frag handler          */
   NULL,			/* frag error control    */
   NULL /*frag_init*/,			/* frag init function    */
   &routing_protocol,			/* next                 */
   IPPROTO_FRAGMENT,		/* protocol ID          */
   0,				/* copy                 */
   NULL,			/* data                 */
   "FRAGMENT"			/* name                 */
};

/*  Encryption security payload extension header - IPPROTO_ESP - 50 */
static struct inet6_protocol esp_protocol =
{
   esp_handler,			/* esp handler          */
   NULL,			/* esp error control    */
   NULL /*esp_init*/,			/* esp init function    */
   &frag_protocol,			/* next                 */
   IPPROTO_ESP,		        /* protocol ID          */
   0,				/* copy                 */
   NULL,			/* data                 */
   "ESP"			/* name                 */
};

/*  Authentication extension header - IPPROTO_AH - 51 */
static struct inet6_protocol ah_protocol =
{
   ah_handler,			/*  ah handler          */
   NULL,			/*  ah error control    */
   NULL /*ah_init*/,			/*  ah init function    */
   &esp_protocol,			/* next                 */
   IPPROTO_AH,		        /* protocol ID          */
   0,				/* copy                 */
   NULL,			/* data                 */
   "AH"			        /* name                 */
};

/*  None extension header - IPPROTO_NONE - 59 */
static struct inet6_protocol none_protocol =
{
   NULL,			/*  none handler          */
   NULL,			/*  none error control    */
   NULL /*none_init*/,			/*  none init function    */
   &ah_protocol,			/* next                 */
   IPPROTO_NONE,		/* protocol ID          */
   0,				/* copy                 */
   NULL,			/* data                 */
   "NONE"			/* name                 */
};

/*  Destination extension header - IPPROTO_DSTOPTS - 60 */
static struct inet6_protocol dst_protocol =
{
   dstopts_handler,			/* destination handler          */
   NULL,			/* destination error control    */
   NULL /*dst_init*/,			/* destination init function    */
   &none_protocol,			/* next                 */
   IPPROTO_DSTOPTS,		/* protocol ID          */
   0,				/* copy                 */
   NULL,			/* data                 */
   "DSTOPTS"			/* name                 */
};

struct inet6_protocol *inet6_protocol_base = &none_protocol;


struct inet6_protocol *inet6_protos[MAX_INET_PROTOS] =
{
   NULL
};


/*
 *    Find a protocol in the protocol tables given its
 *      IPv6 type (usually equal to IP ones).
 */

struct inet6_protocol *inet6_get_protocol(unsigned char prot)
{
   unsigned char hash;
   struct inet6_protocol *p;

   hash = prot & (MAX_INET_PROTOS - 1);
   for (p = inet6_protos[hash]; p != NULL; p = p->next) {
      if (p->protocol == prot)
	 return ((struct inet6_protocol *) p);
   }
   return (NULL);
}

/*
 *    Add a protocol handler to the hash tables
 */

void inet6_add_protocol(struct inet6_protocol *prot)
{
   unsigned char hash;
   struct inet6_protocol *p2;

   hash = prot->protocol & (MAX_INET_PROTOS - 1);
   prot->next = inet6_protos[hash];
   inet6_protos[hash] = prot;
   prot->copy = 0;

   /*
    *    Set the copy bit if we need to. 
    */

   p2 = (struct inet6_protocol *) prot->next;
   while (p2 != NULL) {
      if (p2->protocol == prot->protocol) {
	 prot->copy = 1;
	 break;
      }
      p2 = (struct inet6_protocol *) p2->next;
   }
}

/*
 *    Remove a protocol from the hash tables.
 */

int inet6_del_protocol(struct inet6_protocol *prot)
{
   struct inet6_protocol *p;
   struct inet6_protocol *lp = NULL;
   unsigned char hash;

   hash = prot->protocol & (MAX_INET_PROTOS - 1);
   if (prot == inet6_protos[hash]) {
      inet6_protos[hash] = (struct inet6_protocol *) inet6_protos[hash]->next;
      return (0);
   }
   p = (struct inet6_protocol *) inet6_protos[hash];
   while (p != NULL) {
      /*
       * We have to worry if the protocol being deleted is
       * the last one on the list, then we may need to reset
       * someone's copied bit.
       */
      if (p->next != NULL && p->next == prot) {
	 /*
	  * if we are the last one with this protocol and
	  * there is a previous one, reset its copy bit.
	  */
	 if (p->copy == 0 && lp != NULL)
	    lp->copy = 0;
	 p->next = prot->next;
	 return (0);
      }
      if (p->next != NULL && p->next->protocol == prot->protocol)
	 lp = p;

      p = (struct inet6_protocol *) p->next;
   }
   return (-1);
}












