/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$ 
 *      
 * Authors:     
 *              Benoit Brodard  <Brodard@sophia.inria.fr>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *              From proc.c written by
 *              Fred N. van Kempen, <waltje@uWalt.NL.Mugnet.ORG>
 *              Gerald J. Heim, <heim@peanuts.informatik.uni-tuebingen.de>
 *              Fred Baumgarten, <dc6iq@insu1.etec.uni-karlsruhe.de>
 *              Erik Schoenfelder, <schoenfr@ibr.cs.tu-bs.de>
 *

 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|  Function_name
|
-------------------------------------------------------------------------*/

#include <asm/system.h>
#include <linux/sched.h>
#include <linux/socket.h>
#include <linux/net.h>
#include <linux/un.h>
#include <linux/in.h>
#include <linux/param.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/udp.h>

#include <net/sock.h>
#include <net/snmp.h>

#include <linux/dret/flow_table.h>
#include <linux/dret/ipv6_debug.h>

#include <net/dret/route6.h>
#include <net/dret/nd.h>
#include <net/dret/udp6.h>
#include <net/dret/proc6.h>
#include <net/dret/ipv6.h>
#include <net/dret/raw6.h>
#include <net/dret/tcp6.h>
#include <net/dret/protocol6.h>

#ifdef IPV6_DEBUG_PROC
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[PROC6]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#else
#define DEBUG(format, args...)
#define PRINT_ADDR(a)
#endif
/*#define IPV6_DEBUG_PROC */
/*#define IPV6_DEBUG_PROC_ND */
/*#define IPV6_DEBUG_PROC_RT */
/*
 * Get__netinfo returns the length of that string.
 *
 * KNOWN BUGS
 *  As in get_unix_netinfo, the buffer might be too small. If this
 *  happens, get__netinfo returns only part of the available infos.
 */
static int 
get__netinfo6(struct proto *pro, char *buffer, int format, char **start, off_t offset, int length)
{
   struct sock **s_array;
   struct sock *sp;
   int i;
   int timer_active;
   int timer_active1;
   int timer_active2;
   unsigned long timer_expires;
   struct in6_addr dest, src;
   unsigned short destp, srcp;
   int len = 0;
   off_t pos = 0;
   off_t begin = 0;
   struct tcp_opt *tp;
   
   s_array = pro->sock_array;
   len += sprintf(buffer, " sl local address                         "
                              "remote address                        "
                              "state   tx_queue "
		  "rx_queue timer tm->when uid inode\n");
/*
 *    This was very pretty but didn't work when a socket is destroyed
 *      at the wrong moment (eg a syn recv socket getting a reset), or
 *      a memory timer destroy. Instead of playing with timers we just
 *      concede defeat and cli().
 */
   for (i = 0; i < SOCK_ARRAY_SIZE; i++) {
      cli();
      sp = s_array[i];
      while (sp != NULL) {
	 tp = &(sp->tp_pinfo.af_tcp);
	 COPY_ADDR6(dest, sp->net_pinfo.af_inet6.pi6_fl.fll_da);
	 COPY_ADDR6(src, sp->net_pinfo.af_inet6.pi6_fl.fll_sa);
	 destp = sp->dummy_th.dest;
	 srcp = sp->dummy_th.source;

	 /* Since we are Little Endian we need to swap the bytes :-( */
	 destp = ntohs(destp);
	 srcp = ntohs(srcp);
	 timer_active1 = del_timer(&sp->retransmit_timer);
	 timer_active2 = del_timer(&sp->timer);
	 if (!timer_active1)
	    sp->retransmit_timer.expires = 0;
	 if (!timer_active2)
	    sp->timer.expires = 0;
	 timer_active = 0;
	 timer_expires = (unsigned) -1;
	 if (timer_active1 &&
	     sp->retransmit_timer.expires < timer_expires) {
	    timer_active = timer_active1;
	    timer_expires = sp->retransmit_timer.expires;
	 }
	 if (timer_active2 &&
	     sp->timer.expires < timer_expires) {
	    timer_active = timer_active2;
	    timer_expires = sp->timer.expires;
	 }

	 len += sprintf(buffer + len, "%2d:"
			"%08lX:%08lX:%08lX:%08lX|%04X "
			"%08lX:%08lX:%08lX:%08lX|%04X"
			" %02X %08X:%08X %02X:%08lX %08X %d %d %ld\n",
			i,
			(unsigned long) ntohl(src.s6_addr32[0]),
			(unsigned long) ntohl(src.s6_addr32[1]),
			(unsigned long) ntohl(src.s6_addr32[2]),
			(unsigned long) ntohl(src.s6_addr32[3]), ntohs(srcp),
			(unsigned long) ntohl(dest.s6_addr32[0]),
			(unsigned long) ntohl(dest.s6_addr32[1]),
			(unsigned long) ntohl(dest.s6_addr32[2]),
			(unsigned long) ntohl(dest.s6_addr32[3]), ntohs(destp),
			sp->state,
	                format == 0 ? sp->write_seq - tp->snd_una : sp->wmem_alloc,
	                format == 0 ? tp->rcv_nxt - sp->copied_seq : sp->rmem_alloc,
			timer_active, 
			timer_expires - jiffies, (unsigned) sp->retransmits,
			(sp->socket && SOCK_INODE(sp->socket)) ? SOCK_INODE(sp->socket)->i_uid : 0,
			timer_active ? sp->timeout : 0,
			sp->socket && SOCK_INODE(sp->socket) ? SOCK_INODE(sp->socket)->i_ino : 0);
			
	 if (timer_active1)
	    add_timer(&sp->retransmit_timer);
	 if (timer_active2)
	    add_timer(&sp->timer);
	 /*
	  * All sockets with (port mod SOCK_ARRAY_SIZE) = i
	  * are kept in sock_array[i], so we must follow the
	  * 'next' link to get them all.
	  */
	 sp = sp->next;
	 pos = begin + len;
	 if (pos < offset) {
	    len = 0;
	    begin = pos;
	 }
	 if (pos > offset + length)
	    break;
      }
      sti();			/* We only turn interrupts back on for a moment,
				   but because the interrupt queues anything built
				   up before this will clear before we jump back
				   and cli(), so it's not as bad as it looks */
      if (pos > offset + length)
	 break;
   }
   *start = buffer + (offset - begin);
   len -= (offset - begin);
   if (len > length)
      len = length;
   return len;
}


int raw6_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{
   return get__netinfo6(&raw6_prot, buffer, 1, start, offset, length);
}

int udp6_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{
   return get__netinfo6(&udp6_prot, buffer, 1, start, offset, length);
}

int tcp6_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{
   return get__netinfo6(&tcp6_prot, buffer, 1, start, offset, length);
}

/*
 *    Report socket allocation statistics [mea@utu.fi]
 */
int afinet6_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{
   /* From  net/socket.c  */
   extern int socket_get_info(char *, char **, off_t, int);
/*      extern struct proto packet_prot; */

   int len = socket_get_info(buffer, start, offset, length);

   len += sprintf(buffer + len, "SOCK_ARRAY_SIZE=%d\n", SOCK_ARRAY_SIZE);

/*      
   len += sprintf(buffer+len,"TCP: inuse %d highest %d\n",
   tcp_prot.inuse, tcp_prot.highestinuse); */

   len += sprintf(buffer + len, "UDP: inuse %d highest %d\n",
		  udp6_prot.inuse, udp6_prot.highestinuse);
   len += sprintf(buffer + len, "RAW: inuse %d highest %d\n",
		  raw6_prot.inuse, raw6_prot.highestinuse);
/*      
   len += sprintf(buffer+len,"PAC: inuse %d highest %d\n",
   packet_prot.inuse, packet_prot.highestinuse);
 */
   *start = buffer + offset;
   len -= offset;
   if (len > length)
      len = length;
   return len;
}


/* 
 *    Called from the PROCfs module. This outputs /proc/net/snmp.
 */

int snmp6_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{
   int len;

   len = sprintf(buffer,
		 "IPv6: Forwarding DefaultTTL InReceives InHdrErrors InAddrErrors ForwDatagrams InUnknownProtos InDiscards InDelivers OutRequests OutDiscards OutNoRoutes ReasmTimeout ReasmReqds ReasmOKs ReasmFails FragOKs FragFails FragCreates\n"
	   "Ip: %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu\n",
	   ip6_statistics.IpForwarding, ip6_statistics.IpDefaultTTL,
	   ip6_statistics.IpInReceives, ip6_statistics.IpInHdrErrors,
	   ip6_statistics.IpInAddrErrors, ip6_statistics.IpForwDatagrams,
	   ip6_statistics.IpInUnknownProtos, ip6_statistics.IpInDiscards,
	   ip6_statistics.IpInDelivers, ip6_statistics.IpOutRequests,
	   ip6_statistics.IpOutDiscards, ip6_statistics.IpOutNoRoutes,
	   ip6_statistics.IpReasmTimeout, ip6_statistics.IpReasmReqds,
	   ip6_statistics.IpReasmOKs, ip6_statistics.IpReasmFails,
	   ip6_statistics.IpFragOKs, ip6_statistics.IpFragFails,
	   ip6_statistics.IpFragCreates);

   if (offset >= len) {
      *start = buffer;
      return 0;
   }
   *start = buffer + offset;
   len -= offset;
   if (len > length)
      len = length;
   return len;
}

/*--START FUNCTION--(pr_route6)--------------------------------------------
  
  print struct leaf to buffer
  called from route6_get_info & route6_get_info_1
  
  Last update:
  (bb)  10/24/96
  ---------------------------------------------------------------------------*/
static void pr_route6(struct rt_dest *leaf, char *buffer, int *len)
{
   char temp[129];


   if ((leaf->rt_flags & RTF_GATEWAY) && (leaf->rt_neigh)) {
      sprintf(temp,
	      "%s\t%08lX:%08lX:%08lX:%08lX\t%08lX:%08lX:%08lX:%08lX\t%04X\t%lu\t%lu\t%d\t%d\t%d\t%lu\t%u",
	      leaf->rt_dev->name,
	     (unsigned long) ntohl(leaf->rt_addr.s6_addr32[0]),
	      (unsigned long) ntohl(leaf->rt_addr.s6_addr32[1]),
	      (unsigned long) ntohl(leaf->rt_addr.s6_addr32[2]),
	      (unsigned long) ntohl(leaf->rt_addr.s6_addr32[3]),
	      (unsigned long) ntohl(leaf->rt_neigh->n_addr.s6_addr32[0]),
	      (unsigned long) ntohl(leaf->rt_neigh->n_addr.s6_addr32[1]),
	      (unsigned long) ntohl(leaf->rt_neigh->n_addr.s6_addr32[2]),
	      (unsigned long) ntohl(leaf->rt_neigh->n_addr.s6_addr32[3]),
	      (unsigned int) leaf->rt_flags,
	      (unsigned long) leaf->rt_ref,
	      (unsigned long) leaf->rt_use,
	      (int) leaf->rt_metric,
	      (int) leaf->rt_prefixlen,
	      (int) leaf->rt_pmtu, (unsigned long) 0, (unsigned int) 0);
   } else {
      sprintf(temp,
	      "%s\t%08lX:%08lX:%08lX:%08lX\t%08lX:%08lX:%08lX:%08lX\t%04X\t%d\t%lu\t%d\t%d\t%d\t%lu\t%u",
	      leaf->rt_dev->name,
	      (unsigned long) ntohl(leaf->rt_addr.s6_addr32[0]),
	      (unsigned long) ntohl(leaf->rt_addr.s6_addr32[1]),
	      (unsigned long) ntohl(leaf->rt_addr.s6_addr32[2]),
	      (unsigned long) ntohl(leaf->rt_addr.s6_addr32[3]),
	      (unsigned long) 0,
	      (unsigned long) 0,
	      (unsigned long) 0,
	      (unsigned long) 0,
	      (unsigned int) leaf->rt_flags,
	      (int) leaf->rt_ref,
	      (unsigned long) leaf->rt_use,
	      (int) leaf->rt_metric,
	      (int) leaf->rt_prefixlen,
	      (int) leaf->rt_pmtu, 
	      (unsigned long) 0, 
	      (unsigned int) 0);
   }
   sprintf(buffer + (*len), "%-127s\n", temp);
   *len += 128;
}

/*--END FUNCTION--(pr_route6)-----------------------------------------------*/


/*--START FUNCTION--(route6_get_info_1)--------------------------------------
  
  Called from route6_get_info. Same input parameters
  
  Last update:
  (bb): 10/25/96
  
  --------------------------------------------------------------------------*/
static int route6_get_info_1(char *buffer, char **start, off_t offset,
			     int length, int dummy, struct table_node *node,
			     int *len, off_t * pos)
{
   struct table_node *n;
   struct table_leaf *leaf;

   DEBUG("[PROC6]route6_get_info_1: IN\n");

   if (*pos >= offset + length) {
      DEBUG("[PROC6]route6_get_info_1: OUT1\n");
      return (*len);
   }
   /* Depth first search */
   if ((n = node->t_left) != NULL) {
      route6_get_info_1(buffer, start, offset, length, dummy, n, len, pos);
   }
   if ((n = node->t_right) != NULL) {
      route6_get_info_1(buffer, start, offset, length, dummy, n, len, pos);
   }
   if ((!node->t_leaf) || (node == &routing_root)) {
      DEBUG("[PROC6]route6_get_info_1: OUT4\n");
      return (*len);
   }
   if ((node->t_flags & RTN_BACKTRACK) != 0) {
      DEBUG("[PROC6]route6_get_info_1: print node->rn_bit= %d\n",
	     node->rn_bit);
      for (leaf = node->t_leaf; leaf; leaf = leaf->tl_next) {
	 *pos += 128;
	 if (*pos <= offset) {
	    *len = 0;
	    continue;
	 }
	 pr_route6((struct rt_dest *)(leaf->tl_data), buffer, len);
      }
   }
   DEBUG("[PROC6]route6_get_info_1: OUT5\n");
   return (*len);
}


/*--START FUNCTION--(route6_get_info)-------------------------------------
 |	Called from the PROCfs module. This outputs /proc/net/route.
 |
 |	We preserve the old format but pad the buffers out. This means that
 |	we can spin over the other entries as we read them. Remember the
 |	gated BGP4 code could need to read 60,000+ routes on occasion (that's
 |	about 7Mb of data). To do that ok we will need to also cache the
 |	last route we got to (reads will generally be following on from
 |	one another without gaps).
 -------------------------------------------------------------------------- */
int route6_get_info(char *buffer, char **start, off_t offset, int length, int dummy)
{
   struct table_node *node;
   int len = 0;
   off_t pos = 0;

   DEBUG("[PROC6]route6_get_info: IN\n");

   pos = 128;
   if (offset < 128) {
      sprintf(buffer, "%-127s\n", "Iface\tDestination\t\t\t\tGateway\t\t\t\t\t \tFlags\tRefCnt\tUse\tMetric\tPref\tMTU\tWindow\tIRTT");
      len = 128;
   }
   pos += 128;

   while (routing_table.gt_lock)
      sleep_on(&(routing_table.gt_wait));
   table_fast_lock(&routing_table);

   if (routing_table.gt_default) {
      DEBUG("[PROC6]route6_get_info: Printing default route");
      pr_route6((routing_table.gt_default)->tl_data, buffer, &len);
   } else {
      len += sprintf(buffer + len, "%-127s\n", "No default route registered.");
   }
   node = routing_table.gt_root;
   route6_get_info_1(buffer, start, offset, length, dummy, node, &len, &pos);
   if (pos >= offset + length)
      goto done;

 done:
   table_unlock(&routing_table);
   wake_up(&(routing_table.gt_wait));

   *start = buffer + len - (pos - offset);
   len = pos - offset;
   if (len > length)
      len = length;
   DEBUG("[PROC6]route6_get_info: OUT\n");
   return len;
}
/*--END FUNCTION---------------------------------------------------------*/


int igmpv6_procinfo(char *buffer, char **start, off_t offset,
		    int length, int dummy)
{
   off_t pos = 0, begin = 0;
   struct ipv6_mc_list *mc;
   unsigned long flags;
   int len = 0;
   struct device *dev;
   struct inet6_dev *idev;

#ifdef IPV6_DEBUG_PROC
   printk(KERN_DEBUG "[PROC6]igmpv6_procinfo: IN\n");
#endif

   len = sprintf(buffer, "Device    : Count\tGroup\t\t\tUsers Timer\n");
   save_flags(flags);
   cli();

   for (dev = dev_base; dev; dev = dev->next) {
#ifdef IPV6_DEBUG_PROC
      printk("%s \n", dev->name);
#endif
      if ((dev->flags & IFF_UP) && (dev->flags & IFF_MULTICAST) && (dev->dev6_lst)) {
	 idev = dev->dev6_lst;
	 len += sprintf(buffer + len, "%-10s: %5d\n",
			dev->name, dev->mc_count);
#ifdef IPV6_DEBUG_PROC
	 if (idev->mc_list == NULL)
	    printk("No multicast address for this device...\n");
#endif

	 for (mc = idev->mc_list; mc; mc = mc->if_next) {
	    len += sprintf(buffer + len,
			   "\t\t\t%08lX:%08lX:%08lX:%08lX %5d %d:%08lX\n",
			   (unsigned long) ntohl(mc->addr.s6_addr32[0]),
			   (unsigned long) ntohl(mc->addr.s6_addr32[1]),
			   (unsigned long) ntohl(mc->addr.s6_addr32[2]),
			   (unsigned long) ntohl(mc->addr.s6_addr32[3]),
			   mc->users,
			   mc->tm_running,
			   mc->timer.expires - jiffies);
	    pos = begin + len;
	    if (pos < offset) {
	       len = 0;
	       begin = pos;
	    }
	    if (pos > offset + length)
	       break;
	 }
      }
   }
   restore_flags(flags);
   *start = buffer + (offset - begin);
   len -= (offset - begin);
   if (len > length)
      len = length;
   DEBUG("[PROC6]igmpv6_procinfo: OUT\n");
   return len;
}

static int ndcache_procinfo_1(char *buffer, char **start, off_t offset,
			    int length, int dummy, struct table_node *node,
			      int *len, int *begin, off_t * pos)
{
   struct table_node *n;
   struct table_leaf *leaf;
   struct neighbor *leaf_data;
   
   char cstate[12], hbuff[MAX_HADDR_LEN * 3], *ptr;
   int i, j, l;

   DEBUG("[PROC6]ndcache_procinfo_1: IN\n");

   if ((n = node->t_left) != NULL) {
      ndcache_procinfo_1(buffer, start, offset, length, dummy, n,
			 len, begin, pos);
   }
   if ((n = node->t_right) != NULL) {
      ndcache_procinfo_1(buffer, start, offset, length, dummy, n,
			 len, begin, pos);
   }
   if ((!node->t_leaf) || (node == nd_table.gt_root)) {
      DEBUG("[PROC6]ndcache_procinfo_1: OUT4\n");
      return (*len);
   }
   if ((node->t_flags & RTN_BACKTRACK) != 0) {
      for (leaf = node->t_leaf; leaf; leaf = leaf->tl_next) {
	 leaf_data = (struct neighbor *)(leaf->tl_data);
	 *len += sprintf(buffer + *len,
			 "%s\t%08lX:%08lX:%08lX:%08lX",
			 leaf_data->n_dev->name,
			 (unsigned long) ntohl(leaf_data->n_addr.s6_addr32[0]),
			 (unsigned long) ntohl(leaf_data->n_addr.s6_addr32[1]),
			 (unsigned long) ntohl(leaf_data->n_addr.s6_addr32[2]),
			 (unsigned long) ntohl(leaf_data->n_addr.s6_addr32[3]));

	 switch (leaf_data->n_state) {
	 case 0x11:
	    strcpy(cstate, "INCOMPLETE");
	    break;
	 case 0x02:
	    strcpy(cstate, "REACHABLE");
	    break;
	 case 0x04:
	    strcpy(cstate, "STALE");
	    break;
	 case 0x09:
	    strcpy(cstate, "DELAY");
	    break;
	 case 0x21:
	    strcpy(cstate, "PROBE");
	    break;
	 default:
	    strcpy(cstate, "*");
	 }

	 if ((leaf_data->n_flags & NCF_NOARP) == 0) {
	    ptr = leaf_data->n_hh_data;
	    l = 0;
	    for (i = 0, j = 0; i < (MAX_HADDR_LEN * 3) && j < leaf_data->n_dev->addr_len; j++) {
	       l = sprintf(&hbuff[i], "%02hx:", (unsigned char) ptr[j]);
	       i += l;
	    }
	    hbuff[--i] = 0;
	 } else {
	    hbuff[0] = '*';
	    hbuff[1] = 0;
	 }
	 *len += sprintf(buffer + *len, "\t%s\t%s\t%2X\n",
			 hbuff, cstate, leaf_data->n_flags);
	 *pos = *begin + *len;
	 if (*pos < offset) {
	    *len = 0;
	    *begin = *pos;
	 }
	 if (*pos > offset + length) {
	    DEBUG("[PROC6] *pos > offset + length\n");
	    break;
	 }
      }
   } else {
      DEBUG("[PROC6] node->rn_flags & RTN_BACKTRACK == 0\n");
   }
   DEBUG("[PROC6]route6_get_info_1: OUT5\n");
   return (*len);
}


int ndcache_procinfo(char *buffer, char **start, off_t offset, int length, int dummy)
{
   struct table_node *node = nd_table.gt_root;
   int len = 0, begin = 0;
   off_t pos = 0;

#ifdef IPV6_DEBUG_PROC_ND
   printk(KERN_DEBUG "[PROC6]ndcache_procinfo: IN\n");
#endif

   pos += 128;
   len = sprintf(buffer, "%-127s\n", "Iface\tNeighbor Addr.\t\t\t\tLLA \t\tState\tFlags\n");

   ndcache_procinfo_1(buffer, start, offset, length, dummy,
		      node, &len, &begin, &pos);

   *start = buffer + (offset - begin);
   len -= (offset - begin);
   if (len > length)
      len = length;
#ifdef IPV6_DEBUG_PROC
   printk(KERN_DEBUG "[PROC6]ndcache_procinfo: OUT\n");
#endif
   return len;
}
/*--END FUNCTION---------------------------------------------------------*/


static int flow_cache_procinfo_1(char *buffer, char **start, off_t offset,
				 int length, int dummy, struct table_node *node,
				 int *len, int *begin, off_t * pos)
{
   struct table_node *n;
   struct table_leaf *leaf;
   struct flow       *leaf_data;
   int ll;

   DEBUG("[PROC6]flow_cache_procinfo_1: IN\n");

   if ((n = node->t_left) != NULL) {
      flow_cache_procinfo_1(buffer, start, offset, length, dummy, n,
			 len, begin, pos);
   }
   if ((n = node->t_right) != NULL) {
      flow_cache_procinfo_1(buffer, start, offset, length, dummy, n,
			 len, begin, pos);
   }
   if ((!node->t_leaf) || (node == flow_table.gt_root)) {
      DEBUG("[PROC6]flow_cache_procinfo_1: OUT4\n");
      return (*len);
   }
   if ((node->t_flags & RTN_BACKTRACK) != 0) {
      for (leaf = node->t_leaf; leaf; leaf = leaf->tl_next) {
	 struct flow_entry *fe;
	 
	 leaf_data = (struct flow *)(leaf->tl_data);
	 for (ll = 0; ll < 16; ll++) {
	    if((fe = leaf_data->fl_fa[ll])) {
	       for(; fe; fe = fe->fe_next) {
	          *len += sprintf(buffer + *len,
			 "%6X\t%08lX:%08lX:%08lX:%08lX",
			 fe->fe_label,
			 (unsigned long) ntohl(leaf_data->fl_src.s6_addr32[0]),
			 (unsigned long) ntohl(leaf_data->fl_src.s6_addr32[1]),
			 (unsigned long) ntohl(leaf_data->fl_src.s6_addr32[2]),
			 (unsigned long) ntohl(leaf_data->fl_src.s6_addr32[3]));
		  *len += sprintf(buffer + *len, "\t%4X\t%d\t%d\t%d\n",
				  fe->fe_opt.fo_flags, fe->fe_stats.fs_count,
				  fe->fe_stats.fs_bytes, fe->fe_stats.fs_jitter); 
		  *pos = *begin + *len;
		  if (*pos < offset) {
		     *len = 0;
		     *begin = *pos;
		  }
		  if (*pos > offset + length) {
		     DEBUG("[PROC6]flow_cache_procinfo_1: *pos > offset + length\n");
		     break;
		  }
	       }
	       if (*pos > offset + length) {
		     DEBUG("[PROC6]flow_cache_procinfo_1: *pos > offset + length\n");
		     break;
	       }
	    }
	 }	       
      }
   } else {
      DEBUG("[PROC6]flow_cache_procinfo_1: node->rn_flags & RTN_BACKTRACK == 0\n");
   }
   DEBUG("[PROC6]flow_cache_procinfo_1: OUT5\n");
   return (*len);
}


int flow_cache_procinfo(char *buffer, char **start, off_t offset, int length, int dummy)
{
   struct table_node *node = flow_table.gt_root;
   int len = 0, begin = 0;
   off_t pos = 0;

#ifdef IPV6_DEBUG_PROC_ND
   printk(KERN_DEBUG "[PROC6]flow_cache_procinfo: IN\n");
#endif

   pos += 128;
   len = sprintf(buffer, "%-127s\n", "Flow\tSource address\t\t\t\tOptions\tCount\tBytes\tJitter\n");

   flow_cache_procinfo_1(buffer, start, offset, length, dummy,
		      node, &len, &begin, &pos);

   *start = buffer + (offset - begin);
   len -= (offset - begin);
   if (len > length)
      len = length;
#ifdef IPV6_DEBUG_PROC
   printk(KERN_DEBUG "[PROC6]flow_cache_procinfo: OUT\n");
#endif
   return len;
}









