/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$ 
 *	
 * Authors:	
 *	        Benoit Brodard	<Brodard@sophia.inria.fr>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/** Contents *************************************************************


   
 ************************************************************************/

#include <asm/uaccess.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/sched.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/netdevice.h>

#include <net/sock.h>
#include <net/snmp.h>
#include <net/tcp.h>
#include <net/udp.h>

#include <net/dret/ipv6.h>
#include <linux/dret/icmpv6.h>
#include <linux/dret/ipv6_debug.h>
#include <net/dret/icmpv6.h>
#include <net/dret/nd.h>
#include <net/protocol.h>
#include <net/dret/route6.h>
#include <net/dret/addrconf.h>
#include <net/dret/sit.h>


#ifdef IPV6_DEBUG_OPTIONS
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[OPTIONS]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#define PRINT_ETH_ADDR(a) printk(KERN_DEBUG "[OPTIONS]%2X:%2X:%2X:%2X:%2X:%2X\n",\
				 a[0], a[1], a[2], a[3], a[4], a[5]);
#else
#define DEBUG(format, args...) 
#define PRINT_ADDR(a)
#define PRINT_ETH_ADDR(a)
#endif

int hopopts_handler(struct sk_buff *skb, __u8 *optr) {
   __u8 olen = (((struct opt6_hdr *)optr)->oh_len +1) << 3;
   __u8 cur_len = 2;
   __u8 *hop_ptr = optr;

   do {
      switch(((struct opt6_hdr *)hop_ptr)->oh_nexthdr) {
      case PAD0_HOPOPT:
	 DEBUG("[OPTIONS]hopopts_handler: PAD0_HOPOPT \n");
	 hop_ptr++;
	 cur_len++;
	 break;
	 
      case PADN_HOPOPT:
	 DEBUG("[OPTIONS]hopopts_handler: PADN_HOPOPT (%d)\n",
	       ((struct opt6_hdr *)hop_ptr)->oh_len + 2);
	 cur_len += ((struct opt6_hdr *)hop_ptr)->oh_len + 2;
	 hop_ptr += ((struct opt6_hdr *)hop_ptr)->oh_len + 2;
	 break;
	 
      case JUMBO_HOPOPT:
	 DEBUG("[OPTIONS]hopopts_handler: JUMBO_HOPOPT not supported\n");
	 break;
      default:
	 DEBUG("[OPTIONS]hopopts_handler: %d not supported yet\n",
	       ((struct opt6_hdr *)optr)->oh_nexthdr);
      }
      cur_len += ((struct opt6_hdr *)hop_ptr)->oh_len;
   } while (cur_len < olen);
   
   /* move to head of next header */
   optr += olen;
   return 0;
}

/*
  Returns -1 in case skb is gone.
  Returns 0 otherwise;
*/
int srcrt_handler(struct sk_buff *skb, __u8 *optr, __u8 flow) {
   __u8 next_hdr;
   __u8 len;
   __u8 addr_num;
   __u8 index;
   __u8 strict_loose;
   
   struct in6_addr *a, swap;
   struct ipv6hdr * ip6hdr = (struct ipv6hdr *)skb->nh.ipv6h;
   struct ipv6_rt_hdr * rthdr = (struct ipv6_rt_hdr *)optr;
   
   DEBUG("[OPTIONS]srcrt_handler: IN ptr = %p\n", optr);
   
   if (rthdr->type) {
      DEBUG("[OPTIONS]srcrt_handler: type = %d not supported \n",
	    rthdr->type);
      return 0;
   }
   
   next_hdr = rthdr->nexthdr;
   len      = rthdr->hdrlen;
   if (rthdr->segments_left == 0) {  /* We finally got there! */     
      optr += 8 + (len << 3);
      skb->ip6_srcrt = (unsigned char *)rthdr;  /* To be passed back to user */
      DEBUG("[OPTIONS]srcrt_handler: OUT @DEST ptr = %p\n", optr);
      return 0;
   }
   
   if ((len > 46)||(len%2)) { /* ! Red lights ! */
      DEBUG("[OPTIONS]srcrt_handler: OUT ICMP6_PARAMPROB len\n");
      icmpv6_send_error(skb, ICMP6_PARAMPROB, 0, 
			(__u32)&rthdr->hdrlen, skb->dev);
      kfree_skb(skb, FREE_READ);
      return -1;
   }
   
   addr_num = len / 2;
   if (rthdr->segments_left > addr_num) { /* ! Red lights ! */
      DEBUG("[OPTIONS]srcrt_handler: OUT ICMP6_PARAMPROB seg_left\n");
      icmpv6_send_error(skb, ICMP6_PARAMPROB, 0, 
			(__u32)&rthdr->segments_left, skb->dev);
      kfree_skb(skb, FREE_READ);
      return -1;
   }
   
   rthdr->segments_left--;
   index = addr_num - rthdr->segments_left;
   /* Retrieve next destination */
   a = (struct in6_addr *)(optr + (index<<4) - 8);
   if (IS_MULTIADDR6(*a) || IS_MULTIADDR6(ip6hdr->daddr)) {
      kfree_skb(skb, FREE_READ);
      return -1;
   }
   /*Swap old and new dest. in header */
   COPY_ADDR6(swap, ip6hdr->daddr);
   COPY_ADDR6(ip6hdr->daddr, *a);
   COPY_ADDR6(*a, swap);
   
   strict_loose = *(__u32 *)(optr + 4) && (htonl(1<<(23-index)));
   
   if (flow) return 0;
   
#ifdef CONFIG_IPV6_FORWARD
   if (ipv6_fwd(skb, skb->dev, 0, &ip6hdr->daddr, strict_loose)) {
      kfree_skb(skb, FREE_READ);
      return -1;
   }
#else
   kfree_skb(skb, FREE_READ);
#endif
   return -1;
}
						   
   
   
   
   


