/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$
 *      
 * Authors:     
 *              Benoit Brodard  <Brodard@sophia.inria.fr>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*
 * Neighbor discovery mechanisms are triggered by three types of events:
 *
 * Namely:
 *     - A ND message is received.
 *           
 *     - A ND message is sent.
 *
 *     - The ND timer expired.
 *
 * These events affect the state of a neighbor cache entry:
 *     -INCOMPLETE, REACHABLE, STALE, DELAY, PROBE.
 *
 * All the functions below handle these events and modify the neighbor cache
 * state and the timer expiration time accordingly.
 */
/*---------- Contents ----------------------------------------------------
|
|  Function_name
|
-------------------------------------------------------------------------*/


#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/sched.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/dret/ipv6.h>
#include <linux/dret/icmpv6.h>
#include <linux/dret/route6.h>
#include <linux/dret/ipv6_debug.h>

#include <net/sock.h>
#include <net/snmp.h>

#include <net/dret/ipv6.h>
#include <net/protocol.h>
#include <net/dret/ipv6_route.h>
#include <net/dret/addrconf.h>
#include <net/dret/nd.h>
#include <net/dret/table.h>
#include <net/dret/route6.h>
#include <linux/dret/netdevice6.h>

#include <net/checksum.h>
#include <linux/proc_fs.h>


#ifdef IPV6_DEBUG_ND
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[ND]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#define PRINT_ETH_ADDR(a) printk(KERN_DEBUG "[ND]%2X:%2X:%2X:%2X:%2X:%2X\n",\
				 a[0], a[1], a[2], a[3], a[4], a[5]);
#else
#define DEBUG(format, args...) 
#define PRINT_ADDR(a)
#define PRINT_ETH_ADDR(a)
#endif

#define NEIGHBOR_PLEN 128

#define min(a,b) (((a)<(b))? a : b)

/*
 * ND table root
 */
struct table_node nd_root = {
NULL, NULL, NULL, {{{0}}}, NULL, 0, RTN_ROOT
};

struct gen_table nd_table = {
   &nd_root, 0, 0
};

extern struct socket icmpv6_socket;

/* nd_timer (insert in table struct ?) */
static struct timer_list nd_timer;

static void nd_event_send_ns(struct device *dev, struct neighbor *neigh,
			     struct in6_addr *s, struct in6_addr *d,
			     struct in6_addr *sol_addr,
			     int inc_opt, int type);

static struct neighbor *nd_new_neigh(struct device *dev,
				     struct in6_addr *addr,
				     __u8 state, __u32 tstamp);

/*static __inline__ void neighbor_release(struct neighbor *n);*/

/*------------------------------------------------
   ND Timer characteristics
   Tunable via sysctl?     6.3.2
--------------------------------------------------*/
int BaseReachableTime = REACHABLE_TIME;
int ReachableTime     = REACHABLE_TIME;
int RetransTimer      = RETRANSTIMER;

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
                       NEIGHBOR DISCOVERY TIMER
  
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/*--START FUNCTION--(nd_add_timer)---------------------------------------
  
  Sets a new expiration time for neigh's timer.

-------------------------------------------------------------------------*/
static __inline__ void nd_add_timer(struct neighbor *n, int timer)
{
   unsigned long now = jiffies;
   unsigned long ctime;

   DEBUG("[ND]nd_add_timer: IN neigh = %p timer = %d addr:\n", n, timer);
   PRINT_ADDR(&n->n_addr);
   
   now = jiffies;
   n->n_expires = now + timer;
   n->n_state  |= ND_TIMED;

   ctime = del_timer(&nd_timer);

   if (ctime) {
      ctime = ((ctime < n->n_expires) ? ctime : n->n_expires);
   } 
   else ctime = n->n_expires;

   nd_timer.expires = ctime;
   add_timer(&nd_timer);
   DEBUG("[ND]nd_add_timer: OUT\n");
}
/*--END FUNCTION--(nd_add_timer)------------------------------------------*/

/*--START FUNCTION--(nd_del_timer)---------------------------------------

-------------------------------------------------------------------------*/
static void nd_del_timer(struct neighbor *n)
{
   struct table_node *fn;
   unsigned long ctime;
   
   DEBUG("[ND]nd_del_timer: IN neigh = %p addr:\n", n);
   PRINT_ADDR(&n->n_addr);

   ctime = del_timer(&nd_timer);
   if (ctime == n->n_expires) {
      ctime = ~0UL;
      fn = &nd_root;
      /* search the entire tree */
      /* why not maintain a list of all expiration dates? */
   }
   if (ctime == (~0UL)) return;
   nd_timer.expires = ctime;
   add_timer(&nd_timer);
   DEBUG("[ND]nd_del_timer: OUT\n");
}
/*--END FUNCTION--(nd_del_timer)-----------------------------------------*/

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
                          MAC ADDRESS RESOLUTION
  
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/*--START FUNCTION--(nd_eth_resolve)---------------------------------------

  Called from net/ethernet/eth.c: eth_rebuild_header.
  This function is used to resolve link layer addresses.
  
  Last update:
  07/26/96: bb
---------------------------------------------------------------------------*/
int nd_eth_resolve(unsigned char *eth_dest, struct device *dev,
		   struct sk_buff *skb)
{
   struct neighbor *n;
   struct sk_buff  *skb1;  
   struct inet6_ifaddr *ifa;
   struct in6_addr msol_addr;
   unsigned long   flags;
   unsigned long   now = jiffies;

   DEBUG("[ND]nd_eth_resolve: IN skb = %p neigh = %p daddr = \n"
	 , skb, skb->nexthop);
   PRINT_ADDR(&skb->nh.ipv6h->daddr);

   n = skb->nexthop;
   DEBUG("[ND]nd_eth_resolve: IN n = %p\n", n);

   if (n == NULL) {
      if IS_MULTIADDR6(skb->nh.ipv6h->daddr) {
	 ipv6_mc_map(&skb->nh.ipv6h->daddr, eth_dest);
	 DEBUG("*** Multicast address *** eth mcast addr = \n");
	 PRINT_ETH_ADDR(eth_dest);
	 return 0;
	 }
      DEBUG("[ND]nd_eth_resolve: OUT nexthop is NULL- Can't resolve! \n");
      dev_kfree_skb(skb, FREE_WRITE);
      return 1;
   }
   /*
    *  Must we keep trying?
    */
   save_flags(flags);
   cli();
   if ((n->n_flags & NCF_INVALID)) {
      n->n_state = ND_INCOMPLETE;
      n->n_flags &= ~NCF_INVALID;
      restore_flags(flags);
      ifa = ipv6_get_saddr(dev, &n->n_addr);
      MULTISOL_ADDR6(msol_addr, n->n_addr);
      nd_event_send_ns(dev, n, &ifa->addr, &msol_addr, &n->n_addr, 1,
		       ND_NEIGHBOR_SOLICITATION);
      nd_add_timer(n, RETRANS_TIMER);
      DEBUG("[ND]nd_eth_resolve: invalid --> incomplete + ns\n");
   } else {
      restore_flags(flags);
   }

   switch (n->n_state) {
   case ND_INCOMPLETE:
      DEBUG("[ND]nd_eth_resolve: *** Incomplete State ***\n");
      if (skb_queue_len(&n->n_queue) >= MAX_ND_QLEN) {
	 skb1 = n->n_queue.prev;
	 skb_unlink(skb1);
	 DEBUG("[ND]nd_eth_resolve: unlink and free skb1 %p\n", skb1);
	 dev_kfree_skb(skb1, FREE_WRITE);
      }
      skb_queue_head(&n->n_queue, skb);
      DEBUG("[ND]nd_eth_resolve: OUT\n");
      return 1;

   case ND_REACHABLE:
      DEBUG("[ND]nd_eth_resolve: *** Reachable State ***\n");
      if (now - n->n_tstamp < ReachableTime)
	 break;

   case ND_STALE:
      DEBUG("[ND]nd_eth_resolve: *** Stale State ***\n");
      n->n_state = ND_DELAY;
      nd_add_timer(n, DELAY_FIRST_PROBE_TIME);
      break;

   case ND_DELAY:
      DEBUG("[ND]nd_eth_resolve: *** Delay State ***\n");
      break;

   case ND_PROBE:
      DEBUG("[ND]nd_eth_resolve: *** Probe State ***\n");
      
   default:
      DEBUG("[ND]nd_eth_resolve: *** Default!!! = %d ***\n", n->n_state);
      break;
   }

   if ((n->n_flags & NCF_INVALID) || n->n_hh_data == NULL) {
      DEBUG("[ND]nd_eth_resolve: OUT invalid neighbor\n");
      dev_kfree_skb(skb, FREE_WRITE);
      return 1;
   }
   
   memcpy(eth_dest, n->n_hh_data, dev->addr_len);
   DEBUG("[ND]nd_eth_resolve: OUT eth_dest : \n");
   PRINT_ETH_ADDR(eth_dest);
   return 0;
}
/*--END FUNCTION--(nd_eth_resolve)----------------------------------------*/


/*--START_FUNCTION--(nd_store_hh_data)-----------------------------------------

   Possibly update the link layer address destination of a cache entry.

---------------------------------------------------------------------------*/
static int nd_store_hh_data(struct device *dev, char *h_dest, char *opt,
			    __u8 olen, __u8 otype) {
   DEBUG("[ND]nd_store_hh_data: IN\n");
   while (*opt != otype && olen) {
      int len;

      len = opt[1] << 3;
      if (len == 0) {
	 DEBUG("[ND]nd_store_hh_data: OUT EINVAL option has 0 len\n");
	 return -EINVAL;
      }
      opt += len;
      olen -= len;
   }
   if (*opt == otype) {
      memcpy(h_dest, opt + 2, dev->addr_len);
      DEBUG("[ND]nd_store_hh_data: OUT\n");
      return 0;
   }
   DEBUG("[ND]nd_store_hh_data: OUT EINVAL \n");
   return -EINVAL;
}
/*--END FUNCTION--(nd_store_hh_data)---------------------------------------*/


/*--START_FUNCTION--(nd_update_hh_data)----------------------------------------

   Possibly update the link layer address destination of a cache entry.

---------------------------------------------------------------------------*/
static int nd_update_hh_data(struct device *dev, char *h_dest, char *opt,
			     __u8 olen, __u8 otype)
{
   int len = 0;

   DEBUG("[ND]nd_update_hh_data: IN\n");

   while ((*opt != otype) && olen) {
      DEBUG("[ND]nd_update_hh_data: Not wanted option: %d \n", opt[0]);
      if (opt[0] > ND_OPT_MAXNUM) {
	 DEBUG("[ND]nd_update_hh_data: option has invalid type.\n");
	 return -EINVAL;
      }
      if ((len = opt[1] << 3) == 0) {
	 DEBUG("[ND]nd_update_hh_data: option has 0 length.\n");
	 return -EINVAL;
      }
      opt += len;
      olen -= len;
   }
   if (*opt == otype) {
      len = opt[1] << 3;
      switch (otype) {
      case ND_OPT_SLLA:
	 DEBUG("[ND]nd_update_hh_data: SLLA\n");
	 if (memcmp(h_dest, opt + 2, len) == 0) {
	    DEBUG("[ND]nd_update_hh_data: OUT match\n");
	    return 0;
	 }
      case ND_OPT_TLLA:
      default:
	 if (len < dev->addr_len) {
	    break;
	 }
	 memcpy(h_dest, opt + 2, dev->addr_len);
	 DEBUG("[ND]nd_update_hh_data: OUT with update\n");
	 return 0;
      }
   }
   DEBUG("[ND]nd_update_hh_data: OUT with error otype = %d\n", *opt);
   return -EINVAL;
}
/*--END FUNCTION------------------------------------*/
/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
                               MESSAGE HANDLING 
  
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/*--START_FUNCTION--(nd_event_rcv_ns)-----------------------------------------

   FIXME: Double check this one.
   Last Update: 
   07/25/96: bb.
---------------------------------------------------------------------------*/
static struct neighbor *nd_event_rcv_ns(struct in6_addr *s, 
					struct sk_buff *skb)
{
   struct table_leaf *tl;
   struct neighbor *n;
   __u8 *option;
   int olen;
   /* char *eth; */
   
   DEBUG("[ND]nd_event_rcv_ns: IN addr:\n");
   PRINT_ADDR(s);
   
   if (SAME_ADDR6(*s, in6addr_any)) {
      DEBUG("[ND]nd_event_rcv_ns: OUT\n");
      return NULL;	/* 7.2.3 */
   }

   option = skb->h.raw + sizeof(struct n_adv_sol);
   olen   = skb->tail - option;
   
   while (nd_table.gt_lock) {
            sleep_on(&nd_table.gt_wait);
   }
   table_fast_lock(&nd_table);

   tl = table_lookup_1(s, &nd_table, 1);

   table_unlock(&nd_table);
   wake_up(&nd_table.gt_wait);
   
   if (tl) { /* ============== NEIGH PRESENT IN CACHE ================ */
      n = (struct neighbor *)tl->tl_data;   
      switch (n->n_state) {
      case ND_REACHABLE:
      case ND_STALE:
      case ND_DELAY:
	 if (n->n_state & ND_TIMED) 
	    nd_del_timer(n);
      default:
	 n->n_flags &= ~NCF_HHVALID;
	 if (nd_update_hh_data(n->n_dev, n->n_hh_data, option, olen,
			       ND_OPT_SLLA)) {
	    DEBUG("[ND]nd_event_rcv_ns: failed to update cache\n");
	 } else {
	    n->n_flags |= NCF_HHVALID;
	    n->n_state = ND_STALE;	/* FIXME: what if no update? */
	    n->n_tstamp = jiffies;
	    n->n_probes = 0;
	 }
	 DEBUG("[ND]nd_event_rcv_ns: OUT\n");
	 return (n);
      }
   }
   else { /* =============== CACHE MISS ========================== */
      n = nd_new_neigh(skb->dev, s, ND_REACHABLE, jiffies - ReachableTime);
      if (n == NULL) {
	 DEBUG("[ND]nd_event_rcv_ns: OUT can't create neigh\n");
	 return (NULL);
      }
      n->n_flags &= ~NCF_HHVALID;
      if (nd_store_hh_data(skb->dev, n->n_hh_data, 
			   option, olen, ND_OPT_SLLA) == 0)
	 n->n_flags |= NCF_HHVALID;
      DEBUG("[ND]nd_event_rcv_ns: OUT new neigh %p eth:\n", n);
      PRINT_ETH_ADDR(n->n_hh_data);
      return (n);
   } 
}
/*--END_FUNCTION--(nd_event_rcv_ns)-----------------------------------------*/

/*--START_FUNCTION--(nd_event_rcv_na)-----------------------------------------

 *option : points to first byte in option data.
    olen : total length of option data in packet.

 Last Update:
 07/25/96: bb   
---------------------------------------------------------------------------*/
static void nd_event_rcv_na(struct neighbor *n, char *option, __u8 olen,
			    __u8 solicited, __u8 override)
{
   struct sk_buff *skb;
   int state = n->n_state;

   DEBUG("[ND]nd_event_rcv_na: IN sol = %d; over = %d;\n", 
	 solicited, override);
   /*
    * Should we update the cache?
    */
   if (override || (state == ND_INCOMPLETE)) {
      n->n_flags &= ~NCF_HHVALID;
      if (nd_update_hh_data(n->n_dev, n->n_hh_data, option, olen, ND_OPT_TLLA)) {
	 DEBUG("[ND]nd_event_na: Bad options in received NA\n");
	 return;		/*FIXME: What kind of error is that? */
      }
      n->n_flags |= NCF_HHVALID;
   }
   if (override || (state == ND_INCOMPLETE) || solicited) {
      n->n_probes = 0;
      n->n_tstamp = jiffies;

      if (n->n_state & ND_TIMED) {
	 nd_del_timer(n);
      }
   }
   n->n_state = (solicited ? ND_REACHABLE : ND_STALE);
   if (state == ND_INCOMPLETE) {
      while ((skb = skb_dequeue(&n->n_queue))) {
	 dev_queue_xmit(skb); 
      }
   }
   DEBUG("[ND]nd_event_rcv_na: OUT\n");
}
/*--END_FUNCTION--(nd_event_rcv_na)-----------------------------------------*/
/*--START_FUNCTION--(nd_event_send_na)-----------------------------------------

   Send ND messages 
   Last Update:
   07/25/96: bb.
---------------------------------------------------------------------------*/
static void nd_event_send_na(struct device *dev, struct neighbor *n,
			     struct in6_addr *s, struct in6_addr *d,
			     struct in6_addr *sol_addr,
			     int router, int solicited, int override,
			     int inc_opt, int type)
{
   struct sock       *sk = icmpv6_socket.sk;
   struct sk_buff    *skb;
   struct n_adv_sol  *msg;
   struct nd_opt     *opt;
   int               olen = 0;
   unsigned int      payload, plen, csum;
   int               err;

   DEBUG("[ND]nd_event_send_na: IN dev: %s \n", dev->name);

   plen = sizeof(struct ipv6hdr) + sizeof(struct n_adv_sol);

   if (inc_opt) {
      olen = ((dev->addr_len + 1) >> 3) + 1;
      plen += (olen << 3);
   } else {
      override = 0;
   }

   if ((skb = sock_alloc_send_skb(sk, plen + 15 + dev->hard_header_len,
				  0, 0, &err)) == NULL) {
      DEBUG("[ND]nd_event_send: skb alloc failed \n");
      return;
   }
   skb->dev      = dev;
   skb->protocol = htons(ETH_P_IPV6);
   skb->when     = jiffies;
   skb->sk       = sk;
   skb->nexthop  = n;	/* The 1 we've  just created */
   skb->arp      = 0;		/* How is it used with IPv6? TOHANDLE */
 
   if (build_header_from_neigh(skb, dev, n, sol_addr, d, IPPROTO_ICMPV6,
			       plen, 15) < 0) {
      kfree_skb(skb, FREE_WRITE);
      DEBUG("[ND]nd_event_send_na: build header failed...\n");
      return;
   }
   payload = skb->tail - ((unsigned char *) skb->nh.ipv6h) - sizeof(struct ipv6hdr);

/* =================== FILL IN ICMPv6 HEADER =========================== */
   msg = (struct n_adv_sol *) (skb->nh.ipv6h + 1);
   msg->hdr.type = type;
   msg->hdr.code = 0;
   msg->hdr.csum = 0;
   msg->u2.n_adv.router    = router;
   msg->u2.n_adv.solicited = solicited;
   msg->u2.n_adv.override  = override;

/* reserved  = (router ? 1 << 31 : 0) | (solicited ? 1 << 30 : 0) |
   (override ? 1 << 29 : 0); */

   COPY_ADDR6(msg->target, *sol_addr);

   if (inc_opt) {
      opt = (struct nd_opt *) (msg + 1);
      opt->type = ND_OPT_TLLA;
      opt->len = olen;
      memcpy(opt->lla, dev->dev_addr, dev->addr_len);
   }
   csum = csum_partial((char *) msg, payload, 0);
   msg->hdr.csum = csum_ipv6_magic(sol_addr, d, payload, IPPROTO_ICMPV6, csum);
   ipv6_queue_xmit(sk, skb, 1);
   DEBUG("[ND]nd_event_send_na: OUT\n");
   return;
}
/*--END FUNCTION--(nd_event_send_na)----------------------------------------*/

/*--START_FUNCTION--(nd_event_send_ns)-----------------------------------------

   Send NDS messages

   Called from:
        -nd_new_neigh().
 
   Last Update:
   07/25/96: bb.
---------------------------------------------------------------------------*/
static void nd_event_send_ns(struct device *dev, struct neighbor *n,
			     struct in6_addr *s, struct in6_addr *d,
			     struct in6_addr *sol_addr,
			     int inc_opt, int type)
{
   struct sock      *sk = icmpv6_socket.sk;
   struct sk_buff   *skb;
   struct n_adv_sol *msg;
   struct nd_opt    *opt;
   int              olen = 0;
   unsigned int     payload, plen, csum;
   int              err;

   DEBUG("[ND]nd_event_send_ns: IN dev = %s\n", dev->name);

   plen = sizeof(struct ipv6hdr) + sizeof(struct n_adv_sol);

   if (inc_opt) {
      olen = ((dev->addr_len + 1) >> 3) + 1;
      plen += (olen << 3);
   }
   if ((skb = sock_alloc_send_skb(sk, plen + 15 + dev->hard_header_len, 0, 0, &err)) == NULL) {
      DEBUG("[ND]nd_event_send_ns: skb alloc failed \n");
      return;
   }
   skb->dev = dev;
   skb->protocol = __constant_htons(ETH_P_IPV6);
   /* skb->free = 1; */
   skb->when = jiffies;
   skb->sk = sk;
   skb->nexthop = (IS_MULTIADDR6(*d) ? NULL : n);
   skb->arp = 0;
   /*
     COPY_ADDR6(skb->saddr6, *s);
     COPY_ADDR6(skb->daddr6, *d);
   */
   
   if (build_header_from_neigh(skb, dev, n, s, d, IPPROTO_ICMPV6,
			       plen, 15) < 0) {
      kfree_skb(skb, FREE_WRITE);
      DEBUG("[ND]nd_event_send_ns: build header failed...\n");
      return;
   }
   payload = skb->tail - ((unsigned char *) skb->nh.ipv6h)
       - sizeof(struct ipv6hdr);

/* =================== FILL IN ICMPv6 HEADER =========================== */
   msg = (struct n_adv_sol *) (skb->nh.ipv6h + 1);
   msg->hdr.type = type;
   msg->hdr.code = 0;
   msg->hdr.csum = 0;
   msg->u2.reserved = 0;

   COPY_ADDR6(msg->target, *sol_addr);

   if (inc_opt) {
      opt = (struct nd_opt *) (msg + 1);
      opt->type = ND_OPT_SLLA;
      opt->len = olen;
      memcpy(opt->lla, (char *) dev->dev_addr, (int) dev->addr_len);
   }
   csum = csum_partial((char *) msg, payload, 0);
   msg->hdr.csum = csum_ipv6_magic(&skb->nh.ipv6h->saddr, d, payload,
				   IPPROTO_ICMPV6, csum);
   ipv6_queue_xmit(sk, skb, 1);
   n->n_probes++;
   DEBUG("[ND]nd_event_send_ns: OUT\n");
}
/*--END FUNCTION--(nd_event_send_ns)----------------------------------------*/

/*--START_FUNCTION--(nd_event_rcv)-----------------------------------------
   
   Called from icmpv6_rcv() which is suppose to perform simple
   validation checks on the incoming message.
   
   Last Update:
   07/25/96: bb: ND_NEIGHBOR_SOLICITATION.
                 ND_NEIGHBOR_ADVERTISEMENT.
		 
   NB: skb is freed by icmpv6_rcv
---------------------------------------------------------------------------*/
int nd_event_rcv(struct sk_buff *skb, struct device *dev,
		 struct in6_addr *s, struct in6_addr *d,
		 struct options *opt, unsigned len)
{
   struct table_leaf    *tl;
   struct n_adv_sol     *msg = (struct n_adv_sol *) skb->h.raw;
   struct neighbor      *n = NULL;
   struct inet6_ifaddr  *ia;
   int                  type;
   __u8                 solicited, override, router, mask;
   char                 *ptr;

   DEBUG("[ND]nd_event_rcv: IN dev = %s\n", dev->name);

   switch (msg->hdr.type) {
   case ND_NEIGHBOR_SOLICITATION:
      DEBUG("[ND]nd_event_rcv: ND_NEIGHBOR_SOLICITATION\n");
      PRINT_ADDR(&msg->target);
      if ((ia = get_ifaddr6(&msg->target))) {
	 type = get_ipv6_addr_type(s);
	 if ((type == IPV6_ADDR_UNICAST) ||
	     (type == IPV6_ADDR_LINKLOCAL) ||
	     (type == IPV6_ADDR_SITELOCAL)) {
	    /*
	     * Statistics
	     */

	    /*
	     * State management
	     */
	    if ((n = nd_event_rcv_ns(s, skb))) {
	       /*
	        * Reply
	        */
	       nd_event_send_na(dev, n, d, s, &ia->addr,
				ia->idev->router, 1, 1,
			   IS_MULTIADDR6(*d), ND_NEIGHBOR_ADVERTISEMENT);
	    } else {
	       DEBUG("[ND]nd_event_rcv: find/create neighbor/n");
	    }
	 } else {
	    /* 
	     * Unspecified source address used by hosts performing
	     * duplicated address detection.
	     */
	    if (type == IPV6_ADDR_ANY) {
	       nd_event_send_na(dev, n, d, &ia->addr, &ia->addr,
				ia->idev->router, 1, 1,
			   IS_MULTIADDR6(*d), ND_NEIGHBOR_ADVERTISEMENT);
	       break;
	    }
	    DEBUG("[ND]nd_event_rcv: NS from address, type = %d\n", type);
	 }
      } else {
	 DEBUG("[ND]nd_event_rcv: Not for me\n");\
      }
      /* FIXME: 
         Ooooopppppssss: target address not found on host!
         discard sk_buff?
         Sthg seriously broken if this happens. 
       */
      break;

   case ND_NEIGHBOR_ADVERTISEMENT:
      DEBUG("[ND]nd_event_rcv: ND_NEIGHBOR_ADVERTISEMENT\n");
      PRINT_ADDR(s);
      
      while (nd_table.gt_lock) {
	 sleep_on(&nd_table.gt_wait);
      }
      table_fast_lock(&nd_table);

      tl = table_lookup_1(s, &nd_table, 1);

      table_unlock(&nd_table);
      wake_up(&nd_table.gt_wait);
   
      if (tl) { /* ============== NEIGH PRESENT IN CACHE ================ */
	 n = (struct neighbor *)tl->tl_data;
	 /*
	  *  Find out message flags. FIXME 
	  */
	 mask = ((msg->u2.reserved >> 24) & 0xFF);
	 override = (mask & 0x20);
	 solicited = (mask & 0x40);
	 router = (mask & 0x80);

	 /* 
	  * Any change concerning a router ? 
	  */
	 if (router == 0) {
	    if (n->n_flags & NDF_ISROUTER) {
	       /*
	        * Handle default router list
	        */
	    }
	 } else {
	    if (!(n->n_flags & NDF_ISROUTER)) {
	       /*
	        * Handle default router list
	        */
	       /* Remove entry from default list
	          Update all destinations entry */
	    }
	 }
	 ptr = (unsigned char *) (msg + 1);
	 nd_event_rcv_na(n, ptr, (char *) skb->tail - ptr,
	  (__u8) msg->u2.n_adv.solicited, (__u8) msg->u2.n_adv.override);
      } else {
	 /* if neigh == NULL, silently discard at end of icmpv6_rcv */
	 DEBUG("That's a silent discard, 'coz neighbor's not in cache\n");
      }
      break;

   case ND_ROUTER_SOLICITATION:
      while (nd_table.gt_lock) {
	 sleep_on(&nd_table.gt_wait);
      }
      table_fast_lock(&nd_table);

      tl = table_lookup_1(s, &nd_table, 1);

      table_unlock(&nd_table);
      wake_up(&nd_table.gt_wait);
   
      if (tl) { /* ============== NEIGH PRESENT IN CACHE ================ */
	 n = (struct neighbor *)tl->tl_data;
         ptr = ((unsigned char *)msg) + sizeof(struct r_sol);
         nd_update_hh_data(n->n_dev, n->n_hh_data, ptr, 
			   (char *) skb->tail - ptr, ND_OPT_SLLA);
      }
      break;

   case ND_ROUTER_ADVERTISEMENT:
      while (nd_table.gt_lock) {
	 sleep_on(&nd_table.gt_wait);
      }
      table_fast_lock(&nd_table);

      tl = table_lookup_1(s, &nd_table, 1);

      table_unlock(&nd_table);
      wake_up(&nd_table.gt_wait);
      
      if (tl) { /* ============== NEIGH PRESENT IN CACHE ================ */
	 n = (struct neighbor *)tl->tl_data;
         ptr = ((unsigned char *)msg) + sizeof(struct r_adv);
         nd_update_hh_data(n->n_dev, n->n_hh_data, ptr, (char *) skb->tail - ptr, ND_OPT_SLLA);
      }
      break;

   case ND_REDIRECT:
      break;
   }
   DEBUG("[ND]nd_event_rcv: OUT\n");
   return 0;
}
/*--END FUNCTION--(nd_event_rcv)--------------------------------------------*/

/*--START FUNCTION--(nd_event_send_packet)-----------------------------------
  
    Manage neighbor state when sending IPv6 packets.
    Called from: 
    
    Last update:
    08/01/96: bb.
 ---------------------------------------------------------------------------*/
void nd_event_send_packet(struct neighbor *n)
{
   unsigned long now = jiffies;

   DEBUG("[ND]nd_event_send_packet: IN\n");
   if (n->n_flags & NCF_NOARP)
      return;

   switch (n->n_state) {
   case ND_REACHABLE:
      if (REACHABLE_TIME > now - n->n_tstamp)
	 break;
   case ND_STALE:
      n->n_state = ND_DELAY;
      nd_add_timer(n, DELAY_FIRST_PROBE_TIME);
      break;
   default:
   }
   DEBUG("[ND]nd_event_send_packet: OUT\n");
   return;
} 
/*--END FUNCTION--(nd_event_send_packet)------------------------------------*/

/*--START FUNCTION--(nd_add6)-----------------------------------------------*/
int nd_add6(struct in6_addr *addr, struct neighbor *n) {    
   int err;
   DEBUG("[ND]nd_add6: IN %p\n", n);
   PRINT_ADDR(addr);
   
   while (nd_table.gt_lock) {
            sleep_on(&nd_table.gt_wait);
   }
   table_fast_lock(&nd_table);

   err = table_add_1(addr, 128, n, sizeof(struct neighbor), &nd_table);

   table_unlock(&nd_table);
   wake_up(&nd_table.gt_wait);
   DEBUG("[ND]nd_add6: OUT %p\n", n);
   return(err);
}
/*--END FUNCTION--(nd_add6)-------------------------------------------------*/

static int nd_del(struct neighbor *n)
{
   int err = 0;
   struct device *dev = n->n_dev;

   DEBUG("[ROUTE]nd_del:IN device:%s\n", n->n_dev->name);
   PRINT_ADDR(&n->n_addr);

   while (nd_table.gt_lock) {
      sleep_on(&nd_table.gt_wait);
   }
   table_fast_lock(&nd_table);
   err = table_del_1(&n->n_addr, 128, 
		     NULL, dev, &nd_table);
   table_unlock(&nd_table);
   wake_up(&nd_table.gt_wait);

   DEBUG("[ROUTE]nd_del: OUT dev = %p\n", dev);
   return (err);
}
/*--START FUNCTION--(nd_new_neigh)--------------------------------------------
  
    Create a new cache entry.

    FIXME: What kind of source address do we use?
    Called by nd_get_neighbor if no neighbor found.

    Last update:
    07/26/96: bb.
 ---------------------------------------------------------------------------*/
struct neighbor *nd_new_neigh(struct device *dev, struct in6_addr *addr,
			      __u8 state, __u32 tstamp) {
   struct neighbor *n;
   struct inet6_ifaddr *ifa;
   struct in6_addr msol_addr;

   unsigned short *proto;
   unsigned long flags;
   
   int err;

   DEBUG("[ND]nd_new_neigh: IN --> %s\n", dev->name);
   PRINT_ADDR(addr);

   n = (struct neighbor *) kmalloc(sizeof(struct neighbor), GFP_ATOMIC);
   if (n == NULL) {
      DEBUG("[ND]nd_new_neigh: out of memory \n");
      return NULL;
   }

   /*
    * Allocate and add neighbor 
    */
   memset(n, 0, sizeof(struct neighbor));
   skb_queue_head_init(&n->n_queue);
   COPY_ADDR6(n->n_addr, *addr);
   n->n_plen   = NEIGHBOR_PLEN;
   n->n_type   = get_ipv6_addr_type(addr);
   n->n_dev    = dev;
   n->n_tstamp = (tstamp ? tstamp : jiffies);
   n->n_state  = state;
   n->n_ref    = 1;

   cli();
   if ((dev->flags & IFF_NOARP)||(dev->flags & IFF_LOOPBACK)) {
      n->n_flags |= NCF_NOARP;
      DEBUG("[ND]nd_new_neigh: set flag NCF_NOARP\n");
   } else {
      DEBUG("[ND]nd_new_neigh: NCF_NOARP not set\n");
   }
   sti();

   /* hh_data */
   memcpy(n->n_hh_data + dev->addr_len, dev->dev_addr, dev->addr_len);
   proto = (unsigned short *) (n->n_hh_data + 2 * dev->addr_len);
   *proto = htons(ETH_P_IPV6);

   err = nd_add6(addr, n);

   /*
    *     Sollicit link layer address from neighbors.
    */
   if ((state == ND_INCOMPLETE)) {
      if ((ifa = ipv6_get_saddr(dev, addr))) {
	 MULTISOL_ADDR6(msol_addr, *addr);
	 nd_event_send_ns(dev, n, &ifa->addr, &msol_addr, addr,
			  1, ND_NEIGHBOR_SOLICITATION);
      } else {
	 DEBUG("[ND]nd_new_neigh:Can't determine source address! \n");
	 nd_del(n);
	 return NULL;
      }
      nd_add_timer(n, RETRANS_TIMER);
   }
   DEBUG("[ND]nd_new_neigh: OUT %p\n", n);
   return n;
}
/*--END_FUNCTION--(nd_new_neigh)-----------------------------------------*/

/*--START FUNCTION--(nd_get_neighbor)-------------------------------------
    
    Retrieve neighbor entry with address *addr from cache.
    
    Called from:
         - route_add:   in case a gateway was specified.
         - route_check: when adding a new cache entry.

    Last update:
    07/26/96: bb
    
-------------------------------------------------------------------------*/
struct neighbor *nd_get_neighbor(struct device *dev, struct in6_addr *addr)
{
   struct table_leaf *tl;
   struct neighbor *n;

   DEBUG("[ND]nd_get_neighbor: IN -->\n");
   PRINT_ADDR(addr);
   while (nd_table.gt_lock) {
      sleep_on(&nd_table.gt_wait);
   }
   table_fast_lock(&nd_table);
   
   tl = table_lookup_1(addr, &nd_table, 1);

   table_unlock(&nd_table);
   wake_up(&nd_table.gt_wait);
      
   if (tl) /* ============== neigh IN cache ================ */
      n = (struct neighbor *)tl->tl_data;
   else { /* ============== neigh NOT IN cache ================ */
      if ((dev->flags & IFF_NOARP) || (dev->flags & IFF_LOOPBACK)) {
	 n = nd_new_neigh(dev, addr, ND_REACHABLE, 0);
      } else {
	 n = nd_new_neigh(dev, addr, ND_INCOMPLETE, 0);
      }
   } 
   DEBUG("[ND]nd_get_neighbor: OUT %p\n", n);
   return n;
}
/*--END FUNCTION--(nd_get_neighbor)----------------------------------------*/

/* %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   TIMER MANAGEMENT

   What about linking together all timed neighbors? Wouldn't that be cute?
   That would avoid to go through all intermediary nodes, and lookup is
   still very convenient cause of the tree structure...
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */

/*--START FUNCTION--(nd_event_timer)-----------------------------------------
|
|   Handles the expiration of a neighbor time.
|   Last update:
|   (bb) 07/30/96
|
---------------------------------------------------------------------------*/
static int nd_event_timer(struct neighbor *n)
{
   struct inet6_ifaddr *ifa;
   struct device *dev = n->n_dev;
   struct in6_addr msol_addr;
   unsigned long flags;

   DEBUG("[ND]nd_event_timer: IN neigh = %p addr:\n", n);
   PRINT_ADDR(&n->n_addr);

   switch (n->n_state) {
   case ND_INCOMPLETE:
      if (n->n_probes == MAX_MULTICAST_SOLICIT) {
	 /* 
	  * Return destination unreachable message.
	  */
	 DEBUG("[ND]nd_event_timer: OUT Address Resolution failed\n");
	 save_flags(flags);
	 cli();
	 n->n_flags |= NCF_INVALID;
	 n->n_probes = 0;
	 restore_flags(flags);
	 return 0;
      } else {
	 ifa = ipv6_get_saddr(dev, &n->n_addr);
	 MULTISOL_ADDR6(msol_addr, n->n_addr);
	 nd_event_send_ns(dev, n, &ifa->addr, &msol_addr, &n->n_addr, 1,
			  ND_NEIGHBOR_SOLICITATION);
	 DEBUG("[ND]nd_event_timer: OUT probe = %d\n", n->n_probes);
	 return RETRANS_TIMER;
      }
      break;

   case ND_DELAY:
      DEBUG("[ND]nd_event_timer: ND_DELAY\n");
      n->n_state = ND_PROBE;

   case ND_PROBE:
      DEBUG("[ND]nd_event_timer: ND_DELAY\n");
      if (n->n_probes == MAX_UNICAST_SOLICIT) {
	 /* 
	  * Return destination unreachable message.
	  */
         DEBUG("[ND]nd_event_timer: OUT Address Resolution failed\n");
	 save_flags(flags);
	 cli();
	 n->n_flags |= NCF_INVALID;
	 n->n_state = ND_INCOMPLETE;
	 n->n_probes = 0;
	 restore_flags(flags);
	 return 0;
      } else {
	 ifa = ipv6_get_saddr(dev, &n->n_addr);
	 nd_event_send_ns(dev, n, &ifa->addr, &n->n_addr, &n->n_addr, 1,
			  ND_NEIGHBOR_SOLICITATION);
	 DEBUG("[ND]nd_event_timer: OUT probe = %d\n", n->n_probes);
	 return RETRANS_TIMER;
      }
      break;
   default:
      DEBUG("[ND]nd_event_timer: default !!!\n");
   }
   DEBUG("[ND]nd_event_timer: OUT\n");
   return (~0UL);
}
/*--END FUNCTION--(nd_event_timer)-----------------------------------------*/

/*--START FUNCTION--(nd_timer_handler_1)-----------------------------------
|
|   Recursive function called by nd_timer_handler to
|   handle all "expired" neighbors after a time interrupt.
|   Last update:
|   (bb) 07/30/96
|
---------------------------------------------------------------------------*/
static void nd_timer_handler_1(struct table_node *node, unsigned long *now,
			       unsigned long *ntimer)
{
   struct table_node *rn;
   struct table_leaf *tl;
   struct neighbor *n = NULL;
   int pass = 0;

   DEBUG("[ND]nd_timer_handler_1: IN\n");
/* Depth first search */
/* Try go left first */
   if ((rn = node->t_left)) {
      nd_timer_handler_1(rn, now, ntimer);
      pass++;
   }
/* Try go right */
   if ((rn = node->t_right)) {
      nd_timer_handler_1(rn, now, ntimer);
      pass++;
   }
   if ((!pass) && 
       node->t_leaf &&
       (n = (struct neighbor *)node->t_leaf->tl_data)) {	
      /* Neighbors preflen is 128 */
      DEBUG("[ND]nd_timer_handler_1: ND_TIMED n = %p\n", n);
      if (n->n_state & ND_TIMED) {
	 int time;
	 PRINT_ADDR(&n->n_addr);
	 if (n->n_expires <= *now) {
	    time = nd_event_timer(n);
	 } else
	    time = n->n_expires - *now;

	 if (time == 0) {	/* Leaf can be deleted */
	    unsigned long flags;
	    DEBUG("[ND]nd_timer_handler_1: delete leaf \n");
	    save_flags(flags);
	    cli();

	    tl = node->t_leaf;
	    node->t_leaf = tl->tl_next;
	    leaf_release(&nd_table, tl);

	    restore_flags(flags);
	    return;
	 }
	 *ntimer = min(*ntimer, time);
      }
   }
   DEBUG("[ND]nd_timer_handler_1: OUT\n");
}
/*--END FUNCTION--(nd_timer_handler_1)--------------------------------------*/

/*--START FUNCTION--(nd_timer_handler)----------------------------------------
|
|  Called by interrupt when nd_timer expires.
|  Last update:
|  (bb) 07/30/96
|
---------------------------------------------------------------------------*/
static void nd_timer_handler(unsigned long arg)
{
   unsigned long now = jiffies;
   unsigned long ntimer = ~0UL;

   DEBUG("[ND]nd_timer_handler: IN\n");

   while (nd_table.gt_lock)
      sleep_on(&nd_table.gt_wait);
   table_fast_lock(&nd_table);

   nd_timer_handler_1(&nd_root, &now, &ntimer);

   if (ntimer != (~0UL)) {
      nd_timer.expires = jiffies + ntimer;
      add_timer(&nd_timer);
   }
   table_unlock(&nd_table);
   wake_up(&nd_table.gt_wait);

   DEBUG("[ND]nd_timer_handler: OUT\n");
   return;
}

/* 
 * Called by upper layers to validate neighbour cache entries. 
 * From Roque's tcp.
 */
void nd_validate(struct neighbor *n)
{ 
        if (n->n_state == ND_INCOMPLETE)
                return;

        if (n->n_state == ND_DELAY) 
        {
                nd_del_timer(n);
        }

        /* nd_stats.rcv_upper_conf++; */
        n->n_state = ND_REACHABLE;
        n->n_tstamp = jiffies;
}

/*--START FUNCTION--(nd_init)---------------------------------------------

   Initializes the neighbor discovery timer.

---------------------------------------------------------------------------*/
void nd_init(void)
{
   DEBUG("[ND]nd_init: ND INITIALISATION\n");
   init_timer(&nd_timer);
   nd_timer.function = nd_timer_handler;
   nd_timer.data = 0L;
   nd_timer.expires = 0L;
   return;
}






