/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$ 
 *      
 * Original authors:    
 *              Pedro Roque             <roque@di.fc.ul.pt>     
 *              Peter Belding           <pbelding@qualcomm.com>
 *
 *
 * Some fixes and present version:
 *              Benoit Brodard          <bbrodard@sophia.inria.fr>
 *
 * Description:
 *
 *     Routing Table
 *
 *     simplified version of a radix tree
 *
 *     - every node shares its ancestors prefix
 *     - the tree is ordered from less to most specific mask
 *     - default routes are handled apart
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|  ipv6_route_init
|  ipv6_route_ioctl
|     ipv6_route_add
|        rt_add6   
|     ipv6_route_del
|  dcache_check
|     route_check
|        rt_lookup
|
|  Last Update: 05.13.97 Table modularization
|  TODO       : Garbage collection
|               Maintain more statistic vars, 
|               make them availiable through /proc
-------------------------------------------------------------------------*/
#include <asm/uaccess.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/sched.h>
#include <linux/net.h>
#include <linux/route.h>
#include <linux/netdevice.h>
#include <linux/dret/netdevice6.h>
#include <linux/in.h>
#include <net/netlink.h>
#include <net/sock.h>
#include <net/snmp.h>

#include <net/dret/ipv6.h>
#include <net/dret/nd.h>
#include <net/dret/protocol6.h>
#include <net/dret/route6.h>
#include <net/dret/addrconf.h>
#include <net/dret/sit.h>
#include <net/dret/table_tools.h>

#ifdef IPV6_DEBUG_ROUTE
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[ROUTE6]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#else
#define DEBUG(format, args...)
#define PRINT_ADDR(a)
endif

/*
 * Routing table root
 */
struct table_node routing_root = {
   NULL, NULL, NULL, {{{0}}}, NULL, 0, RTN_ROOT
};

struct gen_table routing_table = {
   &routing_root, 0, 0
   };

struct rt_dest null_entry =
{
   NULL, NULL,
   {
      {
	 {0}}},
   1, 1,
   NULL, NULL,
   0, 0, RTF_REJECT
};


/*--START FUNCTION--(rt_add6)---------------------------------------



  ------------------------------------------------------------------*/
int rt_add6(struct in6_addr *addr, __u16 plen, struct rt_dest *rt) {
   int err;
   while (routing_table->gt_lock) {
            sleep_on(&routing_table->gt_wait);
   }
   table_fast_lock(&routing_table);

   err = table_add_1(&rt1->rd, rt1->rt_prefixlen, 
		     rt, sizeof(struct rt_dest), &routing_table);

   table_unlock(&routing_table->gt_table);
   wake_up(&routing_table->gt_wait);
	 
#ifdef CONFIG_NET_RTNETLINK6
   ipv6_netlink_msg(RTMSG_NEWROUTE, &rt1->rd,
		    (rt1->neigh ? &rt1->neigh->naddr : 0),
		    rt1->rt_prefixlen, dev->flags, dev->metric, dev->name);
#endif
   
   if(!err) atomic_inc(&routing_table->gt_entries);
   return err;
}
/*--END FUNCTION--(rt_add6)---------------------------------------*/

/*--START FUNCTION--(ipv6_route_add)---------------------------------------
 | 
 | Call after an ioctl call with XXXXXXXXXXX
 | Input: a route message
 |        -> type-destination-gateway-prefix length-metric-device-flags
 |
 | Return:
 |
--------------------------------------------------------------------------*/
int ipv6_route_add(struct in6_rtmsg *rtmsg)
{
   struct rt_dest *rt;
   struct device *dev = NULL;
   int flags = rtmsg->rtmsg_flags;


   DEBUG("[ROUTE]ipv6_route_add: IN device: %s\n", rtmsg->rtmsg_device);
   PRINT_ADDR(&rtmsg->rtmsg_dst);
   PRINT_ADDR(&rtmsg->rtmsg_gateway);
   
   /* Retrieve device */
   if (rtmsg->rtmsg_device) {
      if (!(dev = dev_get(rtmsg->rtmsg_device))) {
	 DEBUG("[ROUTE]Can't find device %s\n", rtmsg->rtmsg_device); 
	 return -ENODEV;
      }
      if (!dev->dev6_lst) {
	 DEBUG("[ROUTE]No IPv6 support on device %s\n", dev->name);
	 return -EAFNOSUPPORT;
      }
      DEBUG("[ROUTE]ipv6_route_add: device: %s %p %04X\n",
	    dev->name, dev, dev->flags);
   }
   /*
    * BSD emulation: Permits route add someroute gw one-of-my-addresses
    *   to indicate which iface.
    */
   if (!dev && (flags & RTF_GATEWAY)) {
      struct inet6_ifaddr *ifa;

      if ((ifa = get_ifaddr6(&rtmsg->rtmsg_gateway))) {
	 if (ifa->idev->dev->flags & IFF_UP) {
	    flags &= ~RTF_GATEWAY;
	    dev = ifa->idev->dev;
	    DEBUG("[ROUTE]ipv6_route_add: device: %s %p %04X\n",
	    dev->name, dev, dev->flags);
	 }
      }
   }
   /* Set gateway for tunneled routes */
   if (dev) {
      if (dev->flags & IFF_TUNNEL) {
	 if (!(flags & RTF_GATEWAY)) {
	    if (!IS_IPV4ADDR6(rtmsg->rtmsg_dst)) {
	       flags |= RTF_GATEWAY;
	       memset(&rtmsg->rtmsg_gateway, 0, sizeof(struct in6_addr));
	       rtmsg->rtmsg_gateway.s6_addr32[3] = get_sit_end_point(dev);
	    }
	 }
      }
   }
   
   /* Alloc route entry */
   rt = (struct rt_dest *) kmalloc(sizeof(struct rt_dest), GFP_KERNEL);
   if (rt == NULL)
      return (-ENOMEM);
   memset(rt, 0, sizeof(struct rt_dest));

   COPY_ADDR6(rt->rt_addr, rtmsg->rtmsg_dst);
   rt->rt_prefixlen = rtmsg->rtmsg_prefixlen;
   rt->rt_metric    = rtmsg->rtmsg_metric;
   rt->rt_flags     = rtmsg->rtmsg_flags;
   rt->rt_use       = 0;

   /* Check gateway is OK */
   if (flags & RTF_GATEWAY) {
      struct rt_dest *gw;
      
      if (IS_LINKLADDR6(rtmsg->rtmsg_gateway) && dev) {
	 /*
	  * Gateway is link-local - Set route
	  */
	 struct rt_dest *rt1;
	 rt->rt_neigh = nd_get_neighbor(dev, &rtmsg->rtmsg_gateway);
	 
	 rt1 = (struct rt_dest *) kmalloc(sizeof(struct rt_dest), GFP_KERNEL);
         if (rt1 == NULL)
            return (-ENOMEM);
	 memset(rt1, 0, sizeof(struct rt_dest));
         
         rt1->rt_metric    = 1;
         rt1->rt_flags     = RTF_HOST;
         rt1->rt_use       = 0; /* incremented when packet sent */
	 rt1->rt_ref       = 1;
	 COPY_ADDR6(rt1->rt_addr, rtmsg->rtmsg_gateway);
	 rt1->rt_dev       = dev;
         rt1->rt_pmtu      = dev->mtu;
	 rt1->rt_prefixlen = 128;
	 rt1->rt_neigh     = rt->neigh;
	 
	 /* add route entry */
	 rt_add6(&rt1->rt_addr, rt1->rt_prefixlen, rt1);
         
	 if (!rt->rt_neigh) {
	    kfree(rt);
	    return -ENETUNREACH;
	 }
      } else if ((gw = rt_lookup(&rtmsg->rtmsg_gateway, 0))) {
	 /* Gw is not link-local but route exists */
	 rt->rt_neigh = gw->neigh;
	 atomic_inc(&gw->rt_ref);
	 dev = gw->dev;
      } else {
	 /* No route to referenced gateway - Abort */
	 DEBUG("No route to gateway...\n");
	 kfree(rt);
	 return -EHOSTUNREACH;
      }      
      rt->rt_flags |= RTF_GATEWAY;
      if ((!rt->rt_neigh) ||
	  !(SAME_ADDR6(rt->rt_neigh->naddr, rtmsg->rtmsg_gateway))) {
	 rt->rt_neigh = nd_get_neighbor(dev, &rtmsg->rtmsg_gateway);
	 if (!rt->rt_neigh) {
	    kfree(rt);
	    return -ENETUNREACH;
	 }
      }
   } else if (rt->rt_prefixlen == 128) {
      if (!(rt->rt_neigh = nd_get_neighbor(dev, &rt->rt_addr))) {
	 kfree(rt);
	 return -ENETUNREACH;
      }
   }
   
   /* No device found */
   if (dev == NULL) {
      printk("[ROUTE6]ipv6_rt_add: NULL device\n");
      kfree(rt);
      return -ENETUNREACH;
   }
   
   if (dev->flags & IFF_LOOPBACK) 
      if (!lo_route) 
	 lo_route = (struct destination *) rt;
      
   rt->rt_dev  = dev;
   rt->rt_pmtu = dev->mtu;

   rt_add6(&rt->rt_addr, rt->rt_prefixlen, rt);
   DEBUG("[ROUTE]ipv6_route_add: OUT\n");
   return 0;
}
/*--END FUNCTION--(ipv6_route_add)-----------------------------------------*/

/*--START FUNCTION--(ipv6_route_del)---------------------------------------
 | 
 | Call after an ioctl call with XXXXXXXXXXX
 | Input: a route message
 |        -> type-destination-gateway-prefix length-metric-device-flags
 |
 | Return: 0;
 |
--------------------------------------------------------------------------*/
int ipv6_route_del(struct in6_rtmsg *rtm)
{
   int err = 0;
   struct device *dev = dev_get(rtm->rtmsg_device);

   DEBUG("[ROUTE]ipv6_route_del:IN device:%s\n", rtm->rtmsg_device);
   PRINT_ADDR(&rtm->rtmsg_dst);

   while (routing_table->gt_lock) {
      sleep_on(&routing_table->gt_wait);
   }
   table_fast_lock(&routing_table);
   err = table_del_1(&rtm->rtmsg_dst, rtm->rtmsg_prefixlen, 
		     &rtm->rtmsg_gateway, dev, &routing_table);
   table_unlock(&routing_table);
   wake_up(&routing_table->gt_wait);

   if ((err == 0) && dev) {
#ifdef CONFIG_NET_RTNETLINK6      
      ipv6_netlink_msg(RTMSG_DELROUTE, &rtm->rtmsg_dst, 0,
	       rtm->rtmsg_prefixlen, dev->flags, dev->metric, dev->name);
#endif
   }  
   DEBUG("[ROUTE]ipv6_route_del: OUT dev = %p\n", dev);
   return (err);
}

/*--START FUNCTION--(rt_lookup)-------------------------------------------
 | 
 | 
 |
 |
 ------------------------------------------------------------------------*/
struct rt_dest *rt_lookup(struct in6_addr *addr, int local)
{
   struct rt_dest *rt;
   struct table_leaf *tl;
   struct gen_table *table = &routing_table
   int err = 0;
   DEBUG("[ROUTE]rt_lookup: IN\n");
   
     
   while (routing_table->gt_lock) {
            sleep_on(&routing_table->gt_wait);
   }
   table_fast_lock(&routing_table);

   tl = table_lookup_1(addr, table, 0);

   table_unlock(&routing_table->gt_table);
   wake_up(&routing_table->gt_wait);tl = table_lookup_1(addr, table, 0);
    
   if (tl) {
      rt = (struct rt_dest *)tl->tl_data;
      atomic_inc(&rt->rt_ref);
      DEBUG("[ROUTE]rt_lookup: addr / neigh OUT\n");      
      PRINT_ADDR(&rt->rt_addr);
      if (rt->rt_neigh) {
	 PRINT_ADDR(&rt->rt_neigh->naddr);
      }      
      return rt;
   }
   return NULL;
}
/*--END FUNCTION--(rt_lookup)-----------------------------------------------*/



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   ROUTE RETRIEVAL

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */

/*--START FUNCTION--(dcache_check)----------------------------------------
 *  check cache entry for validity...
 *  Last update:
    07/15/96: simple validity check...
              FIXME
 ---------------------------------------------------------------------------*/
struct destination *dcache_check(struct destination *dest,
				 struct in6_addr *d, int local) {
   /*
    *      Check route validity
    */
   DEBUG("[ROUTE6]dcache_check: IN\n");
   PRINT_ADDR6(dest);
   if (dest) {
      if (SAME_ADDR6(dest->d_addr, *d)) {
	 if ((dest->d_flags & DC_INVALID) == 0) {
	    if ((dest->d_neigh) && !(dest->d_neigh->flags & NCF_NOARP)) {
	       nd_event_send_packet(dest->d_neigh);
	    }
	    DEBUG("[ROUTE6]dcache_check: OUT\n");
	    return dest;
	 } else {
	    /* 
	     * drop usage counter on route 
	     */
	    atomic_dec(&(dest->d_ref));
	    /*if (dest->d_ref == 0) {
	       kfree(dest);
	       } */
	    /* pmtu value is now unkown */
	    dest->d_flags &= ~DCF_PMTU;
	 }
      }
   }
      
   if (check_host_addr6(d)) {
      DEBUG("[ROUTE6]dcache_check: OUT\n");
      return lo_route;
   }
   DEBUG("[ROUTE6]dcache_check: OUT\n");
   return (route_check(d, local));
}
/*--END FUNCTION--(dcache_check)---------------------------------------*/

/*--START FUNCTION--(route_check)-------------------------------------

  Last update:
   07/15/96: 

-------------------------------------------------------------------------*/
struct destination *route_check(struct in6_addr *d, int local)
{
   struct destination *dc;
   struct rt_dest *rt;

   DEBUG("[ROUTE6]route_check: IN\n");
   while (routing_table->gt_lock)
      sleep_on(&routing_table->gt_wait);
   table_fast_lock(&routing_table);
   
   rt = rt_lookup(d, local);
   
   if (rt == NULL) {
      table_unlock(&routing_table);
      wake_up(&table_wait);
      return NULL;
   }
   
   if ((rt->rt_flags & RTF_GATEWAY) || (rt->rt_prefixlen == 128)) {
      dc = (struct destination *) rt;
      atomic_inc(&rt->rt_ref);
   } else {
      /* local route */
      dc = (struct destination *)
	   kmalloc(sizeof(struct destination), GFP_ATOMIC);
      if (dc == NULL) {
	 DEBUG("[ROUTE6]route_check: kmalloc failed OUT\n");
	 return NULL;
      }
      memset(dc, 0, sizeof(struct destination));

      memcpy(&dc->d_addr, d, sizeof(struct in6_addr));
      dc->d_ref = 1;
      dc->d_prefixlen = 128;
      dc->d_metric = rt->rt_metric;
      dc->d_flags = (rt->rt_flags | RTF_HOST | RTF_DCACHE | DCF_PMTU);
      dc->d_dev = rt->rt_dev;
      dc->d_pmtu = rt->d_dev->mtu;
      dc->d_tstamp = jiffies;
      dc->d_neigh = nd_get_neighbor(rt->rt_dev, d);
      if (dc->d_neigh == NULL) {
	 DEBUG("[ROUTE6]route_check: neigh = NULL OUT\n");
	 kfree(dc);
	 return NULL;
      }
      rt_add6(d, 128, (struct rt_dest *) dc);	/* FIXED 10/24/96 */
   }
   table_unlock(&routing_table);
   wake_up(&routing_table->gt_wait);

   DEBUG("[ROUTE6]route_check: OUT\n");
   PRINT_ADDR(&(dc->route.rt_addr));
   return dc;
}
/*--END FUNCTION--(route_check)----------------------------------------------*/


/*--START FUNCTION--(mcast_check)----------------------------------------



---------------------------------------------------------------------------*/
struct destination *mcast_check(struct device *dev, struct in6_addr *d)
{
   struct destination *dest;
   struct rt_dest *rd;
   __u32 hash;


   hash = ipv6_addr_hash(d);
   atomic_inc(&dcache_lock);
   for (dest = dest_cache[hash]; dest; dest = (struct destination *)
	dest->route.next) {
      if (SAME_ADDR6(dest->route.rt_addr, *d)) {
	 if (dev && dev != (dest->route.neigh)->dev)
	    continue;
	 atomic_inc(&dest->refcnt);
	 atomic_dec(&dcache_lock);
	 return dest;
      }
   }
   /* 
    *  not found 
    */
   if (dev == NULL) {
      rd = rt_lookup(d, 0);

      if (rd == NULL) {
	 atomic_dec(&dcache_lock);
	 return NULL;
      }
      dev = rd->dev;
   }
   dest = (struct destination *) kmalloc(sizeof(struct destination),
					 GFP_ATOMIC);
   if (dest == NULL) {
      atomic_dec(&dcache_lock);
      return NULL;
   }
   memset(dest, 0, sizeof(struct destination));
   memcpy(&dest->route.rt_addr, d, sizeof(struct in6_addr));
   /*dest->n = ncache_get_neigh(dev, d); */
   dest->pmtu = dev->mtu;
   dest->refcnt = 1;
   /* insert at the head of the list */
   dest->route.next = (struct rt_dest *) dest_cache[hash];
   dest_cache[hash] = dest;
   atomic_dec(&dcache_lock);
   return dest;
}
/*--END FUNCTION--(mcast_check)---------------------------------------------*/


/*** TO DO
  
  - Destination cache garbage collection

  ****/

/*--START FUNCTION--(ipv6_route_ioctl)----------------------------------*/
int ipv6_route_ioctl(unsigned int cmd, void *arg)
{
   struct in6_rtmsg rtmsg;
   int err;

   switch (cmd) {
   case SIOCADDRT:
   case SIOCDELRT:
      if (!suser())
	 return -EPERM;
      err = copy_from_user(&rtmsg, arg, sizeof(struct in6_rtmsg));
      if (err)
	 return -EFAULT;
      DEBUG("route_ioctl#: type= %d; prefix = %d; metric = %d; dev = %s flags: %04X\n",
	    rtmsg.rtmsg_type, rtmsg.rtmsg_prefixlen, rtmsg.rtmsg_metric,
	    rtmsg.rtmsg_device, rtmsg.rtmsg_flags);
      return
	  (cmd == SIOCDELRT) ? ipv6_route_del(&rtmsg) : ipv6_route_add(&rtmsg);
   default:
      return -EINVAL;
   }
}
/*--END FUNCTION--(ipv6_route_ioctl)--------------------------------------*/

/*--START FUNCTION--(ipv6_route_init)-------------------------------------

 I Init destination cache.... not really route table....

--------------------------------------------------------------------------*/
void ipv6_route_init(void)
{
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]ipv6_route_init IN\n");
#endif
   init_timer(&dcache_timer);
   dcache_timer.function = dcache_bh;
   dcache_timer.data = 0;
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]ipv6_route_init OUT\n");
#endif
}
