/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:     $Id$ 
 *
 * Authors:     
 *              Benoit Brodard  <Brodard@sophia.inria.fr>
 *
 *
 * Fixes:
 *              Pascal Anelli   add sit_init in ipv6_init
 *
 *
 * Description:
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|  ipv6_build_mac_header
|  ipv6_queue_xmit
|  ipv6_build_xmit
|  ipv6_init
|  ipv6_xmit
|
-------------------------------------------------------------------------*/

#include <asm/segment.h>
#include <asm/system.h>
#include <asm/uaccess.h>

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/errno.h>

#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>

#include <net/snmp.h>
#include <net/ip.h>
#include <net/protocol.h>
#include <net/route.h>
#include <net/tcp.h>
#include <net/udp.h>
#include <linux/skbuff.h>
#include <net/sock.h>
#include <net/arp.h>
#include <net/icmp.h>
#include <net/raw.h>
#include <net/checksum.h>
#include <linux/igmp.h>
#include <linux/ip_fw.h>
#include <linux/firewall.h>
#include <linux/mroute.h>
#include <net/netlink.h>

#include <linux/dret/ipv6.h>
#include <linux/dret/ipv6_debug.h>

#include <net/dret/ipv6.h>
#include <net/dret/protocol6.h>
#include <net/dret/route6.h>
#include <net/dret/nd.h>
#include <net/dret/addrconf.h>
#include <net/dret/sit.h>


#ifdef IPV6_DEBUG_OUTPUT
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[OUTPUT]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#define PRINT_ETH_ADDR(a) printk(KERN_DEBUG "[OUTPUT]%2X:%2X:%2X:%2X:%2X:%2X\n",\
				 a[0], a[1], a[2], a[3], a[4], a[5]);
#else
#define DEBUG(format, args...) 
#define PRINT_ADDR(a)
#define PRINT_ETH_ADDR(a)
#endif


/*--START FUNCTION--(ipv6_build_mac_header)----------------------------------
| 
| ipv6_build_mac_header: build hard header into skbuff.
| Last modified: 05/07/96
|
-----------------------------------------------------------------------------*/
int __inline__ ipv6_build_mac_header(struct sk_buff *skb,
				     struct neighbor *n,
				     int len)
{
   int mac;
   int hdrlen = 0;
   struct device *dev = skb->dev;

   DEBUG("[OUTPUT]ipv6_build_mac_header(1): IN skb = %p neigh = %p\n",
	 skb, n);
   /* Presumed resolved */
   skb->arp = 1;		
   /* Reserve room in skb */
   if (dev->hard_header_len) {
      skb_reserve(skb, (dev->hard_header_len + 15) & ~15);

      /* Neighbor exists */
      if (n && (n->n_flags & NCF_HHVALID)) {
	 memcpy(skb_push(skb, dev->hard_header_len),
		n->n_hh_data, dev->hard_header_len);
	 DEBUG("[OUTPUT]ipv6_build_mac_header(2a): OUT used n: %d eth_addr:\n",
	       dev->hard_header_len);
	 PRINT_ETH_ADDR(n->n_hh_data);
	 return dev->hard_header_len;
      }
      
      /* Method exists */
      if (dev->hard_header) {
	 mac = dev->hard_header(skb, dev, ETH_P_IPV6, NULL, NULL, len);
	 if (mac < 0) {
	    skb->arp = 0;
	    hdrlen = -mac;
	 } else {
	    hdrlen = mac;
	 }
	 DEBUG("[OUTPUT]ipv6_build_mac_header(2b): OUT dev->hard_header %d\n",
	       hdrlen);
	 return hdrlen;
      }
      
      /* NoArp? */
      hdrlen = dev->hard_header_len;
      skb->arp = ((dev->flags && IFF_NOARP) ? 1 : 0);
      DEBUG("[OUTPUT]ipv6_build_mac_header(2c): OUT NoArp %d\n",hdrlen);
      return hdrlen;
   } /* dev->hard_header_len */
   
   DEBUG("[OUTPUT]ipv6_build_mac_header(3): OUT 0 skb->arp = %d\n", skb->arp);
   return hdrlen;
}

/*--START FUNCTION--(build_header_from_neigh)----------------------------------
| 
| build_header_from_neigh: build ipv6 header.
| Last modified: 05/07/96
|
-----------------------------------------------------------------------------*/
int build_header_from_neigh(struct sk_buff *skb, struct device *dev,
			    struct neighbor *neigh,
			    struct in6_addr *s, struct in6_addr *d,
			    int proto, int plen, int priority)
{
   struct ipv6hdr *hdr;
   int hdrlen = 0;

   DEBUG("[OUTPUT]build_header_from_neigh: IN\n");
   skb->dev = dev;

   hdrlen += ipv6_build_mac_header(skb, neigh, plen);

   if (proto == IPPROTO_RAW)
      return hdrlen;

   hdr = (struct ipv6hdr *) skb_put(skb, plen);
   skb->nh.ipv6h = hdr;
   /* =================== FILL IN IPv6 HEADER ======================== */
   hdr->vrprflow = IPV6_VERSION |
       IPV6_SET_FLOWLABEL(0) |
       IPV6_SET_PRIORITY(priority);
   /* payload taken care of in ipv6_queue_xmit */
   hdr->hop_limit = 255;

   if (s == NULL) {
      struct inet6_ifaddr *ifp;

      ifp = ipv6_get_saddr(dev, d);
      if (ifp == NULL)
	 return -ENETUNREACH;
      s = &ifp->addr;
   }
   memcpy(&hdr->saddr, s, sizeof(struct in6_addr));
   memcpy(&hdr->daddr, d, sizeof(struct in6_addr));
   hdrlen += sizeof(struct ipv6hdr);
   hdr->next_hdr = proto;

   DEBUG("[OUTPUT]build_header_from_neigh: OUT hdrlen = %d\n", hdrlen);
   return hdrlen;
}

/*--START FUNCTION--(ipv6_queue_xmit)------------------------------------------
| 
| ipv6_queue_xmit: build header and call dev_queue_xmit()
| Last modified: 05/22/96
| Author: B. Brodard
| Last update:
  07/24/96: (bb) called by nd_event_send_na, ns.
-----------------------------------------------------------------------------*/
void ipv6_queue_xmit(struct sock *sk, struct sk_buff *skb, int free)
{
   struct ipv6hdr *hdr;
   /*struct destination *dest = sk->net_pinfo.af_inet6.dest; */
   __u32 payload;
   struct device *dev;

#ifdef IPV6_DEBUG_OUTPUT
   printk(KERN_DEBUG "[OUTPUT]ipv6_queue_xmit: IN \n");
#endif
   hdr = skb->nh.ipv6h;
   dev = skb->dev;

   /* sanity check dev */

   /* 
    * payload len. 
    * note: the pseudo-header checksum depends of the 
    * payload length. Transport protocols must keep track
    * of the packet len.
    */
   payload = skb->tail - ((unsigned char *) hdr) - sizeof(struct ipv6hdr);
   hdr->payload = htons(payload);
   /* if (payload > dest->pmtu)
      {

      kfree_skb(skb, FREE_WRITE);
      return;
      } */

   ip6_statistics.IpOutRequests++;

   /*
    *      Multicast loopback (FIXME)
    */
   if (dev->flags & IFF_UP) {
      dev_queue_xmit(skb);
     
#ifdef IPV6_DEBUG_OUTPUT
      printk(KERN_DEBUG "QXMIT ==============================\n");
      PRINT_ADDR(&skb->nh.ipv6h->saddr);
      PRINT_ADDR(&skb->nh.ipv6h->daddr);
      printk(KERN_DEBUG "QXMIT ==============================\n");
#endif      
   } else {
      if (sk)
	 sk->err = ENETDOWN;
      ip6_statistics.IpOutDiscards++;
      kfree_skb(skb, FREE_WRITE);
   }
#ifdef IPV6_DEBUG_OUTPUT
   printk(KERN_DEBUG "[OUTPUT]ipv6_queue_xmit: OUT\n");
#endif
}
/*--START FUNCTION--(ipv6_queue_xmit)----------------------------------------*/

/*      From Pedro Roque
 *	xmit an sk_buff (used by TCP)
 *	sk can be NULL (for sending RESETs)
 */
int ipv6_xmit(struct sock *sk, struct sk_buff *skb, struct in6_addr *saddr,
	      struct in6_addr *daddr, struct options *opt, int proto)
{
	struct ipv6hdr *hdr;
	struct destination *dc= NULL;
	struct ipv6_pinfo *np = NULL;
	struct device *dev = skb->dev;
	int seg_len;
	int addr_type;
	int rt_flags = 0;

	printk(KERN_DEBUG "[OUTPUT]ipv6_xmit IN\n");
	addr_type = get_ipv6_addr_type(daddr);

	if (addr_type & (IPV6_ADDR_LINKLOCAL|IPV6_ADDR_SITELOCAL))
	{
		/*
		 *	force device match on route lookup
		 */
		
	   /*rt_flags |= RTF_DEVRT; FIXME */ 
	}

	if (sk && sk->localroute)
		rt_flags |= RTF_GATEWAY;

	hdr = skb->nh.ipv6h;
	

	if (sk)
	{
		np = &sk->net_pinfo.af_inet6;
	}
	
	if (np && np->dest)
	{
		dc = dcache_check(np->dest, daddr, rt_flags);
	} else {
	        dc = dcache_check(NULL, daddr, rt_flags);
	}
	printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: back from dcache_check\n");
	if (dc == NULL)
	{
	   /*ipv6_statistics.Ip6OutNoRoutes++;*/
	   printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: Achhhh... no route...\n");
	   return(-ENETUNREACH);
	}
	printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: Found a route\n");
	dev = dc->d_dev;
	
	if (saddr == NULL)
	{
	   struct inet6_ifaddr *ifa;
		
	   ifa = ipv6_get_saddr(dev, daddr);
		
	   if (ifa == NULL)
	      {
		 printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: get_saddr failed\n");
		 return -ENETUNREACH;
	      }
	   
	   saddr = &ifa->addr;
	   
	   if (np)
	      {
		 COPY_ADDR6(np->pi6_fl.fll_sa, *saddr);
	      }
	}
	printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: Found a saddr\n");
	seg_len = skb->tail - ((unsigned char *) hdr);

	/*
	 *	Link Layer headers
	 */

	skb->protocol = __constant_htons(ETH_P_IPV6);
	skb->dev = dev;
	printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: @4\n");
	skb->nexthop = dc->d_neigh;
	/* skb->dev = skb->nexthop->dev; */
	skb->arp = 1;
	skb_pull(skb, (unsigned char *) skb->nh.ipv6h - skb->data);
	if (skb->dev->hard_header)
        {
	   int mac;
	   
                mac = skb->dev->hard_header(skb, skb->dev, ETH_P_IPV6, 
                                       NULL, NULL, 
				       skb->tail - (u8*) skb->nh.ipv6h);
                
                if (mac < 0)
                {                               
                        skb->arp = 0;
                }

        }
        skb->mac.raw = skb->data;
	printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: built hard header \n");
	/* ipv6_redo_mac_hdr(skb, dc->nexthop, seg_len);*/
	
	/*
	 *	Fill in the IPv6 header
	 */
        hdr->vrprflow = IPV6_VERSION |
	                IPV6_SET_FLOWLABEL(np ? np->pi6_fl.fll_label : 0) |
	                IPV6_SET_PRIORITY(np ? np->pi6_fl.fll_prio : 0);	
	hdr->payload  = htons(seg_len - sizeof(struct ipv6hdr));
	hdr->next_hdr = proto;
	hdr->hop_limit = (np ? np->hop_limit : IPV6_DEFAULT_HOP_LIMIT);
	
	memcpy(&hdr->saddr, saddr, sizeof(struct in6_addr));
	memcpy(&hdr->daddr, daddr, sizeof(struct in6_addr));

	printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: built ipv6 header \n");
	/*
	 *	Options
	 */


	/*
	 *	Output the packet
	 */
	
	/*  ipv6_statistics.Ip6OutRequests++; */

	/*
	 *	Update serial number of cached dest_entry or
	 *	release destination cache entry
	 */
	printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: @5\n");
	if (dev->flags & IFF_UP)
	   {
	      /*
	       *      If we have an owner use its priority setting,
	       *      otherwise use NORMAL
	       */
	 printk(KERN_DEBUG "IPV6_XMIT skb = %p \n", skb);
#ifdef IPV6_DEBUG_OUTPUT
	 printk(KERN_DEBUG "IPV6_XMIT ==============================\n");
	 PRINT_ADDR(&(hdr->saddr));
	 PRINT_ADDR(&(hdr->daddr));
	 printk(KERN_DEBUG "IPV6_XMIT ==============================\n");
#endif	      
	      dev_queue_xmit(skb);
	   }
        else
	   {
	      if(sk)
		 sk->err = ENETDOWN;
	      
	      /* ipv6_statistics.Ip6OutDiscards++;*/
	      printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: dev->flags & IFF_UP == 0\n");
	      kfree_skb(skb, FREE_WRITE);
	   }
	
	if (np)
	   {
	      printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: np\n");
	      np->dest = dc;
	      /*
		if (dc->rt.fib_node)
		{
		np->dc_sernum = dc->rt.fib_node->fn_sernum;
		}
		  */
	   }
	else
	   {
	      printk(KERN_DEBUG "[OUTPUT]ipv6_xmit: !np\n");
	      /*ipv6_dst_unlock(dc);*/
	   }
	printk(KERN_DEBUG "[OUTPUT]ipv6_xmit OUT\n");
	return 0;
}

/*--START FUNCTION--(ipv6_flow_xmit)--------------------------------------
| 
| ipv6_flow_xmit: build header and call dev_queue_xmit()
| INPUT         : a flowlist structure
| Last modified: 05/06/97
|    
--------------------------------------------------------------------------*/
int ipv6_flow_xmit(struct sock *sk,
		    void getfrag(const void *, struct in6_addr *,
				 char *, unsigned int, unsigned int),
		    const void *frag,
		    unsigned short int length,	/* fragment's length */
		    struct flowlist *fl,
		    int flags,
		    int protocol,
		    int noblock) {
   struct destination *rt    = NULL;
   struct neighbor    *neigh = NULL;
   struct device      *dev   = NULL;
   struct ipv6hdr     *iph;
   struct ipv6_pinfo  *info  = &sk->net_pinfo.af_inet6; 

   int plen = length;	   /* actual packet length, with hdr & options */
   int mc_hl = 0;							   
							   
   DEBUG("[OUTPUT]ipv6_flow_xmit: IN\n"); PRINT_ADDR(&fl->fll_da);	
   ip6_statistics.IpOutRequests++;
   
   /*
    *  Destination ?
    */
   if (!IS_MULTIADDR6(fl->fll_da)) {
      rt = dcache_check(info->dest, &fl->fll_da,
			sk->localroute||(flags&MSG_DONTROUTE));
      if (rt) {
	 info->dest = rt;
	 neigh      = rt->d_neigh;	/* Set neighbor */
	 dev        = rt->d_dev;	/* Set device   */
	 /* Set pmtu     */
      } 
   } else {
      DEBUG("[OUTPUT]sock %p multi? %d info->mc_if %p \n", sk,
	    IS_MULTIADDR6(fl->fll_da), info->mc_if);
      dev   = info->mc_if;
      mc_hl = info->mcast_hops;
      /* FIXME: pmtu */
   }
   
   /*
    * No route...
    */
   if ((rt == NULL) && (dev == NULL)) {
      ip6_statistics.IpOutNoRoutes++;
      DEBUG("[OUTPUT]ipv6_flow_xmit: OUT - No Route\n");
      return (-ENETUNREACH);
   }
   
   /*
    * Len ?
    */
   plen += sizeof(struct ipv6hdr);
   
   /*
    *      FIXME: option headers 
    */
   plen += fl->fll_opt.opt_srcrt_len;

   
     /*
    *      Packets meeting MTU requirements
    */
   if (plen <= dev->mtu) {
      int error;
      /* 
       *  Alloc sk_buff 
       */
      struct sk_buff *skb = sock_alloc_send_skb(sk,
				   plen + 15 + dev->hard_header_len,
						0, noblock, &error);
      if (skb == NULL) {
	 ip6_statistics.IpOutDiscards++;
	 return error;
      }
      
      skb->dev = dev;
      skb->protocol = htons(ETH_P_IPV6);
      skb->when = jiffies;
      skb->sk = sk;
      skb->nexthop = neigh;
      skb->arp = 0;
      /*
       *      Build hard header
       */
      ipv6_build_mac_header(skb, neigh, plen);
      
      /*
       *      Build IPv6 header
       */
      iph = (struct ipv6hdr *) skb_put(skb, plen);
      if(!iph) {
	 kfree_skb(skb, FREE_WRITE);
	 return -1;
      }
      
      skb->nh.ipv6h = iph;
      dev_lock_wait(); dev_lock_list();
      if (!sk->ip_hdrincl) {	/* IP header not included yet */
	 iph->vrprflow = IPV6_VERSION |
	     IPV6_SET_FLOWLABEL(fl->fll_label) |
	     IPV6_SET_PRIORITY(fl->fll_prio);
	 iph->payload = htons(plen - sizeof(struct ipv6hdr));
	 iph->next_hdr = protocol;
	 iph->hop_limit = (mc_hl ? mc_hl : info->hop_limit);
	 COPY_ADDR6(iph->saddr, fl->fll_sa);
	 COPY_ADDR6(iph->daddr, fl->fll_da);
      }
      /*
       *      Insert options.
       */
      DEBUG("[OUTPUT]ipv6_flow_xmit: add options\n");
      if(fl->fll_opt.opt_srcrt) {	    
	 __u8 *rthdr = (__u8 *)fl->fll_opt.opt_srcrt;
	 
	 DEBUG("[OUTPUT]ipv6_flow_xmit: IPV6_SRCRT\n");
	 /* Set new next hdr for routing header and IPv6 header */
	 *rthdr = protocol;
	 iph->next_hdr = IPPROTO_ROUTING;
	 /* Set addr */
	 
	 memcpy(iph + 1, fl->fll_opt.opt_srcrt, 
		fl->fll_opt.opt_srcrt_len);
      }
      
      /*
       *      Copy data from user space.
       */
      getfrag(frag, &fl->fll_sa, ((char *) iph) + plen - length, 0, length);
      dev_unlock_list();
      
      /*
       *      Send.
       */
      if (dev->flags & IFF_UP) {	/* interface is up: send */
	 DEBUG("FXMIT ==============================\n");
	 PRINT_ADDR(&(fl->fll_sa));
	 PRINT_ADDR(&(fl->fll_da));
	 DEBUG("FXMIT ==============================\n");
	 dev_queue_xmit(skb);
      } else {
	 ip6_statistics.IpOutDiscards++;
	 kfree_skb(skb, FREE_WRITE);
      }
      DEBUG("[OUTPUT]ipv6_flow_xmit: OUT\n");
      return 0;
   } else {
      /*
       *   Fragment
       */
      DEBUG("[OUTPUT]ipv6_build_xmit: invalid packet - 10-05-96\n");
      return -1;
   }
}

/*--START FUNCTION--(ipv6_build_xmit)------------------------------------------
| 
| ipv6_build_xmit: build header and call dev_queue_xmit()
| Last modified: 05/15/96
|     07/15/96:   Made routing operational... 
-----------------------------------------------------------------------------*/
int ipv6_build_xmit(struct sock *sk,
		    void getfrag(const void *, struct in6_addr *,
				 char *, unsigned int, unsigned int),
		    const void *frag,
		    unsigned short int length,	/* fragment's length */
		    struct in6_addr *d,		/* destination addr. */
		    struct in6_addr *us,	/* user source addr. */
		    struct options *opt,
		    int flags,
		    int protocol,
		    int noblock)
{
   struct destination *rt    = NULL;
   struct neighbor    *neigh = NULL;
   struct device      *dev   = NULL;
   struct ipv6hdr     *iph;
   struct ipv6_pinfo  *info  = &sk->net_pinfo.af_inet6;	/* from sock.h */
   struct in6_addr    *s     = NULL;	                /* source      */
   int plen = length;	   /* actual packet length, with hdr & options */
   int mc_hl = 0;

   DEBUG("[OUTPUT]ipv6_build_xmit: IN\n"); PRINT_ADDR(d);
   ip6_statistics.IpOutRequests++;	/*ipv6.h, snmp.h */

   /*
    *  Destination ?
    */
   if (!IS_MULTIADDR6(*d)) {
      rt = dcache_check(info->dest, d, sk->localroute||(flags&MSG_DONTROUTE));
      if (rt) {
	 info->dest = rt;
	 neigh      = rt->d_neigh;	/* Set neighbor */
	 dev        = rt->d_dev;	/* Set device   */
	 /* Set pmtu     */
      } 
   } else {
      DEBUG("[OUTPUT]sock %p multi? %d info->mc_if %p \n", sk,
	    IS_MULTIADDR6(*d), info->mc_if);
      dev   = info->mc_if;
      mc_hl = info->mcast_hops;
      /* FIXME: pmtu */
   } 
   /*
    * No route...
    */
   if ((rt == NULL) && (dev == NULL)) {
      ip6_statistics.IpOutNoRoutes++;
      DEBUG("[OUTPUT]ipv6_build_xmit: OUT - No Route\n");
      return (-ENETUNREACH);
   }
   
   /*
    * Source ?
    */
   if (us && !IS_MULTIADDR6(*us))
      s = us;
   else if (!IS_ANYADDR6(info->pi6_fl.fll_sa) &&
	    (!IS_LOOPADDR6(info->pi6_fl.fll_sa) || IS_LOOPADDR6(*d)))
      s = &info->pi6_fl.fll_sa;

   if (s == NULL) {
      struct inet6_ifaddr *ifa;

      if ((ifa = ipv6_get_saddr(dev, d))) {
	 COPY_ADDR6(info->pi6_fl.fll_sa, ifa->addr);
	 s = &info->pi6_fl.fll_sa;
      } else if (IS_MULTIADDR6(*d)) {
	 s = (struct in6_addr *) kmalloc(sizeof(struct in6_addr), GFP_KERNEL);
	 memset((char *) s, 0, sizeof(struct in6_addr));
      } else {
	 DEBUG("[OUTPUT]ipv6_build_xmit: OUT - Can't determine source address\n");
	 return (-ENETUNREACH);
      }
   }
   PRINT_ADDR(s);
   
   /*
    *      Now compute the buffer space we require
    */
   if (!sk->ip_hdrincl)
      plen += sizeof(struct ipv6hdr);
   /*
    *      FIXME: option headers 
    */
   if(opt) {
      plen += opt->opt_srcrt_len;
   }
   
   /*
    *      Packets meeting MTU requirements
    */
   if (plen <= dev->mtu) {
      int error;
      /* 
       *  Alloc sk_buff 
       */
      struct sk_buff *skb = sock_alloc_send_skb(sk,
				   plen + 15 + dev->hard_header_len,
						0, noblock, &error);
      if (skb == NULL) {
	 ip6_statistics.IpOutDiscards++;
	 return error;
      }
      
      skb->dev = dev;
      skb->protocol = htons(ETH_P_IPV6);
      skb->when = jiffies;
      skb->sk = sk;
      skb->nexthop = neigh;
      skb->arp = 0;
      /*
      COPY_ADDR6(skb->saddr6, *s);
      COPY_ADDR6(skb->daddr6, *d);
      COPY_ADDR6(skb->raddr6, (neigh ? neigh->naddr : *d));
      */
      
      /*
       *      Build hard header
       */
      ipv6_build_mac_header(skb, neigh, plen);
      
      /*
       *      Build IPv6 header
       */
      iph = (struct ipv6hdr *) skb_put(skb, plen);
      if(!iph) {
	 kfree_skb(skb, FREE_WRITE);
	 return -1;
      }
      
      skb->nh.ipv6h = iph;
      dev_lock_wait(); dev_lock_list();
      DEBUG("[OUTPUT]ipv6_build_xmit: vrprflow: %8X\n", iph->vrprflow);
      if (!sk->ip_hdrincl) {	/* IP header not included yet */
	 iph->vrprflow = IPV6_VERSION |
	     IPV6_SET_FLOWLABEL(info->pi6_fl.fll_label) |
	     IPV6_SET_PRIORITY(info->pi6_fl.fll_prio);
	 iph->payload = htons(plen - sizeof(struct ipv6hdr));
	 iph->next_hdr = protocol;
	 iph->hop_limit = (mc_hl ? mc_hl : info->hop_limit);
	 COPY_ADDR6(iph->saddr, *s);
	 COPY_ADDR6(iph->daddr, *d);
      }
      DEBUG("[OUTPUT]ipv6_build_xmit: vrprflow: %8X\n", iph->vrprflow);

      /*
       *      Insert options.
       */
      if (opt) {
	 DEBUG("[OUTPUT]ipv6_build_xmit: add options\n");
	 if(opt->opt_srcrt)
	    copy_from_user(iph + 1, opt->opt_srcrt, opt->opt_srcrt_len);
      }
      DEBUG("[OUTPUT]ipv6_build_xmit: vrprflow: %8X\n", iph->vrprflow);
	    
      /*
       *      Copy data from user space.
       */
      getfrag(frag, s, ((char *) iph) + plen - length, 0, length);
      if (IS_MULTIADDR6(iph->saddr)) {
	 kfree_skb(skb, FREE_WRITE);
	 dev_unlock_list();
	 return -1;
      }
      dev_unlock_list();
      DEBUG("[OUTPUT]ipv6_build_xmit: vrprflow: %8X\n", iph->vrprflow);

      /*
       *      Send.
       */
      if (dev->flags & IFF_UP) {	/* interface is up: send */
	 DEBUG("BXMIT ==============================\n");
	 PRINT_ADDR(&(skb->nh.ipv6h->saddr));
	 PRINT_ADDR(&(skb->nh.ipv6h->daddr));
	 DEBUG("BXMIT ==============================\n");
	 dev_queue_xmit(skb);
      } else {
	 ip6_statistics.IpOutDiscards++;
	 kfree_skb(skb, FREE_WRITE);
      }
#ifdef IPV6_DEBUG_OUTPUT
      printk(KERN_DEBUG "[OUTPUT]ipv6_build_xmit: OUT\n");
#endif
      return 0;
   } else {
      /*
       *   Fragment
       */
      printk(KERN_DEBUG "[OUTPUT]ipv6_build_xmit: invalid raw packet - 10-05-96\n");
      return -1;
   }
}
/*--END FUNCTION--(ipv6_build_xmit)-----------------------------------------*/

/*--START FUNCTION--(ipv6_rt_event)-----------------------------------------
|
| Description:
| 	Notifies any action on devices to interested routines.
| 	Inactive for now.
|
| Authors: 
|         nom
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
static int ipv6_rt_event(struct notifier_block *this,
			 unsigned long event, void *ptr)
{
   /* struct device *dev = ptr; */
   if (event == NETDEV_DOWN) {
      /*  ip_netlink_msg(RTMSG_DELDEVICE, 0,0,0,0,0,dev->name);
         ip_rt_flush(dev); */
   }
/*
 *      Join the intial group if multicast.
 */
   if (event == NETDEV_UP) {
      /*#ifdef CONFIG_IP_MULTICAST      
         ip_mc_allhost(dev);
         #endif          
         ip_netlink_msg(RTMSG_NEWDEVICE, 0,0,0,0,0,dev->name);
       */
   }
   return NOTIFY_DONE;
}
/*--END FUNCTION--(ipv6_rt_event)---------------------------------------*/

/*
 *    IP registers the packet type and then calls the subprotocol initialisers
 */

struct notifier_block ip6_rt_notifier =
{
   ipv6_rt_event,		/*  *notifier_call()   *//* TOHANDLE */
   NULL,			/*  *next */
   0				/* priority */
};


/*
 *      IP protocol layer initialiser
 */
static struct packet_type ipv6_packet_type =
{
   0,				/* MUTTER ntohs(ETH_P_IPV6), */
   NULL,			/* All devices */
   ipv6_rcv,
   NULL,
   NULL,
};

/*--START FUNCTION--(ipv6_init)-----------------------------------------
|
| Description:
|    ipv6 protocol initialization
|
| Authors: 
|         Brodard
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
void ipv6_init(void)
{
#ifdef IPV6_DEBUG_OUTPUT
   printk(KERN_DEBUG "************ IPV6_INIT **************IN\n");
#endif
   
   ipv6_packet_type.type = htons(ETH_P_IPV6);
   dev_add_pack(&ipv6_packet_type);
   /* 
    *   So that we flush routes when a device is downed 
    */
   register_netdevice_notifier(&ip6_rt_notifier);
   /*
    *   Address initialization (bound to change)
    */
   ipv6_addr_init();
   /* 
    *   Simple Interface Transition Initialization
    */
   sit_init();
   /*
    *   Neighbor Discovery initialization
    */
   nd_init();
#ifdef IPV6_DEBUG_OUTPUT
   printk(KERN_DEBUG "************ IPV6_INIT **************OUT\n");
#endif
}

/*--END FUNCTION--(ipv6_init)---------------------------------------*/
