/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:   $Id: ipv6_input.c,v 2.0 1997/01/23 12:20:37 eh Exp $ 
 *      
 * Authors:     
 *              Benoit Brodard  <Brodard@sophia.inria.fr>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|  ipv6_rcv
|
-------------------------------------------------------------------------*/


/*--START INCLUDE---------------------------------------------------------*/
#include <asm/segment.h>
#include <asm/system.h>
#include <asm/uaccess.h>

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/config.h>

#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>

#include <net/snmp.h>
#include <net/ip.h>
#include <net/protocol.h>
#include <net/route.h>
#include <net/tcp.h>
#include <net/udp.h>
#include <linux/skbuff.h>
#include <net/sock.h>
#include <net/icmp.h>
#include <net/checksum.h>
#include <linux/igmp.h>

#ifdef CONFIG_IP_MASQUERADE
#include <net/ip_masq.h>
#endif

#include <linux/firewall.h>
#include <linux/mroute.h>
#include <net/netlink.h>

#include <linux/dret/ipv6.h>
#include <linux/dret/ipv6_debug.h>

#include <net/dret/ipv6.h>
#include <net/dret/protocol6.h>
#include <net/dret/raw6.h>
#include <net/dret/addrconf.h>
#include <net/dret/options6.h>
#include <net/dret/flow6.h>
/*--END INCLUDE--------------------------------------------------------------*/


#ifdef IPV6_DEBUG_INPUT
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[INPUT]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#else
#define DEBUG(format, args...) 
#define PRINT_ADDR(a)
#endif


/*extern int last_retran;
   extern void sort_send(struct sock *sk); */

#define min(a,b)	((a)<(b)?(a):(b))

/*
 *    SNMP management statistics
 */
#ifdef CONFIG_IPV6_FORWARD
struct ip_mib ip6_statistics =
{1, 64,};			/* Forwarding=Yes, Default TTL=64 */
#else
struct ip_mib ip6_statistics =
{2, 64,};			/* Forwarding=No, Default TTL=64 */
#endif

void 
ipv6_raw_deliver(struct sk_buff *skb, __u8 next_hdr, struct sock ** raw_sk) {
   __u8 hash;		     
   struct ipv6hdr        *iph    = skb->nh.ipv6h;
   
   DEBUG("[INPUT]ipv6_raw_deliverIN next_hdr = %d\n", next_hdr);
   hash = next_hdr & (SOCK_ARRAY_SIZE - 1);
   /* 
    *  Is there a raw socket? 
    */
   if ((*raw_sk = raw6_prot.sock_array[hash]) != NULL) {
      struct sock *sknext = NULL;
      struct sk_buff *skb1 = NULL;
      *raw_sk = get_sock_raw6(*raw_sk, next_hdr, &iph->saddr, &iph->daddr);
      if (*raw_sk) {		/* At least one... */
	 DEBUG("Found 1st RAW socket\n");
	 do {
	    /* Others?... */
	    sknext = get_sock_raw6((*raw_sk)->next, next_hdr, &iph->saddr,
				   &iph->daddr);
	    if (sknext) {
	       DEBUG("Found an other one   \n");
	       skb1 = skb_clone(skb, GFP_ATOMIC);
	       if (skb1)
		  raw6_rcv(*raw_sk, skb1);	
	    } else
	       /* 
		* One pending raw socket left to 
		* avoid 1 copy.
		*/
	       break;
	    *raw_sk = sknext;
	 }
	 while (*raw_sk != NULL);
	 /*
	  *   Now, raw_sk is either the last raw socket or NULL. 
	  *   Will receive a copy AFTER the protocol checks
	  */
      }
   }
   DEBUG("[INPUT]ipv6_raw_deliver OUT\n");
} 

int ipv6_proto_deliver(struct sk_buff *skb, __u8 next_hdr, struct sock *raw_sk) {
   __u8 hash;
   int flag = 0;
   struct inet6_protocol *ipprot;
   struct ipv6hdr        *iph    = skb->nh.ipv6h;
   DEBUG("[INPUT]ipv6_proto_deliver IN\n");
   /*
    * Deliver to transport layer
    */
   hash = next_hdr & (MAX_INET_PROTOS - 1);
   for (ipprot = (struct inet6_protocol *) inet6_protos[hash];
	ipprot != NULL;
	ipprot = (struct inet6_protocol *) ipprot->next) {
      struct sk_buff *skb2;
      if (ipprot->protocol != next_hdr)
	 continue;
      /*
       *      Need to make a copy of it? 
       *      Set only if more than one protocol wants 
       *      it but the last one. 
       *      If there is a pending raw delivery wait for that
       */
      if (ipprot->copy || raw_sk) {
	 skb2 = skb_clone(skb, GFP_ATOMIC);
	 if (skb2 == NULL)
	    continue;
      } else {
	 skb2 = skb;
      }
      flag = 1;
      
      /*
       *     Pass on the datagram to that protocol 
       */
      DEBUG("[INPUT]ipv6_rcv: passed datagram"
	    "to protocol %d\n", next_hdr);
      ipprot->handler(skb2, skb->dev,  &iph->saddr, &iph->daddr, 
		      NULL /*opt */ , ntohs(iph->payload),  0, ipprot);
   }
   DEBUG("[INPUT]ipv6_proto_deliver OUT\n");
   return(flag);
}

/*--START FUNCTION--(ipv6_rcv)-------------------------------------------
| 
| ipv6_rcv: receives all incoming IP datagrams.On entry skb->data points
|           to the start of the IP header and the MAC header has been removed.
| Last modified: 05/07/96
| Author: B. Brodard
-----------------------------------------------------------------------------*/
int ipv6_rcv(struct sk_buff *skb, struct device *dev, struct packet_type *pt)
{
   struct sock           *raw_sk = NULL;	     /* raw socket         */
   struct inet6_protocol *ipprot;	             /* inet prot. struct. */
   struct ipv6hdr        *iph    = skb->nh.ipv6h;    /* ipv6 header        */
   struct in6_addr d;
   
   __u8 next_hdr;
   __u8 hash;
   __u8 flag = 0;
   __u32 flabel;
   int err;

   DEBUG("[INPUT]&&&&&&&&&&& dev = %s &&&&&&&&&&&&\n", dev->name);
   PRINT_ADDR(&iph->saddr);
   PRINT_ADDR(&iph->daddr);
   DEBUG("[INPUT]&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&\n");
   DEBUG("[INPUT]&&&& %p %p %d\n", 
	 skb->tail, iph, skb->tail - (unsigned char *) iph);
   
   /*
    *  MIB Accounting
    */
   ip6_statistics.IpInReceives++;
   
   /*
    *    Tag the ip header of this packet so we can find it
    */
   
   /*
    *  Is the datagram acceptable?
    *     1. Doesn't have a bogus length
    *     2. Version = 6.      
    *  for now...
    */
   if ((skb->len < (sizeof(struct ipv6hdr) + ntohs(iph->payload))) ||
        ((iph->vrprflow & IPV6_FLOWINFO_VERSION) != IPV6_VERSION)) {
      ip6_statistics.IpInHdrErrors++;
      kfree_skb(skb, FREE_WRITE);
      DEBUG("[INPUT]ipv6_rcv: OUT, IpInHdrErrors: %1X %1X\n", 
	    (skb->len < (sizeof(struct ipv6hdr) + ntohs(iph->payload))),
	    ((iph->vrprflow & IPV6_FLOWINFO_VERSION) != IPV6_VERSION));
      return (0);
   };

   /*
    *  Our transport medium may have padded the buffer out. Now we know it
    *  is IP we can trim to the true length of the frame.
    *  Note this now means skb->len holds ntohs(iph->tot_len).
    */
   /*  void skb_trim(struct sk_buff *skb, int datalen) 
    *  Sets the tail pointer to be datalen bytes after the data pointer. 
    *  The len field is set to datalen. Used to trim off information at 
    *  the end of the "useful data" region. 
    */
   skb_trim(skb, (ntohs(iph->payload) + sizeof(struct ipv6hdr)));

   /*
    * Labelled packet
    */
   DEBUG("[INPUT]ipv6_rcv: vrprflow: %8X\n", ntohl(iph->vrprflow));
   if ((flabel = IPV6_GET_FLOWLABEL(iph->vrprflow))) {
      flow6_rcv(skb, dev, flabel);
      return(0);
   }
   
   /*
    *    Packet's headed to an address of ours!
    */
   COPY_ADDR6(d, iph->daddr);
  
   if ((check_host_addr6(&d)) || IS_MULTIADDR6(d)) {	/* FIXME */
      __u8 stage = 0;
      __u8 done  = 1;
      __u8 *optr;
      
      skb->h.raw   += sizeof(struct ipv6hdr);
      skb->dev      = dev;
      next_hdr      = iph->next_hdr;
      optr          = skb->h.raw;
      /*
	___________   <-  nh.ipv6h
	| IPv6 hdr |
	|__________|  <-  h.raw
	|Opts,trspt|
	|          |
	*/
      /*
       * Scan options
       */
      skb->ip6_hopopt = NULL;             /* Set if option found and user */
      skb->ip6_dstopt1 = NULL;            /* asked for it                 */
      skb->ip6_srcrt = NULL;
      skb->ip6_dstopt2 =NULL;
      
      while (done) { 
	 switch(next_hdr) {
	 case IPPROTO_HOPOPTS:
	    DEBUG("[INPUT]ipv6_rcv: IPPROTO_HOPOPTS\n");
	    err = hopopts_handler(skb, optr); 
	    next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
	    stage ++;
	    break;
	 case IPPROTO_ROUTING:
	    DEBUG("[INPUT]ipv6_rcv: IPPROTO_ROUTING\n");
	    err = srcrt_handler(skb, optr, 0); 
	    if (err == -1) {
	       DEBUG("[INPUT]ipv6_rcv: IPPROTO_ROUTING - skb gone\n");
	       return(0);
	    }
	    next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
	    stage ++;
	    DEBUG("[INPUT]ipv6_rcv: IPPROTO_ROUTING - @ destination\n");
	    break;
	 case IPPROTO_FRAGMENT:
	    DEBUG("[INPUT]ipv6_rcv: IPPROTO_FRAGMENT not implemented\n"); 
	    next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
	    optr    += ((struct opt6_hdr *)optr)->oh_len; 
	    stage ++;
	    break;
	 case IPPROTO_ESP:
	    DEBUG("[INPUT]ipv6_rcv: IPPROTO_ESP not implemented\n"); 
	    next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr; 
	    optr    += ((struct opt6_hdr *)optr)->oh_len;
	    stage ++;
	    break;
	 case IPPROTO_AH:
	    DEBUG("[INPUT]ipv6_rcv: IPPROTO_AH not implemented\n"); 
	    next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
	    optr    += ((struct opt6_hdr *)optr)->oh_len; 
	    stage ++;
	    break;
	 case IPPROTO_DSTOPTS:
	    DEBUG("[INPUT]ipv6_rcv: IPPROTO_DSTOPTS not implemented\n"); 
	    next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
	    optr    += ((struct opt6_hdr *)optr)->oh_len; 
	    stage ++;
	    break;
	 default:
	    DEBUG("[INPUT]ipv6_rcv: no option found\n");
	    done = 0;
	 }
      } 
      
      /*
       *    Deliver datagram
       */
      /* To raw sockets */
      ipv6_raw_deliver(skb, next_hdr, &raw_sk); 
      /* To protocol */
      flag = ipv6_proto_deliver(skb, next_hdr, raw_sk);
           
      /*
       *    All protocols checked.
       */
      if (raw_sk)
	 raw6_rcv(raw_sk, skb);	
      else if (!flag) {
	 /*
	  * If this packet was a broadcast, 
	  * we may *not* ICMP-reply to it
	  */
	 DEBUG("[INPUT]ipv6_rcv: Nothing reached...%d\n", iph->next_hdr);
      }
      DEBUG("[INPUT]ipv6_rcv: OUT\n");
      return (0);
   }
   
   /*
    * Packet's not addressed to us. Forward... 
    */
#ifdef CONFIG_IPV6_FORWARD 
   if (ipv6_fwd(skb, dev, 0, &d, 0))
      kfree_skb(skb, FREE_WRITE);
#else
   ip_statistics.IpInAddrErrors++;
   kfree_skb(skb, FREE_WRITE);
   DEBUG("[INPUT]ipv6_rcv: OUT, Error in dest addr");
#endif				/* CONFIG_IPV6_FORWARD */

   DEBUG("[INPUT]ipv6_rcv: OUT\n");
   return (0);
}























