/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$ 
 *      
 * Authors:     
 *              Benoit Brodard  <Brodard@sophia.inria.fr>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|  ipv6_fwd
|
-------------------------------------------------------------------------*/

/* TODO
   01/97: bb
   -Handle extension headers.
   -Multicast Routing.

 */


#include <linux/config.h>

#include <linux/types.h>
#include <linux/mm.h> 
#include <linux/sched.h>
#include <linux/in.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/dret/nd.h>
#include <linux/dret/ipv6.h>
#include <linux/dret/icmpv6.h>
#include <linux/dret/ipv6_debug.h>

#include <net/checksum.h>

#include <net/dret/ipv6.h>
#include <net/dret/icmpv6.h>
#include <net/dret/route6.h>


#ifdef IPV6_DEBUG_FWD
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[FWD]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#define PRINT_ETH_ADDR(a) printk(KERN_DEBUG "[FWD]%2X:%2X:%2X:%2X:%2X:%2X\n",\
				 a[0], a[1], a[2], a[3], a[4], a[5]);
#else
#define DEBUG(format, args...) 
#define PRINT_ADDR(a)
#define PRINT_ETH_ADDR(a)
#endif

#ifdef CONFIG_IPV6_FORWARD

/*
 *    Forward an IPv6 packet to its next destination.
 * return: -1	forward can't be do (error)
 *          0   packet is still in the same skb
 *	    1   packet is send via a new skb (different type of device)
 */
int ipv6_fwd(struct sk_buff *skb, struct device *dev, int is_frag,
	     struct in6_addr *a, __u8 flags)
{
   struct device *dev2;		/* Output device */
   struct ipv6hdr *hdr;		/* Our header */
   struct sk_buff *skb2 = NULL;	/* Output packet */
   struct destination *rt;	/* Route we use */
   unsigned char *ptr;		/* Data pointer */
   struct neighbor *neigh;
   unsigned int flow;
   int err;
   
   /*
    *     According to the RFC, we must first decrease the TTL field. If
    *     that reaches zero, we must reply an ICMP control message telling
    *     that the packet's lifetime expired.
    *
    *     Exception:
    *      We may not generate an ICMP for an ICMP. icmp_send does the
    *      enforcement of this so we can forget it here. It is however
    *      sometimes VERY important.
    */

   DEBUG("[FWD]ipv6_fwd: IN addr =\n");
   PRINT_ADDR(a);
   
   if (IS_LINKLADDR6(*a)) {
      DEBUG("[FWD]ipv6_fwd: OUT can't forward link local\n");
      return -1;
   }
   hdr = skb->nh.ipv6h;
   hdr->hop_limit--;		/* Any option that says "Do not decrease?" */

   if (hdr->hop_limit <= 0) {
      /* Tell the sender its packet died... */
      icmpv6_send_error(skb, ICMP6_TIMXCEED,
			ICMP6_TIMXCEED_INTRANS, 0, dev);
      DEBUG("[FWD]ipv6_fwd: OUT Time exceeded\n");
      return -1;
   }
   
   if (IS_MULTIADDR6(hdr->saddr)) {
      printk(KERN_DEBUG "[FWD]ipv6_fwd: OUT saddr is multicast!\n");
      return -1;		/* FIXME */
   }
   
#ifdef CONFIG_IPV6_FLOW_LABEL
   if (flow = IPV6_GET_FLOWLABEL(hdr->vrprfl)) 
      err = flow_handle(skb, &hdr->saddr, flow); 
      return err;
#endif
   /*
    * OK, packet still valid.  Fetch its destination address,
    * and give it to the IP sender for further processing.
    */
   rt = dcache_check(NULL, &hdr->daddr, 0);

   if (rt == NULL) {
      /*
       * Tell the sender its packet cannot be delivered. Again
       *   ICMP is screened later.
       */
      icmpv6_send_error(skb, ICMP6_UNREACH,
			ICMP6_UNREACH_NOROUTE, 0, dev);
      return -1;
   }
   /* Check strict routing option */

   /*
    * Having picked a route we can now send the frame out
    */
   dev2 = rt->d_route.rt_dev; 
   neigh = rt->d_route.rt_neigh; 

#if 0				/* ndef CONFIG_IP_NO_ICMP_REDIRECT */
   COPY_ADDR6(raddr6, (neigh ? neigh->naddr : rt->route.rd)); 
   if (dev == dev2 &&
       !((hdr->saddr ^ dev->pa_addr) & dev->pa_mask) &&
       hdr->daddr != raddr6)
      icmp_send(skb, ICMP_REDIRECT, ICMP_REDIR_HOST, raddr6, dev);
#endif

   /* What about firewall and masquerade ? */

   /*
    * We now may allocate a new buffer, and copy the datagram into it.
    * If the indicated interface is up and running, kick it.
    */

   if (!(dev2->flags & IFF_UP)) {
  	    printk(KERN_DEBUG "ERROR forward packet to a device down\n");
       if (rt) {
	 atomic_dec(&(rt->d_ref));
	 return -1;
       }
       return 1;
   }
   
      IS_SKB(skb);
      
      if (skb->len > dev2->mtu) {
#ifdef IPV6_DEBUG_FORWARD	 
	 printk(KERN_DEBUG "[FWD]skb->len > dev2->mtu \n");
#endif
	 /*
	  * No fragmentation on IPv6 router
	  */

	 ip6_statistics.IpFragFails++;
	 icmpv6_send_error(skb, ICMP6_PKTTOOBIG, 0,
			   htonl(dev2->mtu), dev);
	 if (rt)
	    atomic_dec(&(rt->d_ref));
	 return -1;
      }
      
      if (skb_headroom(skb) < dev2->hard_header_len) {
      
      	 /* 
	  *    Build a new skbuff
	  */

	 DEBUG("[FWD]Not enough room for new hh... \n");	 
	 skb2 = alloc_skb(dev2->hard_header_len + skb->len + 15,
			  GFP_ATOMIC);
	 /*
	  *   This is rare and since IP is tolerant of network failures
	  *   quite harmless.
	  */

	 if (skb2 == NULL) {
	    DEBUG("[FWD6]ipv6_fwd: No memory available\n");
	    if (rt)
	       atomic_dec(&(rt->d_ref));
	    return -1;
	 }
	 
	 IS_SKB(skb2);
	 /*
	  *    Add the physical headers.
	  */
	 skb2->dev = dev2;
	 skb2->protocol = htons(ETH_P_IPV6);
	 skb2->when = jiffies;
	 skb2->nexthop = neigh;
	 skb2->priority = SOPRI_NORMAL;
	         
	 /* FIXME */
	 ipv6_build_mac_header(skb2, neigh, skb->len);

	 /*
	  *We have to copy the bytes over as the new header wouldn't fit
	  *      the old buffer. This should be very rare.
	  */

	 ptr = skb_put(skb2, skb->len);
	 skb2->nh.raw = skb2->h.raw = ptr;
	 /*
	  *    Copy the packet data into the new buffer.
	  */
	 memcpy(ptr, skb->h.raw, skb->len);
	 
      } else { 
         /* room available in the skbuff is enough for the new header */
	 /* 
	  *    Build a new MAC header. 
	  */

	 skb->dev = dev2;
	 skb->nexthop = neigh;
	 skb->arp = 1;
	 skb->priority = SOPRI_NORMAL;
	 skb2 = skb;
	 
	 /* FIXME  (hh)*/
	 if (neigh && (neigh->n_flags & NCF_HHVALID)) {
	    DEBUG("[FWD]Have a valid neigh...\n");
	    memcpy(skb_push(skb, dev2->hard_header_len),
		   neigh->n_hh_data, 
		   dev2->hard_header_len);  
	 } else {
	    if (dev2->hard_header) {
	       skb_push(skb, dev2->hard_header_len);	       
	       if (dev2->hard_header(skb, dev2, ETH_P_IPV6,
				     NULL, NULL, skb->len)) {
		  /* At that point skb has been freed */
		  DEBUG("[FWD]Couldn't dev2->hard_header\n");
		  return 0;
	       }
	       
	    } /* dev2->hard_header == NULL */
	 } /* Mac header built */

	 ip6_statistics.IpForwDatagrams++;

	 /* 
	  *  Any options? Extension headers handling ? 
	  */

      } /* skbuff is ready */
      
      DEBUG("[FWD]going to dev_queue_xmit... \n");
      
      dev_queue_xmit(skb2);
      
     if (rt)
	    atomic_dec(&(rt->d_ref));
     
   /*
    *    Tell the caller if their buffer is free.
    */
    DEBUG("[FWD]ipv6_fwd: OUT -> %d\n", (skb == skb2 ? 0 : 1));    
    return (!(skb == skb2));
}
#endif

