/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:     $Id$ 
 *      
 * Authors:     
 *              Pedro Roque             <roque@di.fc.ul.pt>     
 *
 *
 * Fixes:
 *
 *
 * Description:
 *              IPv6 Address [auto]configuration
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|  Function_name
|
-------------------------------------------------------------------------*/


#include <asm/segment.h>
#include <asm/system.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <net/ip.h>
#include <net/protocol.h>
#include <net/route.h>
#include <linux/skbuff.h>
#include <net/sock.h>
#include <linux/igmp.h>
#include <net/checksum.h>

#include <linux/dret/netdevice6.h>
#include <net/dret/addrconf.h>

/* 
#define IPV6_DEBUG_IGMP 
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[OUTPUT]%X:%X:%X:%X \n",\
		      (a)->s6_addr32[0], (a)->s6_addr32[1], \
                      (a)->s6_addr32[2], (a)->s6_addr32[3]);
*/

/*--START FUNCTION--(igmpv6_init_timer)-------------------------------------
 |
 |
 ---------------------------------------------------------------------------*/
static void igmpv6_init_timer(struct ipv6_mc_list *mc)
{
   mc->tm_running = 0;
   init_timer(&mc->timer);
   mc->timer.data = (unsigned long) mc;
   mc->timer.function = NULL;	/* FIXME */
}

/*--END FUNCTION--(igmpv6_init_timer)---------------------------------------*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

 *                          Multicast lists managers
 *
 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/*--START FUNCTION--(ipv6_dev_mc_inc)----------------------------------
 |
 |	Join a mcast address.
 |
 ------------------------------------------------------------------------*/
int ipv6_dev_mc_inc(struct device *dev, struct in6_addr *addr)
{
   struct ipv6_mc_list *mc;
   struct inet6_dev *idev;
   char buf[6];
   __u8 hash;

#ifdef IPV6_DEBUG_IGMP
   printk("[IGMP]ipv6_dev_mc_inc: IN dev = %s\n", dev->name);
   PRINT_ADDR(addr);
#endif

   if ((idev = dev->dev6_lst) == NULL) {
      printk("[IGMP]ipv6_dev_mc_inc: Device not found... \n");
      return -EINVAL;
   }
   for (mc = idev->mc_list; mc; mc = mc->if_next) {
      if (SAME_ADDR6(mc->addr, *addr)) {
	 atomic_inc(&mc->users);
	 return 0;
      }
   }

   /*
    *  No ipv6_mc_list structure found for this device| address
    *  combination...
    */
   if ((mc = (struct ipv6_mc_list *)
	kmalloc(sizeof(struct ipv6_mc_list), GFP_KERNEL)) == NULL)
       return -ENOMEM;

   memset(mc, 0, sizeof(struct ipv6_mc_list));
   memcpy(&mc->addr, addr, sizeof(struct in6_addr));
   mc->dev = dev;
   mc->users = 1;

   /* Add entry to global list of mcast addresses */
   hash = ipv6_addr_hash(addr);
   mc->next = inet6_mcast_lst[hash];
   inet6_mcast_lst[hash] = mc;

   /* Add entry to device list of mcast addresses */
   mc->if_next = idev->mc_list;
   idev->mc_list = mc;

   igmpv6_init_timer(mc);
   ipv6_mc_map(addr, buf);
   /* Add link layer multicast address */
   dev_mc_add(dev, buf, ETH_ALEN, 0);
#ifdef IPV6_DEBUG_IGMP
   printk("[IGMP]ipv6_dev_mc_inc: OUT");
   /* dump_hex(buf, ETH_ALEN);*/
#endif
   return 0;
}
/*--END FUNCTION--(ipv6_dev_mc_inc)------------------------------------*/

/*--START FUNCTION--(ipv6_dev_mc_dec)----------------------------------
 |
 |	A socket has left a multicast group on device dev
 |
 ----------------------------------------------------------------------*/
int ipv6_dev_mc_dec(struct device *dev, struct in6_addr *addr)
{
   struct ipv6_mc_list *mc1, *mc1_prev = NULL, *mc2, *mc2_prev = NULL;
   struct inet6_dev *idev;
   char buf[6];
   __u8 hash;

   if ((idev = dev->dev6_lst) == NULL) {
      printk("[IGMP]ipv6_dev_mc_dec: Device not found... \n");
      return -EINVAL;
   }
   for (mc1 = idev->mc_list; mc1; mc1 = mc1->if_next) {
      if (SAME_ADDR6(mc1->addr, *addr)) {
	 atomic_dec(&mc1->users);
	 if (mc1->users == 0) {	/* FIXME */
	    del_timer(&mc1->timer);
	    /* Forget info stuff */
	    ipv6_mc_map(addr, buf);
	    dev_mc_delete(dev, buf, ETH_ALEN, 0);
	    if (mc1_prev == NULL)
	       idev->mc_list = mc1->if_next;
	    else
	       mc1_prev->if_next = mc1->if_next;
	    /* Unlink in device list */
	    hash = ipv6_addr_hash(&mc1->addr);
	    for (mc2 = inet6_mcast_lst[hash]; mc2; mc2 = mc2->next) {
	       if (mc2 == mc1)
		  break;
	       mc2_prev = mc2;
	    }
	    if (mc2) {
	       if (mc2_prev == NULL) {
		  inet6_mcast_lst[hash] = mc2->next;
	       } else {
		  mc2_prev->next = mc2->next;
	       }
	    }
	    kfree_s(mc1, sizeof(struct ipv6_mc_list));
	 }
	 return 0;
      }
      mc1_prev = mc1;
   }
   return 0;			/* FIXME, should be NOADDR or somethn' */
}
/*--END FUNCTION--(ipv6_dev_mc_dec)---------------------------------------*/

/*--START FUNCTION--(ipv6_mc_drop_device)---------------------------------
 |
 |  Device going down: Clean up.
 |
 --------------------------------------------------------------------------*/
int ipv6_mc_drop_device(struct device *dev)
{
   struct ipv6_mc_list *mc1, *mc2, *mc2_prev = NULL;
   struct inet6_dev *idev;
   char buf[6];
   __u8 hash;

   if ((idev = dev->dev6_lst) == NULL) {
      printk("[IGMP]ipv6_dev_mc_dec: Device not found... \n");
      return -ENODEV;
   }
   /* Clean out all multicast addresses attached to the device */
   for (mc1 = idev->mc_list; mc1; mc1 = idev->mc_list) {
      ipv6_mc_map(&mc1->addr, buf);
      dev_mc_delete(dev, buf, ETH_ALEN, 0);
      /* Unlink in device list */
      idev->mc_list = mc1->if_next;
      /* Unlink in global list */
      hash = ipv6_addr_hash(&mc1->addr);
      for (mc2 = inet6_mcast_lst[hash]; mc2; mc2 = mc2->next) {
	 if (mc2 == mc1)
	    break;
	 mc2_prev = mc2;
      }

      if (mc2) {		/* should always happen! */
	 if (mc2_prev == NULL)
	    inet6_mcast_lst[hash] = mc1->next;
	 else
	    mc2_prev->next = mc1->next;
      }
      kfree_s(mc1, sizeof(ipv6_mc_list));
   }
   idev->mc_list = NULL;
   return 0;
}
/*--END FUNCTION--(ipv6_mc_drop_device)----------------------------------*/

/*--START FUNCTION--(ipv6_mc_join_group)--------------------------------
 |
 |	Join a socket to a group
 |
 -------------------------------------------------------------------------*/
int ipv6_sock_join_group(struct sock *sk, struct device *dev,
			 struct in6_addr *addr)
{
   int err;
   struct ipv6_mc_socklist *mc_list, *l;
   struct ipv6_pinfo *pinfo = &sk->net_pinfo.af_inet6;

   /* Multicast address ? */
   if (!IS_MULTIADDR6(*addr))
      return -EINVAL;
   /* Multicast enabled on device ? */
   if (!(dev->flags & IFF_MULTICAST))
      return -EADDRNOTAVAIL;

   for (l = pinfo->mc_list; l; l = l->next) {
      if (SAME_ADDR6(l->addr, *addr)
	  && (l->dev == dev))
	 return -EADDRINUSE;
   }

   if ((mc_list = (struct ipv6_mc_socklist *)
	kmalloc(sizeof(*mc_list), GFP_KERNEL)) == NULL)
      return -ENOMEM;

   memcpy(&mc_list->addr, addr, sizeof(struct in6_addr));
   mc_list->dev = dev;
   mc_list->next = NULL;

   if ((err = ipv6_dev_mc_inc(dev, addr))) {
      kfree(mc_list);
      return err;
   }
   /* Insert at head of list */
   mc_list->next = pinfo->mc_list;
   pinfo->mc_list = mc_list;
   return 0;
}
/*--END FUNCTION--(ipv6_sock_join_group)---------------------------*/

/*--START FUNCTION--(ipv6_sock_leave_group)---------------------------
 |
 |	Ask a socket to leave a group.
 |
 --------------------------------------------------------------------*/
int ipv6_sock_leave_group(struct sock *sk, struct device *dev,
			  struct in6_addr *addr)
{
   struct ipv6_mc_socklist *mc = NULL, *l;
   struct ipv6_pinfo *pinfo = &sk->net_pinfo.af_inet6;

   if (!IS_MULTIADDR6(*addr))
      return -EINVAL;
   if (!(dev->flags & IFF_MULTICAST))
      return -EADDRNOTAVAIL;
   if (pinfo->mc_list == NULL)
      return -EADDRNOTAVAIL;
   for (l = pinfo->mc_list; l; l = l->next) {
      if (SAME_ADDR6(l->addr, *addr) && (l->dev == dev)) {
	 if (mc == NULL)
	    pinfo->mc_list = l->next;
	 else
	    mc->next = l->next;
	 ipv6_dev_mc_dec(dev, addr);
	 return 0;
      }
      mc = l;
   }
   return -EADDRNOTAVAIL;
}

/*--START FUNCTION--(ipv6_mc_sock_drop)-------------------------------------*/
void ipv6_mc_sock_drop(struct sock *sk)
{
   struct ipv6_mc_socklist *mc1, *mc2;
   struct ipv6_pinfo *pinfo = &sk->net_pinfo.af_inet6;

   if (pinfo->mc_list == NULL)
      return;

   mc2 = NULL;
   for (mc1 = pinfo->mc_list; mc1; mc1 = mc2) {
      ipv6_dev_mc_dec(mc1->dev, &mc1->addr);
      mc2 = mc1->next;
      kfree_s(mc1, sizeof(*ipv6_mc_socklist));
   }
   kfree_s(pinfo->mc_list, sizeof(*ipv6_mc_socklist));
   pinfo->mc_list = NULL;
}

/*--START FUNCTION--(find_addr6_mcast)-------------------------------------*/
int find_addr6_mcast(struct device *dev, struct in6_addr *addr)
{
   struct ipv6_mc_list *mc;

   for (mc = dev->dev6_lst->mc_list; mc; mc = mc->if_next)
      if (SAME_ADDR6(mc->addr, *addr)) {
	 return 1;
      }
   return 0;
}

/*--START FUNCTION--(ipv6_mc_init)------------------------------------------
 |
 |  Initialisation of mc addresses when device goes up.
 |
 |--------------------------------------------------------------------------*/
void ipv6_mc_init(struct device *dev)
{
   struct in6_addr addr;
#ifdef IPV6_DEBUG
   printk("[IGMP]ipv6_mc_init: IN\n");
#endif
   /* FIXME: Make that clean... */
   /*
    * Init local link multicast address
    */
   addr.s6_addr32[0] = __constant_htonl(0xFF020000);
   addr.s6_addr32[1] = addr.s6_addr32[2] = 0;
   addr.s6_addr32[3] = __constant_htonl(0x00000001);
   ipv6_dev_mc_inc(dev, &addr);

   /*
    * Init local host multicast address
    */
   addr.s6_addr32[0] = __constant_htonl(0xFF010000);
   addr.s6_addr32[1] = addr.s6_addr32[2] = 0;
   addr.s6_addr32[3] = __constant_htonl(0x00000001);
   ipv6_dev_mc_inc(dev, &addr);
#ifdef IPV6_DEBUG
   printk("[IGMP]ipv6_mc_init: OUT\n");
#endif
}
