/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$ 
 *      
 * Authors:     
 *              name    <E_mail>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */
/*      NetBSD: ip_icmp.c,v 1.10 1994/06/29 06:38:16 cgd Exp              */

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/fcntl.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/timer.h>
#include <linux/skbuff.h>

#include <linux/dret/icmpv6.h>
#include <linux/dret/netdevice6.h>
#include <linux/dret/ipv6_debug.h>

#include <asm/system.h>
#include <asm/segment.h>

#include <net/snmp.h>
#include <net/checksum.h>
#include <net/sock.h>

#include <net/dret/ipv6.h>
#include <net/dret/route6.h>
#include <net/dret/protocol6.h>
#include <net/dret/icmpv6.h>
#include <net/dret/addrconf.h>
#include <net/dret/nd.h>


#define PRINT_ADDR(a) printk(KERN_DEBUG  "[ICMPv6]%X:%X:%X:%X \n",\
		      (a)->s6_addr32[0], (a)->s6_addr32[1], \
                      (a)->s6_addr32[2], (a)->s6_addr32[3]);


/* 
 * Fragment to be passed to ipv6_build_xmit 
 */

struct icmpv6_bxm {
   void *data_ptr;
   int data_len;
   struct icmpv6_error hdr;
   struct in6_addr *d;
   unsigned long csum;
};

struct icmp_mib icmp6_statistics;	/* ICMPv6 statistics */

unsigned long dummy;
struct socket icmpv6_socket;	/* socket used by ICMPv6 to send msgs */


/*--START STRUCTURE--(icmpv6_control)-------------------------------------*/
/* NOT USED */
struct icmpv6_control {
   unsigned long *ouptut;	/* address to increment on output */
   unsigned long *input;	/* address to increment on input */
   void (*handler) (struct icmpv6 * hdr,
		    struct sk_buff skb,
		    struct device * dev,
		    struct in6_addr * s,
		    struct in6_addr * d,
		    int len);
   unsigned long error;
   /* struct icmpv6_xrlim *xrlim; */
};

/* static struct icmpv6_control icmpv6_pointers[19]; */
/*--END STRUCTURE--(icmpv6_control)-------------------------------------*/

/*--START FUNCTION--(get_icmpv6_frag)------------------------------------
|
| Send ICMPv6 packets: the skbuff already contains an ipv6 header.
|
| Called from ipv-_output.
-------------------------------------------------------------------------*/
static void get_icmpv6_frag(const void *p, struct in6_addr *s, char *to, unsigned int offset, unsigned int fraglen)
	/* *p is in fact a skbuff */
{
   unsigned int csum;
   struct icmpv6 *hdr = (struct icmpv6 *) to;

#ifdef IPV6_DEBUG_ICMP2
   printk(KERN_DEBUG "\n get_frag IN\n");
#endif	/* IPV6_DEBUG_ICMP2 */
   memcpy(to, ((char *) (((struct sk_buff *) p)->h.raw)),
	  fraglen /* + optlen */ );
   /* Take care of options later */
   hdr->icmp6_type = ICMP6_ECHOREPLY;
   csum = csum_partial(to, fraglen, 0);
   hdr->icmp6_cksum = csum_ipv6_magic(s,
			      &(((struct sk_buff *) p)->nh.ipv6h->daddr),
				      fraglen, IPPROTO_ICMPV6, csum);
   /* kfree_skb((struct sk_buff *)p, FREE_READ);  check... */
#ifdef IPV6_DEBUG_ICMP2
   printk(KERN_DEBUG "get_frag OUT\n");
#endif	/* IPV6_DEBUG_ICMP2 */
   return;
}

/*--END FUNCTION--(get_icmpv6_frag)------------------------------------------*/

/*--START FUNCTION--(icmpv6_reflect)------------------------------------
|
| Reflects ICMPv6 packets
| Called from icmpv6_rcv.
-------------------------------------------------------------------------*/
static void icmpv6_reflect(struct sk_buff *skb, struct device *dev)
{
   struct sock *sk = (struct sock *) (icmpv6_socket.sk);
   struct ipv6hdr *iphdr = skb->nh.ipv6h;
   struct icmpv6 *icmphdr = (struct icmpv6 *) skb->h.raw;
   struct destination *dest = NULL;
/*, **rt_cache = NULL; */
   struct in6_addr t;
   /*   register struct inet6_dev *idev = NULL;
      register struct inet6_ifaddr *ia = NULL; */

   unsigned int len;
   int hdrlen;

#ifdef IPV6_DEBUG_ICMP
   printk(KERN_DEBUG "[ICMPV6]icmpv6_reflect  IN\n");
#endif

/*
 * Return address
 */
   if (IS_ANYADDR6(iphdr->saddr) || IS_MULTIADDR6(iphdr->saddr)) {
#ifdef IPV6_DEBUG_ICMP      
      printk(KERN_DEBUG "[ICMPv6]icmpv6_reflect: bad return address \n");
      PRINT_ADDR(&iphdr->saddr);
#endif
      kfree_skb(skb, FREE_READ);
      return;
   }
   COPY_ADDR6(t, iphdr->daddr);
   COPY_ADDR6(iphdr->daddr, iphdr->saddr);
/*
 * Options
 */

/*
 * Source address
 */
/* Find inet6_dev struct */
/* We may add a pointer in the device structure... */
/*printk(KERN_DEBUG "Before device...\n");
   for (idev = inet6_dev_lst; idev; idev = idev->next)
   {
   if (idev->dev == dev)
   break;
   } */
/* Find inet6_ifaddr struct */
/*printk(KERN_DEBUG "Before address... \n);
   for (ia = idev->addr_list; ia; ia = ia->next)
   {
   if (SAME_ADDR6(t, ia->addr))
   break;
   }
 */
#ifdef IPV6_DEBUG_ICMP2
   if (ia == NULL)
      printk(KERN_DEBUG "Everything normal. ia==NULL \n");
#endif	/* IPV6_DEBUG_ICMP2 */
/* FIXME
   ia = ipv6_get_saddr(dev, &t); 
   if (ia == NULL)
   {
   printk(KERN_DEBUG "[ICMPV6]icmpv6_reflect: no source address...\n");
   #ifdef IPV6_DEBUG_ICMP
   printk(KERN_DEBUG "[ICMPV6]icmpv6_reflect  OUT\n");
   #endif 
   return;
   }
   COPY_ADDR6(iphdr->saddr, ia->addr); */
   COPY_ADDR6(iphdr->saddr, t);
   iphdr->hop_limit = MAXTTL;
/*
 * Turn off flags (broadcast, multicast)?
 */
/*
 * Check PMTU
 */
/*dest = ipv6_dst_route(&iphdr->daddr, 0); */
   if (dest == NULL) {
/* FIXME       return; */
   }
   len = skb->tail - skb->h.raw - sizeof(struct icmpv6);
   hdrlen = sizeof(struct ipv6hdr) + sizeof(struct icmpv6);
   /* + optlen */
   /*
    * Compare payload and MTU.
    * Reduce ICMP content if necessary.
    * Discard if too small.
    */
/* FIXME    if (len + hdrlen > dest->pmtu)
   len = dest->pmtu - hdrlen; */
   if (len < 0) {
#ifdef IPV6_DEBUG_ICMP
      printk(KERN_DEBUG "[ICMPV6]icmpv6_reflect: MTU too small  OUT\n");
#endif
      return;
   }
/*
 * Compute checksum
 */
   icmphdr->icmp6_cksum = 0;
   len += sizeof(struct icmpv6);
/* 
 * We now have a skbuff containing both updated icmpv6 and ipv6 header 
 * but no checksum...
 * The question is: what about hh and available size in sk_buff?
 * This is why we need to allocate a new skbuff and go through
 * ipv6_build_xmit... Sigh...
 */
   ipv6_build_xmit(sk, get_icmpv6_frag, skb, len, &iphdr->daddr,
		 &iphdr->saddr, NULL /*offset */ , 0, IPPROTO_ICMPV6, 1);
   kfree_skb(skb, FREE_READ);
#ifdef IPV6_DEBUG_ICMP
   printk(KERN_DEBUG "[ICMPV6]icmpv6_reflect  OUT\n");
#endif
   return;
}
/*--END FUNCTION--(icmpv6_reflect)------------------------------------------*/

/*
   if !(dest = ipv6_check_dest(rt_cache, &hdr->saddr, 0))
   {
   printk(KERN_DEBUG "[ICMPv6]icmpv6_reflect: no route to host \n");
   return;
   }
   if (skb->tail - skb->ipv6_hdr) > dest->pmtu)
   {
   hdrlen=sizeof(ipv6hdr)+sizeof(icmpv6); 
   len = dest->pmtu - hdrlen;
   if !(len) return;

   } */

/*--START FUNCTION--(getfrag_icmpv6_error)--------------------------------
 |
 |	
 |
 |   Last update:
 |   08/29/96: bb
 ---------------------------------------------------------------------*/
static void getfrag_icmpv6_error(const void *p, struct in6_addr *s, char *to,
			       unsigned int offset, unsigned int fraglen)
{
   struct icmpv6_bxm *frag = (struct icmpv6_bxm *) p;
   struct icmpv6_hdr *hdr;
   unsigned long csum;

   if (offset) {
      frag->csum = csum_partial_copy(
		   frag->data_ptr + offset - sizeof(struct icmpv6_error),
				       to, fraglen, frag->csum);
      return;
   }
   /*
    *    First fragment includes header. Note that we've done
    *      the other fragments first, so that we get the checksum
    *      for the whole packet here.
    */
   csum = csum_partial_copy((void *) &frag->hdr,
			    to, sizeof(struct icmpv6_error),
			    frag->csum);
   csum = csum_partial_copy(frag->data_ptr,
			    to + sizeof(struct icmpv6_error),
			    frag->data_len, csum);
   hdr = (struct icmpv6_hdr *) to;
   hdr->csum = csum_ipv6_magic(s, frag->d, fraglen, IPPROTO_ICMPV6, csum);
}

/*--START FUNCTION--(icmpv6_build_xmit)--------------------------------
 |
 |	Driving logic for building and sending ICMPv6 messages.
 |
 |   Last update:
 |   08/29/96: bb Now unused.
 -------------------------------------------------------------------
static void icmpv6_build_xmit(struct icmpv6_bxm *frag, 
			      struct in6_addr *s, 
			      struct in6_addr *d)
{
	struct sock *sk=icmpv6_socket.data;
}
---------------------------------------------------*/
/*--START FUNCTION--(icmpv6_send_error)-------------------------------------
|
| Send icmpv6 error packets.
| 
| Last update:
| 08/29/96: bb
---------------------------------------------------------------------*/
void icmpv6_send_error(struct sk_buff *skb, __u8 type, __u8 code,
		       __u32 info, struct device *dev)
{
   struct icmpv6_hdr *icmp_hdr;
   struct icmpv6_bxm frag;
   struct inet6_ifaddr *ifa;
   struct in6_addr *a;
   struct sock *sk = icmpv6_socket.sk;
   struct ipv6hdr *ip_hdr = skb->nh.ipv6h;
   int len1, len2, room;

   a = &ip_hdr->daddr;
   
   if (IS_MULTIADDR6(*a) && 
       (type != ICMP6_PKTTOOBIG) &&
       (code != ICMP6_PARAMPROB_OPT))        
      return;
   
   if (IS_ANYADDR6(ip_hdr->saddr))
      return;
   
   /* Anycast source address */
   
   /* Only reply to fragment 0 (FIXME with fragmentation) */


   /* FIXME when options */
   len1 = sizeof(*ip_hdr);

   /*
    * If packet in error is itself an error message, forgive him :-o
    */
   if (ip_hdr->next_hdr == IPPROTO_ICMPV6) {
      icmp_hdr = (struct icmpv6_hdr *) ((char *) ip_hdr + len1);
      if (icmp_hdr->type < 128)
	 return;
   }
   /* Source address ? */
   if ((ifa = ipv6_get_saddr(dev, &ip_hdr->saddr)) == NULL)
      /* Can't determine source address */
      return;

   room = 576 - len1 - sizeof(struct icmpv6_error);
   len2 = ntohs(ip_hdr->payload) + sizeof(*ip_hdr);
#ifdef IPV6_DEBUG_ICMP   
   printk(KERN_DEBUG "[ERROR]icmpv6_send_error: %p %p len2 = %d\n", 
	             skb->tail, (unsigned char *) ip_hdr, len2);
#endif
   frag.hdr.head.type = type;
   frag.hdr.head.code = code;
   frag.hdr.unused = info;	/* MTU or pointer or unused */
   frag.hdr.head.csum = 0;
   frag.csum = 0;
   frag.data_ptr = ip_hdr;
   frag.data_len = ((len2 > room) ? room : len2);
   frag.d = &ip_hdr->saddr;	/* to compute pseudo csum */

   /*
    * Build and send the packet.
    */

   /* FIXME: implement me :) 
      icmp_out_count(icmp_param->icmph.type); */
   ipv6_build_xmit(sk, getfrag_icmpv6_error, &frag,
		   frag.data_len + sizeof(struct icmpv6_error),
		   &ip_hdr->saddr, &ifa->addr, NULL /*options */ , 0,
		   IPPROTO_ICMPV6, 1);

   /* Useless icmp_build_xmit(&frag, &ifa->addr, &ip_hdr->saddr); */
}

/*--END FUNCTION--( )--------------------------------*/

/*--START FUNCTION--(icmpv6_rcv)-------------------------------------
|
| Handles incoming icmpv6 packets.
| Called from ipv6_rcv (*handler).
|
---------------------------------------------------------------------*/
int icmpv6_rcv(struct sk_buff *skb,
	       struct device *dev,
	       struct in6_addr *saddr,
	       struct in6_addr *daddr,
	       struct options *opt,
	       unsigned short len,
	       int redo,
	       struct inet6_protocol *protocol)
{
   int icmp_len = 0;
   __u32 csum;
   struct icmpv6 *hdr = (struct icmpv6 *) skb->h.raw;

   icmp6_statistics.IcmpInMsgs++;

#ifdef IPV6_DEBUG_ICMP
   printk(KERN_DEBUG "[ICMPV6]icmpv6_rcv:  IN type = %d\n", hdr->icmp6_type);
#endif
   /*
    * Validate packet (length)
    */
   if (((icmp_len = skb->tail - (unsigned char *) (hdr)) < ICMP6_MINLEN) &&
       skb->len < ICMP6_MINLEN) /* hack ATM */{
      icmp6_statistics.IcmpInErrors++;
      printk(KERN_DEBUG "[ICMPv6]icmpv6_rcv: bad length %d OUT\n", icmp_len);
      kfree_skb(skb, FREE_READ);
      return (0);
   } 
      
   /*
    * Validate packet
    */
   csum = 0;
   switch (skb->ip_summed) {
   case CHECKSUM_NONE:		/* no checksum computed so far */
      csum = csum_partial((char *) hdr, len, 0);
   case CHECKSUM_HW:		/* pseudo header not part of checksum so far */
      if ((csum = csum_ipv6_magic(saddr, daddr, len, IPPROTO_ICMPV6, csum))) {
	 /* checksum <> 0: error */
	 printk(KERN_DEBUG "[ICMPv6]icmpv6_rcv: checksum failed...: %u \n", csum);
	 printk(KERN_DEBUG "Saddr: ");
	 /* PRINT_ADDR(s); */
	 printk(KERN_DEBUG "Daddr: ");
	 /* PRINT_ADDR(d); */
	 printk(KERN_DEBUG "Len: %u\nType: %u\n, Csum: %u\n", len, IPPROTO_ICMPV6, csum);
	 /* dump_hex((char *) hdr, len); */
	 printk(KERN_DEBUG "====================================\n");
	 hdr->icmp6_cksum = 0;
	 printk(KERN_DEBUG "csum_partial = %u \n", csum_partial((char *) hdr, len, 0));
	 icmp6_statistics.IcmpInErrors++;
	 kfree_skb(skb, FREE_READ);
	 return (0);
      }
      break;
   default:
      /* CHECKSUM_UNNECESSARY should not be reached? */
   }

   switch (hdr->icmp6_type) {
   case ICMP6_UNREACH:
      /*  icmp6stat.icp6s_rcv_unreach++; */
      switch (hdr->icmp6_code) {
      case ICMP6_UNREACH_NOROUTE:
#ifdef IPV6_DEBUG_ICMP
	 printk(KERN_DEBUG "UNREACH. DEST. - NOROUTE");
#endif
	 break;

      case ICMP6_UNREACH_ADMIN:
	 #ifdef IPV6_DEBUG_ICMP
	 printk(KERN_DEBUG "UNREACH. DEST. - ADMIN");
	 #endif
	 break;

      case ICMP6_UNREACH_RTFAIL:
	 #ifdef IPV6_DEBUG_ICMP
	 printk(KERN_DEBUG "UNREACH. DEST. - RTFAIL");
	 #endif
	 break;

      case ICMP6_UNREACH_ADDRESS:
	 #ifdef IPV6_DEBUG_ICMP
	 printk(KERN_DEBUG "UNREACH. DEST. - ADDRESS");
	 #endif
	 break;

      case ICMP6_UNREACH_PORT:
	 #ifdef IPV6_DEBUG_ICMP
	 printk(KERN_DEBUG "UNREACH. DEST. - PORT");
	 #endif
	 break;

      default:
	 goto badcode;
      }
      break;

   case ICMP6_PKTTOOBIG:
      break;

   case ICMP6_TIMXCEED:
      #ifdef IPV6_DEBUG_ICMP
      printk(KERN_DEBUG "ICMPV6 TIME EXCEEDED\n");
      #endif
      break;

   case ICMP6_PARAMPROB:

      break;

   case ICMP6_SOLICITATION_RT:
      if (!skb->dev->dev6_lst->router) {
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      if ((skb->nh.ipv6h->hop_limit != ICMP6_ND_HOPS) ||
	  (hdr->icmp6_code)) {
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      break;

   case ICMP6_ADVERTISMENT_RT:
      if (ipv6_addr_scope(&skb->nh.ipv6h->saddr) != IFA_LINK) {
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      if ((skb->nh.ipv6h->hop_limit != ICMP6_ND_HOPS) ||
	  (hdr->icmp6_code)) {
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      break;

   case ICMP6_SOLICITATION_ND:	/* 135 */
      /*
       * Validate message
       */
      if ((skb->nh.ipv6h->hop_limit != ICMP6_ND_HOPS) ||
	  (hdr->icmp6_code)) {
	 /* FIXME MIB */
	 printk(KERN_DEBUG "[ICMPV6]icmpv6_rcv: nds - bad hop limit\n");
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      if (icmp_len < ICMP6_NSLEN) {
	 /* FIXME MIB */
	 printk(KERN_DEBUG "[ICMPV6]icmpv6_rcv: nds - bad length\n");
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      if (get_ipv6_addr_type(&hdr->icmp6_tgt) == IPV6_ADDR_MULTICAST) {
	 /* FIXME MIB */
	 printk(KERN_DEBUG "[ICMPV6]icmpv6_rcv: Multicast address as target. Pckt discarded\n");
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      /* FIXME MIB icmp6stat.icp6s_rcv_ndsol++; */
      nd_event_rcv(skb, dev, saddr, daddr, opt, len);
      kfree_skb(skb, FREE_READ);
      break;

   case ICMP6_ADVERTISMENT_ND:	/* 136 */
      /*
       * Validate message
       */
      if ((skb->nh.ipv6h->hop_limit != ICMP6_ND_HOPS) ||
	  (hdr->icmp6_code)) {
	 /* FIXME MIB */
	 printk(KERN_DEBUG "[ICMPV6]icmpv6_rcv: nda - bad hop limit\n");
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      if (icmp_len < ICMP6_NSLEN) {
	 /* FIXME MIB */
	 printk(KERN_DEBUG "[ICMPV6]icmpv6_rcv: nda - bad length\n");
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      if (get_ipv6_addr_type(&hdr->icmp6_tgt) == IPV6_ADDR_MULTICAST) {
	 /* FIXME MIB */
	 printk(KERN_DEBUG "[ICMPV6]icmpv6_rcv: nda - Multicast address as target. Pckt discarded\n");
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      if ((get_ipv6_addr_type(&(skb->nh.ipv6h->daddr))
	   == IPV6_ADDR_MULTICAST)
	  && (hdr->icmp6_ndflag & 0x40)) {
	 /* FIXME MIB */
	 printk(KERN_DEBUG "[ICMPV6]icmpv6_rcv:\n");
	 kfree_skb(skb, FREE_READ);
	 break;
      }
      /* FIXME MIB */
      nd_event_rcv(skb, dev, saddr, daddr, opt, len);
      break;

   case ICMP6_REDIRECT:

      break;

   case ICMP6_GROUPMEM_QUERY:
      
      break;


   case ICMP6_GROUPMEM_REPORT:
 
      break;

   case ICMP6_GROUPMEM_TERM:

       break;

   case ICMP6_ECHO:		/* 128 */
      icmp6_statistics.IcmpInEchos++;
      icmp6_statistics.IcmpOutEchoReps++;
      icmpv6_reflect(skb, dev);
      return (0);

      /*
       * No kernel processing for the following;
       * Must have been delivered to raw listener in ipv6_input()
       */
   case ICMP6_ECHOREPLY:	/* 129 */
      icmp6_statistics.IcmpInEchoReps++;
      kfree_skb(skb, FREE_READ);
      break;

   default:
    badcode:
      printk("[ICMPV6]icmpv6_rcv: unknown type %d code %d\n", hdr->icmp6_type, hdr->icmp6_code);
      icmp6_statistics.IcmpInErrors++;
      kfree_skb(skb, FREE_READ);
   }
#ifdef IPV6_DEBUG_ICMP
   printk(KERN_DEBUG "[ICMPV6]icmpv6_rcv: OUT\n");
#endif
   return (0);
}
/*--END FUNCTION--(icmpv6_rcv)----------------------------------------------*/

/*--START FUNCTION--(icmpv6_init)--------------------------------------------
|
| Initialize icmpv6 layer..
| Called from 
|
----------------------------------------------------------------------------*/
void icmpv6_init(struct net_proto_family *ops)
{
   struct sock *sk;
   int err;

#ifdef IPV6_DEBUG_ICMP
   printk(KERN_DEBUG "[ICMPV6]icmpv6_init IN\n");
#endif

   icmpv6_socket.type = SOCK_RAW;
   
   if ((err = ops->create(&icmpv6_socket, IPPROTO_ICMPV6)) < 0)
      printk("Failed to create the ICMPv6 control socket... \n");

   sk = icmpv6_socket.sk;	        /* link structures sock and socket */
   sk->allocation = GFP_ATOMIC;
   sk->num        = 256;		/* Does not receive any data */
#ifdef IPV6_DEBUG_ICMP
   printk(KERN_DEBUG "[ICMPV6]icmpv6_init OUT\n");
#endif
}

/*--END FUNCTION--(icmpv6_init)---------------------------------------------*/

/*
 * From Linux official code, for TCP
 */

static struct icmp6_err {
        int err;
        int fatal;
} tab_unreach[] = {
        { ENETUNREACH,  0},     /* NOROUTE              */
        { EACCES,       1},     /* ADM_PROHIBITED       */
        { EOPNOTSUPP,   1},     /* NOT_NEIGHBOUR        */
        { EHOSTUNREACH, 0},     /* ADDR_UNREACH         */
        { ECONNREFUSED, 1},     /* PORT_UNREACH         */
};

int icmpv6_err_convert(int type, int code, int *err)
{
        int fatal = 0;

        *err = 0;

        switch (type) {
        case ICMP6_UNREACH:
                if (code <= ICMP6_UNREACH_PORT)
                {
                        *err  = tab_unreach[code].err;
                        fatal = tab_unreach[code].fatal;
                }
                break;

        case ICMP6_PKTTOOBIG:
                *err = EMSGSIZE;
                break;
                
        case ICMP6_PARAMPROB:
                *err = EPROTO;
                fatal = 1;
                break;
        };

        return fatal;
}
