#include <linux/sched.h>
#include <asm/uaccess.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include <linux/route.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>

#include <linux/dret/netdevice6.h>
#include <linux/dret/table.h>
#include <linux/dret/route6.h>
#include <linux/dret/flow6.h>
#include <linux/dret/flow_table.h>

#include <linux/in.h>
#include <net/netlink.h>
#include <net/sock.h>
#include <net/snmp.h>

#include <net/dret/ipv6.h>
#include <net/dret/nd.h>
#include <net/dret/protocol6.h>
#include <net/dret/route6.h>
#include <net/dret/addrconf.h>
#include <net/dret/sit.h>
#include <net/dret/table.h>
#include <net/dret/table_tools.h>
#include <net/dret/flow6.h>
#include <net/dret/icmpv6.h>
#include <net/dret/options6.h>
#include <net/dret/raw6.h>
#include <linux/dret/ipv6_debug.h>

/*---------- CONTENTS ---------------
  OUTGOING FLOWS (source)
  flow_hash_code(__u32 label)
  flow_uniqueness(struct inet6_ifaddr *ifa, __u32 label)
  flow_label_pick(struct inet6_ifaddr *ifa)
  flow_add6(struct flow_entry *fe, struct in6_addr *addr, __u8 hash)
  flow6_ioctl(struct sock *sk, unsigned int cmd, void *arg)
  INCOMING FLOWS
  flow6_create(struct sk_buff *skb, struct flow *f, __u32 flabel, __u8 hash)
  flow6_fwd(struct sk_buff *skb, struct flow_entry * fe)
  flow6_to_dest(struct sk_buff *skb, struct flow_entry * fe)
  flow6_rcv(skb, dev, flabel)
  
  ----------------------------------*/

#ifdef IPV6_DEBUG_FLOW
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[FLOW6]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#else
#define DEBUG(format, args...) 
#define PRINT_ADDR(a)
#endif

/*
 * We cannot assume that all options are fixed length (Can we?).
 * Therefore, we only keep track of what kind of options are present...
 */


/********************************************************
  
  FLOW TABLE MANAGEMENT
  
  *******************************************************/

/*--START FUNCTION--(flow_add6)---------------------------------------



  ------------------------------------------------------------------*/
int flow_add6(struct flow_entry *fe, struct in6_addr *addr, __u8 hash) {
   int err;
   struct flow *f;

   DEBUG("[FLOW]fl_add6: IN\n");
   PRINT_ADDR(addr);
   
   f = (struct flow *)kmalloc(sizeof(struct flow), GFP_ATOMIC);
   memset(f, 0, sizeof(struct flow));
   
   COPY_ADDR6(f->fl_src, *addr);
   atomic_inc(&f->fl_ref);
   fe->fe_next = NULL;
   f->fl_fa[hash] = fe;
      
   while (flow_table.gt_lock) {
            sleep_on(&flow_table.gt_wait);
   }
   table_fast_lock(&flow_table);

   err = table_add_1(addr, 128, f, sizeof(struct flow), &flow_table);

   table_unlock(&flow_table);
   wake_up(&flow_table.gt_wait);
	 
   if(!err) atomic_inc(&flow_table.gt_entries);
   DEBUG("[FLOW]rt_add6: OUT: %d\n", err);
   return err;
}
/*--END FUNCTION--(rt_add6)---------------------------------------*/
/********************************************************
  
  FLOW MANAGEMENT
  
  ********************************************************/

/*--START FUNCTION--(flow_hash_code)----------------------------------------
 |
 | Input: a flow label
 | Output: a hash key between 0 and 15.
 |
 -------------------------------------------------------------------------*/
static __inline__ unsigned flow_hash_code(__u32 label) {
        unsigned hash = ((label&0x00F0F0F0)>>4)|((label&0x000F0F0F)<<4);
        hash = hash^(hash>>16);
        return (hash^(hash>>8)) & 0xF;
}

/*--START FUNCTION--(flow_uniqueness)----------------------------------------
 |
 | Input: an interface address structure, a flow label
 | Output: 0 if label does not already exist, -1 otherwise.
 |
 -------------------------------------------------------------------------*/
int flow_uniqueness(struct inet6_ifaddr *ifa, __u32 label) {
   __u32 hash;
   struct flowlist *fl;
   
   /*
    * Hash from flow label
    */
   DEBUG("[FLOW]flow_uniqueness: IN\n");
   hash = flow_hash_code(label);
   /*
    * Lookup
    */
   for(fl = ifa->fl[hash]; fl; fl = fl->fll_next_ifa) {
      if(label == fl->fll_label) {
	 DEBUG("[FLOW]flow_uniqueness: Uniqueness negative\n");
	 return -1;
      }
   }
   DEBUG("[FLOW]flow_uniqueness: Uniqueness positive\n");
   return 0;
}

/*--START FUNCTION--(flow_label_pick)----------------------------------------
 |
 | Input: an interface address structure.
 | Output: a new, unique flow label for this source address.
 |
 -------------------------------------------------------------------------*/
__u32 flow_label_pick(struct inet6_ifaddr *ifa) {
   __u32 label;
   __u8 is_ok = 0;
   
   DEBUG("[FLOW]flow_label_pick: IN\n");
   do {
      label = random_flow_label();
      if (flow_uniqueness(ifa, label) == 0)
	  is_ok++;
   } while (!is_ok);
   DEBUG("[FLOW]flow_label_pick: OUT %d\n", label);
   return(label);
}

/*--START FUNCTION--(flow6_ioctl)----------------------------------------
 |
 | Input: a socket information block (sock), a cmd, a void * 
 |        (struct flowdesc*)
 | Output: 0 if Ok, else error num.
 |
 -------------------------------------------------------------------------*/
int flow6_ioctl(struct sock *sk, unsigned int cmd, void *arg) {
   int    err, len;
   struct flowdesc fld;
   struct flowlist *fll;
   struct ipv6_pinfo  *info  = &sk->net_pinfo.af_inet6;
   struct options *opt = NULL;
   struct cmsghdr *cmsgptr;
   struct inet6_ifaddr *ifa;
   struct destination * dc;
   __u8   *buf = NULL, 
          is_connected = (sk->state == TCP_ESTABLISHED),
          hash;
   
   /* Get base structure */
   err = copy_from_user(&fld, arg, sizeof(struct flowdesc));
   if (err)
      return -EFAULT;  
   
   /* FLL structure */
   if (fld.fld_flags &  FLD_DEFAULT) { 
      if ((!is_connected)||
	  (info->pi6_fl.fll_flags &&  FLL_USED)||
	  (!SAME_ADDR6(info->pi6_fl.fll_da, fld.fld_sin6.sin6_addr)))
	 return -EFAULT;
      fll = &info->pi6_fl;
   } else {
      /* Alloc a flow structure, bound to socket ... */
      fll = (struct flowlist *)kmalloc(sizeof(struct flowlist), GFP_KERNEL);
      memset(fll, 0, sizeof(struct flowlist));
      COPY_ADDR6(fll->fll_da, fld.fld_sin6.sin6_addr);
   }    
      
   /* Any option bound to label ? */
   if(fld.fld_len) {
      if ((fld.fld_len <= sizeof(struct cmsghdr)) || 
	  (fld.fld_len > 1024)) {/*FIXME*/
	 if(!is_connected)
	    kfree(fll);
	 return -EFAULT;
      }
      DEBUG("[FLOW]flow6_ioctl: ancillary data length = %d\n", fld.fld_len);
      /* Get buffer with options */
      buf = (char *)kmalloc(fld.fld_len, GFP_KERNEL);
      err = copy_from_user(buf, fld.fld_buf, fld.fld_len);
      if (err) 
	 goto efault1;
      
      opt = &fll->fll_opt;
      memset(opt, 0, sizeof(struct options));
      opt->opt_base = (void *)buf;
      len = opt->opt_len  = fld.fld_len;

      /* Handle options */
      for (cmsgptr = (struct cmsghdr *)buf ; 
	   (cmsgptr != NULL) && (len > 0);
	   cmsgptr = (struct cmsghdr *)((__u8 *)cmsgptr + cmsgptr->cmsg_len)) {
	 if (cmsgptr->cmsg_level == IPPROTO_IPV6) {
	    switch(cmsgptr->cmsg_type) {
	    case IPV6_PKTINFO:
		  COPY_ADDR6(fll->fll_sa, (((struct in6_pktinfo *)cmsgptr->cmsg_data)->ipi6_addr));	     
	       break;
	    case IPV6_HOPLIMIT:
	       break;
	    case IPV6_NEXTHOP:
	       break;
	    case IPV6_HOPOPTS:
	       break;
	    case IPV6_DSTOPTS:
	       break;
	    case IPV6_SRCRT:
	       {
		  int llen, seg_num, cnter = 1;
		  struct ipv6_rt_hdr0 *rthdr = (struct ipv6_rt_hdr0 *)(cmsgptr + 1);
		  struct in6_addr *rt_addr, *rt2_addr, tmp_addr;
		  
		  DEBUG("[FLOW]flow6_ioctl: IPV6_SRCRT len = %d\n", 
			cmsgptr->cmsg_len);
		  
		  /* Set new daddr, + shift */
		  seg_num = rthdr->segments_left;
		  
		  rt_addr = (struct in6_addr *)(rthdr + 1);
		  COPY_ADDR6(tmp_addr, fll->fll_da);
		  COPY_ADDR6(fll->fll_da, *rt_addr);
		  
		  while (cnter != seg_num) {
		     rt2_addr = rt_addr + 1;
		     COPY_ADDR6(*rt_addr, *rt2_addr);
		     rt_addr ++; cnter ++;
		  }
		 
		  COPY_ADDR6(*rt_addr, tmp_addr);
		  
		  /* Update option structure */
		  llen = cmsgptr->cmsg_len - sizeof(struct cmsghdr);	       
		  opt->opt_srcrt = (struct ipv6_rt_hdr *)(cmsgptr + 1);
		  opt->opt_srcrt_len = llen;
		  /*
		   * Option is ready to use by ipv6_flow_xmit. . .
		   */
	       }
	    break;
	    default:
	    }
	 } else {
	    DEBUG("[FLOW]: ancillary data with bad level\n");
	    goto efault2;
	 }
	 len -= cmsgptr->cmsg_len;
      }
   }
   
   /*
    * Flow SADDR
    */
   if(IS_ANYADDR6(fll->fll_sa)){
      dc = dcache_check(NULL, &fld.fld_sin6.sin6_addr, 0);
      if (dc == NULL)
	 goto efault2;
      ifa = ipv6_get_saddr(dc->d_dev, &fld.fld_sin6.sin6_addr);
      COPY_ADDR6(fll->fll_sa, ifa->addr);
   } else {
      ifa = get_ifaddr6(&fll->fll_sa);
   }
   
   /*
    * Flow LABEL
    */
   fll->fll_label = fld.fld_sin6.sin6_flowinfo = flow_label_pick(ifa);
   
   err = copy_to_user(arg, &fld, sizeof(struct flowdesc));
   if (!err) {          
      /*
       * Bind FLL
       */
      if(!is_connected) {
	 fll->fll_next = info->fl;
	 info->fl = fll;
      }   
      /* Bind */
      hash = flow_hash_code(fll->fll_label);
      fll->fll_next_ifa = ifa->fl[hash];
      ifa->fl[hash] = fll;
      
      return 0;
   } else {
      hash = flow_hash_code(fll->fll_label);
      ifa->fl[hash] = ifa->fl[hash]->fll_next_ifa;
   }
        
efault2:        
    kfree(opt);
    
       
efault1: 
    if(!is_connected)
       kfree(fll);
    if(fld.fld_len)
       kfree_s(buf, fld.fld_len);    
    return -EFAULT;   
}
   
/*
 * Flow table root
 */
struct table_node flow_root = {
   NULL, NULL, NULL, {{{0}}}, NULL, 0, RTN_ROOT
};

struct gen_table flow_table = {
   &flow_root, 0, 0
};

/*--START FUNCTION--(flow6_create)-----------------------------------------
  
  Called when a flow is seen for the first time
  
  -----------------------------------------------------------------------*/
struct flow_entry *flow6_create(struct sk_buff *skb, struct flow *f, 
				__u32 flabel, __u8 hash) {
   struct flow_entry *fe;
   struct ipv6hdr    *iph = skb->nh.ipv6h;
   struct destination *dc;
   struct ipv6_rt_hdr0 *rh0 = NULL;
   struct in6_addr *d;
   __u8 next_hdr,
      done = 1,
      final_dest = 0,
      rh_len = 0,
      *optr;
   unsigned int flags;
   
   DEBUG("[FLOW]flow6_create: IN label = %d\n", flabel);
   d = &iph->daddr;
   fe = (struct flow_entry *)kmalloc(sizeof(struct flow_entry), GFP_ATOMIC);
   memset(fe, 0, sizeof(struct flow_entry));
   
   fe->fe_timestamp = jiffies;
   fe->fe_label     = flabel;
   
   fe->fe_stats.fs_timestamp = jiffies;
   fe->fe_stats.fs_count ++;
   fe->fe_stats.fs_bytes += ntohs(iph->payload) + sizeof(struct ipv6hdr);
   /* jitter is 0 for first packet */
   
   /*
    * Parse options
    */
   next_hdr      = iph->next_hdr;
   optr          = skb->h.raw;
   
   while (done) {
      __u8 d1_2 = 0;
      switch(next_hdr) {
      case IPPROTO_HOPOPTS:
	 DEBUG("[FLOW]flow6_create: IPPROTO_HOPOPTS\n");
	 fe->fe_opt.fo_flags |= FO_HOPOPTS;
	 next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
	 optr += ((struct opt6_hdr *)optr)->oh_len;
	 break;
      case IPPROTO_DSTOPTS:
	 DEBUG("[FLOW]flow6_create: IPPROTO_DSTOPTS 1 not implemented\n");
	 if(!d1_2)
	    fe->fe_opt.fo_flags |= FO_DSTOPTS1;
	 else
	    fe->fe_opt.fo_flags |= FO_DSTOPTS2;
	 next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
	 optr    += ((struct opt6_hdr *)optr)->oh_len;
	 break;
	 
      case IPPROTO_ROUTING:
	 DEBUG("[FLOW]flow6_create: IPPROTO_ROUTING\n");
	 fe->fe_opt.fo_flags |= FO_SRCRT;
	 if (((struct ipv6_rt_hdr*)optr)->type == 0) {
	    rh0 = (struct ipv6_rt_hdr0*)optr;
	    final_dest = (rh0->segments_left == 0);
	    DEBUG("[FLOW]flow6_create: IPPROTO_ROUTING seg: %d\n",
		  (rh0->segments_left));
	 }
         next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
	 optr    += ((struct opt6_hdr *)optr)->oh_len * 8 + 8;
	 d1_2 ++;
	 break;

      case IPPROTO_AH:
	 DEBUG("[FLOW]flow6_create: IPPROTO_AH not implemented\n"); 
	 fe->fe_opt.fo_flags |= FO_AH;
	 next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr; 
	 optr    += ((struct opt6_hdr *)optr)->oh_len;
	 break;
	 
      case IPPROTO_ESP:
	 DEBUG("[FLOW]flow6_create: IPPROTO_ESP not implemented\n");
	 fe->fe_opt.fo_flags |= FO_ESP;
	 next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
	 optr    += ((struct opt6_hdr *)optr)->oh_len; 
	 break;
	 
      default:
	 DEBUG("[FLOW]flow6_create: no other option found\n");
	 done = 0;
      }
   }    
   /*
    * Are we the final destination?
    */
   if (check_host_addr6(&iph->daddr)) { /* FIXME multicast */
      if (!(fe->fe_opt.fo_flags & FO_SRCRT) || final_dest) { 
	 /* dest || src routing last */
         fe->fe_flags |= FE_DEST;
	 goto out;
      } 
   }
   
   /*
    * Lookup route   
    */      
   if (fe->fe_opt.fo_flags & FO_SRCRT) {
      d = (struct in6_addr *)((__u8 *)(rh0 + 1) + 
			      ((rh0->hdrlen / 2) - rh0->segments_left)*8);
      PRINT_ADDR(d);
   }

   
   fe->fe_dest = dcache_check(NULL, d, 0); 
   DEBUG("[FLOW]flow6_create: fe->fe_dest = %p\n", fe->fe_dest);
   if (fe->fe_dest == NULL) {
      kfree(fe);
      return NULL;
   }
   
   fe->fe_dev = skb->dev = fe->fe_dest->d_dev;
   
   out:
   /*
    * Add fe to flow list for the source
    */
   if (f) {
      save_flags(flags);
      cli();      
      fe->fe_next = f->fl_fa[hash];
      f->fl_fa[hash] = fe;
      restore_flags(flags);
   } else {
   /*
    * Add a new flow list for the source
    */   
   flow_add6(fe, &iph->saddr, hash);
   }
   DEBUG("[FLOW]flow6_create: OUT fe = %p\n", fe);
   return fe;
}
/*--END FUNCTION--(flow6_create)------------------------------------------*/


/*--START FUNCTION--(flow6_fwd)---------------------------------------
  
  Forward packet flow
  
  -----------------------------------------------------------------------*/
int flow6_fwd(struct sk_buff *skb, struct flow_entry * fe) {
   __u8 *optr = (__u8 *)skb->h.raw;
   __u8 *ptr;
   __u8 next_hdr;
   int err = 0;
   struct device *dev = skb->dev;
   struct neighbor * neigh;
   struct destination *rt;
   struct ipv6hdr *ipv6h = skb->nh.ipv6h;
   struct sk_buff *skb2 = NULL;
   
   DEBUG("[FLOW]flow6_fwd: IN fe = %p\n", fe);   
   ipv6h->hop_limit--;
     
   if (ipv6h->hop_limit <= 0) {
      icmpv6_send_error(skb, ICMP6_TIMXCEED,
			ICMP6_TIMXCEED_INTRANS, 0, dev);
      DEBUG("[FWD]flow6_fwd: OUT Time exceeded\n");
      return -1;
   }
   
   if (IS_MULTIADDR6(ipv6h->saddr)) {
      printk(KERN_DEBUG "[FWD]flow6_fwd: OUT saddr is multicast!\n");
      return -1;		/* FIXME */
   }
   

   next_hdr = ipv6h->next_hdr;
   /*
    *   OPTIONS
    */
   if (fe->fe_opt.fo_flags & FO_HOPOPTS) {
      err = hopopts_handler(skb, optr); 
      next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;   
   }
   
   if (fe->fe_opt.fo_flags & FO_DSTOPTS1) {
      next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
      optr    += ((struct opt6_hdr *)optr)->oh_len; 
   }
   
   if (fe->fe_opt.fo_flags & FO_SRCRT) {
      err = srcrt_handler(skb, optr, 1); 
      if (err == -1) {
	 DEBUG("[INPUT]ipv6_rcv: IPPROTO_ROUTING - skb gone\n");
	 return(0);
      }
   }
   DEBUG("[FLOW]flow6_fwd: fe->fe_dest = %p\n", fe->fe_dest);
   rt = dcache_check(fe->fe_dest, &ipv6h->daddr, 0);
   if (rt == NULL) {
      icmpv6_send_error(skb, ICMP6_UNREACH,
			ICMP6_UNREACH_NOROUTE, 0, dev);
      return -1;
   }
   
   skb->dev = dev   = rt->d_route.rt_dev; 
   neigh = rt->d_route.rt_neigh;
   
   if (!(dev->flags & IFF_UP)) {
      printk(KERN_DEBUG "ERROR forward packet to a device down\n");
      if (rt) {
	 atomic_dec(&(rt->d_ref));
	 return -1;
      }
      return 1;
   }
      
   /*
    *   Now forward
    */
   IS_SKB(skb);
   
   /*
    *   Frag?
    */
   if (skb->len > dev->mtu) {
      DEBUG("[FLOW]skb->len > dev->mtu \n");
      ip6_statistics.IpFragFails++;
      icmpv6_send_error(skb, ICMP6_PKTTOOBIG, 0,
			htonl(dev->mtu), dev);
      if (rt)
	 atomic_dec(&(rt->d_ref));
      return -1;
   }
   
   /*
    *   Need a larger skb?
    */
   if (skb_headroom(skb) < dev->hard_header_len) {      
      DEBUG("[FLOW]Not enough room for new hh... \n");	 
      skb2 = alloc_skb(dev->hard_header_len + skb->len + 15,
		       GFP_ATOMIC);
      if (skb2 == NULL) {
	 DEBUG("[FLOW]flow6_fwd: No memory available\n");
	 if (rt)
	    atomic_dec(&(rt->d_ref));
	 return -1;
      }
      
      IS_SKB(skb2);
      /*
       *    Add the physical headers.
       */
      skb2->dev = dev;
      skb2->protocol = htons(ETH_P_IPV6);
      skb2->when = jiffies;
      skb2->nexthop = neigh;
      skb2->priority = SOPRI_NORMAL;
      
      /* FIXME */
      ipv6_build_mac_header(skb2, neigh, skb->len);
            
      ptr = skb_put(skb2, skb->len);
      skb2->nh.raw = skb2->h.raw = ptr;
      memcpy(ptr, skb->h.raw, skb->len);      
   } else { 
      /*
       *   Use current skb
       */
      skb->dev = dev;
      skb->nexthop = neigh;
      skb->arp = 1;
      skb->priority = SOPRI_NORMAL;
      skb2 = skb;
      
      /* FIXME  (hh)*/
      if (neigh && (neigh->n_flags & NCF_HHVALID)) {
	 DEBUG("[FLOW]Have a valid neigh...\n");
	 memcpy(skb_push(skb, dev->hard_header_len),
		neigh->n_hh_data, 
		dev->hard_header_len);  
      } else {
	 if (dev->hard_header) {
	    skb_push(skb, dev->hard_header_len);	       
	    if (dev->hard_header(skb, dev, ETH_P_IPV6,
				  NULL, NULL, skb->len)) {
	       /* At that point skb has been freed */
	       DEBUG("[FWD]Couldn't dev->hard_header\n");
	       return 0;
	    }	    
	 } 
      }
   }
   
   ip6_statistics.IpForwDatagrams++;
   fe->fe_stats.fs_count ++;
   fe->fe_stats.fs_bytes += ntohs(ipv6h->payload);
   fe->fe_stats.fs_jitter += jiffies - fe->fe_stats.fs_timestamp;
   fe->fe_stats.fs_timestamp = jiffies;
   
   dev_queue_xmit(skb2);
   if (rt)
      atomic_dec(&(rt->d_ref));   
   /*
    *    Tell the caller if their buffer is free.
    */
   DEBUG("[FWD]flow6_fwd: OUT -> %d\n", (skb == skb2 ? 0 : 1));    
   return (!(skb == skb2));
}


/*--START FUNCTION--(flow6_to_dest)---------------------------------------
  
  Packet flow got to destination
  
  -----------------------------------------------------------------------*/
void flow6_to_dest(struct sk_buff *skb, struct flow_entry * fe) {
   struct sock * raw_sk = NULL;
   __u8        *optr    = skb->h.raw,
               next_hdr = skb->nh.ipv6h->next_hdr;
   int err =0;
   
   DEBUG("[FLOW]flow6_to_dest: IN fe = %p\n", fe);
   /*
    * Scan options
    */
   skb->ip6_hopopt = NULL;             /* Set if option found and user */
   skb->ip6_dstopt1 = NULL;            /* asked for it                 */
   skb->ip6_srcrt = NULL;
   skb->ip6_dstopt2 =NULL;
   
   if (fe->fe_opt.fo_flags & FO_HOPOPTS) {
      err = hopopts_handler(skb, optr); 
      next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;   
   }
   
   if (fe->fe_opt.fo_flags & FO_DSTOPTS1) {
      next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
      optr    += ((struct opt6_hdr *)optr)->oh_len; 
   }
   
   if (fe->fe_opt.fo_flags & FO_SRCRT) {
      err = srcrt_handler(skb, optr, 1); 
      if (err == -1) {
	 DEBUG("[INPUT]ipv6_rcv: IPPROTO_ROUTING - skb gone\n");
	 return;
      }
      next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
      optr    += (((struct opt6_hdr *)optr)->oh_len<<3) +8;
   }
   
   if (fe->fe_opt.fo_flags & FO_DSTOPTS2) {
      next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr; 
      optr    += ((struct opt6_hdr *)optr)->oh_len;
   }
   
   if (fe->fe_opt.fo_flags & FO_AH) {
      next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
      optr    += ((struct opt6_hdr *)optr)->oh_len; 
   }
   
   if (fe->fe_opt.fo_flags & FO_ESP) {
      next_hdr = ((struct opt6_hdr *)optr)->oh_nexthdr;
      optr    += ((struct opt6_hdr *)optr)->oh_len; 
   }
   
   skb->h.raw = optr;
    
   ipv6_raw_deliver(skb, next_hdr, &raw_sk); 

   ipv6_proto_deliver(skb, next_hdr, raw_sk);
       
   /*
    *    All protocols checked.
    */
   if (raw_sk != NULL)
      raw6_rcv(raw_sk, skb);
   
   fe->fe_stats.fs_count ++;
   fe->fe_stats.fs_bytes += ntohs(skb->nh.ipv6h->payload);
   fe->fe_stats.fs_jitter += jiffies - fe->fe_stats.fs_timestamp;
   fe->fe_stats.fs_timestamp = jiffies;
}
   
/*--START FUNCTION--(flow6_rcv)--------------------------------------------
  
  Called when incoming packet has non-zero flow label
  
  -----------------------------------------------------------------------*/
void flow6_rcv(struct sk_buff *skb, struct device *dev, __u32 flabel) {
   struct flow       *f   = NULL;
   struct table_leaf *tl  = NULL;
   struct flow_entry *fe  = NULL;
   struct ipv6hdr    *iph = skb->nh.ipv6h;
   __u8 hash = flow_hash_code(flabel);
   unsigned int flags;
   
   DEBUG("[FLOW]flow6_rcv: IN\n");
   skb->h.raw   += sizeof(struct ipv6hdr);
   skb->dev = dev;
   /*
    *   Flow lookup
    */
   while (flow_table.gt_lock) {
      sleep_on(&flow_table.gt_wait);
   }
   table_fast_lock(&flow_table);

   tl = table_lookup_1(&iph->saddr, &flow_table, 1);

   table_unlock(&flow_table);
   wake_up(&flow_table.gt_wait);
   
   if (tl) {
      f = (struct flow *)tl->tl_data;         
   
      save_flags(flags);
      cli();
      for (fe = f->fl_fa[hash]; fe; fe = fe->fe_next) {
	 if (fe->fe_label == flabel) {
	    DEBUG("[FLOW]flow6_rcv: existing entry %p\n", fe);
	    break;
	 }
      }
      restore_flags(flags);
   }
      
   /* 
    *  Flow creation
    */
   if (fe == NULL) {
      fe = flow6_create(skb, f, flabel, hash);
      if (fe == NULL) {
	 kfree_skb(skb, FREE_READ);
	 return;
      }
   }
   
   /*
    * At that time, we have a flow descriptor, *fe
    */
   if(fe->fe_flags & FE_DEST) {
      DEBUG("[FLOW]flow6_rcv: flow6_to_dest\n");
      flow6_to_dest(skb, fe);
      return;
   } 
   
   if(flow6_fwd(skb, fe)) {
      DEBUG("[FLOW]flow6_rcv: flow6_to_dest\n");
      kfree_skb(skb, FREE_READ);
   }
   return;
}
      

   









