/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:     $Id: devinet6.c,v 1.5 1997/03/21 15:14:50 pan Exp $ 
 *
 * Authors:     
 *              Benoit Brodard  <Brodard@sophia.inria.fr>
 *
 *
 * Fixes:
 *              
 *
 *
 * Description:
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|  ipv6_build_mac_header
|  ipv6_queue_xmit
|  ipv6_build_xmit
|  ipv6_init
|
-------------------------------------------------------------------------*/
#include <asm/uaccess.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/in.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/if_ether.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/notifier.h>
#include <net/ip.h>
#include <net/route.h>
#include <linux/skbuff.h>
#include <net/sock.h>
#include <net/arp.h>
#include <net/slhc.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>
#include <net/br.h>
#include <linux/net_alias.h>
#ifdef CONFIG_KERNELD
#include <linux/kerneld.h>
#endif
#ifdef CONFIG_NET_RADIO
#include <linux/wireless.h>
#endif	/* CONFIG_NET_RADIO */
#ifdef CONFIG_NET_IPV6_DIM
#include <linux/dret/ipv6.h>
#include <linux/dret/netdevice6.h>
#include <net/dret/addrconf.h>
#include <net/dret/sit.h>
#endif

extern struct notifier_block *netdev_chain;
extern int dev_ifconf(char *arg);
extern int dev_ifsioc(void *arg, unsigned int getset);
/*
 *	Perform the SIOCxIFxxx calls. 
 *
 *	The socket layer has seen an ioctl the address family thinks is
 *	for the device. At this point we get invoked to make a decision
 */
 
int devinet6_ifsioc(unsigned int cmd, void *arg)
{
   struct ifreq        ifr;
   struct device       *dev;
   struct inet6_ifaddr *ifp;
   int                 err;

   /*
    *	Fetch the caller's info block into kernel space
    */
   printk(KERN_DEBUG "[DEVINET6]devinet6_ifsioc IN\n");
   err = copy_from_user(&ifr, arg, sizeof(struct ifreq));
   if (err)
      return -EFAULT; 
   
   /*
    *	See which interface the caller is talking about. 
    */
   
#ifdef CONFIG_KERNELD
   dev_load(ifr.ifr_name);
#endif	
	
   if ((dev = dev_get(ifr.ifr_name)) == NULL) 	
      return(-ENODEV);
   
   switch(cmd) 
      {   
	 /* 
	  * Get interface address (and family) 
	  */	
      case SIOCGIFADDR:      
	 {
	    struct ifreq6        *req6ptr= (struct ifreq6 *) ifr.ifr_buf6;
	    struct ifreq6        req6;
	    struct inet6_ifaddr  *addr6;
	    int                  room = ifr.ifr_len6;
	    
	    printk(KERN_DEBUG "Room1 = %d \n" , ifr.ifr_len6);
	    /* size of space filled with addrs */
	    ifr.ifr_len6 = 0; 
	    if (!dev->dev6_lst) {
	       struct inet6_dev *idev;
	       
	       if (!(idev = ipv6_add_dev(dev)))
		  return -ENXIO;
	    }
	    
	    for (addr6 = dev->dev6_lst->addr_list; addr6;
		 addr6 = addr6->if_next)
	       {
		  if (room < sizeof(struct ifreq6))
		     break;
		  
		  COPY_ADDR6(req6.in6, addr6->addr);
		  req6.preflen = addr6->plen;
		  
		  /*
		   *  Write this block to the caller's space. 
		   */
		  err = copy_to_user(req6ptr, &req6, sizeof(struct ifreq6));
		  if (err) 
		     return -EFAULT;
		  
		  req6ptr++;
		  ifr.ifr_len6 += sizeof(struct ifreq6);
		  printk(KERN_DEBUG "Room2 = %d \n" , ifr.ifr_len6);
		  room         -= sizeof(struct ifreq6);
	       }
	    goto rarok;
	 }
      
      /* 
       * Set interface address (and family) 
       */
      case SIOCSIFADDR:      
	 {
	    struct
	       ifreq6 req6;
	    
	    if (ifr.ifr_len6!=sizeof(struct ifreq6))
	       return(-EINVAL);
	    
	    err = copy_from_user(&req6, ifr.ifr_buf6,
				 sizeof(struct ifreq6));
	    if (err)
	       return -EFAULT;
	    
	    if (req6.preflen >128 || req6.preflen < 0)
	       return (-EINVAL);
	    
	    if (!dev->dev6_lst) {
	       struct inet6_dev *idev;
	       
	       if (!(idev = ipv6_add_dev(dev)))
		  return -ENXIO;
	    }
	    
	    ifp = ipv6_add_addr(dev->dev6_lst, &req6.in6,
				ipv6_addr_scope(&req6.in6),
				req6.preflen);
	    
	    return (ifp?0:-ENXIO);	
	 }
      
      
      /* 
       * Get the destination address (for point-to-point links) 
       */   
      case SIOCGIFDSTADDR:
	 {
	    struct ifreq6        *req6ptr= (struct ifreq6 *) ifr.ifr_buf6;
	    struct ifreq6        req6;
	    struct inet6_ifaddr  *addr6;
	    int                  room = ifr.ifr_len6;
	    
	    printk(KERN_DEBUG "SIOCGIFDSTADDR: Room1 = %d \n" , ifr.ifr_len6);
	    /* size of space filled with addrs */ 
	    if (!dev->dev6_lst) {
	       struct inet6_dev *idev;
	       
	       if (!(idev = ipv6_add_dev(dev)))
		  return -ENXIO;
	    }
	    
	    if (room < sizeof(struct ifreq6))
	       return -EINVAL;
	    
	    COPY_ADDR6(req6.in6, dev->dev6_lst->p2p_addr6);
	    req6.preflen = 128;
	    
	    /*
	     *  Write this block to the caller's space. 
	     */
	    err = copy_to_user(req6ptr, &req6, sizeof(struct ifreq6));
	    if (err) 
	       return -EFAULT;
	    goto rarok;
	 }
      
      /*
       * Set the destination address (for point-to-point links) 
       */  
      case SIOCSIFDSTADDR:
	 {
	    struct ifreq6 req6;
	    
	    if (ifr.ifr_len6!=sizeof(struct ifreq6))
	       return(-EINVAL);
	    
	    err = copy_from_user(&req6, ifr.ifr_buf6,
				 sizeof(struct ifreq6));
	    if (err)
	       return -EFAULT;
	    
	    if (req6.preflen >128 || req6.preflen < 0)
	       return (-EINVAL);
	    
	    if (!dev->dev6_lst) {
	       struct inet6_dev *idev;
	       
	       if (!(idev = ipv6_add_dev(dev)))
		  return -ENXIO;
	    }
	       
	    if (IS_ANYADDR6(dev->dev6_lst->p2p_addr6)) {
	       printk(KERN_DEBUG "Adding p2p dest addr\n");
	       COPY_ADDR6(dev->dev6_lst->p2p_addr6, req6.in6);
	       return 0;
	    } else {
	       printk(KERN_DEBUG "Tried to overwrite p2p dest addr\n");
	       return -EINVAL;
	    } 
	 }
	   
	/* 
	 * Get the netmask for the interface 
	 */
	case SIOCGIFNETMASK:	
	   goto rarok;
	   
	/* 
	 * Set the netmask for the interface 
	 */   
	case SIOCSIFNETMASK: 	
	   break;
	   
	case SIOCADDTUNNEL:
	case SIOCDELTUNNEL:
	   {
	      struct ifreq6 req6;   
	      printk(KERN_DEBUG "[DEV]SIOCADDTUNNEL, SIOCDELTUNNEL\n");
	      
	      if (ifr.ifr_len6 != sizeof(struct ifreq6))
		 return(-EINVAL);
	      
	      err = copy_from_user(&req6, ifr.ifr_buf6,
				   sizeof(struct ifreq6));
	      if (err)
		 return -EFAULT;
	      
	      if (req6.preflen != 96)
		 return (-EINVAL);
	      
	      if (cmd == SIOCADDTUNNEL)
		 return (ipv6_add_tunnel(dev, &req6.in6));
	      else
		 return (ipv6_del_tunnel(dev, &req6.in6));
	      break;
	   }

	/*
	 * Delete an IPv6 adress
	 */
	case SIOCDELADDR:
	   {
	      struct ifreq6 req6;
	      
	      if (ifr.ifr_len6 != sizeof(struct ifreq6))
		 return(-EINVAL); 	
	      
	      err = copy_from_user(&req6, ifr.ifr_buf6, sizeof(struct ifreq6));
	      
	      if (err)
		 return -EFAULT;	
	      
	      return (ipv6_del_addr(dev->dev6_lst, &req6.in6));
	   }
	      
	default:
	   return(-EINVAL);
	}

/*
 *	The load of calls that return an ifreq and ok (saves memory).
 */
rarok:
	err = copy_to_user((void *)arg, &ifr, sizeof(struct ifreq));
	return 0;
}


/*
 *	This function handles all "interface"-type I/O control requests.
 *      The actual 'doing' part of this is devinet6_ifsioc above.
 */
int devinet6_ioctl(unsigned int cmd, void *arg)
{
   printk(KERN_DEBUG "[DEVINET6]devinet6_ioctl IN\n");
   switch(cmd) 
	{
		case SIOCGIFCONF:
			(void) dev_ifconf((char *) arg);
			return 0;

		/*
		 *	Ioctl calls that can be done by all.
		 */
		 
		case SIOCGIFFLAGS:
		case SIOCGIFMETRIC:
		case SIOCGIFMTU:
		case SIOCGIFMEM:
		case SIOCGIFHWADDR:
		case SIOCSIFHWADDR:
		case SIOCGIFSLAVE:
		case SIOCGIFMAP:
		case SIOGIFINDEX:
			return dev_ifsioc(arg, cmd);
                
		/*
		 *	Ioctl calls requiring the power of a superuser
		 */
		 
		case SIOCSIFFLAGS:
		case SIOCSIFMETRIC:
		case SIOCSIFMTU:
		case SIOCSIFMEM:
		case SIOCSIFMAP:
		case SIOCSIFSLAVE:
		case SIOCADDMULTI:
		case SIOCDELMULTI:
			if (!suser())
				return -EPERM;
			return dev_ifsioc(arg, cmd);
	        
                case SIOCGIFADDR:
		case SIOCGIFDSTADDR:
		case SIOCGIFBRDADDR:
		case SIOCGIFNETMASK:
			return devinet6_ifsioc(cmd, arg);
			
		case SIOCSIFADDR:
		case SIOCSIFDSTADDR:
		case SIOCSIFBRDADDR:
		case SIOCSIFNETMASK:
	        case SIOCADDTUNNEL:
	        case SIOCDELTUNNEL:
		case SIOCDELADDR:
			if (!suser())
				return -EPERM;
			return devinet6_ifsioc(cmd, arg);
			
                case SIOCSIFLINK:
			return -EINVAL;

		/*
		 *	Unknown or private ioctl.
		 */	
		 
		default:
			if((cmd >= SIOCDEVPRIVATE) &&
			   (cmd <= (SIOCDEVPRIVATE + 15))) {
				return dev_ifsioc(arg, cmd);
			}
			return -EINVAL;
	}
}









