/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$ 
 *      
 * Authors:     
 *              Frank ZAGO      <zago@masi.ibp.fr>
 *
 *
 * Fixes:
 *		to run in 2.1.23 and so on .. Pascal ANELLI
 *
 * Description:
 *              AF_INET6 protocol family socket handler.
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/** Contents *************************************************************

   put_sock6
   get_sock6
   get_sock_raw6
   get_sock6_mcast
   def6_callback3
   inet6_create
   inet6_dup
   inet6_release
   inet6_bind
   
 ************************************************************************/

#include <linux/config.h>

#include <asm/segment.h>
#include <asm/system.h>
#include <asm/uaccess.h>

#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include <linux/fcntl.h>
#include <linux/mm.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/stat.h>

#include <linux/netdevice.h>
#include <linux/skbuff.h>

#include <net/sock.h>
#include <net/udp.h>
#include <net/tcp.h>

#include <linux/dret/ipv6.h>
#include <linux/dret/ipv6_debug.h>

#include <net/dret/addrconf.h>
#include <net/dret/raw6.h>
#include <net/dret/protocol6.h>
#include <net/dret/udp6.h>
#include <net/dret/tcp6.h>
#include <net/dret/proc6.h>
#include <net/dret/ipv6.h>
#include <net/dret/route6.h>
#include <net/dret/flow6.h>


extern void			inet_remove_sock(struct sock *sk1);
extern int			inet_stream_connect(struct socket *sock,
						    struct sockaddr * uaddr,
						    int addr_len, int flags);
extern int			inet_dgram_connect(struct socket *sock, 
						   struct sockaddr * uaddr,
						   int addr_len, int flags);
extern int			inet_accept(struct socket *sock, 
					    struct socket *newsock, int flags);
extern unsigned int             inet_poll(struct socket *sock, 
					  poll_table *wait);
extern int                      devinet6_ioctl(unsigned int cmd, void *arg);
extern int			inet_listen(struct socket *sock, int backlog);
extern int			inet_shutdown(struct socket *sock, int how);
extern int			inet_recvmsg(struct socket *sock, 
					     struct msghdr *ubuf, 
					     int size, int flags, struct scm_cookie *scm);
extern int			inet_sendmsg(struct socket *sock, 
					     struct msghdr *msg, 
					     int size, struct scm_cookie *scm);
/*
   extern struct proto packet_prot;
   extern int raw_get_info(char *, char **, off_t, int, int);
   extern int tcp_get_info(char *, char **, off_t, int, int);
   extern int udp_get_info(char *, char **, off_t, int, int);
   int (*rarp_ioctl_hook)(unsigned int,void*) = NULL;
 */


#define min(a,b)	((a)<(b)?(a):(b))


extern struct proto_ops inet6_proto_ops;

/*
 * Add a socket into the socket tables by number.
 */

void put_sock6(unsigned short num, 
               struct sock *sk)
{
   struct sock **skp;
   unsigned long flags;
   unsigned short t;
   
   if (sk->type == SOCK_PACKET)
      return;

   sk->num = num;
   sk->next = NULL;
   t = num;
   num = num & (SOCK_ARRAY_SIZE - 1);

   /* 
    *    We can't have an interrupt re-enter here. 
    */

   save_flags(flags);
   cli();

   sk->prot->inuse += 1;
   if (sk->prot->highestinuse < sk->prot->inuse)
      sk->prot->highestinuse = sk->prot->inuse;

   if (sk->prot->sock_array[num] == NULL) {
      sk->prot->sock_array[num] = sk;
      restore_flags(flags);
      return;
   }
   restore_flags(flags);

   /*
    * add the socket at the first place in the sock_array[]..
    */

   cli();
   skp = sk->prot->sock_array + num;
   sk->next = *skp;
   *skp = sk;
   sti();
}


/*
 * This routine must find a socket given a TCP or UDP header.
 * Everything is assumed to be in net order.
 *
 * We give priority to more closely bound ports: if some socket
 * is bound to a particular foreign address, it will get the packet
 * rather than somebody listening to any address..
 */

struct sock *get_sock6(struct proto *prot, 
		       unsigned short num,  struct in6_addr *raddr,
		       unsigned short rnum, struct in6_addr *laddr)
{
   struct sock *s;
   struct sock *result = NULL;
   int badness = -1;
   unsigned short hnum;

   hnum = ntohs(num);

   /*
    * SOCK_ARRAY_SIZE must be a power of two.  This will work better
    * than a prime unless 3 or more sockets end up using the same
    * array entry.  This should not be a problem because most
    * well known sockets don't overlap that much, and for
    * the other ones, we can just be careful about picking our
    * socket number when we choose an arbitrary one.
    */

   for (s = prot->sock_array[hnum & (SOCK_ARRAY_SIZE - 1)]; s != NULL; s = s->next) {
      int score = 0;

      if (s->num != hnum)
	 continue;

      if (s->dead && (s->state == TCP_CLOSE)){
		printk(KERN_DEBUG "dead or closed socket\n");
		continue;
      }

      /* local address matches? */
      if (!IS_ANYADDR6(s->net_pinfo.af_inet6.rcv_saddr)) {
	 if (!SAME_ADDR6(s->net_pinfo.af_inet6.rcv_saddr, *laddr))
	    continue;
	 score++;
      }
      /* remote address matches? */
      if (!IS_ANYADDR6(s->net_pinfo.af_inet6.pi6_fl.fll_da)) {
	 if (!SAME_ADDR6(s->net_pinfo.af_inet6.pi6_fl.fll_da, *raddr))
	    continue;
	 score++;
      }
      /* remote port matches? */
      if (s->dummy_th.dest) {
	 if (s->dummy_th.dest != rnum)
	    continue;
	 score++;
      }
      /* perfect match? */
      if (score == 3)
	 return s;

      /* no, check if this is the best so far.. */
      if (score <= badness)
	 continue;
      result = s;
      badness = score;
   }
   return result;
}

/*
 *    Deliver a datagram to raw sockets.
 */

struct sock *get_sock_raw6(struct sock *sk,
			   unsigned short num,
			   struct in6_addr *raddr,
			   struct in6_addr *laddr)
{
   struct sock *s;

   s = sk;

   for (; s != NULL; s = s->next) {
      if (s->num != num)
	 continue;

      if (s->dead && (s->state == TCP_CLOSE))
	 continue;

      if (!IS_ANYADDR6(s->net_pinfo.af_inet6.pi6_fl.fll_da) 
	  && SAME_ADDR6(s->net_pinfo.af_inet6.pi6_fl.fll_da, *raddr))
	 continue;

      if (!IS_ANYADDR6(s->net_pinfo.af_inet6.rcv_saddr) && SAME_ADDR6(s->net_pinfo.af_inet6.rcv_saddr, *laddr))
	 continue;

      return (s);
   }
   return (NULL);
}


/*
 *     Deliver a datagram to broadcast/multicast sockets.
 *     Pass the first sock of a list.
 */
 
struct sock *get_sock6_mcast(struct sock *sk,
			     unsigned short num,
			     unsigned short rnum,
			     struct in6_addr *raddr,
			     struct in6_addr *laddr)
/* num : check byte order... */
{
   struct sock *s;
   struct ipv6_pinfo *np;
   /*unsigned short hnum = ntohs(num); */

   /*
    * SOCK_ARRAY_SIZE must be a power of two.  This will work better
    * than a prime unless 3 or more sockets end up using the same
    * array entry.  This should not be a problem because most
    * well known sockets don't overlap that much, and for
    * the other ones, we can just be careful about picking our
    * socket number when we choose an arbitrary one.
    */
   for (s = sk; s; s = s->next) {
      if (s->num != num)
	 continue;

      if (s->dead && (s->state == TCP_CLOSE))
	 continue;

      if (s->dummy_th.dest)
	 if (s->dummy_th.dest != rnum)
	    continue;

      np = &s->net_pinfo.af_inet6;
      if (!IS_ANYADDR6(np->pi6_fl.fll_da))
	 if (!SAME_ADDR6(np->pi6_fl.fll_da, *raddr))
	    continue;

      if (!IS_ANYADDR6(np->rcv_saddr))
	 if (!SAME_ADDR6(np->rcv_saddr, *laddr))
	    continue;

      return (s);
   }
   return (NULL);
}


/* ---------------------------------------------------------------------
 *    The routines beyond this point handle the behaviour of an AF_INET6
 *      socket object. Mostly it punts to the subprotocols of IP to do
 *      the work.
 */



/*
 *    Default callbacks for user INET6 sockets. These just wake up
 *      the user owning the socket.
 */

static void def6_callback3(struct sock *sk)
{
   if (!sk->dead) {
      wake_up_interruptible(sk->sleep);
      sock_wake_async(sk->socket, 2);
   }
}


/*
 *    Create an inet6 socket.
 */

int inet6_create(struct socket *sock, int protocol)
{
   struct sock *sk;
   struct proto *prot = NULL;
   int err;
   
   printk(KERN_DEBUG "[AF_INET6]inet6_create IN\n");
   
   sk = sk_alloc(GFP_KERNEL);
   if (sk == NULL)
      return (-ENOBUFS);
 
   memset(sk, 0, sizeof(*sk));	/* Efficient way to set most fields to zero */

   /*
    *    Note for tcp that also wiped the dummy_th block for us.
    */

   switch (sock->type) {

   case SOCK_STREAM:
   case SOCK_SEQPACKET:
      if (protocol && protocol != IPPROTO_TCP) {
	 sk_free(sk);
	 return (-EPROTONOSUPPORT);
      }
      protocol = IPPROTO_TCP;
      sk->no_check = TCP_NO_CHECK;
      prot = &tcp6_prot;
      break;

   case SOCK_DGRAM:
      if (protocol && protocol != IPPROTO_UDP) {
	 sk_free(sk);
	 return (-EPROTONOSUPPORT);
      }
      protocol     = IPPROTO_UDP;
      sk->no_check = UDP_NO_CHECK;
      prot         = &udp6_prot;
      break;

   case SOCK_RAW:
      if (!suser()) {
	 sk_free(sk);
	 return (-EPERM);
      }
      if (!protocol) {
	 sk_free(sk);
	 return (-EPROTONOSUPPORT);
      }
      prot      = &raw6_prot;
      sk->reuse = 1;
      sk->num   = protocol;             /* for example IPPROTO_ICMPv6 */
      break;

   default:
      sk_free(sk);
      return (-ESOCKTNOSUPPORT);
   } /* switch */
   
   sock_init_data(sock,sk); 
   
   printk(KERN_DEBUG "[AF_INET6]inet6_proto_ops %p\n", &inet6_proto_ops); 
  
   sock->ops = &inet6_proto_ops;
   
   sk->zapped=0;
#ifdef CONFIG_TCP_NAGLE_OFF
   sk->nonagle = 1;
#endif  
  
   sk->family   = AF_INET6;
   sk->protocol = protocol;

   sk->prot     = prot;
   sk->backlog_rcv = prot->backlog_rcv;
   
   sk->timer.data     = (unsigned long) sk;
   sk->timer.function = &net_timer;

   sk->write_space    = def6_callback3;

	/*
	 *	init the ipv4 part of the socket since
	 *	we can have sockets using v6 API for ipv4
	 */

   sk->ip_ttl=IPDEFTTL;

   /* FIXME 
   if (sk->type==SOCK_RAW && protocol==IPPROTO_RAW)
		sk->ip_hdrincl=1;
   */
   sk->net_pinfo.af_inet6.hop_limit = IPV6_DEFAULT_HOP_LIMIT;
    
#ifdef CONFIG_IP_MULTICAST
   sk->net_pinfo.af_inet6.mcast_hops = IPV6_DEFAULT_MULTICAST_HOPS;
   sk->net_pinfo.af_inet6.mc_loop    = IPV6_DEFAULT_MULTICAST_LOOP;

   sk->ip_mc_loop  = IP_DEFAULT_MULTICAST_LOOP;
   sk->ip_mc_ttl   = IP_DEFAULT_MULTICAST_TTL;
   sk->ip_mc_index = 0;
   sk->ip_mc_list  = NULL;
#endif

    if (sk->num) {
      /*
       * It assumes that any protocol which allows
       * the user to assign a number at socket
       * creation time automatically
       * shares.
       */
      put_sock6(sk->num, sk);
      sk->dummy_th.source = ntohs(sk->num);
   }
   
   if (sk->prot->init) {
      err = sk->prot->init(sk);
      if (err != 0) {
	 destroy_sock(sk);
	 return (err);
      }
   }
   printk(KERN_DEBUG "[AFINET6]inet6_create OUT (ok)\n");
   return (0);
}

/*
 *    Duplicate a socket.
 */

int inet6_dup(struct socket *newsock, struct socket *oldsock)
{
   return (inet6_create(newsock, oldsock->sk->protocol));
}

/*
 *    The peer socket should always be NULL (or else). When we call this
 *      function we are destroying the object and from then on nobody
 *      should refer to it.
 */

int inet6_release(struct socket *sock, struct socket *peer)
{
   unsigned long timeout;
   struct sock *sk =  sock->sk;

   if (sk == NULL)
      return (0);

   sk->state_change(sk);

   /* Start closing the connection.  This may take a while. */

#ifdef CONFIG_IP_MULTICAST
   /* Applications forget to leave groups before exiting */
   ip_mc_drop_socket(sk);
#endif
   /*
    * If linger is set, we don't return until the close
    * is complete.  Otherwise we return immediately. The
    * actually closing is done the same either way.
    *
    * If the close is due to the process exiting, we never
    * linger..
    */
   timeout = 0;
   if (sk->linger) {
      timeout = ~0UL;
      if (!sk->lingertime)
	 timeout = jiffies + HZ * sk->lingertime;
   }
   if (current->flags & PF_EXITING)
      timeout = 0;

   sock->sk = NULL;
   sk->socket = NULL;

   sk->prot->close(sk, timeout);
   return (0);
}


int inet6_bind(struct socket *sock, struct sockaddr *uaddr,
	       int addr_len)
{
   struct sockaddr_in6 *mysin6 = (struct sockaddr_in6 *) uaddr;
   struct sock *sk = sock->sk;
   unsigned short snum = 0;
   __u32 myv4addr = 0;


   /*
    *    If the socket has its own bind function then use it.
    */

   if (sk->prot->bind)
      return sk->prot->bind(sk, uaddr, addr_len);

   /* check this error. */
   if (sk->state != TCP_CLOSE)
      return (-EIO);

   if (addr_len != sizeof(struct sockaddr_in6))
       return (-EINVAL);

   if (sock->type != SOCK_RAW) {
      if (sk->num != 0)
	 return (-EINVAL);
	 
      snum = ntohs(mysin6->sin6_port);
      
      if (snum == 0)
	 snum = get_new_socknum(sk->prot, 0);
	 
      if (snum < PROT_SOCK && !suser())
	 return (-EACCES);
   }
   
   /* We must verify that the type of socket is correct with the adress */

   if (IS_MULTIADDR6 (mysin6->sin6_addr) && sock->type == SOCK_STREAM)
      return (-EINVAL);

	/*
	 *	check if the address belongs to the host
	 */

   if (IS_MAPPEDADDR6(mysin6->sin6_addr)) {
      myv4addr = mysin6->sin6_addr.s6_addr32[3];  
      if (IN_LOOPBACK(myv4addr))
	 return(-EADDRNOTAVAIL);
   } else {
      if (!IS_ANYADDR6 (mysin6->sin6_addr)) {	
	 /*  ipv4 addr of the socket is invalid.
	  *	only the unpecified and mapped address	
	  *	have a v4 equivalent.
	  */
	 myv4addr = LOOPBACK4_IPV6;
	 if (!IS_MULTIADDR6 (mysin6->sin6_addr)){
	    if (!check_host_addr6(&mysin6->sin6_addr))
	       return(-EADDRNOTAVAIL);
	 }
      }
   }

   sk->rcv_saddr = myv4addr;
   sk->saddr = myv4addr;

   /* tells the adress to the socket */
   COPY_ADDR6(sk->net_pinfo.af_inet6.rcv_saddr, mysin6->sin6_addr);

   if (!IS_MULTIADDR6 (mysin6->sin6_addr))
      COPY_ADDR6(sk->net_pinfo.af_inet6.pi6_fl.fll_sa, mysin6->sin6_addr);


   /* FIXME NOT used
   sk->net_pinfo.af_inet6.dest = NULL;
   */
   
   if (sock->type != SOCK_RAW) { /* for datagramm and stream */
      struct sock *sk2;

      /* Make sure we are allowed to bind here. */
      cli();
      for (sk2 = sk->prot->sock_array[snum & (SOCK_ARRAY_SIZE - 1)];
	   sk2 != NULL; sk2 = sk2->next) {
	 
	 /*    Hash collision or real match ?*/
	 if (sk2->num != snum)
	    continue;

	 /*
	  *    Either bind on the port is wildcard means
	  *      they will overlap and thus be in error
	  */

	 if ((!IS_ANYADDR6 (mysin6->sin6_addr)) || (!sk2->rcv_saddr)  ||
	     !IS_ANYADDR6(sk2->net_pinfo.af_inet6.rcv_saddr)) {
	    
	    /* Allow only if both are setting reuse. */
	    if (sk2->reuse && sk->reuse &&
		sk2->state != TCP_LISTEN)
	       continue;
	       
	    sti();
	    return (-EADDRINUSE);
	 }
	 
	 /* Two binds match ? */
	 if (SAME_ADDR6(sk2->net_pinfo.af_inet6.rcv_saddr,
			sk->net_pinfo.af_inet6.rcv_saddr))
	    continue;
	 
	 /*   Reusable port ? */
	 if (!sk->reuse) {
	    sti();
	    return (-EADDRINUSE);
	 }
	 
	 /*   Reuse ? */
	 if (!sk2->reuse || sk2->state == TCP_LISTEN) {
	    sti();
	    return (-EADDRINUSE);
	 }
      } /* for */
      sti();

      inet_remove_sock(sk);

/*              if(sock->type==SOCK_DGRAM)
   			udp_cache_zap();
    		if(sock->type==SOCK_STREAM)
     			tcp_cache_zap();
 */
      
      put_sock6(snum, sk);
      sk->dummy_th.source = ntohs(sk->num);
      sk->daddr = 0;		/*FIXME */
      sk->dummy_th.dest = 0;
   } /* if */
   
   return (0);
}

/*
 *    Connect to a remote host. There is regrettably still a little
 *      TCP 'magic' in here.
 */
int inet6_connect(struct socket *sock, struct sockaddr *uaddr,
		  int addr_len, int flags)
{
   struct sock *sk =  sock->sk;
   int err = 0;
   
   switch(sk->type) {
      case SOCK_STREAM:
      case SOCK_SEQPACKET:
	 err = inet_stream_connect(sock, uaddr, addr_len, flags);
	 break;
      
      case SOCK_DGRAM:
      case SOCK_RAW:
	 err = inet_dgram_connect(sock, uaddr, addr_len, flags);
         break;
	 
      default:
	 return(-ESOCKTNOSUPPORT);
   }
   return (err);
}

int inet6_socketpair(struct socket *sock1, struct socket *sock2)
{
   return (-EOPNOTSUPP);
}

/*
 *    This does both peername and sockname.
 */

int inet6_getname(struct socket *sock, struct sockaddr *uaddr,
		  int *uaddr_len, int peer)
{
   struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *) uaddr;
   struct sock *sk;

   sin6->sin6_family = AF_INET6;
   sk =  sock->sk;
   
   if (peer) {
      if (!tcp_connected(sk->state)) 
              return(-ENOTCONN); 
              
      sin6->sin6_port = sk->dummy_th.dest;
      COPY_ADDR6(sin6->sin6_addr, sk->net_pinfo.af_inet6.pi6_fl.fll_da);
  
   } else {

      sin6->sin6_port = sk->dummy_th.source;
      COPY_ADDR6(sin6->sin6_addr, sk->net_pinfo.af_inet6.rcv_saddr);   
      
   }
   
   *uaddr_len = sizeof(*sin6);
   return (0);
}


unsigned int inet6_select(struct socket *sock, poll_table * wait)
{
   struct sock *sk =  sock->sk;
   int err = 0;
   
   switch(sk->type) {
      case SOCK_STREAM:
      case SOCK_SEQPACKET:
	 err = inet_poll(sock, wait);
	 break;
      
      case SOCK_DGRAM:
      case SOCK_RAW:
	 err = datagram_poll(sock, wait);
         break;
	 
      default:
	 return(-ESOCKTNOSUPPORT);
   }
   return (err);
}
   
/*
 *    ioctl() calls you can issue on an INET socket. Most of these are
 *      device configuration and stuff and very rarely used. Some ioctls
 *      pass on to the socket itself. The same for INET6.
 *
 *      NOTE: I like the idea of a module for the config stuff. ie ifconfig
 *      loads the devconfigure module does its configuring and unloads it.
 *      There's a good 20K of config code hanging around the kernel.
 *  NEED ADAPTATION 
 */

int inet6_ioctl(struct socket *sock, unsigned int cmd, unsigned long arg)
{
   struct sock *sk =  sock->sk;
   int err;
   int pid;

    printk(KERN_DEBUG "[AF_INET6]inet6_ioctl IN \n");
    
    switch (cmd) {
   case FIOSETOWN:
   case SIOCSPGRP:
      err = get_user(pid, (int *) arg);
      if (err)
	 return err;
      /* see inet6_fcntl */
      if (current->pid != pid && current->pgrp != -pid && !suser())
	 return -EPERM;
      sk->proc = pid;
      return (0);
      
   case FIOGETOWN:
   case SIOCGPGRP:
      err = put_user(sk->proc, (int *) arg);
      return err;		/* 0 on success, -EFAULT on error */
      
   case SIOCGSTAMP:
      if (sk->stamp.tv_sec == 0)
	 return -ENOENT;
      err = copy_to_user((void *) arg, &sk->stamp, sizeof(struct timeval));
      if (err)
	 return -EFAULT;
      return (0);
      
   case SIOCADDRT:
   case SIOCDELRT:
      return (ipv6_route_ioctl(cmd, (void *) arg));
      
   case SIOCDARP:
   case SIOCGARP:
   case SIOCSARP:
   case OLD_SIOCDARP:
   case OLD_SIOCGARP:
   case OLD_SIOCSARP:
   case SIOCDRARP:
   case SIOCGRARP:
   case SIOCSRARP:
   case SIOCGIFCONF:
   case SIOCGIFFLAGS:
   case SIOCSIFFLAGS:
   case SIOCGIFADDR:
   case SIOCSIFADDR:
   case SIOCADDMULTI:
   case SIOCDELMULTI:
   case SIOCGIFDSTADDR:    /* for p2p devices */
   case SIOCSIFDSTADDR:    /* for p2p devices */
   case SIOCGIFNETMASK:
   case SIOCSIFNETMASK:
   case SIOCGIFMETRIC:
   case SIOCSIFMETRIC:
   case SIOCGIFMEM:
   case SIOCSIFMEM:
   case SIOCGIFMTU:
   case SIOCSIFMTU:
   case SIOCSIFLINK:
   case SIOCGIFHWADDR:
   case SIOCSIFHWADDR:
   case SIOCSIFMAP:
   case SIOCGIFMAP:
   case SIOCSIFSLAVE:
   case SIOCGIFSLAVE:
   case SIOCDELADDR:
   case SIOCADDTUNNEL:     /* IPv6 in IPv4 */
   case SIOCDELTUNNEL:
   case SIOGIFINDEX:       /* Interface index */
      printk(KERN_DEBUG "[AFINET6]inet6_ioctl devinet6\n");
      return (devinet6_ioctl(cmd, (void *) arg));
   case SIOCGETLABEL:
      printk(KERN_DEBUG "[AFINET6]inet6_ioctl flow label\n");
      return (flow6_ioctl(sk, cmd, (void *) arg)); 
   case SIOCGIFBR:
   case SIOCSIFBR:
   default:
      if ((cmd >= SIOCDEVPRIVATE) &&
	  (cmd <= (SIOCDEVPRIVATE + 15)))
	 return (dev_ioctl(cmd, (void *) arg));

      if (sk->prot->ioctl == NULL)
	 return (-EINVAL);
      return (sk->prot->ioctl(sk, cmd, arg));
   }
   /*NOTREACHED */
   return (0);
}

/*
 *    Set socket options on an inet6 socket.
 */

int inet6_setsockopt(struct socket *sock, int level, int optname,
		     char *optval, int optlen)
{
   struct sock *sk = sock->sk;
   if (level == SOL_SOCKET)
      return sock_setsockopt(sock, level, optname, optval, optlen);
      
   if (sk->prot->setsockopt == NULL)
      return (-EOPNOTSUPP);
   else
      return sk->prot->setsockopt(sk, level, optname, optval, optlen);
}

/*
 *    Get a socket option on an AF_INET6 socket.
 */

int inet6_getsockopt(struct socket *sock, int level, int optname,
		     char *optval, int *optlen)
{
   struct sock *sk = sock->sk;
   if (level == SOL_SOCKET)
      return sock_getsockopt(sock, level, optname, optval, optlen);
      
   if (sk->prot->getsockopt == NULL)
      return (-EOPNOTSUPP);
   else
      return sk->prot->getsockopt(sk, level, optname, optval, optlen);
}


int inet6_fcntl(struct socket *sock, unsigned int cmd, unsigned long arg)
{
   struct sock *sk = sock->sk;

   switch (cmd) {

   case F_SETOWN:
      /*
       * This is a little restrictive, but it's the only
       * way to make sure that you can't send a sigurg to
       * another process.
       */
      if (!suser() && current->pgrp != -arg && current->pid != arg)
	 return (-EPERM);
  
      sk->proc = arg;
      return (0);

   case F_GETOWN:
      return (sk->proc);

   default:
      return (-EINVAL);
   }
}

static struct proto_ops inet6_proto_ops =
{
   AF_INET6,

   inet6_dup,
   inet6_release,
   inet6_bind,
   inet6_connect,
   inet6_socketpair,
   inet_accept,
   inet6_getname,
   inet6_select,
   inet6_ioctl,
   inet_listen,
   inet_shutdown,
   inet6_setsockopt,
   inet6_getsockopt,
   inet6_fcntl,
   inet_sendmsg,
   inet_recvmsg
};

struct net_proto_family inet6_family_ops = {
	AF_INET6,
	inet6_create
};


/*
 *    Called by socket.c on kernel startup.  
 */

extern unsigned long seq_offset;

void inet6_proto_init(struct net_proto *pro)
{
   struct inet6_protocol *p;

   printk("DRET - IPv6 for Linux\n");

   /*
    *    Tell SOCKET that we are alive... 
    */
   (void) sock_register(&inet6_family_ops);

   seq_offset = CURRENT_TIME * 250;

   /*
    *    Add all the protocols. 
    */

   ipv6_init();

   printk("IPv6 Protocols: ");
   for (p = inet6_protocol_base; p != NULL;) {
      struct inet6_protocol *tmp = (struct inet6_protocol *) p->next;
      inet6_add_protocol(p);
      printk("%s%s", p->name, tmp ? ", " : "\n");

      if (p->proto_init)
	 p->proto_init(&inet6_family_ops);

      p = tmp;
   }

   /*
    *    Create all the /proc entries.
    */

#ifdef CONFIG_PROC_FS

   proc_net6_register(&(struct proc_dir_entry) {
		      PROC_NET6_SNMP6, 5, "snmp6",
		      S_IFREG | S_IRUGO, 1, 0, 0,
		      0, &proc_net6_inode_operations,
		      snmp6_get_info
		      });

   proc_net6_register(&(struct proc_dir_entry) {
		      PROC_NET6_SOCKSTAT6, 9, "sockstat6",
		      S_IFREG | S_IRUGO, 1, 0, 0,
		      0, &proc_net6_inode_operations,
		      afinet6_get_info
		      });
   proc_net6_register(&(struct proc_dir_entry) {
		      PROC_NET6_ROUTE6, 6, "route6",
		      S_IFREG | S_IRUGO, 1, 0, 0,
		      0, &proc_net6_inode_operations,
		      route6_get_info
		      });

   proc_net6_register(&(struct proc_dir_entry) {
		      PROC_NET6_IGMPV6, 6, "igmpv6",
		      S_IFREG | S_IRUGO, 1, 0, 0,
		      0, &proc_net6_inode_operations,
		      igmpv6_procinfo
		      });

   proc_net6_register(&(struct proc_dir_entry) {
		      PROC_NET6_NDCACHE, 7, "ndcache",
		      S_IFREG | S_IRUGO, 1, 0, 0,
		      0, &proc_net6_inode_operations,
		      ndcache_procinfo
		      });
   
   proc_net6_register(&(struct proc_dir_entry) {
		      PROC_NET6_FLOWCACHE, 7, "inflows",
		      S_IFREG | S_IRUGO, 1, 0, 0,
		      0, &proc_net6_inode_operations,
		      flow_cache_procinfo
		      });
#endif
}



