/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:     $Id$ 
 *      
 * Authors:     Pedro Roque             <roque@di.fc.ul.pt>     
 *
 *
 * Fixes:
 *              Pascal Anelli           add functions to configure SIT
 *
 * Description:
 *              IPv6 Address [auto]configuration
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */


/*-----------------------------------------------------------------------
|
|  ipv6_add_dev
|  ipv6_add_addr
|  ipv6_get_addr
|  addrconf_notify
|  sit_add_v4_addrs
|  sit_route_add
|
-------------------------------------------------------------------------*/
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/sched.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/if.h>

#include <linux/netdevice.h>
#include <linux/if_arp.h>

#include <net/sock.h>
#include <net/snmp.h>

#include <net/dret/ipv6.h>
#include <net/dret/nd.h>
#include <net/dret/protocol6.h>
#include <net/dret/route6.h>
#include <net/dret/addrconf.h>
#include <linux/dret/netdevice6.h>
#include <net/netlink.h>

#include <linux/dret/ipv6_debug.h>

#ifdef IPV6_DEBUG_ADDRCONF
#define DEBUG(format, args...) printk(KERN_DEBUG format,##args)
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[ADDRCONF]%X:%X:%X:%X\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);
#else
#define DEBUG(format, args...) 
#define PRINT_ADDR(a)
#endif

const struct in6_addr in6addr_any = IN6ADDR_ANY_INIT;
const struct in6_addr in6addr_loopback = IN6ADDR_LOOPBACK_INIT;

/*
 *      Hash list of configured unicast address.
 *      This is a device independent list.
 */
struct inet6_ifaddr *inet6_addr_lst[16];
/*
 *      Hash list of configured multicast addresses
 */
struct ipv6_mc_list *inet6_mcast_lst[16];

struct inet6_dev *inet6_dev_lst;

struct destination *lo_route = NULL;

/* %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   LISTS MANAGEMENT

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */

/*--START FUNCTION--(router_dev_init)------------------------------------
   
    Called from ipv6_add_dev.
    Initialize the router parameter of an interface.

    Last update:
    08/08/96 bb

--------------------------------------------------------------------------*/
static struct router_conf *router_dev_init(struct inet6_dev *idev)
{

   struct router_conf *rc;

   if ((rc = (struct router_conf *) kmalloc(sizeof(*rc), GFP_KERNEL)) == NULL)
      return NULL;

   memset(rc, 0, sizeof(*rc));
   rc->MaxRtrAdvInterval = 600 * HZ;
   rc->MinRtrAdvInterval = 0.33 * rc->MaxRtrAdvInterval;
   rc->AdvCurHopLimit = 255;	/* FIXME */
   rc->AdvDefaultLifetime = 3 * rc->MaxRtrAdvInterval;

   /* 
    * So far, no clue about how the prefix list is initialized...
    */

#ifdef CONFIG_IP_FORWARD
/*   start_advertise(idev); */
#endif

   return (idev->router1 = rc);
}
/*--END FUNCTION--(router_dev_init)---------------------------------------*/


/*--START FUNCTION--(ipv6_add_dev)-----------------------------------------
|
| ipv6_add_dev: add an inet6_dev structure to inet6_dev_lst.
| Authors: P.Roque
|          B. Brodard
| Last modified: 05/22/96.
|
---------------------------------------------------------------------------*/
struct inet6_dev *ipv6_add_dev(struct device *dev)
{
   struct inet6_dev *idev;

#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]ipv6_add_dev: IN \n");
#endif
   /*
    *      called by netdev notifier from a syscall so i guess
    *      GFP_KERNEL is ok here.
    */
   idev = (struct inet6_dev *) kmalloc(sizeof(struct inet6_dev),
				       GFP_KERNEL);

   if (idev == NULL)
      return NULL;

   memset(idev, 0, sizeof(struct inet6_dev));
   idev->dev = dev;
   dev->dev6_lst = idev;

   /*
    *      insert at head.
    */
   idev->next = inet6_dev_lst;
   inet6_dev_lst = idev;

   if (dev->type == ARPHRD_ETHER) {
      idev->router1 = router_dev_init(idev);
   }
#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]ipv6_add_dev: OUT \n");
#endif
   return idev;
}
/*--END FUNCTION--(ipv6_add_dev)-----------------------------------------*/


/*--START FUNCTION--(ipv6_add_addr)-----------------------------------------
|
| ipv6_add_addr: add an inet6_ifaddr structure to a list of inet6_ifaddr.
| Authors: P.Roque
|          B. Brodard
| Last modified: 05/22/96.
|
---------------------------------------------------------------------------*/
struct inet6_ifaddr *ipv6_add_addr(struct inet6_dev *idev,
				   struct in6_addr *addr,
				   int scope, int plen)
{
   struct inet6_ifaddr *ifaddr;
   struct in6_addr smc_addr;
   __u8 hash;

   DEBUG("[ADDRCONF]ipv6_add_addr: IN \n");

   
   for (ifaddr = idev->addr_list; ifaddr; ifaddr = ifaddr->if_next) {
      if (SAME_ADDR6(ifaddr->addr, *addr)) {
	 DEBUG("Address already in use on this device\n");
	 return NULL;
      }
   }

   if (IS_MULTIADDR6(*addr))
      return (NULL);

   if ((plen < 1) || (plen > 128))
      return (NULL);
   
   /*
   if (((IS_IPV4ADDR6(*addr)) && (idev->dev->type != ARPHRD_SIT))
       ||
       ((!IS_IPV4ADDR6(*addr) && (idev->dev->type == ARPHRD_SIT)))) {
      printk(KERN_DEBUG "[ADDRCONF]ipv6_add_addr: Bad addr. assignement\n");
      return NULL;
   }
   */
   /*
    * Allocate space for struct inet6_ifaddr
    */
   ifaddr = (struct inet6_ifaddr *) kmalloc(sizeof(struct inet6_ifaddr),
					    GFP_KERNEL);

   if (ifaddr == NULL) {
      DEBUG("[ADDRCONF]ipv6_add_addr: malloc failed\n");
      return NULL;
   }
   memset(ifaddr, 0, sizeof(struct inet6_ifaddr));
   memcpy(&ifaddr->addr, addr, sizeof(struct in6_addr));


   ifaddr->scope = scope;
   ifaddr->idev = idev;
   ifaddr->plen = plen;
   /* 
    * Add to the global address list 
    */

   hash = ipv6_addr_hash(addr);
   cli();
   ifaddr->lst_next = inet6_addr_lst[hash];
   inet6_addr_lst[hash] = ifaddr;

   /*
    * Add to inet6_dev unicast addr list (list for the device) 
    */
   ifaddr->if_next = idev->addr_list;
   idev->addr_list = ifaddr;
   sti();
   /*
    * Join all solicited-node multicast address
    * 7.2 Address resolution
    */
   /* Make a define of these addresses */
   if ((idev->dev->type == ARPHRD_ETHER) && (scope != IFA_LINK)) {
      MULTISOL_ADDR6(smc_addr, *addr);
      ipv6_dev_mc_inc(idev->dev, &smc_addr);
   }
   /*
    * Netlink
    */
#ifdef CONFIG_NET_RTNETLINK6   
   ipv6_netlink_msg(RTMSG_NEWADDR, addr, 0, plen, 0, 0, idev->dev->name);
#endif
   
#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]ipv6_add_addr: OUT \n");
#endif
   return ifaddr;
}

/*--END FUNCTION--(ipv6_add_addr)-------------------------------------------*/

/*--START FUNCTION--(ipv6_del_addr)-----------------------------------------
|
| ipv6_del_addr: delete an inet6_ifaddr structure to a list of inet6_ifaddr.
|                address is deleted independently of its scope and plen.
| Authors: B. Brodard
| Last modified: 05/22/96.
|
---------------------------------------------------------------------------*/
int ipv6_del_addr(struct inet6_dev *idev, struct in6_addr *addr)
{
   struct inet6_ifaddr *ifaddr, *tmp_addr = NULL, *tmp2_addr = NULL;
   __u8 hash;

#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]ipv6_del_addr: IN \n");
#endif
   if (idev == NULL) {
      printk(KERN_DEBUG "[ADDRCONF]ipv6_del_addr: "
	     "No such IPv6 device initialized\n");
      return (-ENODEV);
   }
   if (idev->addr_list) {
      /* Dequeue from device address list */
      cli();
      for (ifaddr = idev->addr_list; ifaddr;
	   tmp_addr = ifaddr, ifaddr = ifaddr->if_next)
	 if (SAME_ADDR6(ifaddr->addr, *addr))
	    break;

      if (!ifaddr) {
	 sti();
	 return -EADDRNOTAVAIL;
      }

      if (!tmp_addr)
	 idev->addr_list = ifaddr->if_next;
      else
	 tmp_addr->if_next = ifaddr->if_next;
      sti();
   } else
      return -EADDRNOTAVAIL;

   /* Dequeue from global list */
   hash = ipv6_addr_hash(addr);
   cli();
   for (tmp_addr = inet6_addr_lst[hash];
	tmp_addr != ifaddr;
	tmp_addr = tmp_addr->lst_next)
      tmp2_addr = tmp_addr;
   
   if (tmp2_addr)
      tmp2_addr->lst_next = ifaddr->lst_next;
   else
      inet6_addr_lst[hash] = ifaddr->lst_next;
   sti();

   /*
    * Netlink
    */
#ifdef CONFIG_NET_RTNETLINK6   
   ipv6_netlink_msg(RTMSG_DELADDR, addr, 0, ifaddr->plen, 0, 0,
		    idev->dev->name);
#endif
   kfree(ifaddr);

#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]ipv6_del_addr: OUT \n");
#endif
   return 0;
}
/*--END FUNCTION--(ipv6_del_addr)-------------------------------------------*/

/*--START FUNCTION--(check_host_addr6)-----------------------------------------
|
| check_host_addr6
|    Determines if an address belongs to the host.
|
| Authors: 
|          
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
int check_host_addr6(struct in6_addr *a)
{
   struct inet6_ifaddr *ifaddr;
   __u8 hash;

#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]check_host_addr6: IN \n");
   printk(KERN_DEBUG "[ADDRCONF]%X:%X:%X:%X \n", 
	  a->s6_addr32[0],
	  a->s6_addr32[1],
	  a->s6_addr32[2],
	  a->s6_addr32[3]);
#endif

   hash = ipv6_addr_hash(a);
   for (ifaddr = inet6_addr_lst[hash]; ifaddr; ifaddr = ifaddr->lst_next) {
      if SAME_ADDR6 (*a, ifaddr->addr) {
#ifdef IPV6_DEBUG_ADDRCONF
	 printk(KERN_DEBUG "[ADDRCONF]OUT ->1 \n");
#endif
	 return (1);
	 }
   }
#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]OUT ->0 \n");
#endif
   return (0);
}
/*--END FUNCTION--(check_host_addr6)---------------------------------------*/

/*--START FUNCTION--(get_ifaddr6)-----------------------------------------
|
| 
|    Determines if an address belongs to the host.
|    Returns its inet6_ifaddr structure or NULL.
|
| Authors: 
|          B. Brodard
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
struct inet6_ifaddr *get_ifaddr6(struct in6_addr *a)
{
   struct inet6_ifaddr *ifaddr;
   __u8 hash;

#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]get_ifaddr6: IN \n");
#endif

   hash = ipv6_addr_hash(a);
   for (ifaddr = inet6_addr_lst[hash]; ifaddr; ifaddr = ifaddr->lst_next) {
      if SAME_ADDR6 (*a, ifaddr->addr) {
#ifdef IPV6_DEBUG_ADDRCONF
	 printk(KERN_DEBUG "[ADDRCONF]get_ifaddr6: address found OUT \n");
#endif
	 return (ifaddr);
	 }
   }
#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]get_ifaddr6: address not found OUT \n");
#endif
   return (NULL);
}
/*--END FUNCTION--(get_ifaddr6)---------------------------------------*/


/*--START FUNCTION--(ipv6_get_saddr)-----------------------------------------
|
| ipv6_get_addr: 
|      Choose an apropriate source address
|      should do:
|      i)      get an address with an apropriate scope
|         appropriate being:
              a) There is a non-zero prob. that this address can be
	      used by the dest. to reply to the source.
|      ii)     see if there is a specific route for the destination and use
|              an address of the attached interface 
|      iii)    don't use deprecated addresses
|
|      at the moment i believe only iii) is missing.
|
| Authors: P.Roque
|          B. Brodard
| Last modified: 10/22/96.
|
-----------------------------------------------------------------------------*/
struct inet6_ifaddr *ipv6_get_saddr(struct device *dev, struct in6_addr *d)
{
   struct inet6_ifaddr *ifp1, *ifp2 = NULL;
   struct inet6_dev *idev;
   int min_scope = 0, max_scope = 5, scope, i;

   if (dev == NULL) {
      printk(KERN_DEBUG "[ADDRCONF]ipv6_get_saddr: Device parameter is NULL \n");
      return NULL;
   }
   
   /*
    * Unless device is SIT, return an address of the device
    */
#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]ipv6_get_saddr: IN dev = %s \n", dev->name);
   printk(KERN_DEBUG "%8X:%8X:%8X:%8X\n", d->s6_addr32[0],
	  d->s6_addr32[1],
	  d->s6_addr32[2],
	  d->s6_addr32[3]);
#endif
   scope = ipv6_addr_scope(d);
#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "Scope = %d\n", scope);
#endif

   if ((dev->flags & (IFF_TUNNEL)) == 0) {
      if ((idev = dev->dev6_lst) == NULL) {
	 printk(KERN_DEBUG "No idev structure for %s\n", dev->name);
	 return NULL;
      }
      for (ifp1 = idev->addr_list; ifp1; ifp1 = ifp1->if_next) {
	 if (ifp1->scope == scope) {
#ifdef IPV6_DEBUG_ADDRCONF
	    printk(KERN_DEBUG "[ADDRCONF]ipv6_get_saddr: OUT \n");
	    PRINT_ADDR(&ifp1->addr);
#endif
	    return ifp1;
	 }
	 if ((ifp1->scope > scope) && (ifp1->scope < max_scope)) {
	    ifp2 = ifp1;
	    max_scope = ifp1->scope;
	 } else if ((max_scope == 5) && (ifp1->scope > min_scope)) {
	    ifp2 = ifp1;
	    min_scope = ifp1->scope;
	 }
      }

      if (ifp2)
	 if (((ifp2->scope == IFA_HOST) && ((dev->flags & IFF_LOOPBACK) == 0))
	     ||
	     ((ifp2->scope == IFA_LINK) && (scope > IFA_LINK))) {
	    printk(KERN_DEBUG "No saddr with appropriate scope found...\n");
	    return NULL;
	 }
      return ifp2;
   }
   /* Interface is a tunnel or loopback */
   for (i = 0; i < 16; i++) {
      for (ifp1 = inet6_addr_lst[i]; ifp1; ifp1 = ifp1->lst_next) {
	 if (ifp1->scope == scope) {
#ifdef IPV6_DEBUG_ADDRCONF
	    printk(KERN_DEBUG "[ADDRCONF]ipv6_get_saddr: OUT \n");
	    PRINT_ADDR(&ifp1->addr);
#endif
	    return ifp1;
	 }
	 if ((ifp1->scope > scope) && (ifp1->scope < max_scope)) {
	    ifp2 = ifp1;
	    max_scope = ifp1->scope;
	 } else if ((max_scope == 5) && (ifp1->scope > min_scope)) {
	    ifp2 = ifp1;
	    min_scope = ifp1->scope;
	 }
      }
   }
   if (ifp2)
      if ((ifp2->scope == IFA_HOST) 
	   ||
	  ((ifp2->scope == IFA_LINK) && (scope > IFA_LINK))) {
	 printk(KERN_DEBUG "No saddr with appropriate scope found...\n");
	 return NULL;
      }
   return ifp2;
}
/*--END FUNCTION--(ipv6_get_saddr)-------------------------------------------*/

/*--START FUNCTION--(addrconf_cleanup)-----------------------------------------
|
|  addrconf_cleanup: used if binary as module.
|
| Authors: P.Roque
|          B. Brodard
| Last modified: 05/22/96 (bb).
|
-----------------------------------------------------------------------------*/
void addrconf_cleanup(void)
{
   struct inet6_dev *idev, *bidev;
   struct inet6_ifaddr *ifa, *bifa;
   int i;
   /*
    *  Clean dev list.
    */
   for (idev = inet6_dev_lst; idev;) {
      bidev = idev;
      idev = idev->next;
      kfree(bidev);
   }
   /*
    *  Clean addr_list
    */
   for (i = 0; i < 16; i++) {
      for (ifa = inet6_addr_lst[i]; ifa;) {
	 bifa = ifa;
	 ifa = ifa->lst_next;
	 kfree(bifa);
      }
   }
}
/*--END FUNCTION--(addrconf_cleanup)----------------------------------------*/


/*--START FUNCTION--(sit_add_v4_addrs)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         nom
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
static void sit_add_v4_addrs(struct inet6_dev *idev)
{
   struct inet6_ifaddr *ifp;
   struct in6_addr addr;
   struct device *dev;
   /*   int flag; */

#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]sit_add_v4_addrs: IN \n");
#endif
   memset(&addr, 0, sizeof(struct in6_addr));

   for (dev = dev_base; dev != NULL; dev = dev->next) {
      if (dev->family == AF_INET && (dev->flags & IFF_UP)) {
	 int flag = IFA_GLOBAL | IPV6_ADDR_COMPATV4;
#ifdef IPV6_DEBUG_ADDRCONF
	 printk(KERN_DEBUG "[ADDRCONF]sit_add_v4_addrs dev = %s\n", dev->name);
#endif
	 addr.s6_addr32[3] = dev->pa_addr;

	 if (dev->flags & IFF_LOOPBACK)
	    flag = IFA_HOST;
	 else
	    flag = IFA_GLOBAL;

	 ifp = ipv6_add_addr(idev, &addr, flag, 96);

	 if (ifp == NULL)
	    continue;

	 ifp->status = DAD_COMPLETE;
      }
   }
#ifdef IPV6_DEBUG_ADDRCONF   
   printk(KERN_DEBUG "[ADDRCONF]sit_add_v4_addrs: OUT \n");
#endif
}
/*--END FUNCTION--(sit_add_v4_addrs)---------------------------------------*/


/*--START FUNCTION--(sit_route_add)-----------------------------------------
|
| Description:
|    
|
| Authors: 
|         nom
|
| Last modified: 07/17/96
|
-----------------------------------------------------------------------------*/
static void sit_route_add(struct device *dev)
{
   struct in6_rtmsg rtmsg;
   char devname[] = "sit0";
   int err;

#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]sit_route_add: IN \n");
#endif
   rtmsg.rtmsg_type = RTMSG_NEWROUTE;

   memset(&rtmsg.rtmsg_dst, 0, sizeof(struct in6_addr));
   memset(&rtmsg.rtmsg_gateway, 0, sizeof(struct in6_addr));

   /* prefix length - 96 bytes "::d.d.d.d" */
   rtmsg.rtmsg_prefixlen = 96;
   rtmsg.rtmsg_metric = 1;
   rtmsg.rtmsg_flags = RTF_NEXTHOP | RTF_UP;
   strcpy(rtmsg.rtmsg_device, devname);


   err = ipv6_route_add(&rtmsg);

   if (err) {
      printk(KERN_DEBUG "sit_route_add: error in route_add\n");
   }
#ifdef IPV6_DEBUG_ADDRCONF   
   printk(KERN_DEBUG "[ADDRCONF]sit_route_add: OUT \n");
#endif
}
/*--END FUNCTION--(sit_route_add)---------------------------------------*/


/*--START FUNCTION--(start_addr_config)-------------------------------------
|
|
|
----------------------------------------------------------------------------*/
static void start_addr_config(struct inet6_dev *idev)
{
   /*struct inet6_ifaddr * ifp;
      struct in6_addr addr; */
   /*   struct device *dev = idev->dev;
#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]start_addr_config IN dev = %s\n", dev->name);
#endif   */
/* generate link local address */
/*
   memset(&addr, 0, sizeof(struct in6_addr));
   addr.s6_addr[0] = 0xFE;
   addr.s6_addr[1] = 0x80;
   memcpy(addr.s6_addr + (16 - dev->addr_len), dev->dev_addr, dev->addr_len);
   if ((ifp =ipv6_add_addr(idev, &addr, IFA_LINK, 10))==NULL) {  
   return;
   }

   ifp->status = TENTATIVE_ADDR6;
      ifp->plen = 10;        *//* RFC 1884 2.4.8 */

}

/*--END FUNCTION--(start_addr_config)------------------------------------- */

/*--START FUNCTION--(ipv6_lo_init)------------------------------------------
 |
 | Loopback initialization for IPv6.
 | Called from addrconf_init();
 | TODO: idev list atomicity.
 | Last Update:
 | 08/06/96 bb
 ---------------------------------------------------------------------------*/
static void ipv6_lo_init(struct device *dev)
{
   struct inet6_dev *idev;
   struct inet6_ifaddr *ifp;
   struct in6_rtmsg rtmsg;
   char devname[] = "lo";
   int err;

#ifdef IPV6_DEBUG_ADDRCONF_LO
   printk(KERN_DEBUG "[ADDRCONF]ipv6_lo_init: IN\n");
#endif
   /* 
    * Loopback address ::1 
    */
   idev = ipv6_add_dev(dev);

   if (idev == NULL) {
      printk(KERN_DEBUG "[ADDRCONF]ipv6_lo_init: add_dev failed\n");
      return;
   }
   ifp = ipv6_add_addr(idev, (struct in6_addr *) &in6addr_loopback,
		       IFA_HOST, 128);

   if (ifp == NULL) {
      printk(KERN_DEBUG "[ADDRCONF]ipv6_lo_init: add_addr failed\n");
      return;
   }
   memcpy(&rtmsg.rtmsg_dst, &in6addr_loopback, sizeof(struct in6_addr));
   memset(&rtmsg.rtmsg_gateway, 0, sizeof(struct in6_addr));
   rtmsg.rtmsg_prefixlen = 128;
   rtmsg.rtmsg_metric = 0;
   strcpy(rtmsg.rtmsg_device, devname);
   rtmsg.rtmsg_flags = RTF_HOST | RTF_UP;

   if ((err = ipv6_route_add(&rtmsg))) {
      printk(KERN_DEBUG "[ADDRCONF]init_loopback: error in route_add\n");
   }
#ifdef IPV6_DEBUG_ADDRCONF_LO
   printk(KERN_DEBUG "[ADDRCONF]ipv6_lo_init: OUT\n");
#endif
   return;
}
/*--END FUNCTION--(ipv6_lo_init)-------------------------------------------*/


/*--START FUNCTION--(eth_config6)--------------------------------------------
|
|
|
----------------------------------------------------------------------------*/
static void eth_config6(struct device *dev)
{
   struct inet6_dev *idev;
#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]eth_config6 IN dev = %s\n", dev->name);
#endif
   if (dev->dev6_lst)
      return;
   if ((idev = ipv6_add_dev(dev)) == NULL) {
      return;
   }
/* Host & Link local mc addresses. */
   ipv6_mc_init(dev);
   start_addr_config(idev);
}
/*--END FUNCTION--(eth_config6)-------------------------------------------- */

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
 *                                                                         %
 *     addrconf module must be notified of a device going up               %
 *                                                                         %
 %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
static struct notifier_block ipv6_dev_notifier =
{
   addrconf_notify,
   NULL,
   0
};

/*--START FUNCTION--(addrconf_notify)-------------------------------------
|
| belongs to ipv6_dev_notifier notifier block
|
----------------------------------------------------------------------------*/
int addrconf_notify(struct notifier_block *this, unsigned long event, void *data)
{
   struct device *dev;
   struct inet6_dev *idev;

   dev = (struct device *) data;
#ifdef IPV6_DEBUG_ADDRCONF   
   printk(KERN_DEBUG "[ADDRCONF]addrconf_notify IN dev = %s\n", dev->name);
#endif
   switch (event) {
   case NETDEV_UP:
      switch (dev->type) {
      case ARPHRD_SIT:
#ifdef IPV6_DEBUG_ADDRCONF	 
	 printk(KERN_DEBUG "sit device up\n");
#endif
	 idev = ipv6_add_dev(dev);
	 if (memcmp(dev->name, "sit0", 4) == 0) {
	    sit_add_v4_addrs(idev);
	    sit_route_add(dev);
	 }
	 break;

      case ARPHRD_LOOPBACK:
	 ipv6_lo_init(dev);
	 break;

      case ARPHRD_ETHER:
#ifdef IPV6_DEBUG_ADDRCONF	 
	 printk(KERN_DEBUG "Configuring eth interface\n");
#endif
	 eth_config6(dev);
	 break;
      }

#ifdef CONFIG_NET_RTNETLINK6
      ipv6_netlink_msg(RTMSG_NEWDEVICE, 0, 0, 0, dev->flags, dev->metric, dev->name);
#endif
      break;

   case NETDEV_DOWN:
      /*
       *      Remove all addresses from this interface
       *      and take the interface out of the list.
       */
      switch (dev->type) {
      case ARPHRD_SIT:
	 break;

      case ARPHRD_LOOPBACK:
	 break;

      case ARPHRD_ETHER:
	 /*  eth_disable(dev); FIXME */
	 break;
      }
#ifdef CONFIG_NET_RTNETLINK6     
      ipv6_netlink_msg(RTMSG_DELDEVICE, 0, 0, 0, dev->flags, dev->metric, dev->name);
#endif     
      break;
   }
#ifdef IPV6_DEBUG_ADDRCONF   
   printk(KERN_DEBUG "[ADDRCONF]addrconf_notify OUT dev = %s\n", dev->name);
#endif
   return NOTIFY_OK;
}
/*--END FUNCTION--(addrconf_notify)-----------------------------------------*/


/*--START FUNCTION--(ipv6_addr_init)-----------------------------------------
|
|  addrconf_init: initialize address configuration structures
|
| Authors: P.Roque
|          B. Brodard
| Last modified: 05/22/96.
|
-----------------------------------------------------------------------------*/
void ipv6_addr_init(void)
{
#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]ipv6_addr_init: IN\n");
#endif
   /* 
    * Init unicast addr hash list 
    */
   memset(inet6_addr_lst, 0, 16 * sizeof(struct inet6_ifaddr *));
   /* 
    * Init multicast addr hash list 
    */
   memset(inet6_mcast_lst, 0, 16 * sizeof(struct ipv6_mc_list *));
   /*
    * Init device-with-ipv6-addresses list
    */
   inet6_dev_lst = NULL;

   register_netdevice_notifier(&ipv6_dev_notifier);

   /*
    * When programming module, go through the entire device list
    * at load time.
    */

#ifdef IPV6_DEBUG_ADDRCONF
   printk(KERN_DEBUG "[ADDRCONF]ipv6_addr_init: OUT\n");
#endif
}

/*--END FUNCTION--(ipv6_addr_init)-------------------------------------------*/















