/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$
 *      
 * Authors:     
 *              Benoit Brodard  <Brodard@sophia.inria.fr>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|  Function_name
|
-------------------------------------------------------------------------*/


#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/sched.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/dret/ipv6.h>
#include <linux/dret/icmpv6.h>
#include <linux/dret/route6.h>

#include <net/sock.h>
#include <net/snmp.h>

#include <net/dret/ipv6.h>
#include <net/protocol.h>
#include <net/dret/ipv6_route.h>
#include <net/dret/addrconf.h>
#include <net/dret/nd.h>
#include <linux/dret/netdevice6.h>

#include <net/checksum.h>
#include <linux/proc_fs.h>

/*
#define IPV6_DEBUG_ND
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[ND]%X:%X:%X:%X \n",\
		      (a)->s6_addr32[0], (a)->s6_addr32[1], \
                      (a)->s6_addr32[2], (a)->s6_addr32[3]);
*/
#define NEIGHBOR_PLEN 128
#define min(a,b) (((a)<(b))? a : b)

static void nd_event_send_ns(struct device *dev, struct neighbor *neigh,
			     struct in6_addr *s, struct in6_addr *d,
			     struct in6_addr *sol_addr,
			     int inc_opt, int type);
/*
 * Neighbor discovery mechanisms are triggered by three types of events:
 *
 * Namely:
 *     - A ND message is received.
 *           
 *     - A ND message is sent.
 *
 *     - The ND timer expired.
 *
 * These events affect the state of a neighbor cache entry:
 *     -INCOMPLETE, REACHABLE, STALE, DELAY, PROBE.
 *
 * All the functions below handle these events and modify the neighbor cache
 * state and the timer expiration time accordingly.
 */

/*
 * Concerning the neighbor cache, the idea is:
 * Use a tree structure and try to see if it's worth merging it with
 * the routing table.
 * Actually, chances are that every neighbor has a destination cache
 * entry.
 */
static __inline__ void neighbor_release(struct neighbor *n);
static struct neighbor *nd_new_neigh(struct device *dev, struct in6_addr *addr,
				     __u8 state, __u32 tstamp);

/*------------------------------------------------
   ND Timer characteristics
   Tunable via sysctl?     6.3.2
--------------------------------------------------*/

int BaseReachableTime = REACHABLE_TIME;
int ReachableTime = REACHABLE_TIME;
int RetransTimer = RETRANSTIMER;


/*
   struct neighbor null_entry =
   {
   NULL, NULL,
   {
   {
   {0}}},
   1, 1,
   NULL, NULL,
   0, 0, RTF_REJECT
   };
 */

struct rt_neigh nd_root =
{
   NULL, NULL, NULL, NULL,
   0, RTN_ROOT
};

static atomic_t nd_entries = 0;	/* Num of neighbor entries in table */

#define TABLE_BH_FREE	 	2

atomic_t nd_lock;
unsigned nd_bh_mask;

static struct wait_queue *nd_wait;
static struct timer_list nd_timer;

extern struct socket icmpv6_socket;

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   NEIGHBOR CACHE MANAGEMENT TOOLS

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */

/*--START FUNCTION--(naddr_bit)-----------------------------------------
 |
 | Input: an ipv6 address, (in network order)
 |        an integer.
 |
 | Returns: !0 if bit #bit is set in the address, 0 otherwise.
 |
 -------------------------------------------------------------------------*/
static __inline__ int naddr_bit(struct in6_addr *addr, int bit)
{
   int dw;
   __u32 b1;
   __u32 mask;

   /* Find corresponding byte */
   dw = bit >> 0x05;		/* dw = offset of bit's __u32 */
   b1 = addr->s6_addr32[dw];	/* b1 = bit's __u32           */
   mask = htonl((1 << (0x1f - (bit & 0x1f))));	/* bit's position in b1 */
   return (b1 & mask);
}
/*--END FUNCTION--(naddr_bit)---------------------------------------------*/

/*--START FUNCTION--(addr_diff)-----------------------------------------------
 |
 | Input: 2 ipv6 addresses,
 |        an int indicating where to start the comparison from.
 |  
 | Returns: the position of the first mismatching bit after start.
 |          (in network order)
 |
 ---------------------------------------------------------------------------*/
static __inline__ int naddr_diff(struct in6_addr *a1, struct in6_addr *a2,
				 int start)
{
   int dw;
   int i;

   dw = start >> 0x5;		/* byte offset of bit 'start' */
   for (i = dw; i < 4; i++) {
      __u32 b1, b2;
      __u32 xb;

      b1 = a1->s6_addr32[i];
      b2 = a2->s6_addr32[i];

      if ((xb = (b1 ^ b2))) {	/* b1 XOR b2 */
	 int res = 0x20;	/* 32 */

	 xb = htonl(xb);	/* xb put in network order */
	 while (xb) {		/* shift right til 0 */
	    res--;
	    xb >>= 1;
	 }
	 return (i << 5) + res;
      }
   }
   return 128;			/* full match */
}
/*--END FUNCTION--(naddr_diff)---------------------------------------------*/

static __inline__ int pfx_match(struct in6_addr *a1, struct in6_addr *a2,
				int prefixlen)
{
   int pdw;
   int pbi;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]pfx_match: IN, pfx length: %d\n", prefixlen);
   PRINT_ADDR(a1);
   PRINT_ADDR(a2);
#endif
   pdw = prefixlen >> 0x05;	/* # of whole __u32 in prefix */
   pbi = prefixlen & 0x1f;	/* #  of bits in incomplete __u32 in pfx */

   if (pdw) {
      /* Compare prefixes ( 4*pwd bytes) */
      if (memcmp(a1, a2, pdw << 2))
	 return 0;
   }
   if (pbi) {
      __u32 w1, w2;
      __u32 mask;

      w1 = a1->s6_addr32[pdw];
      w2 = a2->s6_addr32[pdw];

      mask = htonl((0xffffffff) << (0x20 - pbi));

      /* printk(KERN_DEBUG"[ROUTE]pfx_match: OUT, %d\n", (((w1 ^ w2) & mask)? 1 : 0) ); */
      if ((w1 ^ w2) & mask)
	 return 0;
   }
   return 1;
}

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   TABLE MANAGEMENT

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */
/*--START FUNCTION--(nd_add_1)--------------------------------------------
 |
 | Adds a route to a destination *addr.
 |
 | Input: an ipv6 address *addr.
 |        its prefix length
 |        a route destination structure.
 | Return: 0
 |
 ------------------------------------------------------------------------ */
static int nd_add_1(struct in6_addr *addr, struct neighbor *n)
{
   struct rt_neigh *fn;
   struct rt_neigh *pn = NULL;
   struct rt_neigh *in;
   struct rt_neigh *ln;
   __u32 bit;
   __u32 dir = 0;
   int plen = 128;
   unsigned long flags;

#ifdef IPV6_DEBUG_ND_1
   printk(KERN_DEBUG "[ND]nd_add_1: IN\n");
#endif
   fn = &nd_root;		/* fn points to root */
   for (;;) {
      if (fn == NULL) {
#ifdef IPV6_DEBUG_ND_1
	 printk(KERN_DEBUG "[ND]nd_add_1: fn==NULL\n");
#endif
	 ln = kmalloc(sizeof(struct rt_neigh), GFP_ATOMIC);
	 if (ln == NULL)
	    return (-ENOMEM);

	 memset(ln, 0, sizeof(struct rt_neigh));
	 ln->rn_bit = NEIGHBOR_PLEN;

	 ln->parent = pn;
	 ln->leaf = n;
	 n->main_node = ln;
	 ln->rn_flags = RTN_BACKTRACK;

	 if (dir)
	    pn->right = ln;
	 else
	    pn->left = ln;
#ifdef IPV6_DEBUG_ND_1
	 printk(KERN_DEBUG "[ND]nd_add_1: OUT\n");
#endif
	 atomic_inc(&nd_entries);
	 return (0);
      }
      /*
       * fn!= NULL
       */
      if (plen >= fn->rn_bit &&
	  pfx_match(&fn->leaf->naddr, addr, fn->rn_bit)) {
	 if (plen == fn->rn_bit) {
#ifdef IPV6_DEBUG_ND_1
	    printk(KERN_DEBUG "[ND]nd_add_1: plen==fn->rn_bit && ....%d\n", plen);
#endif
	    /* clean up an intermediate node */
	    if ((fn->rn_flags & RTN_BACKTRACK) == 0) {
	       save_flags(flags);
	       cli();
	       neighbor_release(fn->leaf);
	       restore_flags(flags);
	       fn->leaf = NULL;
	       fn->rn_flags |= RTN_BACKTRACK;
	    }
	    n->next = fn->leaf;
	    fn->leaf = n;
#ifdef IPV6_DEBUG_ND_1
	    printk(KERN_DEBUG "[ND]nd_add_1: OUT\n");
#endif
	    return 0;
	 }
	 /* plen > ref. bit, 
	  * input addr does match leaf addr up to ref bit
	  */
#ifdef IPV6_DEBUG_ND_1
	 printk(KERN_DEBUG "[ND]nd_add_1: plen>fn->rn_bit &&.... %d \n", plen);
#endif
	 /* walk down on tree */
	 pn = fn;
	 dir = naddr_bit(addr, fn->rn_bit);
	 fn = dir ? fn->right : fn->left;
#ifdef IPV6_DEBUG_ND_1	 
	 printk(KERN_DEBUG "[ND]nd_add_1: dir:%d\n", dir);
#endif
	 continue;
      }
      /*
       * don't have a common prefix anymore
       */
      pn = fn->parent;
      bit = naddr_diff(addr, &fn->leaf->naddr, pn->rn_bit);
      /*
       * We must have plen > bit since NEIGHBOR_PLEN = 128...
       * 0< bit < 127
       */
      /*           (pn)
       *           / or\ 
       *          (in)  ...
       *         /    \
       *       (ln)   (fn) 
       */
      /* allocate new rt_neigh */
      in = kmalloc(sizeof(struct rt_neigh), GFP_ATOMIC);
      if (in == NULL)
	 return (-ENOMEM);
      atomic_inc(&nd_entries);
      memset(in, 0, sizeof(struct rt_neigh));

      /* 
       * new intermediate node. 
       * RTN_BACKTRACK will be off since that an address 
       * that chooses one of the branches would not match 
       * less specific routes in the other branch
       */
      in->rn_bit = bit;		/* diff btw addr & fn->... */
      in->parent = pn;
      in->leaf = n;		/* connect neighbor input param. */
      atomic_inc(&n->refcnt);

      /* update parent pointer */
      if (dir)
	 pn->right = in;
      else
	 pn->left = in;

      /* leaf node */
      ln = kmalloc(sizeof(struct rt_neigh), GFP_ATOMIC);
      if (ln == NULL) {
	 kfree(in);
	 return (-ENOMEM);
      }
      atomic_inc(&nd_entries);
      memset(ln, 0, sizeof(struct rt_neigh));
      ln->rn_bit = NEIGHBOR_PLEN;
      /* We may backtrack and find a less specific route */
      ln->rn_flags = RTN_BACKTRACK;

      ln->parent = in;
      fn->parent = in;

      ln->leaf = n;
      n->main_node = ln;

      if (naddr_bit(addr, bit)) {
	 in->right = ln;
	 in->left = fn;
      } else {
	 in->left = ln;
	 in->right = fn;
      }
#ifdef IPV6_DEBUG_ND_1
      printk(KERN_DEBUG "[ND]nd_add_1: OUT with success new int.: %d %d\n", bit, plen);
#endif
      return 0;
   }
#ifdef IPV6_DEBUG_ND_1
   printk(KERN_DEBUG "[ND]nd_add_1: OUT with failure\n");
#endif
   return (-1);			/* failed to add a rt_neigh */
}
/*--END FUNCTION--(nd_add_1)-----------------------------------------------*/

/*--START FUNCTION--(nd_lookup_1)--------------------------------------------
 |
 | Lookup a route to destination *addr.
 |
 | Input: an ipv6 address
 |
 | Returns the route to destination structure.
 |
 --------------------------------------------------------------------------*/
struct neighbor *nd_lookup_1(struct in6_addr *addr)
{
   struct rt_neigh *fn, *next;
   int dir;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_lookup_1: IN\n");
   PRINT_ADDR(addr);
#endif
   fn = &nd_root;
   for (;;) {
      dir = naddr_bit(addr, fn->rn_bit);
      next = dir ? fn->right : fn->left;
      /* printk(KERN_DEBUG"[ROUTE]nd_lookup_1: lookup dir:%d\n", dir); */
      if (next) {
	 fn = next;
	 continue;
      }
      break;
   }
   if (fn->rn_flags & RTN_BACKTRACK) {
      if (fn->leaf) {
	 if (SAME_ADDR6(*addr, fn->leaf->naddr)) {
#ifdef IPV6_DEBUG_ND
	    printk(KERN_DEBUG "[ND]nd_lookup_1: match OUT\n");
	    PRINT_ADDR(addr);
#endif
	    return fn->leaf;
	 } else {
#ifdef IPV6_DEBUG_ND
	    printk(KERN_DEBUG "[ND]nd_lookup_1: no match");
	    PRINT_ADDR(&fn->leaf->naddr);
#endif
	 }
      } else {
	 printk(KERN_DEBUG "[ND]nd_lookup_1: fn without leaf...\n");
      }
   } else {
      printk(KERN_DEBUG "[ND]nd_lookup_1: BACKTRACK is not set... NULL OUT\n");
      return NULL;
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_lookup_1: no match OUT \n");
#endif
   return NULL;
}
/*--END FUNCTION--(nd_lookup_1)-----------------------------------------*/

/*--START FUNCTION--(nd_del_3)------------------------------------------
 |
 | Called to trim the tree of intermediate nodes when possible
 |
 | INPUT: rt_neigh to be deleted.
 | CALLED BY: nd_del_2.
 -----------------------------------------------------------------------*/
static void nd_del_3(struct rt_neigh *fn)
{
   register struct rt_neigh *tmp;
   int children = 0;
   register int dir = 0;
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ROUTE6]nd_del_3 IN BIT = %d\n", fn->rn_bit);
#endif
   /*
    *      0 or one children:
    *              delete the node
    *      2 children:
    *              move the bit down
    */
   if (fn->left) {
      children++;
   }
   if (fn->right) {
      children++;
      dir = 1;
   }
   if (children < 2) {
      tmp = fn->parent;
      if (tmp->left == fn) {
	 tmp->left = dir ? fn->right : fn->left;
	 if (children)		/* tmp->left != NULL */
	    tmp->left->parent = tmp;
      } else {
	 tmp->right = dir ? fn->right : fn->left;
	 if (children)		/* tmp->right != NULL */
	    tmp->right->parent = tmp;
      }
      /* 
       *      try to colapse on top
       */
      if ((fn->parent->rn_flags & (RTN_BACKTRACK | RTN_ROOT)) == 0)
	 nd_del_3(fn->parent);
      kfree(fn);
      atomic_dec(&nd_entries);
#ifdef IPV6_DEBUG_ND
      printk(KERN_DEBUG "[ROUTE6]nd_del_3: OUT, node deleted\n");
#endif
      return;
   }
   fn->rn_flags &= ~RTN_BACKTRACK;	/* turn off backtrack bit */
   fn->leaf = fn->left->leaf;
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ROUTE6]nd_del_3: OUT, fn_bit updated\n");
#endif
   return;
}
/*--END FUNCTION--(nd_del_3)-----------------------------------------------*/


/*--START FUNCTION--(nd_del_2)---------------------------------------------

--------------------------------------------------------------------------*/
static struct rt_neigh *nd_del_2(struct in6_addr *addr)
{
   struct rt_neigh *fn;
   struct neighbor *lf;
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_del_2: IN \n");
#endif
   for (fn = &nd_root; (fn && fn->rn_bit != NEIGHBOR_PLEN);) {
      int dir;
      dir = naddr_bit(addr, fn->rn_bit);
      fn = dir ? fn->right : fn->left;
   }
   /* 
    *  search matching rt_neigh leaves...
    */
   if (fn) {
      lf = fn->leaf;
      if (SAME_ADDR6(*addr, lf->naddr)) {
	 fn->leaf = NULL;
      }
      return fn;
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_del_2: OUT not found\n");
#endif
   return NULL;
}
/*--END FUNCTION--(nd_del_2)-----------------------------------------------*/

/*--START FUNCTION--(nd_del_1)---------------------------------------------
 |
 | Input: an ipv6 address,
 |        its prefix length
 |        a gateway address for that destination.
 |        the device to that destination.
 |
 |--------------------------------------------------------------------------*/
static int nd_del_1(struct in6_addr *addr)
{
   struct rt_neigh *fn;
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_del_1: IN \n");
#endif
   fn = nd_del_2(addr);

   if (fn == NULL) {
#ifdef IPV6_DEBUG_ND
      printk(KERN_DEBUG "[ND]nd_del_1: OUT(NOENT)\n");
#endif
      return -ENOENT;		/* No entry found by nd_del_2 */
   }
   if (fn->leaf == NULL) {
      nd_del_3(fn);
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_del_1: OUT \n");
#endif
   return 0;
}

/*--START FUNCTION--(neighbor_release)----------------------------------------

  Virtually releases a neighbor entry.

----------------------------------------------------------------------------*/
static __inline__ void neighbor_release(struct neighbor *n)
{
   atomic_dec(&n->refcnt);
   if (n->refcnt == 0) {
      nd_del_1(&n->naddr);
      kfree(n);
      atomic_dec(&nd_entries);
#ifdef IPV6_DEBUG_ND      
      printk(KERN_DEBUG "[ND]neighbor_release: released a leaf... \n");
#endif
   }
}
/*--END FUNCTION--(neighbor_release)-----------------------------------------*/


/*--START FUNCTION--(nd_add_timer)---------------------------------------
  
  Sets a new expiration time for neigh's timer.

-------------------------------------------------------------------------*/
static __inline__ void nd_add_timer(struct neighbor *n, int timer)
{
   unsigned long now = jiffies;
   unsigned long ctime;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_add_timer: IN timer = %d\n", timer);
   PRINT_ADDR(&n->naddr);
#endif

   now = jiffies;
   n->expires = now + timer;
   n->state |= ND_TIMED;

   ctime = del_timer(&nd_timer);

   if (ctime) {
      ctime = ((ctime < n->expires) ? ctime : n->expires);
   } else
      ctime = n->expires;

   nd_timer.expires = ctime;
   add_timer(&nd_timer);
}
/*--END FUNCTION--(nd_add_timer)------------------------------------------*/

/*--START FUNCTION--(nd_del_timer)---------------------------------------

-------------------------------------------------------------------------*/
static void nd_del_timer(struct neighbor *neigh)
{
   unsigned long ctime;
   struct rt_neigh *fn;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_del_timer: IN -->\n");
   PRINT_ADDR(&neigh->naddr);
#endif

   ctime = del_timer(&nd_timer);
   if (ctime == neigh->expires) {
      ctime = ~0UL;
      fn = &nd_root;
      /* search the entire tree */
      /* why not maintain a list of all expiration dates? */
   }
   if (ctime == (~0UL))
      return;

   nd_timer.expires = ctime;
   add_timer(&nd_timer);
}
/*--END FUNCTION--(nd_del_timer)-----------------------------------------*/


#if 0
/*--START FUNCTION--(nd_route_timer_handler)-----------------------------

  Handle routers timers

  Called from timer interrupt. Initialized in route_dev_init.

  Last update:
  8/8/96 bb
---------------------------------------------------------------------------*/
static void nd_route_timer_handler(unsigned long arg)
{



}


/*--START FUNCTION--(start_advertise)--------------------------------------

  Called from router_dev_init...
  Should also be called when router flag is changed.
  Can the forward capability be loaded as a module? or changed on the fly?

  Last update:
  8/8/96 bb
--------------------------------------------------------------------------*/
void start_advertise(struct inet6_dev *idev)
{

   unsigned long now = jiffies;
   unsigned long ntimer = ~0UL;

   struct router_conf *rc = idev->router1;

   if (!rc->flags.AdvSendAdvertisements)
      return;
/*
 * Init router timer
 */
   init_timer(&rc->timer);
   rc->timer.function = nd_route_timer_handler;
   rc->timer.data = 0L;
   rc->timer.expires = 0L;
}
#endif

/*--START FUNCTION--(nd_eth_resolve)---------------------------------------

  Called from net/ethernet/eth.c: eth_rebuild_header.
  This function is used to resolve link layer addresses.
  
  Last update:
  07/26/96: bb
---------------------------------------------------------------------------*/
int nd_eth_resolve(unsigned char *eth_dest, struct device *dev,
		   struct sk_buff *skb)
{
   struct neighbor *neigh;
   struct sk_buff *skb1;
   unsigned long now = jiffies;
   struct inet6_ifaddr *ifa;
   struct in6_addr msol_addr;
   unsigned long flags;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_eth_resolve: IN daddr = \n");
   PRINT_ADDR(&skb->nh.ipv6h->daddr);
#endif

   neigh = skb->nexthop;
   if (neigh == NULL) {
      if IS_MULTIADDR6
	 (skb->nh.ipv6h->daddr) {
#ifdef IPV6_DEBUG_ND
	 printk(KERN_DEBUG "*** Multicast address *** eth mcast addr =\n");
#endif
	 ipv6_mc_map(&skb->nh.ipv6h->daddr, eth_dest);
	 return 0;
	 }
      printk(KERN_DEBUG "[ND]nd_eth_resolve: nexthop is NULL- Can't resolve! \n");
      dev_kfree_skb(skb, FREE_WRITE);
      return 1;
   }
/*
 *  Must we keep trying?
 */
   save_flags(flags);
   cli();
   if ((neigh->flags & NCF_INVALID)) {
      neigh->state = ND_INCOMPLETE;
      neigh->flags &= ~NCF_INVALID;
      restore_flags(flags);
      ifa = ipv6_get_saddr(dev, &neigh->naddr);
      MULTISOL_ADDR6(msol_addr, neigh->naddr);
      nd_event_send_ns(dev, neigh, &ifa->addr, &msol_addr, &neigh->naddr, 1,
		       ND_NEIGHBOR_SOLICITATION);
   } else {
      restore_flags(flags);
   }

   switch (neigh->state) {
   case ND_INCOMPLETE:
#ifdef IPV6_DEBUG_ND     
      printk(KERN_DEBUG "[ND]nd_eth_resolve: *** Incomplete State ***\n");
#endif
      if (skb_queue_len(&neigh->queue) >= MAX_ND_QLEN) {
	 skb1 = neigh->queue.prev;
	 skb_unlink(skb1);
	 dev_kfree_skb(skb1, FREE_WRITE);
      }
      skb_queue_head(&neigh->queue, skb);
      return 1;

   case ND_REACHABLE:
#ifdef IPV6_DEBUG_ND      
      printk(KERN_DEBUG "[ND]nd_eth_resolve: *** Reachable State ***\n");
#endif
      if (now - neigh->tstamp < ReachableTime)
	 break;

   case ND_STALE:
#ifdef IPV6_DEBUG_ND
      printk(KERN_DEBUG "[ND]nd_eth_resolve: *** Stale State ***\n");
#endif
      neigh->state = ND_DELAY;
      nd_add_timer(neigh, DELAY_FIRST_PROBE_TIME);
      break;

   case ND_DELAY:
#ifdef IPV6_DEBUG_ND
      printk(KERN_DEBUG "[ND]nd_eth_resolve: *** Delay State ***\n");
#endif
      break;

   case ND_PROBE:
#ifdef IPV6_DEBUG_ND
      printk(KERN_DEBUG "[ND]nd_eth_resolve: *** Probe State ***\n");
#endif
      
   default:
#ifdef IPV6_DEBUG_ND
      printk(KERN_DEBUG "[ND]nd_eth_resolve: *** Default = %d ***\n", neigh->state);
#endif
      break;
   }

   if ((neigh->flags & NCF_INVALID) || neigh->hh_data == NULL) {
#ifdef IPV6_DEBUG_ND
      printk(KERN_DEBUG "[ND]nd_eth_resolve: invalid neighbor OUT\n");
#endif
      dev_kfree_skb(skb, FREE_WRITE);
      return 1;
   }
   memcpy(eth_dest, neigh->hh_data, dev->addr_len);

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_eth_resolve: OUT\n Eth addr = ");
   /* dump_hex(neigh->hh_data, 14); */
#endif

   return 0;
}
/*--END FUNCTION--(nd_eth_resolve)----------------------------------------*/


/*--START_FUNCTION--(nd_store_hh_data)-----------------------------------------

   Possibly update the link layer address destination of a cache entry.

---------------------------------------------------------------------------*/
static int nd_store_hh_data(struct device *dev, char *h_dest, char *opt,
			    __u8 olen, __u8 otype)
{

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_store_hh_data: IN\n");
#endif
   while (*opt != otype && olen) {
      int len;

      len = opt[1] << 3;
      if (len == 0) {
	 printk(KERN_DEBUG "[ND]nd_store_hh_data: option has 0 len\n");
	 return -EINVAL;
      }
      opt += len;
      olen -= len;
   }
   if (*opt == otype) {
      memcpy(h_dest, opt + 2, dev->addr_len);
      return 0;
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_store_hh_data: OUT\n");
#endif
   return -EINVAL;
}
/*--END FUNCTION--(nd_store_hh_data)---------------------------------------*/


/*--START_FUNCTION--(nd_update_hh_data)----------------------------------------

   Possibly update the link layer address destination of a cache entry.

---------------------------------------------------------------------------*/
static int nd_update_hh_data(struct device *dev, char *h_dest, char *opt,
			     __u8 olen, __u8 otype)
{
   int len = 0;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_update_hh_data: IN\n");
#endif

   while ((*opt != otype) && olen) {
      printk(KERN_DEBUG "[ND]nd_update_hh_data: Not wanted option: %d \n", opt[0]);
      if (opt[0] > ND_OPT_MAXNUM) {
	 printk(KERN_DEBUG "[ND]nd_update_hh_data: option has invalid type.\n");
	 return -EINVAL;
      }
      if ((len = opt[1] << 3) == 0) {
	 printk(KERN_DEBUG "[ND]nd_update_hh_data: option has 0 length.\n");
	 return -EINVAL;
      }
      opt += len;
      olen -= len;
   }
   if (*opt == otype) {
      len = opt[1] << 3;
      switch (otype) {
      case ND_OPT_SLLA:
	 if (memcmp(h_dest, opt + 2, len) == 0)
	    return 0;
      case ND_OPT_TLLA:
      default:
	 if (len < dev->addr_len) {
	    break;
	 }
	 memcpy(h_dest, opt + 2, dev->addr_len);
#ifdef IPV6_DEBUG_ND
	 printk(KERN_DEBUG "[ND]nd_update_hh_data: OUT with update\n");
#endif
	 return 0;
      }
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_update_hh_data: OUT with error otype = %d\n", *opt);
#endif
   return -EINVAL;
}

/*--START_FUNCTION--(nd_event_rcv_ns)-----------------------------------------

   FIXME: Double check this one.
   Last Update: 
   07/25/96: bb.
---------------------------------------------------------------------------*/
static struct neighbor *nd_event_rcv_ns(struct in6_addr *s, struct sk_buff *skb)
{
   struct neighbor *neigh;
   __u8 *option;
   int olen;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_rcv_ns: IN\n");
#endif

   if SAME_ADDR6
      (*s, in6addr_any) return NULL;	/* 7.2.3 */

   option = skb->h.raw + sizeof(struct n_adv_sol);
   olen   = skb->tail - option;


   if ((neigh = nd_lookup_1(s)) == NULL) {
      /* =============== CACHE MISS ========================== */
      neigh = nd_new_neigh(skb->dev, s, ND_REACHABLE, jiffies - ReachableTime);
      if (neigh == NULL)
	 return (NULL);
      neigh->flags &= ~NCF_HHVALID;
      if (nd_store_hh_data(skb->dev, neigh->hh_data, option, olen, ND_OPT_SLLA) == 0)
	 neigh->flags |= NCF_HHVALID;
      return (neigh);
   } else {
      /* ============== NEIGH PRESENT IN CACHE ================ */
      switch (neigh->state) {
      case ND_REACHABLE:
      case ND_STALE:
      case ND_DELAY:

	 if (neigh->state & ND_TIMED) {
	    nd_del_timer(neigh);
	 }
      default:
	 neigh->flags &= ~NCF_HHVALID;
	 if (nd_update_hh_data(neigh->dev, neigh->hh_data, option, olen,
			       ND_OPT_SLLA)) {
	    printk(KERN_DEBUG "[ND]nd_event_rcv_ns: failed to update cache\n");
	 } else {
	    neigh->flags |= NCF_HHVALID;
	    neigh->state = ND_STALE;	/* FIXME: what if no update? */
	    neigh->tstamp = jiffies;
	    neigh->probes = 0;
	 }
#ifdef IPV6_DEBUG_ND
	 printk(KERN_DEBUG "[ND]nd_event_rcv_ns: OUT\n");
#endif
	 return (neigh);
      }
   }
}

/*--START_FUNCTION--(nd_event_rcv_na)-----------------------------------------

 *option : points to first byte in option data.
    olen : total length of option data in packet.

 Last Update:
 07/25/96: bb   
---------------------------------------------------------------------------*/
static void nd_event_rcv_na(struct neighbor *n, char *option, __u8 olen,
			    __u8 solicited, __u8 override)
{
   struct sk_buff *skb;
   int state = n->state;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_rcv_na: IN sol = %d; over = %d;\n", solicited, override);
#endif
   /*
    * Should we update the cache?
    */
   if (override || (state == ND_INCOMPLETE)) {
      n->flags &= ~NCF_HHVALID;
      if (nd_update_hh_data(n->dev, n->hh_data, option, olen, ND_OPT_TLLA)) {
	 printk(KERN_DEBUG "[ND]nd_event_na: Bad options in received NA\n");
	 return;		/*FIXME: What kind of error is that? */
      }
      n->flags |= NCF_HHVALID;
   }
   if (override || (state == ND_INCOMPLETE) || solicited) {
      n->probes = 0;
      n->tstamp = jiffies;

      if (n->state & ND_TIMED) {
	 nd_del_timer(n);
      }
   }
   n->state = (solicited ? ND_REACHABLE : ND_STALE);
   if (state == ND_INCOMPLETE) {
      while ((skb = skb_dequeue(&n->queue))) {
	 dev_queue_xmit(skb); 
	 /*, n->dev, (skb->sk ? skb->sk->priority : SOPRI_NORMAL)); */
      }
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_rcv_na: OUT\n");
#endif
}

/*--START_FUNCTION--(nd_event_send_na)-----------------------------------------

   Send ND messages 
   Last Update:
   07/25/96: bb.
---------------------------------------------------------------------------*/
static void nd_event_send_na(struct device *dev, struct neighbor *neigh,
			     struct in6_addr *s, struct in6_addr *d,
			     struct in6_addr *sol_addr,
			     int router, int solicited, int override,
			     int inc_opt, int type)
{
   struct sock *sk = icmpv6_socket.sk;
   struct sk_buff *skb;
   struct n_adv_sol *msg;
   struct nd_opt *opt;
   int olen = 0;
   unsigned int payload, plen, csum;
   int err;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_send_na: IN\n");
#endif

   plen = sizeof(struct ipv6hdr) +
    sizeof(struct n_adv_sol);

   if (inc_opt) {
      olen = ((dev->addr_len + 1) >> 3) + 1;
      plen += (olen << 3);
   } else {
      override = 0;
   }

   if ((skb = sock_alloc_send_skb(sk, plen + 15 + dev->hard_header_len,
				  0, 0, &err)) == NULL) {
      printk(KERN_DEBUG "[ND]nd_event_send: skb alloc failed \n");
   }
   skb->dev = dev;
   skb->protocol = htons(ETH_P_IPV6);
   /* skb->free = 1; */
   skb->when = jiffies;
   skb->sk = sk;
   skb->nexthop = neigh;	/* The 1 we've  just created */
   skb->arp = 0;		/* How is it used with IPv6? TOHANDLE */
   /*
     COPY_ADDR6(skb->saddr6, *s);
     COPY_ADDR6(skb->raddr6, *d);
   */
   if (build_header_from_neigh(skb, dev, neigh, sol_addr, d, IPPROTO_ICMPV6,
			       plen, 15) < 0) {
      kfree_skb(skb, FREE_WRITE);
      printk(KERN_DEBUG "[ND]nd_event_send_na: build header failed...\n");
   }
   payload = skb->tail - ((unsigned char *) skb->nh.ipv6h)
       - sizeof(struct ipv6hdr);

/* =================== FILL IN ICMPv6 HEADER =========================== */
   msg = (struct n_adv_sol *) (skb->nh.ipv6h + 1);
   msg->hdr.type = type;
   msg->hdr.code = 0;
   msg->hdr.csum = 0;
   msg->u2.n_adv.router = router;
   msg->u2.n_adv.solicited = solicited;
   msg->u2.n_adv.override = override;

/* reserved  = (router ? 1 << 31 : 0) | (solicited ? 1 << 30 : 0) |
   (override ? 1 << 29 : 0); */

   COPY_ADDR6(msg->target, *sol_addr);

   if (inc_opt) {
      opt = (struct nd_opt *) (msg + 1);
      opt->type = ND_OPT_TLLA;
      opt->len = olen;
      memcpy(opt->lla, dev->dev_addr, dev->addr_len);
   }
   csum = csum_partial((char *) msg, payload, 0);
   msg->hdr.csum = csum_ipv6_magic(sol_addr, d, payload, IPPROTO_ICMPV6, csum);

   ipv6_queue_xmit(sk, skb, 1);

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_send_na: OUT\n");
#endif
}

/*--END FUNCTION--(nd_event_send_na)----------------------------------------*/

/*--START_FUNCTION--(nd_event_send_ns)-----------------------------------------

   Send NDS messages

   Called from:
        -nd_new_neigh().
 
   Last Update:
   07/25/96: bb.
---------------------------------------------------------------------------*/
static void nd_event_send_ns(struct device *dev, struct neighbor *neigh,
			     struct in6_addr *s, struct in6_addr *d,
			     struct in6_addr *sol_addr,
			     int inc_opt, int type)
{
   struct sock *sk = icmpv6_socket.sk;
   struct sk_buff *skb;
   struct n_adv_sol *msg;
   struct nd_opt *opt;
   int olen = 0;
   unsigned int payload, plen, csum;
   int err;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_send_ns: IN\n");
#endif

   plen = sizeof(struct ipv6hdr) + sizeof(struct n_adv_sol);

   if (inc_opt) {
      olen = ((dev->addr_len + 1) >> 3) + 1;
      plen += (olen << 3);
   }
   if ((skb = sock_alloc_send_skb(sk, plen + 15 + dev->hard_header_len, 0, 0, &err)) == NULL) {
      printk(KERN_DEBUG "[ND]nd_event_send_ns: skb alloc failed \n");
   }
   skb->dev = dev;
   skb->protocol = __constant_htons(ETH_P_IPV6);
   /* skb->free = 1; */
   skb->when = jiffies;
   skb->sk = sk;
   skb->nexthop = (IS_MULTIADDR6(*d) ? NULL : neigh);
   skb->arp = 0;
   
   /*
     COPY_ADDR6(skb->saddr6, *s);
     COPY_ADDR6(skb->daddr6, *d);
   */
   
   if (build_header_from_neigh(skb, dev, neigh, s, d, IPPROTO_ICMPV6,
			       plen, 15) < 0) {
      kfree_skb(skb, FREE_WRITE);
      printk(KERN_DEBUG "[ND]nd_event_send_ns: build header failed...\n");
   }
   payload = skb->tail - ((unsigned char *) skb->nh.ipv6h)
       - sizeof(struct ipv6hdr);

/* =================== FILL IN ICMPv6 HEADER =========================== */
   msg = (struct n_adv_sol *) (skb->nh.ipv6h + 1);
   msg->hdr.type = type;
   msg->hdr.code = 0;
   msg->hdr.csum = 0;
   msg->u2.reserved = 0;

   COPY_ADDR6(msg->target, *sol_addr);

   if (inc_opt) {
      opt = (struct nd_opt *) (msg + 1);
      opt->type = ND_OPT_SLLA;
      opt->len = olen;
      memcpy(opt->lla, (char *) dev->dev_addr, (int) dev->addr_len);
   }
   csum = csum_partial((char *) msg, payload, 0);
   msg->hdr.csum = csum_ipv6_magic(&skb->nh.ipv6h->saddr, d, payload,
				   IPPROTO_ICMPV6, csum);

   ipv6_queue_xmit(sk, skb, 1);
   neigh->probes++;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_send_ns: OUT\n");
#endif
}

/*--END FUNCTION--(nd_event_send_ns)----------------------------------------*/

/*--START_FUNCTION--(nd_event_rcv)-----------------------------------------
   
   Called from icmpv6_rcv() which is suppose to perform simple
   validation checks on the incoming message.
   
   Last Update:
   07/25/96: bb: ND_NEIGHBOR_SOLICITATION.
                 ND_NEIGHBOR_ADVERTISEMENT.
---------------------------------------------------------------------------*/
int nd_event_rcv(struct sk_buff *skb, struct device *dev,
		 struct in6_addr *s, struct in6_addr *d,
		 struct options *opt, unsigned len)
{

   struct n_adv_sol *msg = (struct n_adv_sol *) skb->h.raw;
   struct neighbor *neigh;
   struct inet6_ifaddr *ia;
   int type;
   __u8 solicited, override, router, mask;
   char *ptr;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_rcv: IN\n");
#endif

   switch (msg->hdr.type) {
   case ND_NEIGHBOR_SOLICITATION:
#ifdef IPV6_DEBUG_ND
      printk(KERN_DEBUG "[ND]nd_event_rcv: ND_NEIGHBOR_SOLICITATION\n");
      PRINT_ADDR(&msg->target);
#endif
      if ((ia = get_ifaddr6(&msg->target))) {
	 type = get_ipv6_addr_type(s);


	 if ((type == IPV6_ADDR_UNICAST) ||
	     (type == IPV6_ADDR_LINKLOCAL) ||
	     (type == IPV6_ADDR_SITELOCAL)) {
	    /*
	     * Statistics
	     */

	    /*
	     * State management
	     */
	    if ((neigh = nd_event_rcv_ns(s, skb))) {
	       /*
	        * Reply
	        */
	       nd_event_send_na(dev, neigh, d, s, &ia->addr,
				ia->idev->router, 1, 1,
			   IS_MULTIADDR6(*d), ND_NEIGHBOR_ADVERTISEMENT);
	    } else {
	       /* Couldn't create neighbor */
	    }
	 } else {
	    /* 
	     * Unspecified source address used by hosts performing
	     * duplicated address detection.
	     */
	    if (type == IPV6_ADDR_ANY) {
	       nd_event_send_na(dev, neigh, d, &ia->addr, &ia->addr,
				ia->idev->router, 1, 1,
			   IS_MULTIADDR6(*d), ND_NEIGHBOR_ADVERTISEMENT);
	       break;
	    }
#ifdef IPV6_DEBUG_ND
	    printk(KERN_DEBUG "[ND]nd_event_rcv: NS from address, type = %d\n", type);
#endif
	 }
      }
      /* FIXME: 
         Ooooopppppssss: target address not found on host!
         discard sk_buff?
         Sthg seriously broken if this happens. 
       */
      break;

   case ND_NEIGHBOR_ADVERTISEMENT:
#ifdef IPV6_DEBUG_ND
      printk(KERN_DEBUG "[ND]nd_event_rcv: ND_NEIGHBOR_ADVERTISEMENT\n");
      PRINT_ADDR(s);
#endif
      if ((neigh = nd_lookup_1(s))) {
	 /*
	  *  Find out message flags. FIXME 
	  */
	 mask = ((msg->u2.reserved >> 24) & 0xFF);
	 override = (mask & 0x20);
	 solicited = (mask & 0x40);
	 router = (mask & 0x80);

	 /* 
	  * Any change concerning a router ? 
	  */
	 if (router == 0) {
	    if (neigh->flags & NDF_ISROUTER) {
	       /*
	        * Handle default router list
	        */
	    }
	 } else {
	    if (!(neigh->flags & NDF_ISROUTER)) {
	       /*
	        * Handle default router list
	        */
	       /* Remove entry from default list
	          Update all destinations entry */
	    }
	 }
	 ptr = (unsigned char *) (msg + 1);
	 nd_event_rcv_na(neigh, ptr, (char *) skb->tail - ptr,
	  (__u8) msg->u2.n_adv.solicited, (__u8) msg->u2.n_adv.override);
      } else {
	 /* if neigh == NULL, silently discard at end of icmpv6_rcv */
	 printk(KERN_DEBUG "That's a silent discard, 'coz neighbor's not in cache\n");
      }
      break;

   case ND_ROUTER_SOLICITATION:
      if ((neigh = nd_lookup_1(s))) {
         ptr = ((unsigned char *)msg) + sizeof(struct r_sol);
         nd_update_hh_data(neigh->dev, neigh->hh_data, ptr, (char *) skb->tail - ptr, ND_OPT_SLLA);
      }
      break;

   case ND_ROUTER_ADVERTISEMENT:
      if ((neigh = nd_lookup_1(s))) {
         ptr = ((unsigned char *)msg) + sizeof(struct r_adv);
         nd_update_hh_data(neigh->dev, neigh->hh_data, ptr, (char *) skb->tail - ptr, ND_OPT_SLLA);
      }
      break;

   case ND_REDIRECT:
      break;
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_rcv: OUT\n");
#endif

   return 0;
}
/*--END FUNCTION--(nd_event_rcv)--------------------------------------------*/

/*--START FUNCTION--(nd_event_send_packet)-----------------------------------
  
    Manage neighbor state when sending IPv6 packets.
    Called from: 
    
    Last update:
    08/01/96: bb.
 ---------------------------------------------------------------------------*/
void nd_event_send_packet(struct neighbor *n)
{
   unsigned long now = jiffies;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_send_packet: IN\n");
#endif

   if (n->flags & NCF_NOARP)
      return;

   switch (n->state) {
   case ND_REACHABLE:
      if (REACHABLE_TIME > now - n->tstamp)
	 break;
   case ND_STALE:
      n->state = ND_DELAY;
      nd_add_timer(n, DELAY_FIRST_PROBE_TIME);
      break;
   default:
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_send_packet: OUT\n");
#endif
   return;
}
/*--END FUNCTION--(nd_event_send_packet)------------------------------------*/


/*--START FUNCTION--(nd_new_neigh)--------------------------------------------
  
    Create a new cache entry.

    FIXME: What kind of source address do we use?
    Called by nd_get_neighbor if no neighbor found.

    Last update:
    07/26/96: bb.
 ---------------------------------------------------------------------------*/
struct neighbor *nd_new_neigh(struct device *dev, struct in6_addr *addr,
			      __u8 state, __u32 tstamp)
{
   struct neighbor *n;
   struct inet6_ifaddr *ifa;
   struct in6_addr msol_addr;

   unsigned short *proto;
   unsigned long flags;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_new_neigh: IN --> %s\n", dev->name);
   PRINT_ADDR(addr);
#endif

   n = (struct neighbor *) kmalloc(sizeof(struct neighbor), GFP_ATOMIC);
   if (n == NULL) {
      printk("[ND]nd_new_neigh: out of memory \n");
      return NULL;
   }
   nd_entries++;

/*
 * Allocate and add neighbor 
 */
   memset(n, 0, sizeof(struct neighbor));
   skb_queue_head_init(&n->queue);
   COPY_ADDR6(n->naddr, *addr);
   n->len = NEIGHBOR_PLEN;
   n->type = get_ipv6_addr_type(addr);
   n->dev = dev;
   n->tstamp = (tstamp ? tstamp : jiffies);
   n->state = state;
   n->refcnt = 1;

   cli();
   if ((dev->flags & IFF_NOARP) ||
       (dev->flags & IFF_LOOPBACK)) {
      n->flags |= NCF_NOARP;
#ifdef IPV6_DEBUG_ND      
      printk(KERN_DEBUG "[ND]nd_new_neigh: set flag NCF_NOARP\n");
#endif
   } else {
#ifdef IPV6_DEBUG_ND      
      printk(KERN_DEBUG "[ND]nd_new_neigh: NCF_NOARP not set\n");
#endif
   }
   sti();

/* hh_data */
   memcpy(n->hh_data + dev->addr_len, dev->dev_addr, dev->addr_len);
   proto = (unsigned short *) (n->hh_data + 2 * dev->addr_len);
   *proto = htons(ETH_P_IPV6);

   while (nd_lock)
      sleep_on(&nd_wait);
   nd_fast_lock();

   nd_add_1(addr, n);

   nd_unlock();
   wake_up(&nd_wait);

/*
 *     Sollicit link layer address from neighbors.
 */
   if ((state == ND_INCOMPLETE)) {
      if ((ifa = ipv6_get_saddr(dev, addr))) {
	 MULTISOL_ADDR6(msol_addr, *addr);
	 nd_event_send_ns(dev, n, &ifa->addr, &msol_addr, addr,
			  1, ND_NEIGHBOR_SOLICITATION);
      } else {
	 printk(KERN_DEBUG "[ND]nd_new_neigh:Can't determine source address! \n");
	 save_flags(flags);
	 cli();
	 neighbor_release(n);
	 restore_flags(flags);
	 return NULL;
      }
      nd_add_timer(n, RETRANS_TIMER);
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_new_neigh: OUT %p\n", n);
#endif

   return n;
}
/*--END_FUNCTION--(nd_new_neigh)-----------------------------------------*/

/*--START FUNCTION--(nd_get_neighbor)-------------------------------------
    
    Retrieve neighbor entry with address *addr from cache.
    
    Called from:
         - route_add:   in case a gateway was specified.
         - route_check: when adding a new cache entry.

    Last update:
    07/26/96: bb
    
-------------------------------------------------------------------------*/
struct neighbor *nd_get_neighbor(struct device *dev, struct in6_addr *addr)
{
   struct neighbor *neigh;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_get_neighbor: IN -->\n");
   PRINT_ADDR(addr);
#endif

   while (nd_lock)
      sleep_on(&nd_wait);

   nd_fast_lock();
   neigh = nd_lookup_1(addr);
   nd_unlock();
   wake_up(&nd_wait);

   if (neigh == NULL) {
      if ((dev->flags & IFF_NOARP) || (dev->flags & IFF_LOOPBACK)) {
	 neigh = nd_new_neigh(dev, addr, ND_REACHABLE, 0);
      } else {
	 neigh = nd_new_neigh(dev, addr, ND_INCOMPLETE, 0);
      }

   } else {
      atomic_inc(&neigh->refcnt);
   }

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_get_neighbor: OUT %p\n", neigh);
#endif
   return neigh;
}
/*--END FUNCTION--(nd_get_neighbor)----------------------------------------*/

/* %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   TIMER MANAGEMENT

   What about linking together all timed neighbors? Wouldn't that be cute?
   That would avoid to go through all intermediary nodes, and lookup is
   still very convenient cause of the tree structure...
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */

/*--START FUNCTION--(nd_event_timer)-----------------------------------------
|
|   Handles the expiration of a neighbor time.
|   Last update:
|   (bb) 07/30/96
|
---------------------------------------------------------------------------*/
static int nd_event_timer(struct neighbor *n)
{
   struct inet6_ifaddr *ifa;
   struct device *dev = n->dev;
   struct in6_addr msol_addr;
   unsigned long flags;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_timer: IN -->\n");
   PRINT_ADDR(&n->naddr);
#endif

   switch (n->state) {
   case ND_INCOMPLETE:
      if (n->probes == MAX_MULTICAST_SOLICIT) {
	 /* 
	  * Return destination unreachable message.
	  */
#ifdef IPV6_DEBUG_ND
	 printk(KERN_DEBUG "[ND]nd_event_timer: OUT Address Resolution failed\n");
#endif
	 save_flags(flags);
	 cli();
	 n->flags |= NCF_INVALID;
	 n->probes = 0;
	 restore_flags(flags);
	 return 0;
      } else {
	 ifa = ipv6_get_saddr(dev, &n->naddr);
	 MULTISOL_ADDR6(msol_addr, n->naddr);
	 nd_event_send_ns(dev, n, &ifa->addr, &msol_addr, &n->naddr, 1,
			  ND_NEIGHBOR_SOLICITATION);
#ifdef IPV6_DEBUG_ND
	 printk(KERN_DEBUG "[ND]nd_event_timer: OUT probe = %d\n", n->probes);
#endif
	 return RETRANS_TIMER;
      }
      break;

   case ND_DELAY:
      n->state = ND_PROBE;

   case ND_PROBE:
      if (n->probes == MAX_UNICAST_SOLICIT) {
	 /* 
	  * Return destination unreachable message.
	  */
#ifdef IPV6_DEBUG_ND
	 printk(KERN_DEBUG "[ND]nd_event_timer: OUT Address Resolution failed\n");
#endif
	 save_flags(flags);
	 cli();
	 n->flags |= NCF_INVALID;
	 n->state = ND_INCOMPLETE;
	 n->probes = 0;
	 restore_flags(flags);
	 return 0;
      } else {
	 ifa = ipv6_get_saddr(dev, &n->naddr);
	 nd_event_send_ns(dev, n, &ifa->addr, &n->naddr, &n->naddr, 1,
			  ND_NEIGHBOR_SOLICITATION);
#ifdef IPV6_DEBUG_ND
	 printk(KERN_DEBUG "[ND]nd_event_timer: OUT probe = %d\n", n->probes);
#endif
	 return RETRANS_TIMER;
      }
      break;
   default:
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_event_timer: OUT\n");
#endif
   return (~0UL);
}



/*--END FUNCTION--(nd_event_timer)-----------------------------------------*/

/*--START FUNCTION--(nd_timer_handler_1)-----------------------------------
|
|   Recursive function called by nd_timer_handler to
|   handle all "expired" neighbors after a time interrupt.
|   Last update:
|   (bb) 07/30/96
|
---------------------------------------------------------------------------*/
static void nd_timer_handler_1(struct rt_neigh *node, unsigned long *now,
			       unsigned long *ntimer)
{
   struct rt_neigh *rn;
   struct neighbor *n = NULL;
   int pass = 0;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_timer_handler_1: IN\n");
#endif

/* Depth first search */
/* Try go left first */
   if ((rn = node->left)) {
      nd_timer_handler_1(rn, now, ntimer);
      pass++;
   }
/* Try go right */
   if ((rn = node->right)) {
      nd_timer_handler_1(rn, now, ntimer);
      pass++;
   }
   if ((!pass) && (n = node->leaf)) {	/* Neighbors preflen is 128 */
      if (n->state & ND_TIMED) {
	 int time;

	 /* PRINT_ADDR(&n->naddr); */
	 if (n->expires <= *now) {
	    time = nd_event_timer(n);
	 } else
	    time = n->expires - *now;

	 if (time == 0) {	/* Leaf can be deleted */
	    unsigned long flags;

	    save_flags(flags);
	    cli();

	    neighbor_release(n);

	    restore_flags(flags);
	    return;
	 }
	 *ntimer = min(*ntimer, time);
      }
   }
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_timer_handler_1: OUT\n");
#endif
}
/*--END FUNCTION--(nd_timer_handler_1)--------------------------------------*/

/*--START FUNCTION--(nd_timer_handler)----------------------------------------
|
|  Called by interrupt when nd_timer expires.
|  Last update:
|  (bb) 07/30/96
|
---------------------------------------------------------------------------*/
static void nd_timer_handler(unsigned long arg)
{
   unsigned long now = jiffies;
   unsigned long ntimer = ~0UL;

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_timer_handler: IN\n");
#endif

   while (nd_lock)
      sleep_on(&nd_wait);
   nd_fast_lock();

   nd_timer_handler_1(&nd_root, &now, &ntimer);

   if (ntimer != (~0UL)) {
      nd_timer.expires = jiffies + ntimer;
      add_timer(&nd_timer);
   }
   nd_unlock();
   wake_up(&nd_wait);

#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_timer_handler: OUT\n");
#endif

   return;
}

/* 
 * Called by upper layers to validate neighbour cache entries. 
 * From Roque's tcp.
 */

void nd_validate(struct neighbor *n)
{ 
        if (n->state == ND_INCOMPLETE)
                return;

        if (n->state == ND_DELAY) 
        {
                nd_del_timer(n);
        }

        /* nd_stats.rcv_upper_conf++; */
        n->state = ND_REACHABLE;
        n->tstamp = jiffies;
}

/*--START FUNCTION--(nd_init)---------------------------------------------

   Initializes the neighbor discovery timer.

---------------------------------------------------------------------------*/
void nd_init(void)
{
#ifdef IPV6_DEBUG_ND
   printk(KERN_DEBUG "[ND]nd_init: ND INITIALISATION\n");
#endif
   init_timer(&nd_timer);
   nd_timer.function = nd_timer_handler;
   nd_timer.data = 0L;
   nd_timer.expires = 0L;
   return;
}






