/*
 * DRET-IPv6  An implementation of the TCP/IP protocol suite for the LINUX
 *              operating system.  INET6 is implemented using the BSD Socket
 *              interface as the means of communication with the user level.
 *
 * Contacts:    
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id$ 
 *      
 * Original authors:    
 *              Pedro Roque             <roque@di.fc.ul.pt>     
 *              Peter Belding           <pbelding@qualcomm.com>
 *
 *
 * Some fixes and present version:
 *              
 *
 * Description:
 *
 *     Routing Table
 *
 *     simplified version of a radix tree
 *
 *     - every node shares its ancestors prefix
 *     - the tree is ordered from less to most specific mask
 *     - default routes are handled apart
 *
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *              Portions of the software are derived from  various 
 *              networking code publicly available, mainly:
 *                      NRL IPv6 code 
 *                      NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *                      Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */

/*---------- Contents ----------------------------------------------------
|
|     
|  rtd_release
|   dcash_hash_function
|   pfx_match
|   addr_bit
|   addr_diff
|   rt_add_1
|   rt_lookup_1
|   rt_del_3
|   rt_del_2
|   rt_del_1
|   rt_flush_1
|   rt_flush_2
|   rt_flush_3
|   ipv6_route_add
|   ipv6_route_del   
|   rt_lookup
|   route_check
|
|   ipv6_route_init
|   ipv6_route_ioctl
|
|   07/20/96
|         TODO: All links with ND
|         Garbage collection
|
-------------------------------------------------------------------------*/
/*
 *     TODO: 
 *     maintain more statistic vars and make them availiable through /proc
 */

#include <asm/uaccess.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/sched.h>
#include <linux/net.h>
#include <linux/route.h>
#include <linux/netdevice.h>
#include <linux/dret/netdevice6.h>
#include <linux/in.h>
#include <net/netlink.h>
#include <net/sock.h>
#include <net/snmp.h>

#include <net/dret/ipv6.h>
#include <net/dret/nd.h>
#include <net/dret/protocol6.h>
#include <net/dret/route6.h>
#include <net/dret/addrconf.h>
#include <net/dret/sit.h>

#define IPV6_DEBUG_ROUTE
#define IPV6_DEBUG_ROUTE_ND
#define PRINT_ADDR(a) printk(KERN_DEBUG  "[ROUTE6]%X:%X:%X:%X (daddr)\n",\
			     (a)->s6_addr32[0], (a)->s6_addr32[1], \
			     (a)->s6_addr32[2], (a)->s6_addr32[3]);


struct rt_dest null_entry =
{
   NULL, NULL,
   {
      {
	 {0}}},
   1, 1,
   NULL, NULL,
   0, 0, RTF_REJECT
};

struct rt_node rt_root =
{
   NULL, NULL, NULL, &null_entry,
   0, RTN_ROOT
};

struct rt_dest *default_routes = NULL;

static atomic_t rt_entries = 0;	/* Num of rt_dest entries in table   */
static atomic_t rt_nodes = 0;	/* Num of nodes in rt tree           */


#define TABLE_BH_FREE	 	2

atomic_t table_lock;
unsigned table_bh_mask;

/*static struct rt_dest *table_free_queue; */
struct wait_queue *table_wait;

/*
 *     Destination cache works like the route cache used in v4 code...
 */
struct destination *dest_cache[256];
atomic_t dcache_lock = 0;
int dcache_size = 0;
struct timer_list dcache_timer;
int dc_timer_running = 0;



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   TOOLS

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */
/*--START FUNCTION--(rtd_release)-------------------------------------------

  Virtually releases a route entry.

----------------------------------------------------------------------------*/
static __inline__ void rtd_release(struct rt_dest *rt)
{
   printk(KERN_DEBUG "[ROUTE6]rtd_release: rt_use = %d\n", rt->rt_use);
   atomic_dec(&rt->rt_use);
   if (rt->rt_use == 0) {
      kfree(rt);
      atomic_dec(&rt_entries);
      printk(KERN_DEBUG "[ROUTE6]rtd_release: released a leaf... \n");
   }
   printk(KERN_DEBUG "[ROUTE6]rtd_release: rt_use = %d\n", rt->rt_use);
}
/*--END FUNCTION--(rtd_release----------------------------------------------*/

/*--START FUNCTION--(pfx_match)--------------------------------------------
 |
 | Input:   2 ipv6 addresses
 |          a prefix length
 |
 | Returns: 1 if addresses match
 |          0 if addresses mismatch.
 -------------------------------------------------------------------------*/
static __inline__ int pfx_match(struct in6_addr *a1, struct in6_addr *a2,
				int prefixlen)
{
   int pdw;
   int pbi;

#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]pfx_match: IN, pfx length: %d\n", prefixlen);
   PRINT_ADDR(a1);
   PRINT_ADDR(a2);
#endif
   pdw = prefixlen >> 0x05;	/* # of whole __u32 in prefix */
   pbi = prefixlen & 0x1f;	/* #  of bits in incomplete __u32 in pfx */

   if (pdw) {
      /* Compare prefixes ( 4*pwd bytes) */
      if (memcmp(a1, a2, pdw << 2))
	 return 0;
   }
   if (pbi) {
      __u32 w1, w2;
      __u32 mask;

      w1 = a1->s6_addr32[pdw];
      w2 = a2->s6_addr32[pdw];

      mask = htonl((0xffffffff) << (0x20 - pbi));
#ifdef IPV6_DEBUG_ROUTE
      printk(KERN_DEBUG "[ROUTE]pfx_match: OUT, %d\n", 
	                (((w1 ^ w2) & mask) ? 1 : 0));
#endif
      if ((w1 ^ w2) & mask)
	 return 0;
   }
   return 1;
}
/*--END FUNCTION--(pfx_match)----------------------------------------------*/

/*--START FUNCTION--(addr_bit)-----------------------------------------
 |
 | Input: an ipv6 address, (in network order)
 |        an integer.
 |
 | Returns: !0 if bit #bit is set in the address, 0 otherwise.
 |
 -------------------------------------------------------------------------*/
static __inline__ int addr_bit(struct in6_addr *addr, int bit)
{
   int dw;
   __u32 b1;
   __u32 mask;

   /* Find corresponding byte */
   dw = bit >> 0x05;		/* dw = offset of bit's __u32 */
   b1 = addr->s6_addr32[dw];	/* b1 = bit's __u32           */
   mask = htonl((1 << (0x1f - (bit & 0x1f))));	/* bit's position in b1 */
   return (b1 & mask);
}

/*--END FUNCTION--(addr_bit)---------------------------------------------*/

/*--START FUNCTION--(addr_diff)-----------------------------------------------
 |
 | Input: 2 ipv6 addresses,
 |        an int indicating where to start the comparison from.
 |  
 | Returns: the position of the first mismatching bit after start.
 |          (in network order)
 |
 ---------------------------------------------------------------------------*/
static __inline__ int addr_diff(struct in6_addr *a1, struct in6_addr *a2,
				int start)
{
   int dw;
   int i;

   dw = start >> 0x5;		/* byte offset of bit 'start' */
   for (i = dw; i < 4; i++) {
      __u32 b1, b2;
      __u32 xb;

      b1 = a1->s6_addr32[i];
      b2 = a2->s6_addr32[i];

      if ((xb = (b1 ^ b2))) {	/* b1 XOR b2 */
	 int res = 0x20;	/* 32 */

	 xb = htonl(xb);	/* xb put in network order */
	 while (xb) {		/* shift right til 0 */
	    res--;
	    xb >>= 1;
	 }
	 return (i << 5) + res;
      }
   }
   return 128;			/* full match */
}
/*--END FUNCTION--(addr_diff)---------------------------------------------*/

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   TABLE MANAGEMENT

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */
/*--START FUNCTION--(rt_add_1)--------------------------------------------
 |
 | Adds a route to a destination *addr.
 |
 | Input: an ipv6 address *addr.
 |        its prefix length
 |        a route destination structure.
 | Return: 0
 |
 ------------------------------------------------------------------------ */
static int rt_add_1(struct in6_addr *addr, int plen, struct rt_dest *rt)
{
   struct rt_node *fn;
   struct rt_node *pn = NULL;
   struct rt_node *in;
   struct rt_node *ln;
   __u32 bit;
   __u32 dir = 0;

#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_add_1: IN\n");
#endif
   if ((SAME_ADDR6(*addr, in6addr_any)) && (plen == 0)) {   /* default route */
#ifdef IPV6_DEBUG_ROUTE      
      printk(KERN_DEBUG "[ROUTE]rt_add_1: Default route OUT\n");
#endif
      rt->next = default_routes;
      default_routes = rt;
      return 0;
   }
   fn = &rt_root;		/* fn points to root */
   for (;;) {
      if (fn == NULL) {
#ifdef IPV6_DEBUG_ROUTE_1
	 printk(KERN_DEBUG "[ROUTE]rt_add_1: fn==NULL\n");
#endif
	 ln = kmalloc(sizeof(struct rt_node), GFP_ATOMIC);
	 if (ln == NULL)
	    return (-ENOMEM);

	 memset(ln, 0, sizeof(struct rt_node));
	 ln->rn_bit = plen;
	 ln->rn_flags = RTN_BACKTRACK;

	 ln->parent = pn;
	 ln->leaf = rt;
	 rt->route = ln;

	 if (dir)
	    pn->right = ln;
	 else
	    pn->left = ln;
#ifdef IPV6_DEBUG_ROUTE
	 printk(KERN_DEBUG "[ROUTE]rt_add_1: OUT\n");
#endif
	 atomic_inc(&rt_nodes);
	 return (0);
      }
      if (plen >= fn->rn_bit &&
	  pfx_match(&fn->leaf->rd, addr, fn->rn_bit)) {
	 if (plen == fn->rn_bit) {
	    /*struct rt_dest *l1; */
#ifdef IPV6_DEBUG_ROUTE_1
	    printk(KERN_DEBUG "[ROUTE6]rt_add_1: plen==fn->rn_bit && ....%d\n", plen);
#endif
	    /* clean up an intermediate node */
	    if ((fn->rn_flags & RTN_BACKTRACK) == 0) {
	       rtd_release(fn->leaf);
	       fn->leaf = NULL;
	       fn->rn_flags |= RTN_BACKTRACK;
	    } else {
/*	       printk("Route with same prefix already registered \n");*/
               return 0;
	    }
	    /* 
	     * 11/21/96 Eliminating redundant routes and identical routes 
	     *          with different devices.
	     *
	     for (l1 = fn->leaf; l1; l1 = l1->next){
	     if (SAME_ADDR6(rt->rd, l1->rd) &&
	     (rt->dev == l1->dev)) {
	     printk("Route already registered \n");
	     return 0;
	     }
	     }*/
	    rt->next = fn->leaf;
	    fn->leaf = rt;
	    rt->route = fn;
#ifdef IPV6_DEBUG_ROUTE
	    printk(KERN_DEBUG "[ROUTE]rt_add_1: OUT\n");
#endif
	    return 0;
	 }
	 /* plen > ref. bit, 
	  * input addr does match leaf addr up to ref bit
	  */
#ifdef IPV6_DEBUG_ROUTE_1
	 printk(KERN_DEBUG "[ROUTE6]rt_add_1: plen>fn->rn_bit &&.... %d \n", plen);
#endif
	 /* walk down on tree */
	 pn = fn;
	 dir = addr_bit(addr, fn->rn_bit);
	 fn = dir ? fn->right : fn->left;
#ifdef IPV6_DEBUG_ROUTE	 
	 printk(KERN_DEBUG "[ROUTE]rt_add_1: dir:%d\n", dir);
#endif
	 continue;
      }
      /*
       * plen < rn_bit... don't have a common prefix anymore
       * or we have a less significant route.
       * insert an intermediate node on the list.
       * this new rt_node will point to the rt_dest we add
       * and the current rt_node
       */
      pn = fn->parent;

      /* find 1st bit in difference between the 2 addrs 
       * starting at the reference bit of the parent node.
       */
      bit = addr_diff(addr, &fn->leaf->rd, pn->rn_bit);
      if (plen > bit) {
	 /*           (pn)
	  *           / or\ 
	  *          (in)  ...
	  *         /    \
	  *       (ln)   (fn) 
	  */
	 /* allocate new rt_node */
	 in = kmalloc(sizeof(struct rt_node), GFP_ATOMIC);
	 if (in == NULL)
	    return (-ENOMEM);
	 atomic_inc(&rt_nodes);
	 memset(in, 0, sizeof(struct rt_node));

	 /* 
	  * new intermediate node. 
	  * RTN_BACKTRACK will be off since that an address 
	  * that chooses one of the branches would not match 
	  * less specific routes in the other branch
	  */
	 in->rn_bit = bit;	/* diff btw addr & fn->... */
	 in->parent = pn;
	 in->leaf = rt;		/* connect rt_dest input param. */
	 atomic_inc(&rt->rt_use);

	 /* update parent pointer */
	 if (dir)
	    pn->right = in;
	 else
	    pn->left = in;

	 /* leaf node */
	 ln = kmalloc(sizeof(struct rt_node), GFP_ATOMIC);
	 if (ln == NULL) {
	    kfree(in);
	    return (-ENOMEM);
	 }
	 atomic_inc(&rt_nodes);
	 memset(ln, 0, sizeof(struct rt_node));
	 ln->rn_bit = plen;	/* prefix length */
	 /* We may backtrack and find a less specific route */
	 ln->rn_flags = RTN_BACKTRACK;

	 ln->parent = in;
	 fn->parent = in;

	 ln->leaf = rt;
	 rt->route = ln;

	 if (addr_bit(addr, bit)) {
	    in->right = ln;
	    in->left = fn;
	 } else {
	    in->left = ln;
	    in->right = fn;
	 }
#ifdef IPV6_DEBUG_ROUTE
	 printk(KERN_DEBUG "[ROUTE]rt_add_1: OUT with success new int.: %d %d\n", bit, plen);
#endif
	 return 0;
      }
      /*   plen <= bit where the mismatch occurs.  
       *                 (pn)
       *                /
       *              (ln)
       *             /   \
       *           (fn)   NULL
       */
      ln = kmalloc(sizeof(struct rt_node), GFP_ATOMIC);
      if (ln == NULL)
	 return (-ENOMEM);
      atomic_inc(&rt_nodes);
      memset(ln, 0, sizeof(struct rt_node));
      ln->rn_bit = plen;
      ln->rn_flags = RTN_BACKTRACK;

      ln->parent = pn;
      ln->leaf = rt;
      rt->route = ln;

      if (dir)
	 pn->right = ln;
      else
	 pn->left = ln;

      if (addr_bit(&fn->leaf->rd, plen))
	 ln->right = fn;
      else
	 ln->left = fn;

      fn->parent = ln;
#ifdef IPV6_DEBUG_ROUTE
      printk(KERN_DEBUG "[ROUTE]rt_add_1: OUT with success new node %d \n", plen);
#endif
      return (0);
   }
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_add_1: OUT with failure\n");
#endif
   return (-1);			/* failed to add a rt_node */
}
/*--END FUNCTION--(rt_add_1)-----------------------------------------------*/


/*--START FUNCTION--(rt_lookup_1)--------------------------------------------
 |
 | Lookup a route to destination *addr.
 |
 | Input: an ipv6 address
 |
 | Returns the route to destination structure.
 |
 --------------------------------------------------------------------------*/
struct rt_dest *rt_lookup_1(struct in6_addr *addr)
{
   struct rt_node *fn, *next;
   int dir;

#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_lookup_1: IN --->\n");
   PRINT_ADDR(addr);
#endif
   fn = &rt_root;
   for (;;) {			/* walk down the tree as far as possible */
      dir = addr_bit(addr, fn->rn_bit);
      next = dir ? fn->right : fn->left;

      /* printk(KERN_DEBUG "[ROUTE]rt_lookup_1: lookup dir:%d\n", dir); */

      if (next) {
	 fn = next;
	 continue;
      }
      break;
   }

   while ((fn->rn_flags & RTN_ROOT) == 0) {	/* while not at root... */
      if (fn->rn_flags & RTN_BACKTRACK) {	/* we can backtrack... */
	 if (pfx_match(&fn->leaf->rd, addr,
		       fn->leaf->rt_prefixlen)) {
#ifdef IPV6_DEBUG_ROUTE
	    printk(KERN_DEBUG "[ROUTE]rt_lookup_1: OUT with match\n");
#endif
	    return fn->leaf;
	 }
	 /*      printk(KERN_DEBUG "[ROUTE]rt_lookup_1:match fail bit=%d\n",
	    fn->rn_bit);
	    printk(KERN_DEBUG "[ROUTE]rt_lookup_1:  1:%x 2:%x\n",
	    addr->s6_addr32[3],
	    fn->leaf->rd.s6_addr32[3]); */
      }
#ifdef IPV6_DEBUG_ROUTE      
      printk(KERN_DEBUG "|up|\n");	/* FIXME */
#endif
      fn = fn->parent;
   }
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_lookup_1: OUT with no match\n");
#endif
   return NULL;
}
/*--END FUNCTION--(rt_lookup_1)-----------------------------------------*/

/*--START FUNCTION--(rt_del_3)------------------------------------------
 |
 | Called to trim the tree of intermediate nodes when possible
 |
 | INPUT: rt_node to be deleted.
 | CALLED BY: rt_del_2.
 -----------------------------------------------------------------------*/
static void rt_del_3(struct rt_node *fn)
{
   register struct rt_node *tmp;
   int children = 0;
   register int dir = 0;
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE6]rt_del_3 IN BIT = %d\n", fn->rn_bit);
#endif
   /*
    *      0 or one children:
    *              delete the node
    *      2 children:
    *              move the bit down
    */
   if (fn->left) {
      children++;
   }
   if (fn->right) {
      children++;
      dir = 1;
   }
   if (children < 2) {
      tmp = fn->parent;
      if (tmp->left == fn) {
	 tmp->left = dir ? fn->right : fn->left;
	 if (children)		/* tmp->left != NULL */
	    tmp->left->parent = tmp;
      } else {
	 tmp->right = dir ? fn->right : fn->left;
	 if (children)		/* tmp->right != NULL */
	    tmp->right->parent = tmp;
      }
      /* 
       *      try to colapse on top
       */
      if ((fn->parent->rn_flags & (RTN_BACKTRACK | RTN_ROOT)) == 0)
	 rt_del_3(fn->parent);
      kfree(fn);
      atomic_dec(&rt_nodes);
#ifdef IPV6_DEBUG_ROUTE
      printk(KERN_DEBUG "[ROUTE6]rt_del_3: OUT, node deleted\n");
#endif
      return;
   }
   /* Exactly two children ... */
   /* bit = addr_diff(&fn->left->leaf->rd, 
      &fn->right->leaf->rd, fn->rn_bit);

      fn->rn_bit = bit; */
   fn->rn_flags &= ~RTN_BACKTRACK;	/* turn off backtrack bit */
   fn->leaf = fn->left->leaf;
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE6]rt_del_3: OUT, fn_bit updated\n");
#endif
   return;
}
/*--END FUNCTION--(rt_del_3)-----------------------------------------------*/


/*--START FUNCTION--(rt_del_2)---------------------------------------------
 |
 | Removes a route destination entry.
 | INPUT: an ipv6 address *addr, 
 |        its prefix length, 
 |        the route gateway, 
 |        the route device.
 | ASSUMPTION: 
 | Returns: the radix node where the entry was deleted or NULL if not found.
 | CALLED BY: rt_del_1
 |       
--------------------------------------------------------------------------*/
static struct rt_node *rt_del_2(struct in6_addr *addr, __u32 prefixlen,
				struct in6_addr *gw, struct device *dev)
{
   struct rt_node *fn;
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE6]rt_del_2: IN \n");
#endif
   for (fn = &rt_root; fn;) {
      int dir;

      if ((fn->rn_flags & RTN_BACKTRACK) &&
	  prefixlen == fn->rn_bit &&
	  pfx_match(&fn->leaf->rd, addr, fn->rn_bit)) {
	 /* We've found a matching rt_node */
	 break;
      }
      /* Go down the tree */
#ifdef IPV6_DEBUG_ROUTE      
      printk(KERN_DEBUG "BCKTCK: %d, PREF: %d, BIT: %d MATCH %d \n",
	     ((fn->rn_flags & RTN_BACKTRACK) ? 1 : 0),
	     prefixlen, fn->rn_bit,
	     ((pfx_match(&fn->leaf->rd, addr, fn->rn_bit) ? 1 : 0)));
#endif
      dir = addr_bit(addr, fn->rn_bit);
      fn = dir ? fn->right : fn->left;
   }
   /* 
    *  search matching rt_node leaves...
    */
   if (fn) {
      struct rt_dest *back = NULL;
      struct rt_dest *lf;

      for (lf = fn->leaf; lf; lf = lf->next) {
	 if (SAME_ADDR6(*addr, lf->rd)) {
	    /* delete this entry */
	    if (back == NULL)
	       fn->leaf = lf->next;
	    else
	       back->next = lf->next;
	    /* remove leaf */
	    dev = lf->dev;
	    /* FIXME decrement reference count */
	    rtd_release(lf);	    
#ifdef IPV6_DEBUG_ROUTE
	    printk(KERN_DEBUG "[ROUTE6]rt_del_2: OUT found\n");
#endif
	    return fn;
	 }
	 back = lf;
      }
   }
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE6]rt_del_2: OUT not found\n");
#endif
   return NULL;
}
/*--END FUNCTION--(rt_del_2)-----------------------------------------------*/


/*--START FUNCTION--(rt_del_1)---------------------------------------------
 |
 | Input: an ipv6 address,
 |        its prefix length
 |        a gateway address for that destination.
 |        the device to that destination.
 |
 |--------------------------------------------------------------------------*/
static int rt_del_1(struct in6_addr *addr,
		    __u32 prefixlen,
		    struct in6_addr *gw,
		    struct device *dev)
{
   struct rt_node *fn;
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_del_1: IN \n");
#endif
   if (!prefixlen) {		/* default route */
      struct rt_dest *tmp;	/* FIXME: Assume only one default */
      tmp = default_routes;
      if (tmp) {
	 dev = tmp->dev;
	 default_routes = tmp->next;
	 rtd_release(tmp);
	 return (0);
      } else {
	 return -ENXIO;
      }
   }
   fn = rt_del_2(addr, prefixlen, gw, dev);

   if (fn == NULL) {
#ifdef IPV6_DEBUG_ROUTE
      printk(KERN_DEBUG "[ROUTE]rt_del_1: OUT(NOENT)\n");
#endif
      return -ENOENT;		/* No entry found by rt_del_2 */
   }
   if (fn->leaf == NULL) {	/* No more rt_dest struct linked 
				 * Delete rt_node
				 */
      rt_del_3(fn);
   }
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_del_1: OUT \n");
#endif
   return (0);
}

/*--START FUNCTION--(rt_flush_3)------------------------------------------
 |
 | Flush a route node with leaves...
 |
 -------------------------------------------------------------------------*/
void rt_flush_3(struct rt_node *fn)
{
   struct rt_dest *leaf;
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_flush_3 : IN\n");
#endif
   /* Flush all leaves */
   if (fn->rn_flags & RTN_BACKTRACK) {	/* Flush all leaves */
      for (leaf = fn->leaf; leaf;) {
	 fn->leaf = leaf->next;
	 rtd_release(leaf);
	 leaf = fn->leaf;
      }
   }
   /* Flush route node */
   if (fn->parent->left == fn)
      fn->parent->left = NULL;
   else
      fn->parent->right = NULL;
   kfree(fn);
   atomic_dec(&rt_nodes);
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_flush_3 : OUT\n");
#endif
   return;
}

/*--START FUNCTION--(rt_flush_2)------------------------------------------
 |
 | Flush the route table...
 |
 -------------------------------------------------------------------------*/
void rt_flush_2(struct rt_node *fn)
{
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_flush_2 : IN\n");
#endif
   if (fn->left)
      rt_flush_2(fn->left);

   if (fn->right)
      rt_flush_2(fn->right);

   if (fn != &rt_root)
      rt_flush_3(fn);
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_flush_2 : OUT\n");
#endif
   return;
}

/*--START FUNCTION--(rt_flush_1)------------------------------------------
 |
 | Flush the route table...
 | Could be replaced by rt_flush_2(&rt_root)
 -------------------------------------------------------------------------*/
void rt_flush_1(void)
{
   struct rt_node *fn;
   while (table_lock)
      sleep_on(&table_wait);
   table_fast_lock();
   fn = &rt_root;
   rt_flush_2(fn);
   table_unlock();
   wake_up(&table_wait);
   return;
}

/*--START FUNCTION--(ipv6_route_add)---------------------------------------
 | 
 | Call after an ioctl call with XXXXXXXXXXX
 | Input: a route message
 |        -> type-destination-gateway-prefix length-metric-device-flags
 |
 | Return:
 |
--------------------------------------------------------------------------*/
int ipv6_route_add(struct in6_rtmsg *rtmsg)
{
   struct rt_dest *rt;
   struct device *dev = NULL;
   int flags = rtmsg->rtmsg_flags;


#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]ipv6_route_add: IN device: %s\n",
	  rtmsg->rtmsg_device);
   PRINT_ADDR(&rtmsg->rtmsg_dst);
   PRINT_ADDR(&rtmsg->rtmsg_gateway);
#endif

   if (rtmsg->rtmsg_device) {
      if (!(dev = dev_get(rtmsg->rtmsg_device))) {
	 printk(KERN_DEBUG "[ROUTE]Can't find device %s\n", rtmsg->rtmsg_device); 
	 return -ENODEV;
      }
      if (!dev->dev6_lst) {
	 printk(KERN_DEBUG "[ROUTE]No IPv6 support on device %s\n", dev->name);
	 return -EAFNOSUPPORT;
      }
#ifdef IPV6_DEBUG_ROUTE       
      printk(KERN_DEBUG "[ROUTE]ipv6_route_add: device: %s %p %04X\n",
	                                   dev->name, dev, dev->flags);
#endif
   }
   rt = (struct rt_dest *) kmalloc(sizeof(struct rt_dest), GFP_KERNEL);
   if (rt == NULL)
      return (-ENOMEM);
   atomic_inc(&rt_entries);
   memset(rt, 0, sizeof(struct rt_dest));

   COPY_ADDR6(rt->rd, rtmsg->rtmsg_dst);
   rt->rt_prefixlen = rtmsg->rtmsg_prefixlen;
   rt->rt_metric = rtmsg->rtmsg_metric;
   rt->rt_flags = rtmsg->rtmsg_flags;
   rt->rt_use = 1;

   /*
    * BSD emulation: Permits route add someroute gw one-of-my-addresses
    *   to indicate which iface. Not as clean as the nice Linux dev technique
    *   but people keep using it...  (and gated likes it ;))
    */
   if (!dev && (flags & RTF_GATEWAY)) {
      struct inet6_ifaddr *ifa;

      if ((ifa = get_ifaddr6(&rtmsg->rtmsg_gateway))) {
	 if (ifa->idev->dev->flags & IFF_UP) {
	    flags &= ~RTF_GATEWAY;
	    dev = ifa->idev->dev;
	 }
      }
   }
   if (dev) {
      if (dev->flags & IFF_TUNNEL) {
	 if (!(flags & RTF_GATEWAY)) {
	    if (!IS_IPV4ADDR6(rtmsg->rtmsg_dst)) {
	       flags |= RTF_GATEWAY;
	       memset(&rtmsg->rtmsg_gateway, 0, sizeof(struct in6_addr));
	       rtmsg->rtmsg_gateway.s6_addr32[3] = get_sit_end_point(dev);
	    }
	 }
      }
   }
   if (flags & RTF_GATEWAY) {
      struct rt_dest *gw;
      /* Get neighbor cache entry for gateway */
      if (IS_LINKLADDR6(rtmsg->rtmsg_gateway) && dev) {
	 struct rt_dest *rt1;
	 rt->neigh = nd_get_neighbor(dev, &rtmsg->rtmsg_gateway);
	 
	 rt1 = (struct rt_dest *) kmalloc(sizeof(struct rt_dest), GFP_KERNEL);
         if (rt1 == NULL)
            return (-ENOMEM);
	 
         atomic_inc(&rt_entries);
         memset(rt1, 0, sizeof(struct rt_dest));

         rt1->rt_metric = 1;
         rt1->rt_flags = RTF_HOST;
         rt1->rt_use = 1;
	 rt1->refcnt = 1;
	 COPY_ADDR6(rt1->rd, rtmsg->rtmsg_gateway);
	 rt1->dev = dev;
         rt1->rt_pmtu = dev->mtu;
	 rt1->rt_prefixlen = 128;
	 rt1->neigh = rt->neigh;

         while (table_lock) {
            sleep_on(&table_wait);
         }
         table_fast_lock();

         rt_add_1(&rt1->rd, rt1->rt_prefixlen, rt1);

         table_unlock();
         wake_up(&table_wait);
	 
#ifdef CONFIG_NET_RTNETLINK6
         ipv6_netlink_msg(RTMSG_NEWROUTE, &rt1->rd,
		         (rt1->neigh ? &rt1->neigh->naddr : 0),
		          rt1->rt_prefixlen, dev->flags, dev->metric, dev->name);
#endif
	 
	 if (!rt->neigh) {
	    kfree(rt);
	    return -ENETUNREACH;
	 }
      } else if ((gw = rt_lookup(&rtmsg->rtmsg_gateway, 0))) {
	 /* Route to GW exists */
	 atomic_inc(&gw->rt_use);
	 atomic_inc(&gw->refcnt);
	 dev = gw->dev;
	 rt->neigh = gw->neigh;
      } else {
#ifdef IPV6_DEBUG_ROUTE	 
	 printk("No route to gateway...\n");
#endif
	 kfree(rt);
	 return -EHOSTUNREACH;
      }
      rt->rt_flags |= RTF_GATEWAY;
      if ((!rt->neigh) ||
	  !(SAME_ADDR6(rt->neigh->naddr, rtmsg->rtmsg_gateway))) {
	 rt->neigh = nd_get_neighbor(dev, &rtmsg->rtmsg_gateway);
	 if (!rt->neigh) {
	    kfree(rt);
	    return -ENETUNREACH;
	 }
      }
   } else if (rt->rt_prefixlen == 128) {
      if (!(rt->neigh = nd_get_neighbor(dev, &rt->rd))) {
	 kfree(rt);
	 return -ENETUNREACH;
      }
   }
   if (dev == NULL) {
      printk("[ROUTE6]ipv6_rt_add: NULL device\n");
      kfree(rt);
      return -ENETUNREACH;
   }
   if ((lo_route == NULL) && (dev->flags & IFF_LOOPBACK)) {
      lo_route = (struct destination *) rt;
   }
   rt->dev = dev;
   rt->rt_pmtu = dev->mtu;

   while (table_lock) {
      sleep_on(&table_wait);
   }
   table_fast_lock();

   rt_add_1(&rt->rd, rt->rt_prefixlen, rt);

   table_unlock();
   wake_up(&table_wait);
   
#ifdef CONFIG_NET_RTNETLINK6
   ipv6_netlink_msg(RTMSG_NEWROUTE, &rt->rd,
		    (rt->neigh ? &rt->neigh->naddr : 0),
		    rt->rt_prefixlen, dev->flags, dev->metric, dev->name);
#endif
   
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]ipv6_route_add: OUT table_lock = %d\n", table_lock);
#endif
   return 0;
}
/*--END FUNCTION--(ipv6_route_add)-----------------------------------------*/

/*--START FUNCTION--(ipv6_route_del)---------------------------------------
 | 
 | Call after an ioctl call with XXXXXXXXXXX
 | Input: a route message
 |        -> type-destination-gateway-prefix length-metric-device-flags
 |
 | Return: 0;
 |
--------------------------------------------------------------------------*/
int ipv6_route_del(struct in6_rtmsg *rtm)
{
   int err = 0;
   struct device *dev = dev_get(rtm->rtmsg_device);

#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]ipv6_route_del:IN device:%s\n", rtm->rtmsg_device);
   PRINT_ADDR(&rtm->rtmsg_dst);
#endif

   while (table_lock) {
      sleep_on(&table_wait);
   }
   table_fast_lock();
   err = rt_del_1(&rtm->rtmsg_dst,
		  rtm->rtmsg_prefixlen, &rtm->rtmsg_gateway, dev);
   table_unlock();
   wake_up(&table_wait);

   if ((err == 0) && dev) {
#ifdef CONFIG_NET_RTNETLINK6      
      ipv6_netlink_msg(RTMSG_DELROUTE, &rtm->rtmsg_dst, 0,
	       rtm->rtmsg_prefixlen, dev->flags, dev->metric, dev->name);
#endif
   }  
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]ipv6_route_del: OUT table_lock = %d\n", 
	                                                 table_lock);
   printk(KERN_DEBUG "[ROUTE]ipv6_route_del: dev = %p\n", dev);
#endif
   return (err);
}

/*--START FUNCTION--(rt_lookup)-------------------------------------------
 | 
 | 
 |
 |
 ------------------------------------------------------------------------*/
struct rt_dest *rt_lookup(struct in6_addr *addr, int local)
{
   struct rt_dest *rt;
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_lookup: IN\n");
#endif

   if ((rt = rt_lookup_1(addr))) {
#ifdef IPV6_DEBUG_ROUTE
      printk(KERN_DEBUG "[ROUTE]rt_lookup: OUT\n");
      PRINT_ADDR(&rt->rd);
      if (rt->neigh)
	 PRINT_ADDR(&rt->neigh->naddr);
#endif
      return rt;
   }
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]rt_lookup: OUT default_route OUT\n");
   if (default_routes)
      PRINT_ADDR(&default_routes->rd);
#endif
   return default_routes;	/* Assume only one default */
}
/*--END FUNCTION--(rt_lookup)-----------------------------------------------*/

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   ROUTE RETRIEVAL

   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% */
/*--START FUNCTION--(dcache_check)----------------------------------------
 *     check cache entry for validity...
 *     this needs to be done as a inline func that calls
 *     ipv6_slow_dst_check if entry is invalid
 *  Last update:
    07/15/96: simple validity check...
              FIXME
 ---------------------------------------------------------------------------*/
struct destination *dcache_check(struct destination *dest,
				 struct in6_addr *d, int local)
{
   /*
    *      Check route validity
    */
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE6]dcache_check: IN\n");
#endif
   if (dest) {
      if (SAME_ADDR6(dest->route.rd, *d)) {
	 if ((dest->flags & DC_INVALID) == 0) {
	    if ((dest->dneigh) && !(dest->dneigh->flags & NCF_NOARP)) {
	       nd_event_send_packet(dest->dneigh);
	    }
#ifdef IPV6_DEBUG_ROUTE
	    printk(KERN_DEBUG "[ROUTE6]dcache_check: OUT\n");
#endif
	    return dest;
	 } else {
	    /* 
	     * drop usage counter on route 
	     */
	    atomic_dec(&(dest->route).rt_use);

	    /*if ((dest->route).rt_use == 0) {
	       kfree(dest);
	       }  FIXME */
	    /* pmtu value is now unkown */
	    dest->flags &= ~DCF_PMTU;
	 }
      }
   }
/* FIXME */
   if (check_host_addr6(d)) {
#ifdef IPV6_DEBUG_ROUTE
      printk(KERN_DEBUG "[ROUTE6]dcache_check: OUT\n");
#endif
      return lo_route;
   }
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE6]dcache_check: OUT\n");
#endif
   return (route_check(d, local));
}
/*--END FUNCTION--(dcache_check)---------------------------------------*/

/*--START FUNCTION--(route_check)-------------------------------------

  Last update:
   07/15/96: 

-------------------------------------------------------------------------*/
struct destination *route_check(struct in6_addr *d, int local)
{
   struct destination *dc;
   struct rt_dest *rt;

#ifdef IPV6_DEBUG_ROUTE_ND
   printk(KERN_DEBUG "[ROUTE6]route_check: IN\n");
#endif

   while (table_lock)
      sleep_on(&table_wait);
   table_fast_lock();

   rt = rt_lookup(d, local);
   if (rt == NULL) {
      table_unlock();
      wake_up(&table_wait);
      return NULL;
   }
   if ((rt->rt_flags & RTF_GATEWAY) ||
       (rt->rt_prefixlen == 128)) {
      dc = (struct destination *) rt;
      atomic_inc(&rt->rt_use);
   } else {
      dc = (struct destination *) kmalloc(sizeof(struct destination),
					  GFP_ATOMIC);
      if (dc == NULL) {
	 printk(KERN_DEBUG "[ROUTE6]route_check: kmalloc failed OUT\n");
	 return NULL;
      }
      memset(dc, 0, sizeof(struct destination));

      memcpy(&((dc->route).rd), d, sizeof(struct in6_addr));
      dc->route.rt_prefixlen = 128;
      dc->route.rt_use = 1;
      dc->route.rt_metric = rt->rt_metric;
      dc->route.rt_flags = (rt->rt_flags | RTF_HOST
			    | RTF_DCACHE
			    | DCF_PMTU);
      dc->route.dev = rt->dev;
      dc->route.rt_pmtu = rt->dev->mtu;
      dc->tstamp = jiffies;
      dc->route.neigh = nd_get_neighbor(rt->dev, d);
      if (dc->route.neigh == NULL) {
	 printk(KERN_DEBUG "[ROUTE6]route_check: neigh = NULL OUT\n");
	 kfree(dc);
	 return NULL;
      }
      rt_add_1(d, 128, (struct rt_dest *) dc);	/* FIXED 10/24/96 */
   }
   table_unlock();
   wake_up(&table_wait);

#ifdef IPV6_DEBUG_ROUTE_ND
   printk(KERN_DEBUG "[ROUTE6]route_check: OUT\n");
   PRINT_ADDR(&(dc->route.rd));
#endif
   return dc;
}
/*--END FUNCTION--(route_check)----------------------------------------------*/


/*--START FUNCTION--(mcast_check)----------------------------------------



---------------------------------------------------------------------------*/
struct destination *mcast_check(struct device *dev, struct in6_addr *d)
{
   struct destination *dest;
   struct rt_dest *rd;
   __u32 hash;


   hash = ipv6_addr_hash(d);
   atomic_inc(&dcache_lock);
   for (dest = dest_cache[hash]; dest; dest = (struct destination *)
	dest->route.next) {
      if (SAME_ADDR6(dest->route.rd, *d)) {
	 if (dev && dev != (dest->route.neigh)->dev)
	    continue;
	 atomic_inc(&dest->refcnt);
	 atomic_dec(&dcache_lock);
	 return dest;
      }
   }
   /* 
    *  not found 
    */
   if (dev == NULL) {
      rd = rt_lookup(d, 0);

      if (rd == NULL) {
	 atomic_dec(&dcache_lock);
	 return NULL;
      }
      dev = rd->dev;
   }
   dest = (struct destination *) kmalloc(sizeof(struct destination),
					 GFP_ATOMIC);
   if (dest == NULL) {
      atomic_dec(&dcache_lock);
      return NULL;
   }
   memset(dest, 0, sizeof(struct destination));
   memcpy(&dest->route.rd, d, sizeof(struct in6_addr));
   /*dest->n = ncache_get_neigh(dev, d); */
   dest->pmtu = dev->mtu;
   dest->refcnt = 1;
   /* insert at the head of the list */
   dest->route.next = (struct rt_dest *) dest_cache[hash];
   dest_cache[hash] = dest;
   atomic_dec(&dcache_lock);
   return dest;
}
/*--END FUNCTION--(mcast_check)---------------------------------------------*/


/*--END FUNCTION--(ipv6_dst_unlock)----------------------------------------



--------------------------------------------------------------------------*/
void ipv6_dst_unlock(struct destination *dest)
{
   /*
    *      decrement counter and mark entry for deletion
    *      if counter reaches 0. we delay deletions in hope
    *      we can reuse cache entries.
    */
   atomic_dec(&dest->refcnt);
   if (dest->refcnt == 0) {
      dest->tstamp = jiffies;
      /* schedule deletion */
      if (dc_timer_running++ == 0) {
	 dcache_timer.expires = jiffies + DC_TIME_RUN;
	 add_timer(&dcache_timer);
      }
   }
}
/*--END FUNCTION--(ipv6_unlock_dest)--------------------------------------*/


/*--START FUNCTION--(dest_cache_garbage_collect)-------------------------
 *     Use of locking should make this operation atomic in respect 
 *     with the other dest cache hadnling routines.
 

------------------------------------------------------------------------*/
static int dcache_garbage_collect(void)
{
   struct destination *dest, *bp;
   __u32 timeout;
   int more = 0;
   int i;

   if (dcache_size >= DC_WATER_MARK)
      timeout = DC_SHORT_TIMEOUT;
   else
      timeout = DC_LONG_TIMEOUT;
   /*

    */
   for (i = 0; i < 256; i++) {
      bp = NULL;

      for (dest = dest_cache[i]; dest;
	   dest = (struct destination *) dest->route.next) {
	 if (dest->refcnt == 0) {
	    if (jiffies - dest->tstamp >= timeout) {
	       if (bp)
		  bp->route.next = dest->route.next;
	       else
		  dest_cache[i] = (struct destination *) dest->route.next;
	       /* 
	        * FIXME: dec refcount on 
	        * neigh_cache 
	        */
	       kfree(dest);
	       dcache_size--;
	       continue;
	    }
	    more++;
	 }
	 bp = dest;
      }
   }
   return more;			/* Number of deleted entries */
}
/*--END FUNCTION--(dcache_garbage_collect)------------------------------*/

/*--START FUNCTION--(dec_route_use)--------------------------------------

-----------------------------------------------------------------------*/
void dec_route_use(struct destination *rt)
{
   if (rt)
      atomic_dec(&rt->route.rt_use);
}
/*--END FUNCTION--(dec_route_use)----------------------------------------*/

/*--START FUNCTION--(dcache_bh)------------------------------------------

 | Destination cache bottom half.


-----------------------------------------------------------------------*/
static void dcache_bh(unsigned long data)
{
   unsigned long flags;
   int more;

   save_flags(flags);
   cli();
   if (dcache_lock) {
      restore_flags(flags);
      dcache_timer.expires = jiffies + DC_TIME_RETRY;
      add_timer(&dcache_timer);
   }
   more = dcache_garbage_collect();
   dc_timer_running = 0;
   restore_flags(flags);

   if (more) {
      dcache_timer.expires = jiffies + DC_TIME_RUN;
      add_timer(&dcache_timer);
      dc_timer_running++;
   }
}
/*--END FUNCTION--(dcache_bh)-------------------------------------------*/


/*--START FUNCTION--(ipv6_route_ioctl)----------------------------------*/
int ipv6_route_ioctl(unsigned int cmd, void *arg)
{
   struct in6_rtmsg rtmsg;
   int err;

   switch (cmd) {
   case SIOCADDRT:
   case SIOCDELRT:
      if (!suser())
	 return -EPERM;
      err = copy_from_user(&rtmsg, arg, sizeof(struct in6_rtmsg));
      if (err)
	 return -EFAULT;
#ifdef IPV6_DEBUG_ROUTE
      printk(KERN_DEBUG
	     "route_ioctl#: type= %d; prefix = %d; metric = %d; dev = %s flags: %04X\n",
	     rtmsg.rtmsg_type, rtmsg.rtmsg_prefixlen, rtmsg.rtmsg_metric,
	     rtmsg.rtmsg_device, rtmsg.rtmsg_flags);
#endif

      return
	  (cmd == SIOCDELRT) ? ipv6_route_del(&rtmsg) : ipv6_route_add(&rtmsg);
   default:
      return -EINVAL;
   }
}
/*--END FUNCTION--(ipv6_route_ioctl)--------------------------------------*/


void table_run_bh()
{
   unsigned long flags;
   save_flags(flags);
   cli();
   if (table_bh_mask && !table_lock) {
      /*if (table_bh_mask & TABLE_BH_REDIRECT)
         table_kick_backlog(); */
   }
   restore_flags(flags);
}


/*--START FUNCTION--(ipv6_route_init)-------------------------------------

 I Init destination cache.... not really route table....

--------------------------------------------------------------------------*/
void ipv6_route_init(void)
{
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]ipv6_route_init IN\n");
#endif
   init_timer(&dcache_timer);
   dcache_timer.function = dcache_bh;
   dcache_timer.data = 0;
#ifdef IPV6_DEBUG_ROUTE
   printk(KERN_DEBUG "[ROUTE]ipv6_route_init OUT\n");
#endif
}
