/* net/atm/signaling.c - ATM signaling */

/* Written 1995-1997 by Werner Almesberger, EPFL LRC */


#include <linux/errno.h>	/* error codes */
#include <linux/kernel.h>	/* printk, suser */
#include <linux/skbuff.h>
#include <linux/wait.h>
#include <linux/sched.h>	/* jiffies and HZ */
#include <linux/atm.h>		/* ATM stuff */
#include <linux/atmsap.h>
#include <linux/atmsvc.h>
#include <linux/atmdev.h>

#include "tunable.h"
#include "static.h"
#include "signaling.h"


#undef WAIT_FOR_DEMON		/* #define this if system calls on SVC sockets
				   should block until the demon runs.
				   Danger: may cause nasty hangs if the demon
				   crashes. */

#if 0
#define DPRINTK(format,args...) printk(KERN_DEBUG format,##args)
#else
#define DPRINTK(format,args...)
#endif


struct atm_vcc *sigd = NULL;
static struct wait_queue *sigd_sleep = NULL;


static int sigd_send(struct atm_vcc *vcc,struct sk_buff *skb)
{
	struct atmsvc_msg *msg;

	msg = (struct atmsvc_msg *) skb->data;
	atomic_sub(skb->truesize+ATM_PDU_OVHD,&vcc->tx_inuse);
	DPRINTK("sigd_send %d (0x%lx)\n",(int) msg->type,msg->vcc);
	vcc = (struct atm_vcc *) msg->vcc;
	switch (msg->type) {
		case as_okay:
			vcc->reply = msg->reply;
			if (!*vcc->local.sas_addr.prv &&
			    !*vcc->local.sas_addr.pub) {
				vcc->local.sas_family = AF_ATMSVC;
				memcpy(vcc->local.sas_addr.prv,
				    msg->local.sas_addr.prv,ATM_ESA_LEN);
				memcpy(vcc->local.sas_addr.pub,
				    msg->local.sas_addr.pub,ATM_E164_LEN+1);
			}
			if (vcc->vpi || vcc->vci) break;
			vcc->itf = msg->pvc.sap_addr.itf;
			vcc->vpi = msg->pvc.sap_addr.vpi;
			vcc->vci = msg->pvc.sap_addr.vci;
			if (vcc->vpi || vcc->vci) vcc->qos = msg->qos;
			break;
		case as_error:
			vcc->flags &= ~(ATM_VF_REGIS | ATM_VF_READY);
			vcc->reply = msg->reply;
			break;
		case as_indicate:
			vcc = (struct atm_vcc *) msg->listen_vcc;
			DPRINTK("as_indicate!!!\n");
			if (!vcc->backlog_quota) {
				sigd_enq(0,as_reject,(struct atm_vcc *)
				    msg->listen_vcc,NULL,NULL);
				return 0;
			}
			vcc->backlog_quota--;
			skb_queue_tail(&vcc->listenq,skb);
			if (vcc->callback) {
				DPRINTK("waking vcc->sleep 0x%p\n",
				    &vcc->sleep);
				vcc->callback(vcc);
			}
			return 0;
		case as_close:
			vcc->flags |= ATM_VF_RELEASED;
			vcc->flags &= ~ATM_VF_READY;
			vcc->reply = msg->reply;
			break;
		default:
			printk(KERN_ALERT "sigd_send: bad message type %d\n",
			    (int) msg->type);
			return -EINVAL;
	}
	if (vcc->callback) vcc->callback(vcc);
	dev_kfree_skb(skb,FREE_WRITE);
	return 0;
}


void sigd_enq(struct atm_vcc *vcc,enum atmsvc_msg_type type,
    const struct atm_vcc *listen_vcc,const struct sockaddr_atmpvc *pvc,
    const struct sockaddr_atmsvc *svc)
{
	struct sk_buff *skb;
	struct atmsvc_msg *msg;
	struct atm_blli *walk;
	int size,i;
#ifdef WAIT_FOR_DEMON
	static unsigned long silence = 0;
#endif

	DPRINTK("sigd_enq %d (0x%p)\n",(int) type,vcc);
	size = sizeof(struct atmsvc_msg)-sizeof(struct atm_blli);
	if (svc)
		for (walk = svc->sas_addr.blli; walk; walk = walk->next)
			size += sizeof(struct atm_blli);
	while (!(skb = alloc_skb(size,GFP_KERNEL))) schedule();
	skb->len = size;
	msg = (struct atmsvc_msg *) skb->data;
	msg->type = type;
	msg->vcc = (unsigned long) vcc;
	msg->listen_vcc = (unsigned long) listen_vcc;
	if (vcc) {
	    msg->qos = vcc->qos;
	    if (!msg->qos.aal) msg->qos.aal = vcc->aal;
	}
	if (!svc) msg->svc.sas_family = 0;
	else {
		msg->svc = *svc;
		i = 0;
		for (walk = svc->sas_addr.blli; walk; walk = walk->next)
			msg->blli[i++] = *walk;
	}
	if (vcc) msg->local = vcc->local;
	if (!pvc) memset(&msg->pvc,0,sizeof(msg->pvc));
	else msg->pvc = *pvc;
	while (!sigd) {
#ifdef WAIT_FOR_DEMON
		if (silence < jiffies) {
			printk(KERN_INFO "atmsvc: waiting for signaling demon "
			    "...\n");
			silence = jiffies+30*HZ;
		}
		sleep_on(&sigd_sleep);
#else
		printk(KERN_WARNING "atmsvc: no signaling demon\n");
		kfree_skb(skb,FREE_READ);
		return;
#endif
	}
	atomic_add(skb->truesize+ATM_PDU_OVHD,&sigd->rx_inuse);
	skb_queue_tail(&sigd->recvq,skb);
	wake_up(&sigd->sleep);
	if (vcc) vcc->flags |= ATM_VF_REGIS;
}


static void trigger(struct atm_vcc *vcc)
{
	if (vcc->family == PF_ATMSVC && !(vcc->flags & ATM_VF_META)) {
		vcc->flags |= ATM_VF_RELEASED;
		vcc->reply = -EUNATCH;
		wake_up(&vcc->sleep);
	}
}


static void sigd_close(struct atm_vcc *vcc)
{
	struct sk_buff *skb;

	DPRINTK("sigd_close\n");
	sigd = NULL;
	if (skb_peek(&vcc->recvq))
		printk(KERN_ERR "sigd_close: closing with requests pending\n");
	while ((skb = skb_dequeue(&vcc->recvq))) kfree_skb(skb,FREE_READ);
	for_all_vccs(trigger);
}


static struct atmdev_ops sigd_dev_ops = {
	NULL,		/* no open */
	sigd_close,	/* close */
	NULL,		/* no ioctl */
	NULL,		/* no getsockopt */
	NULL,		/* no setsockopt */
	sigd_send,	/* send */
	NULL,		/* no sg_send */
	NULL,		/* no poll */
	NULL,		/* no phy_put */
	NULL,		/* no phy_get */
	NULL		/* no feedback */
};


static struct atm_dev sigd_dev = {
	&sigd_dev_ops,
	NULL,		/* no PHY */
    	"sig",		/* type */
	999,		/* dummy device number */
	NULL,NULL,	/* pretend not to have any VCCs */
	NULL,NULL,	/* no data */
	0,		/* no flags */
	NULL,		/* no local address */
	{ 0 }		/* no ESI, no statistics */
};


int sigd_attach(struct atm_vcc *vcc)
{
	if (sigd) return -EADDRINUSE;
	DPRINTK("sigd_attach\n");
	sigd = vcc;
	vcc->dev = &sigd_dev;
	vcc->flags |= ATM_VF_READY | ATM_VF_META;
	wake_up(&sigd_sleep);
	return 0;
}
