/* net/atm/raw.c - Raw AAL0 and AAL5 transports */

/* Written 1995-1997 by Werner Almesberger, EPFL LRC */


#include <linux/config.h>
#include <linux/sched.h>
#include <linux/atmdev.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/mm.h>
#include <linux/mmuio.h>
#include <linux/uio.h>

#include "common.h"
#include "protocols.h"
#include "tunable.h"		/* tunable parameters */


#if 0
#define DPRINTK(format,args...) printk(KERN_DEBUG format,##args)
#else
#define DPRINTK(format,args...)
#endif


/*
 * SKB == NULL indicates that the link is being closed
 */

void atm_push_raw(struct atm_vcc *vcc,struct sk_buff *skb)
{
	if (skb) {
		DPRINTK("APushR %d += %d\n",vcc->rx_inuse,skb->truesize);
		atomic_add(skb->truesize+ATM_PDU_OVHD,&vcc->rx_inuse);
		skb_queue_tail(&vcc->recvq,skb);
		wake_up(&vcc->sleep);
	}
}


/*
 * A return value of NULL means to discard the PDU
 */


static struct sk_buff *atm_peek_aal0(struct atm_vcc *vcc,unsigned long pdu_size,
    __u32 (*fetch)(struct atm_vcc *vcc,int i))
{
	struct sk_buff *skb;

	if (pdu_size+vcc->rx_inuse+ATM_PDU_OVHD <= vcc->rx_quota) {
		skb = alloc_skb(pdu_size,GFP_ATOMIC);
		if (skb) return skb;
	}
	vcc->stats->rx_drop++;
	return NULL;
}


/*
 * atm_peek_aal5 is currently also used for AAL3/4
 */


static struct sk_buff *atm_peek_aal5(struct atm_vcc *vcc,unsigned long pdu_size,
    __u32 (*fetch)(struct atm_vcc *vcc,int i))
{
	struct sk_buff *skb;

	if (pdu_size+vcc->rx_inuse+ATM_PDU_OVHD <= vcc->rx_quota)
		if (pdu_size < PAGE_SIZE) {
			skb = alloc_skb((pdu_size+3) & ~3,GFP_ATOMIC);
			if (skb) return skb;
			DPRINTK("PEEK: 1\n");
		}
		else {
			skb = alloc_skb(((pdu_size+3) & ~3)+PAGE_SIZE-1,
					GFP_ATOMIC);
			DPRINTK("PEEK: 2\n");
			if (skb) {
				skb->data = (unsigned char *)
				    (((unsigned long) skb->data+PAGE_SIZE-1) &
				    ~(PAGE_SIZE-1));
				DPRINTK("PEEK: data at 0x%p\n",skb->data);
				return skb;
			}
		}
	vcc->stats->rx_drop++;
	return NULL;
}


static void atm_pop_raw(struct atm_vcc *vcc,struct sk_buff *skb)
{
#ifdef CONFIG_MMU_HACKS
	if (skb->atm.iovcnt)
		unlock_user(skb->atm.iovcnt,(struct iovec *) skb->data);
#endif
	DPRINTK("APopR (%d) %d -= %d\n",vcc->vci,vcc->tx_inuse,skb->truesize);
	atomic_sub(skb->truesize+ATM_PDU_OVHD,&vcc->tx_inuse);
	dev_kfree_skb(skb,FREE_WRITE);
#if 0 /* experimental */
if (vcc->dev->sending != 1) printk("sending == %d !!!\n",vcc->dev->sending);
	vcc->dev->sending--;
#endif
	wake_up(&vcc->wsleep);
}


int atm_init_aal0(struct atm_vcc *vcc)
{
	vcc->aal = ATM_AAL0;
	vcc->push = atm_push_raw;
	vcc->peek = atm_peek_aal0;
	vcc->pop = atm_pop_raw;
	vcc->push_oam = NULL;
	return 0;
}


int atm_init_aal34(struct atm_vcc *vcc)
{
	vcc->aal = ATM_AAL34;
	vcc->push = atm_push_raw;
	vcc->peek = atm_peek_aal5; /* same procedure */
	vcc->pop = atm_pop_raw;
	vcc->push_oam = NULL;
	return 0;
}


int atm_init_aal5(struct atm_vcc *vcc)
{
	vcc->aal = ATM_AAL5;
	vcc->push = atm_push_raw;
	vcc->peek = atm_peek_aal5;
	vcc->pop = atm_pop_raw;
	vcc->push_oam = NULL;
	return 0;
}
