#include <linux/types.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <asm/param.h>
#include <asm/atomic.h>
#include <net/route.h>

#include <linux/atmlec.h>

#include "lec.h"
#include "lec_arpc.h"


#define DPRINTK(format,args...)
/*
#define DPRINTK printk
*/
#define DEBUG_ARP_TABLE 0

#define LEC_ARP_REFRESH_INTERVAL (3*HZ)

static void lec_arp_check_expire(unsigned long data);
static __inline__ void lec_arp_expire_arp(unsigned long data);
void dump_arp_table(struct lec_priv *priv);

/* 
 * Arp table funcs
 */

#define HASH(ch) (ch & (LEC_ARP_TABLE_SIZE -1))

static __inline__ void 
lec_arp_lock(struct lec_priv *priv)
{
        atomic_inc(&priv->lec_arp_lock_var);
}

static __inline__ void 
lec_arp_unlock(struct lec_priv *priv)
{
        atomic_dec(&priv->lec_arp_lock_var);
}

/*
 * Initialization of arp-cache
 */
void 
lec_arp_init(struct lec_priv *priv)
{
        unsigned short i;

        for (i=0;i<LEC_ARP_TABLE_SIZE;i++) {
                priv->lec_arp_tables[i] = NULL;
        }        
        init_timer(&priv->lec_arp_timer);
        priv->lec_arp_timer.expires = jiffies+LEC_ARP_REFRESH_INTERVAL;
        priv->lec_arp_timer.data = (unsigned long)priv;
        priv->lec_arp_timer.function = lec_arp_check_expire;
        add_timer(&priv->lec_arp_timer);
}

void
lec_arp_clear_vccs(struct lec_arp_table *entry)
{
        if (entry->vcc) {
                entry->vcc->push = entry->old_push;
                entry->vcc->flags |= ATM_VF_RELEASED;
                entry->vcc->flags &= ~ATM_VF_READY;
                entry->vcc = NULL;
        }
        if (entry->recv_vcc) {
                entry->recv_vcc->push = entry->old_recv_push;
                entry->recv_vcc->flags |= ATM_VF_RELEASED;
                entry->recv_vcc->flags &= ~ATM_VF_READY;
                entry->recv_vcc = NULL;
        }        
}

/*
 * Insert entry to lec_arp_table
 */
static __inline__ void 
lec_arp_put(struct lec_arp_table **lec_arp_tables, 
            struct lec_arp_table *to_put)
{
        unsigned short place;
        unsigned long flags;

        save_flags(flags);
        cli();

        place = HASH(to_put->mac_addr[ETH_ALEN-1]);
        to_put->next = lec_arp_tables[place];
        lec_arp_tables[place] = to_put;
  
        restore_flags(flags);
        DPRINTK("LEC_ARP: Added entry:%2.2x %2.2x %2.2x %2.2x %2.2x %2.2x\n",
                0xff&to_put->mac_addr[0], 0xff&to_put->mac_addr[1],
                0xff&to_put->mac_addr[2], 0xff&to_put->mac_addr[3],
                0xff&to_put->mac_addr[4], 0xff&to_put->mac_addr[5]);
}

/*
 * Remove entry from lec_arp_table
 */
static __inline__ int 
lec_arp_remove(struct lec_arp_table **lec_arp_tables,
               struct lec_arp_table *to_remove)
{
        unsigned short place;
        struct lec_arp_table *tmp;
        unsigned long flags;
        int remove_vcc=1;

        save_flags(flags);
        cli();

        if (!to_remove) {
                restore_flags(flags);
                return -1;
        }
        place = HASH(to_remove->mac_addr[ETH_ALEN-1]);
        tmp = lec_arp_tables[place];
        if (tmp == to_remove) {
                lec_arp_tables[place] = tmp->next;
        } else {
                while(tmp && tmp->next != to_remove) {
                        tmp = tmp->next;
                }
                if (!tmp) {/* Entry was not found */
                        restore_flags(flags);
                        return -1;
                }
        }
        tmp->next = to_remove->next;
        del_timer(&to_remove->timer);
  
        /* If this is the only MAC connected to this VCC, also tear down
           the VCC */
        if (to_remove->status >= ESI_FLUSH_PENDING) {
                /*
                 * ESI_FLUSH_PENDING, ESI_FORWARD_DIRECT
                 */
                for(place=0;place<LEC_ARP_TABLE_SIZE;place++) {
                        for(tmp=lec_arp_tables[place];tmp!=NULL;tmp=tmp->next){
                                if (memcmp(tmp->atm_addr, to_remove->atm_addr,
                                           ATM_ESA_LEN)==0) {
                                        remove_vcc=0;
                                        break;
                                }
                        }
                }
                if (remove_vcc)
                        lec_arp_clear_vccs(to_remove);
        }
        restore_flags(flags);
        DPRINTK("LEC_ARP: Removed entry:%2.2x %2.2x %2.2x %2.2x %2.2x %2.2x\n",
                0xff&to_remove->mac_addr[0], 0xff&to_remove->mac_addr[1],
                0xff&to_remove->mac_addr[2], 0xff&to_remove->mac_addr[3],
                0xff&to_remove->mac_addr[4], 0xff&to_remove->mac_addr[5]);

       return 0;
}

#if DEBUG_ARP_TABLE
static char*
get_status_string(unsigned char st)
{
        switch(st) {
        case ESI_UNKNOWN:
                return "ESI_UNKNOWN";
        case ESI_ARP_PENDING:
                return "ESI_ARP_PENDING";
        case ESI_VC_PENDING:
                return "ESI_VC_PENDING";
        case ESI_FLUSH_PENDING:
                return "ESI_FLUSH_PENDING";
        case ESI_FORWARD_DIRECT:
                return "ESI_FORWARD_DIRECT";
        default:
                return "<UNKNOWN>";
        }
}
#endif

void
dump_arp_table(struct lec_priv *priv)
{
#if DEBUG_ARP_TABLE
        int i,j, offset;
        struct lec_arp_table *rulla;
        char buf[1024];
        struct lec_arp_table **lec_arp_tables =
                (struct lec_arp_table **)priv->lec_arp_tables;
        struct lec_arp_table *lec_arp_empty_ones =
                (struct lec_arp_table *)priv->lec_arp_empty_ones;
        struct lec_arp_table *lec_no_forward =
                (struct lec_arp_table *)priv->lec_no_forward;

        printk("Dump %p:\n",priv);
        for (i=0;i<LEC_ARP_TABLE_SIZE;i++) {
                rulla = lec_arp_tables[i];
                offset = 0;
                offset += sprintf(buf,"%d: %p\n",i, rulla);
                while (rulla) {
                        offset += sprintf(buf+offset,"Mac:");
                        for(j=0;j<ETH_ALEN;j++) {
                                offset+=sprintf(buf+offset,
                                                "%2.2x ",
                                                rulla->mac_addr[j]&0xff);
                        }
                        offset +=sprintf(buf+offset,"Atm:");
                        for(j=0;j<ATM_ESA_LEN;j++) {
                                offset+=sprintf(buf+offset,
                                                "%2.2x ",
                                                rulla->atm_addr[j]&0xff);
                        }      
                        offset+=sprintf(buf+offset,
                                        "Vcc vpi:%d vci:%d, Recv_vcc vpi:%d vci:%d Last_used:%lx, Timestamp:%lx, No_tries:%d ",
                                        rulla->vcc?rulla->vcc->vpi:0, 
                                        rulla->vcc?rulla->vcc->vci:0,
                                        rulla->recv_vcc?rulla->recv_vcc->vpi:0,
                                        rulla->recv_vcc?rulla->recv_vcc->vci:0,
                                        rulla->last_used,
                                        rulla->timestamp, rulla->no_tries);
                        offset+=sprintf(buf+offset,
                                        "Flags:%x, Packets_flooded:%x, Status: %s ",
                                        rulla->flags, rulla->packets_flooded, 
                                        get_status_string(rulla->status));
                        offset+=sprintf(buf+offset,"->%p\n",rulla->next);
                        rulla = rulla->next;
                }
                printk("%s",buf);
        }
        rulla = lec_no_forward;
        if (rulla)
                printk("No forward\n");  
        while(rulla) {
                offset=0;
                offset += sprintf(buf+offset,"Mac:");
                for(j=0;j<ETH_ALEN;j++) {
                        offset+=sprintf(buf+offset,"%2.2x ",
                                        rulla->mac_addr[j]&0xff);
                }
                offset +=sprintf(buf+offset,"Atm:");
                for(j=0;j<ATM_ESA_LEN;j++) {
                        offset+=sprintf(buf+offset,"%2.2x ",
                                        rulla->atm_addr[j]&0xff);
                }      
                offset+=sprintf(buf+offset,
                                "Vcc vpi:%d vci:%d, Recv_vcc vpi:%d vci:%d Last_used:%lx, Timestamp:%lx, No_tries:%d ",
                                rulla->vcc?rulla->vcc->vpi:0, 
                                rulla->vcc?rulla->vcc->vci:0, 
                                rulla->recv_vcc?rulla->recv_vcc->vpi:0,
                                rulla->recv_vcc?rulla->recv_vcc->vci:0,
                                rulla->last_used, 
                                rulla->timestamp, rulla->no_tries);
                offset+=sprintf(buf+offset,
                                "Flags:%x, Packets_flooded:%x, Status: %s ",
                                rulla->flags, rulla->packets_flooded, 
                                get_status_string(rulla->status));
                offset+=sprintf(buf+offset,"->%lx\n",(long)rulla->next);
                rulla = rulla->next;
                printk("%s",buf);
        }
        rulla = lec_arp_empty_ones;
        if (rulla)
                printk("Empty ones\n");  
        while(rulla) {
                offset=0;
                offset += sprintf(buf+offset,"Mac:");
                for(j=0;j<ETH_ALEN;j++) {
                        offset+=sprintf(buf+offset,"%2.2x ",
                                        rulla->mac_addr[j]&0xff);
                }
                offset +=sprintf(buf+offset,"Atm:");
                for(j=0;j<ATM_ESA_LEN;j++) {
                        offset+=sprintf(buf+offset,"%2.2x ",
                                        rulla->atm_addr[j]&0xff);
                }      
                offset+=sprintf(buf+offset,
                                "Vcc vpi:%d vci:%d, Recv_vcc vpi:%d vci:%d Last_used:%lx, Timestamp:%lx, No_tries:%d ",
                                rulla->vcc?rulla->vcc->vpi:0, 
                                rulla->vcc?rulla->vcc->vci:0, 
                                rulla->recv_vcc?rulla->recv_vcc->vpi:0,
                                rulla->recv_vcc?rulla->recv_vcc->vci:0,
                                rulla->last_used, 
                                rulla->timestamp, rulla->no_tries);
                offset+=sprintf(buf+offset,
                                "Flags:%x, Packets_flooded:%x, Status: %s ",
                                rulla->flags, rulla->packets_flooded, 
                                get_status_string(rulla->status));
                offset+=sprintf(buf+offset,"->%lx\n",(long)rulla->next);
                rulla = rulla->next;
                printk("%s",buf);
        }

#endif
}

/*
 * Destruction of arp-cache
 */
void
lec_arp_destroy(struct lec_priv *priv)
{
        struct lec_arp_table *entry, *next;
        unsigned long flags;
        int i;

        save_flags(flags);
        cli();

        del_timer(&priv->lec_arp_timer);
        
        /*
         * Remove all entries
         */
        for (i=0;i<LEC_ARP_TABLE_SIZE;i++) {
                for(entry =priv->lec_arp_tables[i];entry != NULL; entry=next) {
                        next = entry->next;
                        lec_arp_remove(priv->lec_arp_tables, entry);
                        kfree(entry);
                }
        }
        entry = priv->lec_arp_empty_ones;
        while(entry) {
                next = entry->next;
                del_timer(&entry->timer);
                lec_arp_clear_vccs(entry);
                kfree(entry);
                entry = next;
        }
        priv->lec_arp_empty_ones = NULL;
        entry = priv->lec_no_forward;
        while(entry) {
                next = entry->next;
                del_timer(&entry->timer);
                lec_arp_clear_vccs(entry);
                kfree(entry);
                entry = next;
        }
        priv->lec_no_forward = NULL;
        priv->mcast_vcc = NULL;
        memset(priv->lec_arp_tables, 0, 
               sizeof(struct lec_arp_table*)*LEC_ARP_TABLE_SIZE);
        restore_flags(flags);
}


/* 
 * Find entry by mac_address
 */
static __inline__ struct lec_arp_table*
lec_arp_find(struct lec_priv *priv,
             unsigned char *mac_addr)
{
        unsigned short place;
        struct lec_arp_table *to_return;

        DPRINTK("LEC_ARP: lec_arp_find :%2.2x %2.2x %2.2x %2.2x %2.2x %2.2x\n",
                mac_addr[0]&0xff, mac_addr[1]&0xff, mac_addr[2]&0xff, 
                mac_addr[3]&0xff, mac_addr[4]&0xff, mac_addr[5]&0xff);
        lec_arp_lock(priv);
        place = HASH(mac_addr[ETH_ALEN-1]);
  
        to_return = priv->lec_arp_tables[place];
        while(to_return) {
                if (memcmp(mac_addr, to_return->mac_addr, ETH_ALEN) == 0) {
                        lec_arp_unlock(priv);
                        return to_return;
                }
                to_return = to_return->next;
        }
        lec_arp_unlock(priv);
        return NULL;
}

static struct lec_arp_table*
make_entry(struct lec_priv *priv, unsigned char *mac_addr)
{
        struct lec_arp_table *to_return;

        to_return=(struct lec_arp_table *)kmalloc(sizeof(struct lec_arp_table),
                                                  GFP_ATOMIC);
        if (!to_return) {
                printk("LEC: Arp entry kmalloc failed\n");
                return NULL;
        }
        memset(to_return,0,sizeof(struct lec_arp_table));
        memcpy(to_return->mac_addr, mac_addr, ETH_ALEN);
        init_timer(&to_return->timer);
        to_return->timer.function = lec_arp_expire_arp;
        to_return->timer.data = (unsigned long)to_return;
        to_return->last_used = jiffies;
        to_return->priv = priv;
        return to_return;
}

/*
 *
 * Arp sent timer expired
 *
 */
static void
lec_arp_expire_arp(unsigned long data)
{
        struct lec_arp_table *entry;

        entry = (struct lec_arp_table *)data;

        del_timer(&entry->timer);

        DPRINTK("lec_arp_expire_arp\n");
        if (entry->status == ESI_ARP_PENDING) {
                if (entry->no_tries <= entry->priv->max_retry_count) {
                        send_to_lecd(entry->priv, l_arp_xmt, 
                                     entry->mac_addr, NULL);
                        entry->no_tries++;
                }
                entry->timer.expires = jiffies + (1*HZ);
                add_timer(&entry->timer);
        }
}

/*
 *
 * Unknown/unused vcc expire, remove associated entry
 *
 */
static void
lec_arp_expire_vcc(unsigned long data)
{
        struct lec_arp_table *to_remove = (struct lec_arp_table*)data;
        struct lec_priv *priv = (struct lec_priv *)to_remove->priv;
        struct lec_arp_table *entry = NULL;

        del_timer(&to_remove->timer);

        DPRINTK("LEC_ARP %p %p: lec_arp_expire_vcc vpi:%d vci:%d\n",
                to_remove, priv, 
                to_remove->vcc?to_remove->recv_vcc->vpi:0,
                to_remove->vcc?to_remove->recv_vcc->vci:0);
        DPRINTK("eo:%p nf:%p\n",priv->lec_arp_empty_ones,priv->lec_no_forward);
        if (to_remove == priv->lec_arp_empty_ones)
                priv->lec_arp_empty_ones = to_remove->next;
        else {
                entry = priv->lec_arp_empty_ones;
                while (entry && entry->next != to_remove)
                        entry = entry->next;
                if (entry)
                        entry->next = to_remove->next;
        }
        if (!entry)
                if (to_remove == priv->lec_no_forward) {
                        priv->lec_no_forward = to_remove->next;
                } else {
                        entry = priv->lec_no_forward;
                        while (entry && entry->next != to_remove)
                                entry = entry->next;
                        if (entry)
                                entry->next = to_remove->next;
                }
        lec_arp_clear_vccs(to_remove);
        kfree(to_remove);
}

/*
 * Expire entries.
 * 1. Re-set timer
 * 2. For each entry, delete entries that have aged past the age limit.
 * 3. For each entry, depending on the status of the entry, perform
 *    the following maintenance.
 *    a. If status is ESI_VC_PENDING or ESI_ARP_PENDING then if the
 *       tick_count is above the max_unknown_frame_time, clear
 *       the tick_count to zero and clear the packets_flooded counter
 *       to zero. This supports the packet rate limit per address
 *       while flooding unknowns.
 *    b. If the status is ESI_FLUSH_PENDING and the tick_count is greater
 *       than or equal to the path_switching_delay, change the status
 *       to ESI_FORWARD_DIRECT. This causes the flush period to end
 *       regardless of the progress of the flush protocol.
 */
static void
lec_arp_check_expire(unsigned long data)
{
        struct lec_priv *priv = (struct lec_priv *)data;
        struct lec_arp_table **lec_arp_tables =
                (struct lec_arp_table **)priv->lec_arp_tables;
        struct lec_arp_table *entry, *next;
        unsigned long now;
        unsigned long time_to_check;
        int i;

        del_timer(&priv->lec_arp_timer);

        DPRINTK("lec_arp_check_expire %p,%d\n",priv,priv->lec_arp_lock_var);
        DPRINTK("expire: eo:%p nf:%p\n",priv->lec_arp_empty_ones,
                priv->lec_no_forward);
        if (!priv->lec_arp_lock_var) {
                lec_arp_lock(priv);
                now = jiffies;
                for(i=0;i<LEC_ARP_TABLE_SIZE;i++) {
                        for(entry = lec_arp_tables[i];entry != NULL;) {
                                if ((entry->flags) & LEC_REMOTE_FLAG && 
                                    priv->topology_change)
                                        time_to_check=priv->forward_delay_time;
                                else
                                        time_to_check = priv->aging_time;

                                DPRINTK("About to expire: %lx - %lx > %lx\n",
                                        now,entry->last_used, time_to_check);
                                if((now-entry->last_used > time_to_check) && 
                                   !(entry->flags & LEC_PERMANENT_FLAG)) {
                                        /* Remove entry */
                                        DPRINTK("LEC:Entry timed out\n");
                                        next = entry->next;      
                                        lec_arp_remove(lec_arp_tables, entry);
                                        kfree(entry);
                                        entry = next;
                                } else {
                                        /* Something else */
                                        if ((entry->status == ESI_VC_PENDING ||
                                             entry->status == ESI_ARP_PENDING) 
                                            &&
                                            now-entry->timestamp >= 
                                            priv->max_unknown_frame_time) {
                                                entry->timestamp = jiffies;
                                                entry->packets_flooded = 0;
                                        }
                                        if (entry->status == ESI_FLUSH_PENDING 
                                            &&
                                            now-entry->timestamp >= 
                                            priv->path_switching_delay) {
                                                entry->last_used = jiffies;
                                                entry->status = 
                                                        ESI_FORWARD_DIRECT;
                                        }
                                        entry = entry->next;
                                }
                        }
                }
                lec_arp_unlock(priv);
        }
        priv->lec_arp_timer.expires = jiffies + LEC_ARP_REFRESH_INTERVAL;
        add_timer(&priv->lec_arp_timer);
}
/*
 * Try to find vcc where mac_address is attached.
 * 
 */
struct atm_vcc*
lec_arp_resolve(struct lec_priv *priv, unsigned char *mac_to_find)
{
        struct lec_arp_table *entry;

        if (mac_to_find[0]&0x01) {
                return priv->mcast_vcc;
        }
        entry = lec_arp_find(priv, mac_to_find);
  
        if (entry) {
                if (entry->status == ESI_FORWARD_DIRECT) {
                        /* Connection Ok */
                        entry->last_used = jiffies;
                        return entry->vcc;
                }
                if (entry->status == ESI_UNKNOWN) {
                        del_timer(&entry->timer);
                        goto make_arp_entry;
                }
                /* Data direct VC not yet set up, check to see if the unknown
                   frame count is greater than the limit. If the limit has
                   not been reached, allow the caller to send packet to
                   BUS. */
                if (entry->status != ESI_FLUSH_PENDING &&
                    entry->packets_flooded<priv->maximum_unknown_frame_count) {
                        entry->packets_flooded++;
                        DPRINTK("LEC_ARP: Flooding..\n");
                        return priv->mcast_vcc;
                }
                return NULL;
        } else {
                /* No matching entry was found */
                entry = make_entry(priv, mac_to_find);
                DPRINTK("LEC_ARP: Making entry\n");
                if (!entry) {
                        return priv->mcast_vcc;
                }
                lec_arp_put(priv->lec_arp_tables, entry);
        make_arp_entry:
                /* We want arp-request(s) to be sent */
                entry->packets_flooded =1;
                entry->status = ESI_ARP_PENDING;
                entry->no_tries = 1;
                entry->last_used = entry->timestamp = jiffies;
                send_to_lecd(priv, l_arp_xmt, mac_to_find, NULL);
                entry->timer.expires = jiffies + (1*HZ);
                entry->timer.function = lec_arp_expire_arp;
                add_timer(&entry->timer);
                return priv->mcast_vcc;
        }
}

int
lec_addr_delete(struct lec_priv *priv, unsigned char *atm_addr, 
                unsigned long permanent)
{
        struct lec_arp_table *entry, *next;
        int i;

        lec_arp_lock(priv);
        DPRINTK("lec_addr_delete\n");
        for(i=0;i<LEC_ARP_TABLE_SIZE;i++) {
                for(entry=priv->lec_arp_tables[i];entry != NULL; entry=next) {
                        next = entry->next;
                        if (!memcmp(atm_addr, entry->atm_addr, ATM_ESA_LEN)
                            && (permanent || 
                                !(entry->flags & LEC_PERMANENT_FLAG))) {
                                lec_arp_remove(priv->lec_arp_tables, entry);
                                kfree(entry);
                        }
                        lec_arp_unlock(priv);
                        return 0;
                }
        }
        lec_arp_unlock(priv);
        return -1;
}

/*
 * Notifies:  Response to arp_request (atm_addr != NULL) 
 */
void
lec_arp_update(struct lec_priv *priv, unsigned char *mac_addr,
               unsigned char *atm_addr, unsigned long remoteflag)
{
        struct lec_arp_table *entry, *tmp;
        int i;

        DPRINTK("LEC:lec_arp_update mac:%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x\n",
                mac_addr[0],mac_addr[1],mac_addr[2],mac_addr[3],
                mac_addr[4],mac_addr[5]);
        lec_arp_lock(priv);
        if (priv->lec_arp_empty_ones) {
                entry = priv->lec_arp_empty_ones;
                if (!memcmp(entry->atm_addr, atm_addr, ATM_ESA_LEN)) {
                        priv->lec_arp_empty_ones = entry->next;
                } else {
                        while(entry->next && memcmp(entry->next->atm_addr, 
                                                    atm_addr, ATM_ESA_LEN))
                                entry = entry->next;
                        if (entry->next) {
                                tmp = entry;
                                entry = entry->next;
                                tmp->next = entry->next;
                        } else
                                entry = NULL;
                        
                }
                if (entry) {
                        del_timer(&entry->timer);
                        tmp = lec_arp_find(priv, mac_addr);
                        if (tmp) {
                                del_timer(&tmp->timer);
                                tmp->status = ESI_FORWARD_DIRECT;
                                memcpy(tmp->atm_addr, atm_addr, ATM_ESA_LEN);
                                tmp->vcc = entry->vcc;
                                tmp->old_push = entry->old_push;
                                tmp->last_used = jiffies;
                                kfree(entry);
                                entry=tmp;
                        } else {
                                entry->status = ESI_FORWARD_DIRECT;
                                memcpy(entry->mac_addr, mac_addr, ETH_ALEN);
                                entry->last_used = jiffies;
                                lec_arp_put(priv->lec_arp_tables, entry);
                        }
                        if (remoteflag)
                                entry->flags|=LEC_REMOTE_FLAG;
                        else
                                entry->flags&=~LEC_REMOTE_FLAG;
                        lec_arp_unlock(priv);
                        DPRINTK("After update\n");
                        dump_arp_table(priv);
                        return;
                }
        }
        entry = lec_arp_find(priv, mac_addr);
        if (!entry) {
                entry = make_entry(priv, mac_addr);
                entry->status = ESI_UNKNOWN;
                lec_arp_put(priv->lec_arp_tables, entry);
                /* Temporary, changes before end of function */
        }
        memcpy(entry->atm_addr, atm_addr, ATM_ESA_LEN);
        del_timer(&entry->timer);
        for(i=0;i<LEC_ARP_TABLE_SIZE;i++) {
                for(tmp=priv->lec_arp_tables[i];tmp;tmp=tmp->next) {
                        if (entry != tmp &&
                            !memcmp(tmp->atm_addr, atm_addr,
                                    ATM_ESA_LEN)) { 
                                /* Vcc to this host exists */
                                if (tmp->status > ESI_VC_PENDING) {
                                        /*
                                         * ESI_FLUSH_PENDING,
                                         * ESI_FORWARD_DIRECT
                                         */
                                        entry->vcc = tmp->vcc;
                                        entry->old_push=tmp->old_push;
                                }
                                entry->status=tmp->status;
                                break;
                        }
                }
        }
        if (remoteflag)
                entry->flags|=LEC_REMOTE_FLAG;
        else
                entry->flags&=~LEC_REMOTE_FLAG;
        if (entry->status == ESI_ARP_PENDING ||
            entry->status == ESI_UNKNOWN) {
                entry->status = ESI_VC_PENDING;
                send_to_lecd(priv, l_svc_setup, NULL, atm_addr);
        }
        DPRINTK("After update2\n");
        dump_arp_table(priv);
        lec_arp_unlock(priv);
}

/*
 * Notifies: Vcc setup ready 
 */
void
lec_vcc_added(struct lec_priv *priv, struct atmlec_ioc *ioc_data,
              struct atm_vcc *vcc,
              void (*old_push)(struct atm_vcc *vcc, struct sk_buff *skb))
{
        struct lec_arp_table *entry;
        int i, found_entry=0;
        unsigned char bus_mac[ETH_ALEN] = { 0xff, 0xff, 0xff, 0xff, 0xff,0xff};

        lec_arp_lock(priv);
        if (ioc_data->receive == 2) {
                /* Vcc for BUS distribute */
                DPRINTK("LEC_ARP: Attaching mcast distribute\n");
                entry = lec_arp_find(priv, bus_mac);
                if (!entry) {
                        printk("LEC_ARP: Multicast entry not found!\n");
                        lec_arp_unlock(priv);
                        return;
                }
                memcpy(entry->atm_addr, ioc_data->atm_addr, ATM_ESA_LEN);
                entry->recv_vcc = vcc;
                entry->old_recv_push = old_push;
                lec_arp_unlock(priv);
                return;
        } else if (ioc_data->receive == 1) {
                /* Vcc which we don't want to make default vcc, attach it
                   anyway. */
                DPRINTK("LEC_ARP:Attaching data direct, not default :%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x\n",
                        ioc_data->atm_addr[0],ioc_data->atm_addr[1],
                        ioc_data->atm_addr[2],ioc_data->atm_addr[3],
                        ioc_data->atm_addr[4],ioc_data->atm_addr[5],
                        ioc_data->atm_addr[6],ioc_data->atm_addr[7],
                        ioc_data->atm_addr[8],ioc_data->atm_addr[9],
                        ioc_data->atm_addr[10],ioc_data->atm_addr[11],
                        ioc_data->atm_addr[12],ioc_data->atm_addr[13],
                        ioc_data->atm_addr[14],ioc_data->atm_addr[15],
                        ioc_data->atm_addr[16],ioc_data->atm_addr[17],
                        ioc_data->atm_addr[18],ioc_data->atm_addr[19]);
                entry = make_entry(priv, bus_mac);
                memcpy(entry->atm_addr, ioc_data->atm_addr, ATM_ESA_LEN);
                memset(entry->mac_addr, 0, ETH_ALEN);
                entry->recv_vcc = vcc;
                entry->old_recv_push = old_push;
                entry->status = ESI_UNKNOWN;
                entry->timer.expires = jiffies + priv->vcc_timeout_period;
                entry->timer.function = lec_arp_expire_vcc;
                add_timer(&entry->timer);
                entry->next = priv->lec_no_forward;
                priv->lec_no_forward = entry;
                lec_arp_unlock(priv);
		dump_arp_table(priv);
                return;
        }
        DPRINTK("LEC_ARP:Attaching data direct, default:%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x%2.2x\n",
                        ioc_data->atm_addr[0],ioc_data->atm_addr[1],
                        ioc_data->atm_addr[2],ioc_data->atm_addr[3],
                        ioc_data->atm_addr[4],ioc_data->atm_addr[5],
                        ioc_data->atm_addr[6],ioc_data->atm_addr[7],
                        ioc_data->atm_addr[8],ioc_data->atm_addr[9],
                        ioc_data->atm_addr[10],ioc_data->atm_addr[11],
                        ioc_data->atm_addr[12],ioc_data->atm_addr[13],
                        ioc_data->atm_addr[14],ioc_data->atm_addr[15],
                        ioc_data->atm_addr[16],ioc_data->atm_addr[17],
                ioc_data->atm_addr[18],ioc_data->atm_addr[19]);
        for (i=0;i<LEC_ARP_TABLE_SIZE;i++) {
                for (entry = priv->lec_arp_tables[i];entry;entry=entry->next) {
                        if (memcmp(ioc_data->atm_addr, entry->atm_addr, 
                                   ATM_ESA_LEN)==0) {
                                DPRINTK("LEC_ARP: Attaching data direct\n");
                                DPRINTK("Currently -> Vcc: %d, Rvcc:%d\n",
                                        entry->vcc?entry->vcc->vci:0,
                                        entry->recv_vcc?entry->recv_vcc->vci:0);
                                found_entry=1;
                                del_timer(&entry->timer);
                                if (vcc) {
                                        lec_arp_clear_vccs(entry);
                                }
                                entry->vcc = vcc;
                                entry->old_push = old_push;
                                if (entry->status == ESI_VC_PENDING) {
                                        if(priv->maximum_unknown_frame_count
                                           ==0)
                                                entry->status = 
                                                        ESI_FORWARD_DIRECT;
                                        else {
                                                entry->timestamp = jiffies;
                                                entry->status = 
                                                        ESI_FLUSH_PENDING;
#if 0
                                                send_to_lecd(priv,l_flush_xmt,
                                                             NULL,
                                                             entry->atm_addr);
#endif
                                        }
                                } else {
                                        /* They were forming a connection
                                           to us, and we to them. Our
                                           ATM address is numerically lower
                                           than theirs, so we make connection
                                           we formed into default VCC (8.1.11).
                                           Connection they made gets torn
                                           down. This might confuse some
                                           clients. Can be changed if
                                           someone reports trouble... */
                                        ;
                                }
                        }
                }
        }
        if (found_entry) {
                lec_arp_unlock(priv);
                DPRINTK("After vcc was added\n");
                dump_arp_table(priv);
                return;
        }
        /* Not found, snatch address from first data packet that arrives from
           this vcc */
        entry = make_entry(priv, bus_mac);
        entry->vcc = vcc;
        entry->old_push = old_push;
        memcpy(entry->atm_addr, ioc_data->atm_addr, ATM_ESA_LEN);
        memset(entry->mac_addr, 0, ETH_ALEN);
        entry->status = ESI_UNKNOWN;
        entry->next = priv->lec_arp_empty_ones;
        priv->lec_arp_empty_ones = entry;
        entry->timer.expires = jiffies + priv->vcc_timeout_period;
        entry->timer.function = lec_arp_expire_vcc;
        add_timer(&entry->timer);
        lec_arp_unlock(priv);
        DPRINTK("After vcc was added\n");
	dump_arp_table(priv);
}

void
lec_flush_complete(struct lec_priv *priv, unsigned long tran_id)
{
        struct lec_arp_table *entry;
        int i;
  
        DPRINTK("LEC:lec_flush_complete %lx\n",tran_id);
        for (i=0;i<LEC_ARP_TABLE_SIZE;i++) {
                for (entry=priv->lec_arp_tables[i];entry;entry=entry->next) {
                        if (entry->flush_tran_id == tran_id &&
                            entry->status == ESI_FLUSH_PENDING) {
                                entry->status = ESI_FORWARD_DIRECT;
                                DPRINTK("LEC_ARP: Flushed\n");
                        }
                }
        }
        dump_arp_table(priv);
}

void
lec_set_flush_tran_id(struct lec_priv *priv,
                      unsigned char *atm_addr, unsigned long tran_id)
{
        struct lec_arp_table *entry;
        int i;

        for (i=0;i<LEC_ARP_TABLE_SIZE;i++)
                for(entry=priv->lec_arp_tables[i];entry;entry=entry->next)
                        if (!memcmp(atm_addr, entry->atm_addr, ATM_ESA_LEN)) {
                                entry->flush_tran_id = tran_id;
                                DPRINTK("Set flush transaction id to %lx for %p\n",tran_id,entry);
                        }
}

int 
lec_mcast_make(struct lec_priv *priv, struct atm_vcc *vcc)
{
        unsigned char mac_addr[] = {
                0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
        struct lec_arp_table *to_add;
  
        lec_arp_lock(priv);
        to_add = make_entry(priv, mac_addr);
        if (!to_add)
                return -ENOMEM;
        to_add->status = ESI_FORWARD_DIRECT;
        to_add->flags |= LEC_PERMANENT_FLAG;
        to_add->vcc = vcc;
        to_add->old_push = vcc->push;
        vcc->push = lec_push;
        priv->mcast_vcc = vcc;
        lec_arp_put(priv->lec_arp_tables, to_add);
        lec_arp_unlock(priv);
        return 0;
}

void
lec_vcc_close(struct lec_priv *priv, struct atm_vcc *vcc)
{
        struct lec_arp_table *entry, *next;
        int i;

        DPRINTK("LEC_ARP: lec_vcc_close vpi:%d vci:%d\n",vcc->vpi,vcc->vci);
        dump_arp_table(priv);
        lec_arp_lock(priv);
        for(i=0;i<LEC_ARP_TABLE_SIZE;i++) {
                for(entry = priv->lec_arp_tables[i];entry; entry=next) {
                        next = entry->next;
                        if (vcc == entry->vcc) {
                                lec_arp_remove(priv->lec_arp_tables,entry);
                                kfree(entry);
                                if (priv->mcast_vcc == vcc) {
                                        priv->mcast_vcc = NULL;
                                }
                        } else if (vcc == entry->recv_vcc) { 
                                /* Bus distribution closed */
                                priv->mcast_vcc = NULL;
                                lec_arp_remove(priv->lec_arp_tables,entry);
				kfree(entry);
                                lec_arp_unlock(priv);
                                vcc->push(vcc, NULL);
                                return;
                        }
                }
        }
        entry=priv->lec_arp_empty_ones;
        while(entry && entry->vcc==vcc) {
                lec_arp_clear_vccs(entry);
                priv->lec_arp_empty_ones=entry->next;
                del_timer(&entry->timer);
                kfree(entry);
                entry=priv->lec_arp_empty_ones;
        }
        for(;entry!=NULL;entry=next) {
                next=entry->next;
                if (vcc == next->vcc) {
                        lec_arp_clear_vccs(next);
                        entry->next = next->next;
                        del_timer(&next->timer);
                        kfree(next);
                }
        }
        entry=priv->lec_no_forward;
        while(entry && entry->recv_vcc==vcc) {
                lec_arp_clear_vccs(entry);
                priv->lec_no_forward=entry->next;
                del_timer(&entry->timer);
                kfree(entry);
                entry=priv->lec_no_forward;
        }
        for(;entry!=NULL;entry=next) {
                next=entry->next;
                if (vcc == next->recv_vcc) {
                        lec_arp_clear_vccs(next);
                        entry->next = next->next;
                        del_timer(&next->timer);
                        kfree(next);
                }
        }	
        lec_arp_unlock(priv);
	dump_arp_table(priv);
}

void
lec_arp_check_empties(struct lec_priv *priv,
                      struct atm_vcc *vcc, struct sk_buff *skb)
{
        struct lec_arp_table *entry, *prev;
        struct lecdatahdr_8023 *hdr = (struct lecdatahdr_8023 *)skb->data;
        unsigned long flags;
  
        lec_arp_lock(priv);
        entry = priv->lec_arp_empty_ones;
        if (vcc == entry->vcc) {
                save_flags(flags);
                cli();
                del_timer(&entry->timer);
                memcpy(entry->mac_addr, hdr->h_source, ETH_ALEN);
                entry->status = ESI_FORWARD_DIRECT;
                entry->last_used = jiffies;
                priv->lec_arp_empty_ones = entry->next;
                restore_flags(flags);
                /* We might have got an entry */
                if ((prev=lec_arp_find(priv,hdr->h_source))) {
                        lec_arp_remove(priv->lec_arp_tables, prev);
                        kfree(prev);
                }
                lec_arp_put(priv->lec_arp_tables, entry);
                lec_arp_unlock(priv);
                return;
        }
        prev = entry;
        entry = entry->next;
        while (entry && entry->vcc != vcc) {
                prev= entry;
                entry = entry->next;
        }
        if (!entry) {
                DPRINTK("LEC_ARP: Arp_check_empties: entry not found!\n");
                lec_arp_unlock(priv);
                return;
        }
        save_flags(flags);
        cli();
        del_timer(&entry->timer);
        memcpy(entry->mac_addr, hdr->h_source, ETH_ALEN);
        entry->status = ESI_FORWARD_DIRECT;
        entry->last_used = jiffies;
        prev->next = entry->next;
        restore_flags(flags);
        if ((prev = lec_arp_find(priv, hdr->h_source))) {
                lec_arp_remove(priv->lec_arp_tables,prev);
                kfree(prev);
        }
        lec_arp_put(priv->lec_arp_tables,entry);
        lec_arp_unlock(priv);  
}

