/*
 * lec.c: Lan Emulation driver 
 * Marko Kiiskila carnil@cs.tut.fi
 *
 */

#include <linux/kernel.h>

/* We are ethernet device */
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <net/sock.h>
#include <linux/skbuff.h>
#include <asm/byteorder.h>
#include <net/arp.h>
#include <linux/proc_fs.h>

/* And atm device */
#include <linux/atmdev.h>
#include <linux/atmlec.h>

#include "lec.h"
#include "lec_arpc.h"
#include "tunable.h"


#define DPRINTK(format,args...)
/*
#define DPRINTK printk
*/
#define DUMP_PACKETS 0 /* 0 = None,
                        * 1 = 30 first bytes
                        * 2 = Whole packet
                        */

static int lec_open(struct device *dev);
static int lec_send_packet(struct sk_buff *skb, struct device *dev);
static int lec_close(struct device *dev);
static struct enet_statistics *lec_get_stats(struct device *dev);
static int lec_init(struct device *dev);

/* will be lec0, lec1, lec2 etc. */
static char myname[] = "lecx";

/* Device structures */
struct device *dev_lec[MAX_LEC_ITF] =
{NULL, NULL, NULL, NULL};

/*
 * Open/initialize the netdevice. This is called (in the current kernel)
 * sometime after booting when the 'ifconfig' program is run.
 *
 * This routine should set everything up anew at each open, even
 * registers that "should" only need to be set once at boot, so that
 * there is non-reboot way to recover if something goes wrong.
 */

static int 
lec_open(struct device *dev)
{
        struct lec_priv *priv = (struct lec_priv *)dev->priv;
        
        dev->tbusy = 0;
        dev->start = 1;
        dev->interrupt = 1;
        memset(&priv->stats,0,sizeof(struct enet_statistics));
        
        return 0;
}

static int 
lec_send_packet(struct sk_buff *skb, struct device *dev)
{
        struct lec_priv *priv = (struct lec_priv *)dev->priv;
        struct lecdatahdr_8023 *lec_h;
        struct atm_vcc *send_vcc;
        unsigned char *nb;
#if DUMP_PACKETS > 0
        char buf[300];
        int i=0;
#endif /* DUMP_PACKETS >0 */
        
        DPRINTK("Lec_send_packet called\n");  
        if (!priv->lecd) {
                printk("%s:No lecd attached\n",dev->name);
                priv->stats.tx_errors++;
                return -EUNATCH;
        } 
        if (dev->tbusy) {
                /*
                 * If we get here, some higher level has decided we are broken.
                 * There should really be a "kick me" function call instead.
                 */
                printk(KERN_WARNING "%s: transmit timed out.\n", dev->name);
                dev->tbusy=0;
        }
        /*
         * If some higher layer thinks we've missed an tx-done interrupt
         * we are passed NULL. Caution: dev_tint() handles the cli()/sti()
         * itself.
         */
        if (skb == NULL) {
                dev_tint(dev);
                return 0;
        }
        /*
         * Block a timer-based transmit from overlapping. This could better be
         * done with atomic_swap(1, dev->tbusy), but set_bit() works as well.
         */

        if (set_bit(0, (void*)&dev->tbusy) != 0) {
                printk(KERN_WARNING "%s: Transmitter access conflict.\n", 
                       dev->name);
        } else {
                DPRINTK("skbuff head:%lx data:%lx tail:%lx end:%lx\n",
                        (long)skb->head, (long)skb->data, (long)skb->tail,
                        (long)skb->end);
                
                /* Put le header to place */
                lec_h = (struct lecdatahdr_8023*)skb->data;
                /*
                lec_h->le_header = htons(priv->lecid); 
                */
#if DUMP_PACKETS > 0
                printk("%s: send datalen:%ld lecid:%4.4x\n", dev->name,
                        skb->len, priv->lecid);
#if DUMP_PACKETS >= 2
                for(i=0;i<skb->len && i <99;i++) {
                        sprintf(buf+i*3,"%2.2x ",0xff&skb->data[i]);
                }
#elif DUMP_PACKETS >= 1
                for(i=0;i<skb->len && i < 30;i++) {
                        sprintf(buf+i*3,"%2.2x ", 0xff&skb->data[i]);
                }
#endif /* DUMP_PACKETS >= 1 */
                if (i==skb->len)
                        printk("%s\n",buf);
                else
                        printk("%s...\n",buf);
#endif /* DUMP_PACKETS > 0 */
                /* Send to right vcc */
                send_vcc = lec_arp_resolve(priv, lec_h->h_dest);
                DPRINTK("%s:send_vcc:%p vcc_flags:%x\n", dev->name,
                        send_vcc, send_vcc?send_vcc->flags:0);
                if (!send_vcc || !(send_vcc->flags & ATM_VF_READY)) {    
                        priv->stats.tx_errors++;
                        DPRINTK("%s:lec_send_packet: handing back...\n",
                                dev->name);
                        dev->tbusy=1;
                        return 1;
                } else {
#if DUMP_PACKETS > 0                    
                        printk("%s:sending to vpi:%d vci:%d\n", dev->name,
                                send_vcc->vpi, send_vcc->vci);
#endif /* DUMP_PACKETS > 0 */
                        /* Minimum ethernet-frame size */
                        if (skb->len <62) {
                                if (skb->truesize < 62) {
                                        printk("%s:data packet %ld / %d\n",
                                               dev->name,
                                               skb->len,skb->truesize);
                                        nb=(unsigned char*)kmalloc(64, 
                                                                   GFP_ATOMIC);
                                        memcpy(nb,skb->data,skb->len);
                                        kfree(skb->head);
                                        skb->head = skb->data = nb;
                                        skb->tail = nb+62;
                                        skb->end = nb+64;
                                        skb->len=62;
                                        skb->truesize = 64;
                                } else {
                                        skb->len = 62;
                                }
                        }
                        atomic_add(skb->truesize, &send_vcc->tx_inuse);
                        skb->atm.iovcnt = 0;
                        send_vcc->dev->ops->send(send_vcc, skb);
                        priv->stats.tx_packets++;
                }
        }
        /* Should we wait for card's device driver to notify us? */
        dev->tbusy=0;
        
        return 0;
}

/* The inverse routine to net_open(). */
static int 
lec_close(struct device *dev) 
{
        dev->tbusy = 1;
        dev->start = 0;
        return 0;
}

/*
 * Get the current statistics.
 * This may be called with the card open or closed.
 */
static struct enet_statistics *
lec_get_stats(struct device *dev)
{
        struct lec_priv *priv = (struct lec_priv *)dev->priv;
        
        return (struct enet_statistics *)&priv->stats;
}

int 
lec_hard_header(struct sk_buff *skb, struct device *dev, 
               unsigned short type, void *daddr, void *saddr, 
               unsigned len)
{
        struct lec_priv *priv = (struct lec_priv *)dev->priv;
        struct lecdatahdr_8023 *hdr = 
                (struct lecdatahdr_8023 *)skb_push(skb, LEC_HEADER_LEN); 
                
        /* Set lecid header */
        if (priv)
                hdr->le_header = htons(priv->lecid);
        else
                hdr->le_header = 0;
        
        /* Set the protocol type. */
        if(type!=ETH_P_802_3)
                hdr->h_type = htons(type);
        else
                hdr->h_type = htons(len);
        
        /* Source hw address */
        if (saddr)
                memcpy(hdr->h_source, saddr, dev->addr_len);
        else
                memcpy(hdr->h_source, dev->dev_addr, dev->addr_len);
        
        /* Destination addr */
        if (daddr) {
                memcpy(hdr->h_dest, daddr, dev->addr_len);
                return dev->hard_header_len;
        }
        return -dev->hard_header_len;
}

int 
lec_rebuild_header(void *buff, struct device *dev, unsigned long dst,
                  struct sk_buff *skb)
{
        struct lecdatahdr_8023 *hdr = (struct lecdatahdr_8023*)buff;
        
        /*
         *      Only ARP/IP is currently supported
         */
        if (hdr->h_type !=  htons(ETH_P_IP)) {
                printk(KERN_DEBUG "%s: unable to resolve type %X addresses.\n",
                       dev->name,(int)hdr->h_type);
                memcpy(hdr->h_source, dev->dev_addr, dev->addr_len);
                return 0;
        }
        /*
         *      Try and get ARP to resolve the header.
         */
#ifdef CONFIG_INET       
        return arp_find(hdr->h_dest, dst, dev, dev->pa_addr, skb)? 1 : 0;
#else
        return 0;
#endif
}

/*
 * Upper level calls this function to bind hardware header cache entry.
 * If the call is successful, then corresponding Address Resolution Protocol
 * (maybe, not ARP) takes responsibility for updating cache content.
 */
void 
lec_header_cache_bind(struct hh_cache ** hhp, struct device *dev,
                     unsigned short htype, __u32 daddr)
{
        struct hh_cache *hh;
        
        if (htype != ETH_P_IP) {
                printk("eth_header_cache_bind: %04x cache is not implemented\n",
                       htype);
                return;
        }
        if (arp_bind_cache(hhp, dev, htype, daddr))
                return;
        if ((hh=*hhp) != NULL) {
                memcpy(hh->hh_data+8, dev->dev_addr, ETH_ALEN);
                hh->hh_data[14] = htype>>8;
                hh->hh_data[15] = htype&0xFF;
        }
}

/*
 * Called by Address Resolution module to notify changes in address.
 */
void 
lec_header_cache_update(struct hh_cache *hh, struct device *dev, 
                       unsigned char *haddr)
{
        if (hh->hh_type != ETH_P_IP) {
                printk("lec_header_cache_update: %04x cache is not implemented\n", 
                       hh->hh_type);
                return;
        }
        memcpy(hh->hh_data+2, haddr, ETH_ALEN);
        hh->hh_uptodate = 1;
}

static int 
lec_atm_send(struct atm_vcc *vcc, struct sk_buff *skb)
{
        struct device *dev = (struct device*)vcc->proto_data;
        struct lec_priv *priv = (struct lec_priv*)dev->priv;
        struct atmlec_msg *mesg;
        int i;

	atomic_sub(skb->truesize+ATM_PDU_OVHD, &vcc->tx_inuse);
        mesg = (struct atmlec_msg *)skb->data;
        DPRINTK("%s: msg from zeppelin:%d\n", dev->name, mesg->type);
        switch(mesg->type) {
        case l_set_mac_addr:
                for (i=0;i<6;i++) {
                        dev->dev_addr[i] = mesg->content.normal.mac_addr[i];
                }    
                break;
        case l_del_mac_addr:
                for(i=0;i<6;i++) {
                        dev->dev_addr[i] = 0;
                }
                break;
        case l_addr_delete:
                lec_addr_delete(priv, mesg->content.normal.atm_addr, 
                                mesg->content.normal.flag);
                break;
        case l_topology_change:
                priv->topology_change = mesg->content.normal.flag;  
                break;
        case l_flush_complete:
                lec_flush_complete(priv, mesg->content.normal.flag);
                break;
        case l_arp_update:
                lec_arp_update(priv, mesg->content.normal.mac_addr,
                               mesg->content.normal.atm_addr,
                               mesg->content.normal.flag);
                break;
        case l_config:
                priv->maximum_unknown_frame_count = 
                        mesg->content.config.maximum_unknown_frame_count;
                priv->max_unknown_frame_time = 
                        (mesg->content.config.max_unknown_frame_time*HZ);
                priv->max_retry_count = 
                        mesg->content.config.max_retry_count;
                priv->aging_time = (mesg->content.config.aging_time*HZ);
                priv->forward_delay_time = 
                        (mesg->content.config.forward_delay_time*HZ);
                priv->arp_response_time = 
                        (mesg->content.config.arp_response_time*HZ);
                priv->flush_timeout = (mesg->content.config.flush_timeout*HZ);
                priv->path_switching_delay = 
                        (mesg->content.config.path_switching_delay*HZ);
                break;
        case l_flush_tran_id:
                lec_set_flush_tran_id(priv, mesg->content.normal.atm_addr,
                                      mesg->content.normal.flag);
                break;
        case l_set_lecid:
                priv->lecid=(unsigned short)(0xffff&mesg->content.normal.flag);
                break;
        default:
                printk("%s: Unknown message type %d\n", dev->name, mesg->type);
                dev_kfree_skb(skb, FREE_WRITE);
                return -EINVAL;
        }
        dev_kfree_skb(skb, FREE_WRITE);
        return 0;
}

static void 
lec_atm_close(struct atm_vcc *vcc)
{
        struct sk_buff *skb;
        struct device *dev = (struct device *)vcc->proto_data;
        struct lec_priv *priv = (struct lec_priv *)dev->priv;

        priv->lecd = NULL;
        /* Do something needful? */

        dev->tbusy = 1;
        dev->start = 0;

        lec_arp_destroy(priv);

        if (skb_peek(&vcc->recvq))
		printk("%s lec_atm_close: closing with messages pending\n",
                       dev->name);
        while ((skb = skb_dequeue(&vcc->recvq))) 
		dev_kfree_skb(skb,FREE_READ);
  
	printk("%s: Shut down!\n", dev->name);
}

static struct atmdev_ops lecdev_ops = {
        NULL, /*open*/
        lec_atm_close, /*close*/
        NULL, /*ioctl*/
        NULL, /*getsockopt */
        NULL, /*setsockopt */
        lec_atm_send, /*send */
        NULL, /*sg_send */
        NULL, /*poll */
        NULL, /*phy_put*/
        NULL, /*phy_get*/
        NULL  /*feedback*/
};

static struct atm_dev lecatm_dev = {
        &lecdev_ops,
        NULL,	    /*PHY*/
        "lec",	    /*type*/
        999,	    /*dummy device number*/
        NULL,NULL,  /*no VCCs*/
        NULL,NULL,  /*no data*/
        0,	    /*no flags*/
        NULL,	    /* no local address*/
        { 0 }	    /*no ESI, no statistics */
};

int 
send_to_lecd(struct lec_priv *priv, atmlec_msg_type type, 
             unsigned char *mac_addr, unsigned char *atm_addr)
{
	struct sk_buff *skb;
	struct atmlec_msg *mesg;

	if (!priv || !priv->lecd) {
		return -1;
	}
	skb = alloc_skb(sizeof(struct atmlec_msg), GFP_ATOMIC);
	if (!skb)
		return -1;
	skb->free = 1;
	skb->len = sizeof(struct atmlec_msg);
	mesg = (struct atmlec_msg *)skb->data;
	mesg->type = type;
	if (mac_addr)
		memcpy(&mesg->content.normal.mac_addr, mac_addr, ETH_ALEN);
	if (atm_addr)
		memcpy(&mesg->content.normal.atm_addr, atm_addr, ATM_ESA_LEN);

	atomic_add(skb->truesize+ATM_PDU_OVHD, &priv->lecd->rx_inuse);
	skb_queue_tail(&priv->lecd->recvq, skb);
        wake_up(&priv->lecd->sleep);
        return 0;
}

static int 
lec_init(struct device *dev)
{
        dev->priv = kmalloc(sizeof(struct lec_priv), GFP_KERNEL);
        if (!dev->priv)
                return -ENOMEM;

        memset(dev->priv,0,sizeof(struct lec_priv));

        ether_setup(dev);
        dev->open = lec_open;
        dev->stop = lec_close;
        dev->hard_start_xmit = lec_send_packet;
        dev->hard_header_len = LEC_HEADER_LEN;

        dev->hard_header = lec_hard_header;
        dev->rebuild_header = lec_rebuild_header;
        dev->header_cache_bind  = lec_header_cache_bind;
        dev->header_cache_update= lec_header_cache_update;

        dev->get_stats = lec_get_stats;
        dev->set_multicast_list = NULL;
        dev->do_ioctl  = NULL;
        printk("%s: Initialized!\n",dev->name);
        return 0;
}

static unsigned char lec_ctrl_magic[] = {
        0xff,
        0x00,
        0x01,
        0x01 };

void 
lec_push(struct atm_vcc *vcc, struct sk_buff *skb)
{
        struct device *dev = (struct device *)vcc->proto_data;
        struct lec_priv *priv = (struct lec_priv *)dev->priv; 
        struct lecdatahdr_8023 *hdr;
        unsigned char *rawp;
#if DUMP_PACKETS >0
        int i=0;
        char buf[300];

        printk("%s: lec_push vcc vpi:%d vci:%d\n", dev->name,
               vcc->vpi, vcc->vci);
#endif
        if (!skb) {
                DPRINTK("%s: null skb\n",dev->name);
                lec_vcc_close(priv, vcc);
                return;
        }
#if DUMP_PACKETS > 0
        printk("%s: rcv datalen:%ld lecid:%4.4x\n", dev->name,
               skb->len, priv->lecid);
#if DUMP_PACKETS >= 2
        for(i=0;i<skb->len && i <99;i++) {
                sprintf(buf+i*3,"%2.2x ",0xff&skb->data[i]);
        }
#elif DUMP_PACKETS >= 1
        for(i=0;i<skb->len && i < 30;i++) {
                sprintf(buf+i*3,"%2.2x ", 0xff&skb->data[i]);
        }
#endif /* DUMP_PACKETS >= 1 */
        if (i==skb->len)
                printk("%s\n",buf);
        else
                printk("%s...\n",buf);
#endif /* DUMP_PACKETS > 0 */
        if (memcmp(skb->data, lec_ctrl_magic, 4) ==0) { /* Control frame, to daemon*/
                DPRINTK("%s: To daemon\n",dev->name);
                atomic_add(skb->truesize+ATM_PDU_OVHD, &vcc->rx_inuse);
                skb_queue_tail(&vcc->recvq, skb);
                wake_up(&vcc->sleep);
        } else { /* Data frame, queue to protocol handlers */
                hdr = (struct lecdatahdr_8023 *)skb->data;
                if (hdr->le_header == htons(priv->lecid) ||
                    !priv->lecd) { 
                        /* Probably looping back, or if lecd is missing,
                           lecd has gone down */
                        DPRINTK("Ignoring loopback frame...\n");
                        dev_kfree_skb(skb, FREE_READ);
                        return;
                }
                if (priv->lec_arp_empty_ones) { /* FILTER DATA!!!! */
                        lec_arp_check_empties(priv, vcc, skb);
                }
                skb->dev = dev;
                skb->mac.raw = (unsigned char *)skb->data;
                if (*hdr->h_dest&1) {
                        if (memcmp(hdr->h_dest,dev->broadcast, ETH_ALEN)==0)
                                skb->pkt_type=PACKET_BROADCAST;
                        else
                                skb->pkt_type=PACKET_MULTICAST;
                } else if (dev->flags&(IFF_PROMISC|IFF_ALLMULTI)) {
                        if (memcmp(hdr->h_dest,dev->dev_addr, ETH_ALEN))
                                skb->pkt_type=PACKET_OTHERHOST;
                } 
                if (ntohs(hdr->h_type)>=1536)
                        skb->protocol = hdr->h_type;
                else 
                        skb->protocol = htons(ETH_P_802_2);
                skb->h.raw = skb_pull(skb, LEC_HEADER_LEN);
                rawp = skb->data;
                /* Magic hack for IPX ... */
                if (*(unsigned short*)rawp == 0xFFFF)
                        skb->protocol = htons(ETH_P_802_3);
                netif_rx(skb);
                priv->stats.rx_packets++;
        }
}

int 
lec_vcc_attach(struct atm_vcc *vcc, void *arg)
{
        struct atmlec_ioc ioc_data;

        /* Lecd must be up in this case */
        memcpy_fromfs(&ioc_data, arg, sizeof(struct atmlec_ioc));
        if (ioc_data.dev_num < 0 || ioc_data.dev_num >= MAX_LEC_ITF || 
            !dev_lec[ioc_data.dev_num])
                return -EINVAL;
        lec_vcc_added(dev_lec[ioc_data.dev_num]->priv, 
                      &ioc_data, vcc, vcc->push);
        vcc->push = lec_push;
        vcc->proto_data = dev_lec[ioc_data.dev_num];
        return 0;
}

int 
lec_mcast_attach(struct atm_vcc *vcc, int arg)
{
        if (arg <0 || arg >= MAX_LEC_ITF || !dev_lec[arg])
                return -EINVAL;
        vcc->proto_data = dev_lec[arg];
        return (lec_mcast_make((struct lec_priv*)dev_lec[arg]->priv, vcc));
}

/* Initialize device. */
int 
lecd_attach(struct atm_vcc *vcc, int arg)
{  
        int i, result;
        struct lec_priv *priv;

        if (arg<0)
                i = 0;
        else
                i = arg;
        if (arg >= MAX_LEC_ITF)
                return -EINVAL;
        if (!dev_lec[i]) {
                dev_lec[i] = (struct device*)kmalloc(sizeof(struct device)+
                                                     sizeof(myname)+1, 
                                                     GFP_KERNEL);
                if (!dev_lec[i])
                        return -ENOMEM;
                memset(dev_lec[i],0, sizeof(struct device)+sizeof(myname)+1);
                dev_lec[i]->name = (char*)(dev_lec[i]+1);
                sprintf(dev_lec[i]->name, "lec%d",i);
                dev_lec[i]->init = lec_init;
                if ((result = register_netdev(dev_lec[i])) !=0)
                        return result;
                priv = (struct lec_priv *)dev_lec[i]->priv;
	} else {
                priv = (struct lec_priv *)dev_lec[i]->priv;
                if (priv->lecd)
                        return -EADDRINUSE;
        }
        lec_arp_init(priv);
        priv->lecd = vcc;
        vcc->dev = &lecatm_dev;
        
        vcc->proto_data = dev_lec[i];
        vcc->flags |= ATM_VF_READY | ATM_VF_META;

        /* Set default values to these variables */
        priv->maximum_unknown_frame_count = 1;
        priv->max_unknown_frame_time = (1*HZ);
        priv->vcc_timeout_period = (1200*HZ);
        priv->max_retry_count = 1;
        priv->aging_time = (300*HZ);
        priv->forward_delay_time = (15*HZ);
        priv->topology_change = 0;
        priv->arp_response_time = (1*HZ);
        priv->flush_timeout = (4*HZ);
        priv->path_switching_delay = (6*HZ);

        return i;
}
