/* net/atm/ipcommon.c - Common items for all ways of doing IP over ATM */

/* Written 1996,1997 by Werner Almesberger, EPFL LRC */


#include <linux/config.h>
#include <linux/string.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/in.h>
#include <linux/atmdev.h>
#include <linux/atmclip.h>

#include "common.h"
#include "ipcommon.h"
#include "atmarp.h"


#if 1
#define DPRINTK(format,args...) printk(KERN_DEBUG format,##args)
#else
#define DPRINTK(format,args...)
#endif


const unsigned char llc_oui[] = {
	0xaa,	/* DSAP: non-ISO */
	0xaa,	/* SSAP: non-ISO */
	0x03,	/* Ctrl: Unnumbered Information Command PDU */
	0x00,	/* OUI: EtherType */
	0x00,
	0x00 };


static int clip_hard_header(struct sk_buff *skb,struct device *dev,
    unsigned short type,void *daddr,void *saddr,unsigned len)
{
	void *here;

	if (!dev->hard_header_len) return 0; /* just in case ... */
	here = skb_push(skb,dev->hard_header_len);
	memcpy(here,llc_oui,sizeof(llc_oui));
	((unsigned short *) here)[3] = htons(type);
	return -RFC1483LLC_LEN;
}


static int clip_rebuild_header(void *buff,struct device *dev,unsigned long dst,
    struct sk_buff *skb)
{
	/* no ARP with this type of IP over ATM */
	return 0;
}


struct sk_buff *atm_peek_clip(struct atm_vcc *vcc,unsigned long pdu_size,
    __u32 (*fetch)(struct atm_vcc *vcc,int i))
{
        struct sk_buff *skb;
	int header_size;

	/* TODO: check against buffer quota (should even check against upper
		 layer protocol socket quota) */
	header_size = 0;
	if (fetch && pdu_size > 28) { /* > IP+UDP header */
		unsigned long type;

/* doesn't work yet */
type = ntohl(fetch(vcc,2));
/*printk("type is 0x%08lx\n",type);*/
		type = ntohl(fetch(vcc,2)) & 0xff0000;
		if (type == IPPROTO_UDP << 8) header_size = 28; /* bytes */
		else if (type == IPPROTO_TCP << 8) header_size = 40; /* bytes */
	}
	if (!header_size) skb = alloc_skb((pdu_size+3) & ~3,GFP_ATOMIC);
	else {
		skb = alloc_skb(((pdu_size+3) & ~3)+PAGE_SIZE+header_size-1,
		    GFP_ATOMIC);
		if (skb) {
			DPRINTK("data before: 0x%p\n",skb->data);
		        skb->data = (unsigned char *) (((unsigned long) skb->
			    data+header_size+PAGE_SIZE-1) & ~(PAGE_SIZE-1))-
			    header_size;
			DPRINTK("data after: 0x%p\n",skb->data);
		}
	}
	if (!skb) {
		CLIP(vcc->proto_data)->stats.rx_dropped++;
		vcc->stats->rx_drop++;
	}
        return skb;
}
 
 
void atm_pop_clip(struct atm_vcc *vcc,struct sk_buff *skb)
{
#if 0 && defined(CONFIG_MMU_HACKS)
	if (skb->atm.iovcnt)
		unlock_user(skb->atm.iovcnt-1,(struct iovec *) skb->data+1);
#endif
/*printk("popping (r:%d,w:%d)\n",skb->sk->rmem_alloc,skb->sk->wmem_alloc);*/
	dev_kfree_skb(skb,FREE_WRITE);
}



static struct enet_statistics *atm_clip_get_stats(struct device *dev)
{
	return &CLIP(dev)->stats;
}


/*@@@static*/ int clip_ioctl(struct device *dev,struct ifreq *rq,int cmd)
{
	if (!suser()) return -EPERM;
	switch (cmd) {
		case CLIP_NULENCAP:
#if 0
			dev->type = ARPHDR_ATMNULL;
#endif
			dev->hard_header_len = 0;
			return 0;
		case CLIP_LLCENCAP:
#if 0
			dev->type = ARPHDR_ATMLLC;
#endif
			dev->hard_header_len = RFC1483LLC_LEN;
			return 0;
		default:
			return -EOPNOTSUPP;
	}
}


static int clip_open(struct device *dev)
{
	DPRINTK("clip_open called\n");
	return 0;
}


static int clip_stop(struct device *dev)
{
	DPRINTK("DOWN! (%s,0x%p)\n",dev->name,CLIP(dev)->vcc);
	atm_release_vcc(CLIP(dev)->vcc,1);
	return 0;
}


void ipcom_init(struct device *dev,
    int (*hard_start_xmit)(struct sk_buff *skb,struct device *dev),
    unsigned short extra_flags)
{
	DPRINTK("ipcom_init\n");
	DPRINTK("configuring %s\n",dev->name);
	dev->hard_start_xmit = hard_start_xmit;
#if 0
#ifdef CONFIG_MMU_HACKS
	dev->sg_xmit = clip_sg_xmit;
#else
	dev->sg_xmit = NULL;
#endif
#endif
	dev->open = clip_open;
	dev->stop = clip_stop;
	ether_setup(dev);
	dev->tbusy = 0; /* @@@ check */
	dev->hard_header = clip_hard_header;
	dev->do_ioctl = clip_ioctl;
	dev->change_mtu = NULL;
	dev->rebuild_header = clip_rebuild_header;
	dev->get_stats = atm_clip_get_stats;
	dev->hard_header_len = RFC1483LLC_LEN;
	dev->flags |= IFF_NOARP | IFF_POINTOPOINT | extra_flags;
#if 0
	dev->type = ARPHDR_ATMLLC;
#endif
	dev->mtu = RFC1626_MTU;
	memset(&CLIP(dev)->stats,0,sizeof(struct enet_statistics));

}


int ipcom_pick_number(int number)
{
#ifdef CONFIG_ATM_ATMARP
	struct device *atmarp;
#endif
#ifdef CONFIG_ATM_CLIP
	struct clip_priv *clip;
#endif

	if (number != -1) {
#ifdef CONFIG_ATM_ATMARP
		for (atmarp = clip_devs; atmarp; atmarp = PRIV(atmarp)->next)
			if (PRIV(atmarp)->number == number) return -EEXIST;
#endif
	}
        else {
                number = 0;

#ifdef CONFIG_ATM_ATMARP
                for (atmarp = clip_devs; atmarp; atmarp = PRIV(atmarp)->next)
                        if (PRIV(atmarp)->number >= number)
                                number = PRIV(atmarp)->number+1;
#endif
#ifdef CONFIG_ATM_CLIP
                for (clip = old_clip_devs; clip; clip = clip->next)
                        if (clip->number >= number) number = clip->number+1;
#endif
        }
	return number;
}
