/* net/atm/common.c - ATM sockets (common part for PVC and SVC) */

/* Written 1995-1997 by Werner Almesberger, EPFL LRC */

#include <linux/config.h>
#include <linux/net.h>		/* struct socket, struct net_proto, struct
				   proto_ops */
#include <linux/atm.h>		/* ATM stuff */
#include <linux/atmdev.h>
#include <linux/atmclip.h>	/* CLIP_*ENCAP */
#include <linux/atmarp.h>	/* manifest constants */
#include <linux/sonet.h>	/* for ioctls */
#include <linux/socket.h>	/* SOL_SOCKET */
#include <linux/errno.h>	/* error codes */
#include <linux/kernel.h>	/* suser */
#include <asm/uaccess.h>		
#include <linux/sched.h>
#include <linux/time.h>		/* struct timeval */
#include <linux/skbuff.h>
#include <linux/mmuio.h>
#include <linux/uio.h>

#ifdef CONFIG_AREQUIPA
#include <linux/arequipa.h>
#endif

#ifdef CONFIG_ATM_LANE
#include <linux/atmlec.h>
#include "lec.h"
#include "lec_arpc.h"
#endif

#include "static.h"		/* atm_find_dev */
#include "common.h"		/* prototypes */
#include "protocols.h"		/* atm_init_<transport> */
#include "tunable.h"		/* tunable parameters */
#include "atmarp.h"		/* for clip_create */
#include "signaling.h"		/* for WAITING and sigd_attach */


#if 1
#define DPRINTK(format,args...) printk(KERN_DEBUG format,##args)
#else
#define DPRINTK(format,args...)
#endif


static struct sk_buff *alloc_tx(struct atm_vcc *vcc,unsigned int size)
{
	struct sk_buff *skb;

	if (vcc->tx_inuse && size+vcc->tx_inuse+ATM_PDU_OVHD > vcc->tx_quota)
		return NULL;
	while (!(skb = alloc_skb(size,GFP_KERNEL))) schedule();
	DPRINTK("AlTx %d += %d\n",vcc->tx_inuse,skb->truesize);
	atomic_add(skb->truesize+ATM_PDU_OVHD,&vcc->tx_inuse);
	return skb;
}


int atm_create(struct socket *sock,int protocol)
{
	struct atm_vcc *vcc;

	ATM_SD(sock) = NULL;
	if (sock->type == SOCK_STREAM) return -EINVAL;
	if (!(vcc = alloc_atm_vcc())) return -ENOMEM;
#ifdef CONFIG_AREQUIPA
	vcc->upper = NULL;
	vcc->sock = sock;
#endif
	vcc->flags = ATM_VF_SCRX | ATM_VF_SCTX;
	vcc->dev = NULL;
	vcc->family = sock->ops->family;
	vcc->alloc_tx = alloc_tx;
	vcc->callback = NULL;
	memset(&vcc->local,0,sizeof(struct sockaddr_atmsvc));
	memset(&vcc->remote,0,sizeof(struct sockaddr_atmsvc));
	vcc->tx_quota = ATM_TXBQ_DEF;
	vcc->rx_quota = ATM_RXBQ_DEF;
	vcc->tx_inuse = vcc->rx_inuse = 0;
	vcc->aal = protocol; /* temporary @@@ */
	vcc->push = NULL;
	vcc->peek = NULL;
	vcc->pop = NULL;
	vcc->push_oam = NULL;
	vcc->vpi = vcc->vci = 0; /* no VCI/VPI yet */
	vcc->atm_options = vcc->aal_options = 0;
	vcc->timestamp.tv_sec = vcc->timestamp.tv_usec = 0;
	vcc->sleep = vcc->wsleep = NULL;
	skb_queue_head_init(&vcc->recvq);
	skb_queue_head_init(&vcc->listenq);
	ATM_SD(sock) = vcc;
	return 0;
}


int atm_release_vcc(struct atm_vcc *vcc,int free_vcc)
{
	struct sk_buff *skb;

	vcc->flags &= ~ATM_VF_READY;
	if (vcc->dev) {
		if (vcc->dev->ops->close) vcc->dev->ops->close(vcc);
		if (vcc->push) vcc->push(vcc,NULL); /* atmarpd has no push */
		while ((skb = skb_dequeue(&vcc->recvq))) {
			atomic_sub(skb->truesize+ATM_PDU_OVHD,&vcc->rx_inuse);
			if (vcc->dev->ops->free_rx_skb)
				vcc->dev->ops->free_rx_skb(vcc,skb);
			else kfree_skb(skb,FREE_READ);
		}
		if (vcc->rx_inuse)
			printk(KERN_WARNING "atm_release_vcc: strange ... "
			    "rx_inuse == %d after closing\n",vcc->rx_inuse);
		if (vcc->prev) vcc->prev->next = vcc->next;
		else vcc->dev->vccs = vcc->next;
		if (vcc->next) vcc->next->prev = vcc->prev;
		else vcc->dev->last = vcc->prev;
	}
	if (free_vcc) free_atm_vcc(vcc);
	return 0;
}


extern void atm_push_clip(struct atm_vcc *vcc,struct sk_buff *skb);


int atm_release(struct socket *sock,struct socket *peer)
{
	struct atm_vcc *vcc;

	vcc = ATM_SD(sock);
	if (!vcc
#ifdef CONFIG_ATM_CLIP
	    || (vcc->push == atm_push_clip && (vcc->vpi || vcc->vci))
#endif
	    )
		return 0; /* ugly */
	return atm_release_vcc(vcc,1);
}


static int adjust_tp(struct atm_trafprm *tp,unsigned char aal)
{
	int max_sdu;

	if (!tp->traffic_class) return 0;
	if (tp->traffic_class != ATM_UBR && !tp->min_pcr && !tp->max_pcr)
		return -EINVAL;
	switch (aal) {
		case ATM_AAL0:
			max_sdu = ATM_CELL_SIZE-1;
			break;
		case ATM_AAL34:
			max_sdu = ATM_MAX_AAL34_PDU;
			break;
		default:
			printk(KERN_WARNING "ATM: AAL problems ... "
			    "(%d)\n",aal);
			/* fall through */
		case ATM_AAL5:
			max_sdu = ATM_MAX_AAL5_PDU;
	}
	if (!tp->max_sdu) tp->max_sdu = max_sdu;
	else if (tp->max_sdu > max_sdu) return -EINVAL;
	if (!tp->max_cdv) tp->max_cdv = ATM_MAX_CDV;
	return 0;
}


static int check_ci(struct atm_vcc *vcc,short vpi,int vci)
{
	struct atm_vcc *walk;

	for (walk = vcc->dev->vccs; walk; walk = walk->next)
		if ((walk->flags & ATM_VF_ADDR) && walk->vpi == vpi &&
		    walk->vci == vci && ((walk->qos.txtp.traffic_class !=
		    ATM_NONE && vcc->qos.txtp.traffic_class != ATM_NONE) ||
		    (walk->qos.rxtp.traffic_class != ATM_NONE &&
		    vcc->qos.rxtp.traffic_class != ATM_NONE)))
			return -EADDRINUSE;
		/* allow VCCs with same VPI/VCI iff they don't collide on
		   TX/RX (but we may refuse such sharing for other reasons,
		   e.g. if protocol requires to have both channels) */
	return 0;
}


int atm_find_ci(struct atm_vcc *vcc,short *vpi,int *vci)
{
	static short p = 0; /* poor man's per-device cache */
	static int c = 0;
	short old_p;
	int old_c;

	if (*vpi != ATM_VPI_ANY && *vci != ATM_VCI_ANY)
		return check_ci(vcc,*vpi,*vci);
	/* last scan may have left values out of bounds for current device */
	if (*vpi != ATM_VPI_ANY) p = *vpi;
	else if (p >= 1 << vcc->dev->ci_range.vpi_bits) p = 0;
	if (*vci != ATM_VCI_ANY) c = *vci;
	else if (c < ATM_NOT_RSV_VCI || c >= 1 << vcc->dev->ci_range.vci_bits)
			c = ATM_NOT_RSV_VCI;
	old_p = p;
	old_c = c;
	do {
		if (!check_ci(vcc,p,c)) {
			*vpi = p;
			*vci = c;
			return 0;
		}
		if (*vci == ATM_VCI_ANY) {
			c++;
			if (c >= 1 << vcc->dev->ci_range.vci_bits)
				c = ATM_NOT_RSV_VCI;
		}
		if ((c == ATM_NOT_RSV_VCI || *vci != ATM_VCI_ANY) &&
		    *vpi == ATM_VPI_ANY) {
			p++;
			if (p >= 1 << vcc->dev->ci_range.vpi_bits) p = 0;
		}
	}
	while (old_p != p || old_c != c);
	return -EADDRINUSE;
}


static int atm_do_connect(struct atm_vcc *vcc,int itf,int vpi,int vci)
{
	volatile struct atm_dev *dev; /* need volatile for sequence */
	int error;

	if (itf >= MAX_ATM_ITF) return -EINVAL;
	dev = &atm_dev[itf];
	if (!dev->ops) return -ENODEV;
	if ((vpi != ATM_VPI_UNSPEC && vpi != ATM_VPI_ANY &&
	    vpi >> dev->ci_range.vpi_bits) || (vci != ATM_VCI_UNSPEC &&
	    vci != ATM_VCI_ANY && vci >> dev->ci_range.vci_bits))
		return -EINVAL;
	if (vci > 0 && vci < ATM_NOT_RSV_VCI && !suser()) return -EPERM;
	if (vcc->qos.aal) vcc->aal = vcc->qos.aal; /* temporary @@@ */
	error = 0;
	switch (vcc->aal) {
		case ATM_AAL0:
			error = atm_init_aal0(vcc);
			vcc->stats = &((struct atm_dev *) dev)->stats.aal0;
			break;
		case ATM_AAL34:
			error = atm_init_aal34(vcc);
			vcc->stats = &((struct atm_dev *) dev)->stats.aal34;
			break;
		case ATM_NO_AAL:
			/* ATM_AAL5 is also used in the "0 for default" case */
			vcc->aal = ATM_AAL5;
			/* fall through */
		case ATM_AAL5:
			error = atm_init_aal5(vcc);
			vcc->stats = &((struct atm_dev *) dev)->stats.aal5;
			break;
		default:
			error = -EPROTOTYPE;
	}
	if (!error) error = adjust_tp(&vcc->qos.txtp,vcc->aal);
	if (!error) error = adjust_tp(&vcc->qos.rxtp,vcc->aal);
	if (error) return error;
	vcc->dev = (struct atm_dev *) dev;
	DPRINTK("VCC %d.%d, AAL %d\n",vpi,vci,vcc->aal);
	DPRINTK("  TX: %d, PCR %d..%d, SDU %d\n",vcc->qos.txtp.traffic_class,
	    vcc->qos.txtp.min_pcr,vcc->qos.txtp.max_pcr,vcc->qos.txtp.max_sdu);
	DPRINTK("  RX: %d, PCR %d..%d, SDU %d\n",vcc->qos.rxtp.traffic_class,
	    vcc->qos.rxtp.min_pcr,vcc->qos.rxtp.max_pcr,vcc->qos.rxtp.max_sdu);
	if (dev->ops->open) {
		error = dev->ops->open(vcc,vpi,vci);
		if (error) {
			vcc->dev = NULL;
			return error;
		}
	}
	vcc->prev = dev->last;
	vcc->next = NULL;
	if (!dev->vccs) dev->vccs = vcc;
	else dev->last->next = vcc;
	dev->last = vcc;
	return 0;
}


int atm_connect(struct socket *sock,int itf,short vpi,int vci)
{
	struct atm_vcc *vcc;
	int error;

	DPRINTK("atm_connect (vpi %d, vci %d)\n",vpi,vci);
	if (sock->state == SS_CONNECTED) return -EISCONN;
	if (sock->state != SS_UNCONNECTED) return -EINVAL;
	if (!(vpi || vci)) return -EINVAL;
	vcc = ATM_SD(sock);
	if (vpi != ATM_VPI_UNSPEC && vci != ATM_VCI_UNSPEC)
		vcc->flags &= ~ATM_VF_PARTIAL;
	else if (vcc->flags & ATM_VF_PARTIAL) return -EINVAL;
	printk("atm_connect (TX: cl %d,bw %d-%d,sdu %d; RX: cl %d,bw %d-%d,"
	    "sdu %d,AAL %d)\n",vcc->qos.txtp.traffic_class,
	    vcc->qos.txtp.min_pcr,vcc->qos.txtp.max_pcr,vcc->qos.txtp.max_sdu,
	    vcc->qos.rxtp.traffic_class,vcc->qos.rxtp.min_pcr,
	    vcc->qos.rxtp.max_pcr,vcc->qos.rxtp.max_sdu,vcc->qos.aal ?
	    vcc->qos.aal : vcc->aal);
	if (!vcc->qos.txtp.traffic_class && !vcc->qos.rxtp.traffic_class)
		return -EINVAL;
	if (vcc->qos.txtp.traffic_class == ATM_ANYCLASS ||
	    vcc->qos.rxtp.traffic_class == ATM_ANYCLASS)
		return -EINVAL;
	if (itf != ATM_ITF_ANY) error = atm_do_connect(vcc,itf,vpi,vci);
	else {
		int i;

		error = -ENODEV;
		for (i = 0; i < MAX_ATM_ITF; i++)
			if (atm_dev[i].ops)
				if (!(error = atm_do_connect(vcc,i,vpi,vci)))
					break;
	}
	if (error) return error;
	if (vpi == ATM_VPI_UNSPEC || vci == ATM_VCI_UNSPEC)
		vcc->flags |= ATM_VF_PARTIAL;
	else sock->state = SS_CONNECTED;
	return 0;
}


int atm_recvmsg(struct socket *sock,struct msghdr *m,int total_len,
    int flags, struct scm_cookie *scm) /*int nonblock,int flags,int *addr_len)*/
{
	struct atm_vcc *vcc;
	struct sk_buff *skb;
	unsigned long cpu_flags;
	int eff_len;

	void *buff;
	int size;

	if (sock->state != SS_CONNECTED) return -ENOTCONN;
	if (flags) return -EOPNOTSUPP;
	if (m->msg_iovlen != 1) return -ENOSYS; /* fix this later @@@ */
	buff = m->msg_iov->iov_base;
	size = m->msg_iov->iov_len;
	/* verify_area is done by net/socket.c */
	vcc = ATM_SD(sock);
	if (vcc->dev->ops->poll && (vcc->flags & ATM_VF_READY))
		while (1) {
			vcc->dev->ops->poll(vcc,0); /* FIXME */
			if (current->signal & ~current->blocked)
				return -ERESTARTSYS;
			if (skb_peek(&vcc->recvq)) break;
		}
	save_flags(cpu_flags);
	cli();
	while (!(skb = skb_dequeue(&vcc->recvq))) {
		if (vcc->flags & ATM_VF_RELEASED) return vcc->reply;
		if (!(vcc->flags & ATM_VF_READY)) return 0;
		interruptible_sleep_on(&vcc->sleep);
		if (current->signal & ~current->blocked) {
			restore_flags(cpu_flags);
			return -ERESTARTSYS;
		}
	}
	restore_flags(cpu_flags);
	vcc->timestamp = skb->atm.timestamp;
	eff_len = skb->len > size ? size : skb->len;
	if (vcc->dev->ops->feedback)
		vcc->dev->ops->feedback(vcc,skb,(unsigned long) skb->data,
		    (unsigned long) buff,eff_len);
	DPRINTK("RcvM %d -= %d\n",vcc->rx_inuse,skb->truesize);
	atomic_sub(skb->truesize+ATM_PDU_OVHD,&vcc->rx_inuse);
	if (skb->atm.iovcnt) { /* @@@ hack */
		/* iovcnt set, use scatter-gather for receive */
		int el, cnt;
		struct iovec *iov = (struct iovec *)skb->data;
		unsigned char *p = (unsigned char *)buff;

		el = eff_len;
		for (cnt = 0; (cnt < skb->atm.iovcnt) && el; cnt++) {
			copy_to_user(p, iov->iov_base,
			    (iov->iov_len > el) ? el : iov->iov_len);
			p += iov->iov_len;
			el -= (iov->iov_len > el)?el:iov->iov_len;
			iov++;
		}
		if (!vcc->dev->ops->free_rx_skb) kfree_skb(skb,FREE_READ);
		else vcc->dev->ops->free_rx_skb(vcc, skb);
		return eff_len;
	}
#ifdef CONFIG_MMU_HACKS
	if (vcc->flags & ATM_VF_SCRX)
		mmucp_tofs((unsigned long) buff,eff_len,skb,
		    (unsigned long) skb->data);
	else
#endif
	{
#ifdef DUMP_PACKETS
	        if (vcc->vci==5 || vcc->vci >32) {
	                char buf[300];
	                int i;
                
	                for(i=0;i<99 && i < skb->len;i++) {
	                        sprintf(buf+i*3,"%2.2x ",
	                                0xff&skb->data[i]);
	                }
	                printk("recv %d:%s\n",vcc->vci,buf);
	        }          
#endif      
		copy_to_user(buff,skb->data,eff_len);
		if (!vcc->dev->ops->free_rx_skb) kfree_skb(skb,FREE_READ);
		else vcc->dev->ops->free_rx_skb(vcc, skb);
	}
	return eff_len;
}


int atm_sendmsg(struct socket *sock,struct msghdr *m,int total_len,
  struct scm_cookie *scm) /*int nonblock,int flags)*/
{
	struct atm_vcc *vcc;
	struct sk_buff *skb;
	int eff,error;

	const void *buff;
	int size;

	if (sock->state != SS_CONNECTED) return -ENOTCONN;
	if (m->msg_name) return -EISCONN;
	if (m->msg_iovlen != 1) return -ENOSYS; /* fix this later @@@ */
	buff = m->msg_iov->iov_base;
	size = m->msg_iov->iov_len;
	vcc = ATM_SD(sock);
	if (vcc->flags & ATM_VF_RELEASED) return vcc->reply;
	if (!(vcc->flags & ATM_VF_READY)) return -EPIPE;
	if (!size) return 0;
	/* verify_area is done by net/socket.c */
#ifdef CONFIG_MMU_HACKS
	if ((vcc->flags & ATM_VF_SCTX) && vcc->dev->ops->sg_send &&
	    vcc->dev->ops->sg_send(vcc,(unsigned long) buff,size)) {
		int res,max_iov;

		max_iov = 2+size/PAGE_SIZE;
		/*
		 * Doesn't use alloc_tx yet - this will change later. @@@
		 */
		while (!(skb = alloc_skb(sizeof(struct iovec)*max_iov,
		    GFP_KERNEL))) {
		   /*if (nonblock) return -EAGAIN; FIXME */
			interruptible_sleep_on(&vcc->wsleep);
			if (current->signal & ~current->blocked)
				return -ERESTARTSYS;
		}
		skb->free = 1;
		skb->len = size;
		res = lock_user((unsigned long) buff,size,max_iov,
		    (struct iovec *) skb->data);
		if (res < 0) {
			kfree_skb(skb,FREE_WRITE);
			if (res != -EAGAIN) return res;
		}
		else {
			DPRINTK("res is %d\n",res);
			DPRINTK("Asnd %d += %d\n",vcc->tx_inuse,skb->truesize);
			atomic_add(skb->truesize+ATM_PDU_OVHD,&vcc->tx_inuse);
			skb->atm.iovcnt = res;
			skb_device_lock(skb);
			error = vcc->dev->ops->send(vcc,skb);
			/* FIXME: security: may send up to 3 "garbage" bytes */
			return error ? error : size;
		}
	}
#endif
	eff = (size+3) & ~3; /* align to word boundary */
	while (!(skb = vcc->alloc_tx(vcc,eff))) {
	   /* if (nonblock) return -EAGAIN; FIXME */
		interruptible_sleep_on(&vcc->wsleep);
		if (current->signal & ~current->blocked)
			return -ERESTARTSYS;
		if (vcc->flags & ATM_VF_RELEASED) return vcc->reply;
		if (!(vcc->flags & ATM_VF_READY)) return -EPIPE;
	}
	/* skb->free =1; */
	skb->len = size;
	skb->atm.iovcnt = 0;
	copy_from_user(skb->data,buff,size);
	if (eff != size) memset(skb->data+size,0,eff-size);
#if 0 /* experimental */
	vcc->dev->sending = 1;
#endif
	dev_lock_list(); /*skb_device_lock(skb);*/
#ifdef DUMP_PACKETS
        if (vcc->vci==5 || vcc->vci >32) {
                char buf[300];
                int i;
                
                for(i=0;i<99 && i < skb->len;i++) {
                        sprintf(buf+i*3,"%2.2x ", 0xff&skb->data[i]);
                }
                printk("send %d:%s\n",vcc->vci,buf);
        }
#endif
	error = vcc->dev->ops->send(vcc,skb);
#if 0 /* experimental */
	while (vcc->dev->sending) sleep_on(&vcc->sleep);
#endif
	return error ? error : size;
}


unsigned int atm_select(struct socket *sock, poll_table *wait)
{
	unsigned int mask;
	struct atm_vcc *vcc;

	/*	vcc = ATM_SD(sock);
	
		case SEL_IN:
			if (vcc->dev && vcc->dev->ops->poll)
				vcc->dev->ops->poll(ATM_SD(sock),1);
			if (sock->state == SS_CONNECTING) break;
			if (!skb_peek(&vcc->recvq) &&
			    !skb_peek(&vcc->listenq) &&
			    !(vcc->flags & ATM_VF_RELEASED)) break;
			if (vcc->dev && vcc->dev->ops->poll) return 0;
			return 1;
		case SEL_OUT:
			if (sock->state != SS_CONNECTING) {
				if (vcc->qos.txtp.traffic_class == ATM_NONE)
					return 1;
				if (vcc->qos.txtp.max_sdu+vcc->tx_inuse+
				    ATM_PDU_OVHD <= vcc->tx_quota) return 1;
			}
			poll_wait(&vcc->wsleep,wait);
			return 0;
		case SEL_EX:
			if (sock->state != SS_CONNECTING) return 0;
			if (vcc->reply == WAITING) break;
			return 1;
	
	poll_wait(&vcc->sleep,wait);*/
	return 0;
}



static int fetch_stats(struct atm_dev *dev,struct atm_dev_stats *arg,int zero)
{
	unsigned long flags;
 
	save_flags(flags);
	cli();
	if (arg)
		copy_to_user(arg,&dev->stats,sizeof(struct atm_dev_stats));
        if (zero)
                memset(&dev->stats,0,sizeof(struct atm_dev_stats));
        restore_flags(flags);
	return 0;
}


extern int clip_ioctl(void *dev,void *rq,int cmd); /* @@@ lie and cheat ... */


#ifdef CONFIG_AREQUIPA

/* @@@ stolen from net/socket.c - should be in a common header file */


struct socket *sockfd_lookup(int fd)
{
	struct file *file;
	struct inode *inode;

	if (fd < 0 || fd >= NR_OPEN || !(file = current->files->fd[fd])) 
		return NULL;
	inode = file->f_inode;
	if (!inode || !inode->i_sock) return NULL;
	return &inode->u.socket_i;
}

#endif


int atm_ioctl(struct socket *sock,unsigned int cmd,unsigned long arg)
{
	struct atm_dev *dev;
	struct atm_vcc *vcc;
	unsigned long eff_arg;
	int rsize,wsize,len;
	int error;
	unsigned long t1, t2;
	vcc = ATM_SD(sock);
	rsize = wsize = 0;
	switch (cmd) {
		case ATM_GETNAMES:
			{
				struct atm_iobuf *buf;
				error = verify_area(VERIFY_READ,(void *) arg,
				    sizeof(struct atm_iobuf));
				if (error) 
				   return error;
				buf = (struct atm_iobuf *) arg;
				
				get_user(t1, (int *) &buf->buffer);
				get_user(t2, (int *) &buf->length);
				error = verify_area(VERIFY_WRITE,
						    (void *)t1,t2);
				if (error) 
				   return error;
				
				error = verify_area(VERIFY_WRITE,
				    (void *) &buf->length,sizeof(int));
				if (error) 
				   return error;
				
				get_user(t1, (int *)&buf->buffer);
				return atm_dev_list((void *)t1, &buf->length);
			}
		case SIOCGSTAMP: /* borrowed from IP */
			if (!vcc->timestamp.tv_sec) return -ENOENT;
			vcc->timestamp.tv_sec += vcc->timestamp.tv_usec/1000000;
			vcc->timestamp.tv_usec %= 1000000;
			error = verify_area(VERIFY_WRITE,(void *) arg,
			    sizeof(struct timeval));
			if (error) return error;
			copy_to_user((void *) arg,&vcc->timestamp,
			    sizeof(struct timeval));
			return 0;
		case ATM_SETSC:
			if (arg & ~(ATM_VF_SCRX | ATM_VF_SCTX)) return -EINVAL;
			/* @@@ race condition - should split flags into
			       "volatile" and non-volatile part */
			vcc->flags = (vcc->flags & ~(ATM_VF_SCRX |
			    ATM_VF_SCTX)) | arg;
			return 0;
		case ATMSIGD_CTRL:
			if (!suser()) return -EPERM;
			error = sigd_attach(vcc);
			if (!error) sock->state = SS_CONNECTED;
			return error;
#ifdef CONFIG_ATM_CLIP
		case CLIP_PVC:
			if (!suser()) return -EPERM;
			return atm_init_clip(vcc);
		case CLIP_NULENCAP:
		case CLIP_LLCENCAP:
			if (!suser()) return -EPERM;
			return clip_ioctl(vcc->proto_data,NULL,cmd);
				/* rather crude hack ... */
#endif
#ifdef CONFIG_ATM_ATMARP
		case SIOCMKCLIP:
			if (!suser()) return -EPERM;
			return clip_create(arg);
		case ATMARPD_CTRL:
			if (!suser()) return -EPERM;
			error = atm_init_atmarp(vcc);
			if (!error) sock->state = SS_CONNECTED;
			return error;
		case ATMARP_MKIP:
			if (!suser()) return -EPERM;
			return atmarp_mkip(vcc,arg);
		case ATMARP_SETENTRY:
			if (!suser()) return -EPERM;
			return atmarp_setentry(vcc,arg);
		case ATMARP_ENCAP:
			if (!suser()) return -EPERM;
			return atmarp_encap(vcc,arg);
#endif
#ifdef CONFIG_AREQUIPA
		case AREQUIPA_PRESET:
			{
				struct socket *upper;

				if (!(upper = sockfd_lookup(arg)))
					return -ENOTSOCK;
				if (upper->ops->family != PF_INET)
					return -EPROTOTYPE;
				return arequipa_preset(sock,
				    (struct sock *) upper->data);
			}
		case AREQUIPA_INCOMING:
			return arequipa_incoming(sock);
		case AREQUIPA_CTRL:
			if (!suser()) return -EPERM;
			error = arequipad_attach(vcc);
			if (!error) sock->state = SS_CONNECTED;
			return error;
		case AREQUIPA_CLS3RD:
			if (!suser()) return -EPERM;
			arequipa_close_vcc((struct atm_vcc *) arg);
			return 0;
#endif
#ifdef CONFIG_ATM_LANE
                case ATMLEC_CTRL:
                        if (!suser()) return -EPERM;
                        error = lecd_attach(vcc, (int)arg);
                        if (error >=0) sock->state = SS_CONNECTED;
                        return error;
                case ATMLEC_MCAST:
                        if (!suser()) return -EPERM;
                        return lec_mcast_attach(vcc, (int)arg);
                case ATMLEC_DATA:
                        if (!suser()) return -EPERM;
                        return lec_vcc_attach(vcc, (void*)arg);
#endif
		case ATM_GETTYPE:
			wsize = -1; /* special - don't check length */
			break;
		case ATM_GETESI:
			wsize = ESI_LEN;
			break;
		case ATM_GETSTATZ:
			if (!suser()) return -EPERM;
			/* fall through */
		case ATM_GETSTAT:
			wsize = sizeof(struct atm_dev_stats);
			break;
		case ATM_GETCIRANGE:
			wsize = sizeof(struct atm_cirange);
			break;
		case ATM_SETCIRANGE:
			if (!suser()) return -EPERM;
			rsize = sizeof(struct atm_cirange);
			break;
		case SONET_GETSTATZ:
			if (!suser()) return -EPERM;
			/* fall through */
		case SONET_GETSTAT:
			wsize = sizeof(struct sonet_stats);
			break;
		case SONET_GETDIAG:
			wsize = sizeof(int);
			break;
		case SONET_SETDIAG:
		case SONET_CLRDIAG:
			if (!suser()) return -EPERM;
			rsize = sizeof(int);
			break;
		case SONET_SETFRAMING:
			rsize = sizeof(int);
			break;
		case SONET_GETFRAMING:
			wsize = sizeof(int);
			break;
		case SONET_GETFRSENSE:
			wsize = SONET_FRSENSE_SIZE;
			break;
		default:
			wsize = -1; /* just in case ... */
	}
	error = verify_area(VERIFY_READ,(void *) arg,
	    sizeof(struct atmif_sioc));
	if (error) return error;
	if (wsize) {
		error = verify_area(VERIFY_WRITE,(void *) arg,
		    sizeof(struct atmif_sioc));
		if (error) return error;
	}
	get_user(t1, (int *)&((struct atmif_sioc *) arg)->number);
	if (!(dev = atm_find_dev(t1))) return -ENODEV;
	get_user(len, (int *)&((struct atmif_sioc *) arg)->length);
	get_user(eff_arg, (int *)&((struct atmif_sioc *) arg)->arg);
	if (!eff_arg && (rsize || wsize)) return -EINVAL;
	if (rsize > 0) {
		if (len != rsize) return -EINVAL;
		error = verify_area(VERIFY_READ,(void *) eff_arg,len);
		if (error) return error;
	}
	if (wsize > 0) {
		if (len != wsize) return -EINVAL;
		error = verify_area(VERIFY_WRITE,(void *) eff_arg,len);
		if (error) return error;
		
		put_user(wsize,&((struct atmif_sioc *) arg)->length);
	}
	switch (cmd) {
		case ATM_GETTYPE:
			{
				int length;

				length = strlen(dev->type);
				if (len <length+1) return -EINVAL;
				error = verify_area(VERIFY_WRITE,
				    (void *) eff_arg,length+1);
				if (error) return error;
				copy_to_user((void *) eff_arg,dev->type,
				    length+1);
				put_user(length,
				    &((struct atmif_sioc *) arg)->length);
				return length;
			}
		case ATM_GETESI:
			copy_to_user((void *) eff_arg,dev->esi,ESI_LEN);
			return wsize;
		case ATM_GETSTATZ:
		case ATM_GETSTAT:
			error = fetch_stats(dev,(void *) eff_arg,
			    cmd == ATM_GETSTATZ);
			return error < 0 ? error : wsize;
		case ATM_GETCIRANGE:
			copy_to_user((void *) eff_arg,&dev->ci_range,
			    sizeof(struct atm_cirange));
			return wsize;
		default:
			if (!dev->ops->ioctl) return -EINVAL;
			error = dev->ops->ioctl(dev,cmd,eff_arg);
			if (error >= 0) 
				put_user(len,
				    &((struct atmif_sioc *) arg)->length);
			return error;
	}
}


int atm_setsockopt(struct socket *sock,int level,int optname,
    char *optval,int optlen)
{
	struct atm_vcc *vcc;

	vcc = ATM_SD(sock);
	if (level == SOL_SOCKET) {
		unsigned long value;

		switch (optname) {
			case SO_SNDBUF:
				if (optlen != sizeof(value)) return -EINVAL;
				get_user(value, (int *)optval);
				if (!value) value = ATM_TXBQ_DEF;
				if (value < ATM_TXBQ_MIN) value = ATM_TXBQ_MIN;
				if (value > ATM_TXBQ_MAX) value = ATM_TXBQ_MAX;
				vcc->tx_quota = value;
				return 0;
			case SO_RCVBUF:
				if (optlen != sizeof(value)) return -EINVAL;
				get_user(value, (int *)optval);
				if (!value) value = ATM_RXBQ_DEF;
				if (value < ATM_RXBQ_MIN) value = ATM_RXBQ_MIN;
				if (value > ATM_RXBQ_MAX) value = ATM_RXBQ_MAX;
				vcc->rx_quota = value;
				return 0;
			default:
				return -EINVAL;
		}
	}
	if (level == SOL_ATM) {
		switch (optname) {
			case SO_ATMQOS:
				if (sock->state != SS_UNCONNECTED)
					return -EBADFD;
				if (optlen != sizeof(struct atm_qos))
					return -EINVAL;
				copy_from_user(&vcc->qos,optval,optlen);
#if 0
				if ((vcc->qos.txtp.traffic_class == ATM_UBR &&
				    (vcc->qos.txtp.min_pcr ||
				    vcc->qos.txtp.max_pcr ||
				    vcc->qos.rxtp.min_pcr ||
				    vcc->qos.rxtp.max_pcr)) ||
				    (vcc->qos.rxtp.traffic_class == ATM_UBR &&
				    (vcc->qos.rxtp.min_pcr ||
				    vcc->qos.rxtp.max_pcr)))
					printk(KERN_WARNING "Warning: "
					  "semantics of ATM_UBR have changed "
					  "with min/max_pcr != 0\n");
#endif
				vcc->flags |= ATM_VF_HASQOS;
				return 0;
			default:
				break;
		}
	}
	if (!vcc->dev || !vcc->dev->ops->setsockopt) return -EINVAL;
	return vcc->dev->ops->setsockopt(vcc,level,optname,optval,optlen);
}


static int putsockopt(void *data,int len,char *optval,int *optlen)
{
	int error;
        int olen;
	
	get_user(olen, optlen);
	if (olen < len) 
	   return -EINVAL;
	put_user(len,optlen);
	error = copy_to_user(optval,data,len);
	return (error ? error: 0);
}


int atm_getsockopt(struct socket *sock,int level,int optname,
    char *optval,int *optlen)
{
	struct atm_vcc *vcc;
	int error;

	error = verify_area(VERIFY_READ,optlen,sizeof(*optlen)); /* paranoia? */
	if (error) return error;
	error = verify_area(VERIFY_WRITE,optlen,sizeof(*optlen));
	if (error) return error;
	vcc = ATM_SD(sock);
	switch (level) {
		case SOL_SOCKET:
			switch (optname) {
				case SO_SNDBUF:
					return putsockopt(&vcc->tx_quota,
					    sizeof(unsigned long),optval,
					    optlen);
				case SO_RCVBUF:
					return putsockopt(&vcc->rx_quota,
					    sizeof(unsigned long),optval,
					    optlen);
				case SO_BCTXOPT:
				case SO_BCRXOPT:
					level = SOL_AAL; /* cheat */
					break;
				default:
					return -EINVAL;
			}
			/* fall through */
		case SOL_AAL:
			switch (optname) {
				case SO_AALTYPE:
					return putsockopt(&vcc->aal,
					    sizeof(unsigned char),optval,
					    optlen);
				default:
					break;
			}
			break;
		case SOL_ATM:
			switch (optname) {
				case SO_ATMQOS:
					if (!(vcc->flags & ATM_VF_HASQOS))
						return -EINVAL;
					return putsockopt(&vcc->qos,
					    sizeof(struct atm_qos),optval,
					    optlen);
				default:
					break;
			}
			/* fall through */
		default:
			break;
	}
	if (!vcc->dev || !vcc->dev->ops->getsockopt) return -EINVAL;
	return vcc->dev->ops->getsockopt(vcc,level,optname,optval,optlen);
}
