/* atmarp.c - RFC1577 ATM ARP */

/* Written 1995-1997 by Werner Almesberger, EPFL LRC */
 

#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kernel.h> /* for UINT_MAX */
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/wait.h>
#include <linux/timer.h>
#include <linux/if_arp.h> /* for some manifest constants */
#include <linux/notifier.h>
#include <linux/atm.h>
#include <linux/atmdev.h>
#include <linux/atmclip.h>
#include <linux/atmarp.h>
#include <linux/ip.h> /* for net/route.h */
#include <linux/in.h> /* for struct sockaddr_in */
#include <net/route.h> /* for struct rtable and ip_rt_route */
#include <asm/param.h> /* for HZ */
#include <asm/byteorder.h> /* for htons etc. */


/*
 * WARNING: This code will eventually become full ATMARP support as defined in
 *	    RFC1577 (actually, the plan is to move on quickly to RFC1577+), but
 *	    right now it's certainly full of bugs and severe design errors.
 *	    Don't use it as a reference for anything.
 */


#include "common.h"
#include "tunable.h"
#include "protocols.h" /* for atm_push_raw */
#include "ipcommon.h"
#include "atmarp.h"


#if 0
#define DPRINTK(format,args...) printk(KERN_DEBUG format,##args)
#else
#define DPRINTK(format,args...)
#endif


/*
 * Relation between address/subddress and private/public address:
 *
 * Caller 	Called			Address		Subaddress
 * -----------  ----------------------- --------------	----------
 * private	same private net	private		-
 * public-only	private via public	public (E.164)	private
 * private net	public UNI		public (NSAP)	-
 *
 * See also: ATM Forum UNI 3.0 (or 3.1), Annex A
 */


struct device *clip_devs = NULL;
struct atm_vcc *atmarpd = NULL;
static struct wait_queue *atmarpd_sleep = NULL;
static struct timer_list idle_timer = { NULL, NULL, 0L, 0L, NULL };
static int start_timer = 1;


#define WAITING 1 /* see also signaling.h */


static int atmarpd_send(struct atm_vcc *vcc,struct sk_buff *skb)
{
	struct atmarp_ctrl *ctrl;
 
	atomic_sub(skb->truesize+ATM_PDU_OVHD,&vcc->tx_inuse);
        ctrl = (struct atmarp_ctrl *) skb->data;
	if (ctrl->magic != ATMARP_CTRL_MAGIC) {
		printk(KERN_ALERT "atmarpd_send: bad magic 0x%x\n",
		    ctrl->magic);
		return -EPROTO;
	}
	if (ctrl->type != act_complete) {
		printk(KERN_ALERT "atmarpd_send: bad type 0x%x\n",ctrl->type);
		return -EPROTO;
	}
	if (!ctrl->reply) {
		printk(KERN_ALERT "atmarpd_send: no reply\n");
		return -EPROTO;
	}
	*ctrl->reply = ctrl->arg;
	wake_up(&atmarpd_sleep);
	dev_kfree_skb(skb,FREE_WRITE);
	return 0;
}


static int send_demon(enum atmarp_ctrl_type type,int itf,unsigned long arg,
    void *data,int length)
{
	struct atmarp_ctrl *ctrl;
	struct sk_buff *skb;
	int size,need_reply;
	volatile int reply;

	DPRINTK("send_demon(%d)\n",type);
	if (!atmarpd) return -EUNATCH;
	size = sizeof(struct atmarp_ctrl)+(data ? length : 0);
	skb = alloc_skb(size,GFP_ATOMIC);
	if (!skb) return -ENOMEM;
	skb->free = 1;
	skb->len = size;
	need_reply = type == act_ioctl || type == act_create;
	ctrl = (struct atmarp_ctrl *) skb->data;
	ctrl->magic = ATMARP_CTRL_MAGIC;
	ctrl->type = type;
	ctrl->reply = need_reply ? &reply : NULL;
	ctrl->itf_num = itf;
	ctrl->arg = arg;
	if (data) memcpy(ctrl->data,data,length);
	reply = WAITING;
	atomic_add(skb->truesize+ATM_PDU_OVHD,&atmarpd->rx_inuse);
	skb_queue_tail(&atmarpd->recvq,skb);
	wake_up(&atmarpd->sleep);
	if (!need_reply) return 0;
	while (reply == WAITING && atmarpd) sleep_on(&atmarpd_sleep);
	return atmarpd ? reply : -EUNATCH;
}


static void remove(struct atmarp_entry *entry) /* @@@ ugly ! */
{
	struct atmarp_entry **walk;

	for (walk = &PRIV(entry->dev)->table; *walk; walk = &(*walk)->next)
		if (*walk == entry) {
			*walk = entry->next;
			return;
		}
	printk(KERN_CRIT "ATMARP: remove failed (0x%08lx)\n",
	    (unsigned long) entry);
}


static inline int time_out_entry(struct atmarp_entry **entry)
{
	struct atmarp_entry *next;

	DPRINTK("VC TIMED OUT\n");
	if ((*entry)->vcc) {
		(*entry)->vcc->flags |= ATM_VF_RELEASED;
		(*entry)->vcc->flags &= ~ATM_VF_READY;
		wake_up(&(*entry)->vcc->sleep);
		return 0;
	}
	if ((*entry)->queued) {
		dev_kfree_skb((*entry)->queued,FREE_WRITE);
		DPRINTK("discarding queued skb\n");
	}
	else printk(KERN_CRIT "atmarp: weird - incomplete entry, but "
		    "nothing queued\n");
	next = (*entry)->next;
	kfree(*entry);
	*entry = next;
	return 1;
}


static void idle_timer_check(unsigned long dummy)
{
	struct device *itf;
	struct atmarp_entry **entry;
	unsigned long expire;

	idle_timer.expires = UINT_MAX;
	for (itf = clip_devs; itf; itf = PRIV(itf)->next) {
		entry = &PRIV(itf)->table;
		while (*entry) {
			if ((*entry)->idle_timeout) {
				expire = (*entry)->last_use+(*entry)->
				    idle_timeout;
				if (expire < jiffies) {
					if (time_out_entry(entry)) continue;
				}
				else if (expire < idle_timer.expires)
						idle_timer.expires = expire;
			}
			entry = &(*entry)->next;
		}
	}
	if (idle_timer.expires < jiffies+CLIP_CHECK_INTERVAL*HZ)
		idle_timer.expires = jiffies+CLIP_CHECK_INTERVAL*HZ;
	del_timer(&idle_timer);
	add_timer(&idle_timer);
}


static void atm_push_ip(struct atm_vcc *vcc,struct sk_buff *skb)
{
#if 0
	DPRINTK("clip push\n");
#endif
	if (!skb) {
		DPRINTK("removing AE\n");
		AE(vcc)->old_push(vcc,NULL);
		if (AE(vcc)->ip) remove(AE(vcc));
		kfree(AE(vcc));
		return;
	}
	AE(vcc)->last_use = jiffies;
	skb->dev = AE(vcc)->dev;
	skb->mac.raw = skb->data;
	if (!skb->dev->hard_header_len) skb->protocol = htons(ETH_P_IP);
	else if (skb->len < RFC1483LLC_LEN || memcmp(skb->data,llc_oui,
		    sizeof(llc_oui))) skb->protocol = 0;
			/* probably wrong encap ... */
		else {
			skb->protocol = ((unsigned short *) skb->data)[3];
			skb_pull(skb,RFC1483LLC_LEN);
			if (vcc && skb->protocol == htons(ETH_P_ARP)) {
				PRIV(skb->dev)->stats.rx_packets++;
				atm_push_raw(vcc,skb);
				return;
			}
		}
	PRIV(skb->dev)->stats.rx_packets++;
	netif_rx(skb);
}


static struct atmarp_entry *new_entry(int timeout)
{
	struct atmarp_entry *entry;

	entry = kmalloc(sizeof(struct atmarp_entry),GFP_ATOMIC);
	if (!entry) return NULL;
	entry->ip = 0;
	entry->vcc = NULL;
	entry->encap = 1;
	entry->dev = clip_devs;
	entry->old_push = NULL;
	entry->last_use = jiffies;
	entry->idle_timeout = timeout*HZ;
	entry->queued = NULL;
	entry->next = NULL;
	return entry;
}


static void attach_entry(struct atm_vcc *vcc,struct atmarp_entry *entry)
{
	AE(vcc) = entry;
	entry->old_push = vcc->push;
	vcc->push = atm_push_ip;
	entry->vcc = vcc;
}


static int clip_hard_header(struct sk_buff *skb,struct device *dev,
    unsigned short type,void *daddr,void *saddr,unsigned len)
{
	void *here;

	here = skb_push(skb,dev->hard_header_len);
	memcpy(here,llc_oui,sizeof(llc_oui));
	((unsigned short *) here)[3] = htons(type);
	skb->atm.encap = 1;
	return -RFC1483LLC_LEN;
}


static int clip_rebuild_header(void *buff,struct device *dev,unsigned long dst,
    struct sk_buff *skb)
{
#if 0
	void *here;

	here = skb->data; /*skb_push(skb,dev->hard_header_len);*/
	memcpy(here,llc_oui,sizeof(llc_oui));
	((unsigned short *) here)[3] = htons(ETH_P_IP);
#endif
	return 0;
}


static int clip_xmit(struct sk_buff *skb,struct device *dev)
{
	struct atmarp_entry *entry;

#if 0
  int i;
	DPRINTK("new clip_xmit (0x%x)\n",skb->raddr);
/*(int *) 0xffff0000 = 42;*/
for (i = 0; i < skb->len; i++) printk("%02X ",skb->data[i]);
printk("\n");
#endif
	for (entry = PRIV(dev)->table; entry; entry = entry->next)
		if (entry->ip == skb->raddr) break;
	if (!entry) {
		DPRINTK("no entry - queuing\n");
		send_demon(act_need,PRIV(dev)->number,skb->raddr,NULL,0);
		entry = new_entry(ATMARP_RETRY_DELAY);
		entry->queued = skb;
		entry->ip = skb->raddr;
		entry->next = PRIV(dev)->table;
		PRIV(dev)->table = entry;
		idle_timer_check(0);
		return 0;
	}
	if (!entry->vcc || !(entry->vcc->flags & ATM_VF_READY)) {
		DPRINTK("not found - discarding\n");
		dev_kfree_skb(skb,FREE_WRITE);
		return 0;
		/* Should return -EHOSTUNREACH, but then it will retry
		   forever, so we just discard the packet. */
	}
	if (!entry->encap) {
		if (skb->atm.encap) {
			skb_pull(skb,RFC1483LLC_LEN);
			skb->atm.encap = 0;
		}
	}
	else {
		memcpy((void *) skb->data,llc_oui,sizeof(llc_oui));
		((unsigned short *) skb->data)[3] = htons(ETH_P_IP);
	}
	DPRINTK("CX(A) %d += %d\n",entry->vcc->tx_inuse,skb->truesize);
	atomic_add(skb->truesize,&entry->vcc->tx_inuse);
	skb->atm.iovcnt = 0;
	AE(entry->vcc)->last_use = jiffies;
	entry->vcc->dev->ops->send(entry->vcc,skb);
	PRIV(dev)->stats.tx_packets++;
	return 0;
}


static struct enet_statistics *atm_clip_get_stats(struct device *dev)
{
	return &PRIV(dev)->stats;
}


int atmarp_mkip(struct atm_vcc *vcc,int timeout)
{
	struct atmarp_entry *entry;

	DPRINTK("MKIP\n");
	if (!vcc->push) return -EBADFD;
	entry = new_entry(timeout);
	if (!entry) return -ENOMEM;
	attach_entry(vcc,entry);
	idle_timer_check(0);
	return 0;
}


int atmarp_setentry(struct atm_vcc *vcc,unsigned long ip)
{
	struct atmarp_entry **walk,**next,*succ;
	struct rtable *route;
	struct sk_buff *queued;

	DPRINTK("SETENTRY 0x%lx\n",ip);
	if (vcc->push != atm_push_ip) {
		printk(KERN_WARNING "atmarp_setentry: VCC has no ARP entry\n");
		return -EBADF;
	}
	if (!ip) {
		if (!AE(vcc)->ip) {
			printk(KERN_ERR "hiding hidden ATMARP entry\n");
			return 0;
		}
		DPRINTK("setentry: remove\n");
		remove(AE(vcc));
		AE(vcc)->ip = 0;
		return 0;
	}
	DPRINTK("setentry: route\n");
	route = ip_rt_route(ip,1);
	if (!route) return -EHOSTUNREACH;
	AE(vcc)->dev = route->rt_dev;
	if (AE(vcc)->ip) {
		DPRINTK("setentry: update\n");
		DPRINTK("(updating)\n");
		AE(vcc)->ip = ip;
		return 0;
	}
	DPRINTK("setentry: add\n");
	queued = NULL;
	for (walk = &PRIV(AE(vcc)->dev)->table; *walk; walk = next) {
		next = &(*walk)->next;
		if ((*walk)->ip == ip) {
			if ((*walk)->vcc) continue;
				/* more than one VC to dest */
			if ((*walk)->queued) {
				DPRINTK("setentry: flushing\n");
				if (queued)
					printk(KERN_CRIT "atmarp: bad news - "
					    "more than one skb queued\n");
				queued = (*walk)->queued;
			}
			succ = (*walk)->next;
			kfree(*walk);
			*walk = succ;
			next = walk;
			continue;
		}
	}
	DPRINTK("(adding)\n");
	AE(vcc)->ip = ip;
	AE(vcc)->next =  PRIV(AE(vcc)->dev)->table;
	PRIV(AE(vcc)->dev)->table = AE(vcc);
	if (queued) clip_xmit(queued,route->rt_dev);
	return 0;
}


int atmarp_encap(struct atm_vcc *vcc,int mode)
{
	AE(vcc)->encap = mode;
	return 0;
}


static int atmarp_ioctl(struct device *dev,unsigned int cmd,void *arg)
{
	struct atmarpreq req;
	__u32 *ip;
	int error;

	DPRINTK("atmarp_ioctl\n");
	error = verify_area(VERIFY_READ,arg,sizeof(struct atmarpreq));
	if (error) return error;
	memcpy_fromfs(&req,arg,sizeof(struct atmarpreq));
	if (req.arp_pa.sa_family != AF_INET) return -EPFNOSUPPORT;
	ip = &((struct sockaddr_in *) &req.arp_pa)->sin_addr.s_addr;
	if (!(*ip & ~dev->pa_mask) && !(req.arp_flags & (ATF_ARPSRV |
	    ATF_DEFQOS)))
		return -EINVAL;
	switch (cmd) {
		case SIOCSARP:
		case SIOCDARP:
		case SIOCGARP:
			return send_demon(act_ioctl,PRIV(dev)->number,cmd,&req,
			    sizeof(struct atmarpreq));
			/* @@@ get will need special treatment */
		default:
			return -EINVAL;
	}
	return 0;
}


static int clip_open(struct device *dev)
{
	DPRINTK("clip_open called\n");
	return 0;
}


static int clip_stop(struct device *dev)
{
	DPRINTK("clip_stop\n");
	/* @@@ just kill it on error ? */
	return 0;
}


static int clip_init(struct device *dev)
{
	DPRINTK("init %s\n",dev->name);
	dev->hard_start_xmit = clip_xmit;
	/* sg_xmit ... */
	dev->open = clip_open;
	dev->stop = clip_stop;
	ether_setup(dev);
	dev->tbusy = 0; /* @@@ check */
	dev->hard_header = clip_hard_header;
	dev->do_ioctl = NULL;
	dev->change_mtu = NULL;
	dev->ip_arp = atmarp_ioctl;
	dev->rebuild_header = clip_rebuild_header;
	dev->get_stats = atm_clip_get_stats;
	dev->hard_header_len = RFC1483LLC_LEN;
	dev->flags &= ~(IFF_BROADCAST | IFF_MULTICAST);
	dev->flags |= IFF_NOARP; /* we do our own ARP ... */
	dev->mtu = RFC1626_MTU;
	return send_demon(act_create,PRIV(dev)->number,0,NULL,0);
}


int clip_create(int number) /* remove by downing */
{
	struct device *dev;

	number = ipcom_pick_number(number);
	if (number < 0) return number;
	dev = kmalloc(sizeof(struct device)+sizeof(struct atmarp_priv),
	    GFP_KERNEL);
	if (!dev) return -ENOMEM;
	memset(dev,0,sizeof(struct device)+sizeof(struct atmarp_priv));
	dev->name = PRIV(dev)->name;
	sprintf(dev->name,"atm%d",number);
	dev->init = clip_init;
	PRIV(dev)->number = number;
	PRIV(dev)->table = NULL;
	if (register_netdev(dev)) return -EIO; /* free dev ? */
	PRIV(dev)->next = clip_devs;
	clip_devs = dev;
	DPRINTK("registered (net:%s)\n",dev->name);
	return number;
}


static int clip_device_event(struct notifier_block *this,unsigned long event,
    void *dev)
{
	/* ignore non-CLIP devices */
	if (((struct device *) dev)->init != clip_init) return NOTIFY_DONE;
	switch (event) {
		case NETDEV_UP:
			(void) send_demon(act_up,PRIV(dev)->number,0,NULL,0);
			break;
		case NETDEV_DOWN:
			DPRINTK("clip_device_event NETDEV_DOWN\n");
			(void) send_demon(act_down,PRIV(dev)->number,0,NULL,0);
			break;
		case NETDEV_REBOOT:
			/* ignore */
			break;
		default:
			printk(KERN_ERR "clip_device_event: unknown event "
			    "%ld\n",event);
			break;
	}
	return NOTIFY_DONE;
}


static struct notifier_block clip_dev_notifier = {
	clip_device_event,
	NULL,
	0
};


static void atmarpd_close(struct atm_vcc *vcc)
{
        struct sk_buff *skb;

        DPRINTK("atmarpd_close\n");
        atmarpd = NULL; /* assumed to be atomic */
	barrier();
	unregister_netdevice_notifier(&clip_dev_notifier);
	wake_up(&atmarpd_sleep);
        if (skb_peek(&vcc->recvq))
                printk(KERN_ERR "atmarpd_close: closing with requests "
		    "pending\n");
        while ((skb = skb_dequeue(&vcc->recvq))) kfree_skb(skb,FREE_READ);
	DPRINTK("(done)\n");
}


static struct atmdev_ops atmarpd_dev_ops = {
	NULL,		/* no open */
	atmarpd_close,	/* close */
	NULL,		/* no ioctl */
	NULL,		/* no getsockopt */
	NULL,		/* no setsockopt */
	atmarpd_send,	/* send */
	NULL,		/* no sg_send */
	NULL,		/* no poll */
	NULL,		/* no phy_put */
	NULL,		/* no phy_get */
	NULL		/* no feedback */
};

static struct atm_dev atmarpd_dev = {
	&atmarpd_dev_ops,
	NULL,		/* no PHY */
	"arpd",		/* type */
	999,		/* dummy device number */
	NULL,NULL,	/* pretend not to have any VCCs */
	NULL,NULL,	/* no data */
	0,		/* no flags */
	NULL,		/* no local address */
	{ 0 }		/* no ESI, no statistics */
};


int atm_init_atmarp(struct atm_vcc *vcc)
{
	if (atmarpd) return -EADDRINUSE;
	if (start_timer) {
		start_timer = 0;
		idle_timer.expires = jiffies+CLIP_CHECK_INTERVAL*HZ;
		idle_timer.function = idle_timer_check;
		add_timer(&idle_timer);
	}
	atmarpd = vcc;
	vcc->flags |= ATM_VF_READY | ATM_VF_META;
	    /* allow replies and avoid getting closed if signaling dies */
	vcc->dev = &atmarpd_dev;
	vcc->aal = ATM_AAL5; /* lie */
	vcc->push = NULL;
	vcc->peek = NULL; /* crash */
	vcc->pop = NULL; /* crash */
	vcc->push_oam = NULL; /* crash */
	register_netdevice_notifier(&clip_dev_notifier);
	return 0;
}
