/* net/atm/arequipa.c - Application requested IP over ATM */
 
/* Written 1996,1997 by Jean-Michel Pittet and Werner Almesberger, EPFL LRC */


#include <linux/config.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/errno.h>
#include <linux/skbuff.h>
#include <linux/in.h>
#include <linux/mmuio.h>
#include <linux/atmdev.h>
#include <linux/atmclip.h>
#include <linux/arequipa.h>
#include <linux/route.h>
#include <net/sock.h>
#include <netinet/in.h>
#include <asm/system.h> /* cli and such */

#include "protocols.h"
#include "tunable.h"
#include "signaling.h"  /* for indirect closing, see below */
#include "common.h"
#include "ipcommon.h"


#if 0
#define DPRINTK(format,args...) printk(KERN_DEBUG format,##args)
#else
#define DPRINTK(format,args...)
#endif


struct device *arequipa_dev = NULL;
    /* must use a different null value if skb->dev can ever be NULL */
struct atm_vcc *aqd = NULL;

struct rtable arequipa_rt = {
	NULL,		/* rt_next */
	0L,		/* rt_dst */
	0L,		/* rt_src */
	0L,		/* rt_gateway */
	2,		/* rt_refcnt */
	1,		/* rt_use */
	0,		/* rt_window */
	0L,		/* rt_lastuse */
	NULL,		/* rt_hh */
	NULL,		/* rt_dev */
	RTF_UP,		/* rt_flags */
	RFC1626_MTU,	/* rt_mtu */
	0,		/* rt_irtt */
	0		/* rt_tos */
};


static struct atm_vcc *aq_list = NULL; /* dangling Arequipa VCs */
static unsigned long aq_generation = 0;


static void arequipa_unuse(struct atm_vcc *vcc)
{
	unsigned long flags;

	save_flags(flags);
	cli();
	if (!(vcc->flags & ATM_VF_AQDANG)) {
		restore_flags(flags);
		return;
	}
	vcc->flags &= ~ATM_VF_AQDANG;
	if (vcc->aq_prev) vcc->aq_prev->aq_next = vcc->aq_next;
	else aq_list = vcc->aq_next;
	if (vcc->aq_next) vcc->aq_next->aq_prev = vcc->aq_prev;
	restore_flags(flags);
}


/*
 * Closing is tricky. Since we may be in an interrupt when executing
 * arequipa_close, we can't just go and call close_fp. So what we do it instead
 * is to ask arequipad nicely to close the VC. arequipad issues an
 * AREQUIPA_CLS3RD ioctl to close us (via arequipa_close_vcc). Now we're in a
 * process context and can sleep, etc. Ain't life sweet ?
 */


static void aqd_enq(struct atm_vcc *vcc)
{
	struct sk_buff *skb;

	if (!aqd) {
		printk(KERN_CRIT "aqd_enq: no Arequipa demon\n");
		return;
	}
	skb = alloc_skb(sizeof(vcc),GFP_ATOMIC);
	if (!skb) {
		printk(KERN_CRIT "adq_enq: out of memory\n");
		return;
	}
	skb->free = 1;
	skb->len = sizeof(vcc);
	*(struct atm_vcc **) skb->data = vcc;
	atomic_add(skb->truesize+ATM_PDU_OVHD,&aqd->rx_inuse);
	skb_queue_tail(&aqd->recvq,skb);
	wake_up(&aqd->sleep);
}


int arequipa_close(struct sock *upper)
{
	struct socket *lower;
	unsigned long flags;

	DPRINTK("arequipa_close\n");
	if (!(lower = upper->arequipa)) return -ENOTCONN;
	save_flags(flags);
	cli();
	arequipa_unuse(ATM_SD(lower));
	ip_rt_put(upper->ip_route_cache);
	upper->ip_route_cache = NULL;
	upper->arequipa = NULL;
	ATM_SD(lower)->upper = NULL;
	if (!(ATM_SD(lower)->flags & ATM_VF_AQREL)) aqd_enq(ATM_SD(lower));
	ATM_SD(lower)->flags |= ATM_VF_AQREL;
	restore_flags(flags);
	return 0;
}


void arequipa_close_vcc(struct atm_vcc *vcc)
{
	if (!(vcc->flags & ATM_VF_AQREL)) {
		printk(KERN_CRIT "arequipa_close_vcc: VCC %p doesn't "
		    "have ATM_VF_AQREL set\n",vcc);
		return;
	}
	arequipa_unuse(vcc);
	close_fp(vcc->sock->file);
}


static void arequipa_callback(struct atm_vcc *vcc)
{
	unsigned long flags;

	DPRINTK("arequipa_callback\n");
	svc_callback(vcc);
	if (!(vcc->flags & ATM_VF_RELEASED)) return;
	vcc->callback = svc_callback; /* paranoia ... */
	save_flags(flags);
	cli();
	arequipa_unuse(vcc);
	if (vcc->upper) {
		if (!vcc->upper->arequipa)
			printk("arequipa_callback: upper pretends not to "
			    "use Arequipa\n");
		ip_rt_put(vcc->upper->ip_route_cache);
		vcc->upper->ip_route_cache = NULL;
		vcc->upper->arequipa = NULL;
	}
	if (vcc->flags & ATM_VF_AQREL) {
		restore_flags(flags);
		return;
	}
	vcc->flags |= ATM_VF_AQREL;
	restore_flags(flags);
	arequipa_close_vcc(vcc);
	return;
}


static int check_aq_vcc(struct socket *lower)
{
	if (lower->ops->family != PF_ATMSVC && lower->ops->family != PF_ATMPVC)
		return -EPROTOTYPE;
	if (lower->state != SS_CONNECTED) return -ENOTCONN;
	if (ATM_SD(lower)->aal != ATM_AAL5) return -EPROTONOSUPPORT;
	return 0;
}


/*static*/ void atm_push_arequipa(struct atm_vcc *vcc,struct sk_buff *skb)
{
	if (!skb) return; /* it's okay to close Arequipa VCs */
	/*DPRINTK("arequipa push(%ld)\n",skb->len);*/
	skb->dev = arequipa_dev;
	skb->atm.generation = vcc->generation;
	ipcom_push(skb);
}


static void make_aq_vcc(struct socket *lower,int incoming)
{
	struct atm_vcc *vcc;
	unsigned long flags;

	save_flags(flags);
	cli();
	vcc = ATM_SD(lower);
	vcc->pop = atm_pop_clip;
	vcc->callback = arequipa_callback;
	vcc->push = atm_push_arequipa;
	vcc->peek = atm_peek_clip;
	vcc->push_oam = NULL;
	if (incoming) {
	    vcc->flags |= ATM_VF_AQDANG;
	    vcc->aq_next = aq_list;
	    vcc->aq_prev = NULL;
	    if (aq_list) aq_list->aq_prev = vcc;
	    aq_list = vcc;
	}
	vcc->generation = aq_generation++;
	restore_flags(flags);
	lower->file->f_count++;
}


static int arequipa_attach_unchecked(struct socket *lower,struct sock *upper)
{
	unsigned long flags;
	struct rtable *rt;
	int error;

	if (upper->arequipa) {
		printk(KERN_WARNING "arequipa_attach_unchecked: upper already "
		    "uses Arequipa\n");
		return -EISCONN;
	}
	error = check_aq_vcc(lower);
	if (error) return error;
	save_flags(flags);
	cli();
	if (ATM_SD(lower)->upper) {
		restore_flags(flags);
		printk(KERN_WARNING "arequipa_attach_unchecked: lower is "
		    "already attached\n");
		return -EISCONN;
	}
	DPRINTK("arequipa_attach_unchecked %p (i_count=%d,f_count=%d)\n",upper,
	  lower->inode->i_count,lower->file->f_count);
	upper->arequipa = lower;
	ATM_SD(lower)->upper = upper;
	rt = upper->ip_route_cache; /* revalidate cache */
	upper->ip_route_cache = NULL;
	restore_flags(flags);
	set_rt_cache(upper,rt);
	/*
	 * The next statement violates RFC1122, because it may change MSS when
	 * both sides have already exchanged their SYNs. Linux doesn't mind if
	 * this happens, but other systems might. Needs to be fixed. @@@
	 */
	if (ATM_SD(lower)->qos.txtp.max_sdu > RFC1483LLC_LEN)
		upper->mtu = ATM_SD(lower)->qos.txtp.max_sdu-RFC1483LLC_LEN;
	return 0;
}


int arequipa_attach(struct socket *lower,struct sock *upper,
    unsigned long generation)
{
	unsigned long flags;
	struct atm_vcc *walk;

	save_flags(flags);
	cli();
	for (walk = aq_list; walk; walk = walk->aq_next)
		if (walk == ATM_SD(lower)) break;
	restore_flags(flags);
	if (walk && walk->generation == generation)
		return arequipa_attach_unchecked(lower,upper);
	printk(KERN_DEBUG "arequipa_attach: avoided close/attach race\n");
	return -ENOTCONN;
}


int arequipa_expect(struct sock *upper,int on)
{
	DPRINTK("arequipa_expect %d\n",on);
	if (!aqd) {
		printk(KERN_ERR "arequipa_expect: no Arequipa demon\n");
		return -EUNATCH;
	}
	if (on) {
		if (upper->aq_route) return 0;
		upper->aq_route = kmalloc(sizeof(struct rtable),GFP_KERNEL);
		return upper->aq_route ? 0 : -ENOMEM;
	}
	if (!upper->aq_route) return 0;
	if (upper->arequipa) return -EBUSY;
	kfree(upper->aq_route);
	upper->aq_route = NULL;
	return 0;
}


int arequipa_preset(struct socket *lower,struct sock *upper)
{
	unsigned long flags;
	int error;

	if (upper->state == TCP_LISTEN) return -EPROTO;
	if (!aqd) {
		printk(KERN_ERR "arequipa_preset: no Arequipa demon\n");
		return -EUNATCH;
	}
	error = arequipa_expect(upper,1);
	if (error) return error;
	save_flags(flags);
	cli();
	error = arequipa_attach_unchecked(lower,upper);
	if (!error) make_aq_vcc(lower,0);
	restore_flags(flags);
	return error;
}


int arequipa_incoming(struct socket *lower)
{
	int error;

	if (!suser()) return -EPERM;
	error = check_aq_vcc(lower);
	if (error) return error;
	ATM_SD(lower)->upper = NULL;
	make_aq_vcc(lower,1);
	DPRINTK("aq_incoming %d\n",lower->file->f_count);
	return 0;
}


static int arequipa_xmit(struct sk_buff *skb,struct device *dev)
{
	struct atm_vcc *vcc;

	/*DPRINTK("arequipa xmit\n");*/
	if (!skb->sk || !skb->sk->arequipa ||
	    !ATM_SD(skb->sk->arequipa)) {
		printk("arequipa_xmit: discarding orphaned packets\n");
	        dev_kfree_skb(skb,FREE_WRITE);
		return 0;
	}
	vcc = ATM_SD(skb->sk->arequipa);
	if (!(vcc->flags & ATM_VF_READY)) {
		printk("arequipa_xmit: not ready\n");
	        dev_kfree_skb(skb,FREE_WRITE);
		return 0;
	}
	ipcom_xmit(arequipa_dev,vcc,skb);
	CLIP(arequipa_dev)->stats.tx_packets++;
	return 0;
}


static int arequipa_init(struct device *dev)
{
	ipcom_init(dev,arequipa_xmit,IFF_UP);
	dev->pa_addr = 0x01020304;
	dev->pa_mask = ~0L;
	dev->pa_alen = 4;
	return 0;
}


int atm_init_arequipa(void)
{
	DPRINTK("atm_init_arequipa\n");
	if (!suser()) return -EPERM;
	arequipa_dev = kmalloc(sizeof(struct device)+sizeof(struct clip_priv),
	    GFP_KERNEL);
	if (!arequipa_dev) return -ENOMEM;
	arequipa_rt.rt_dev = arequipa_dev;
	memset(arequipa_dev,0,sizeof(struct device)+sizeof(struct clip_priv));
	arequipa_dev->name = "arequipa";
	arequipa_dev->init = arequipa_init;
	arequipa_rt.rt_dev = arequipa_dev;
	arequipa_init(arequipa_dev);
	return 0;
}


static void aqd_close(struct atm_vcc *vcc)
{
        struct sk_buff *skb;

        DPRINTK("aqd_close\n");
        aqd = NULL; /* assumed to be atomic */
	barrier();
        if (skb_peek(&vcc->recvq))
                printk(KERN_CRIT "aqd_close: closing with requests "
		    "pending\n");
        while ((skb = skb_dequeue(&vcc->recvq))) kfree_skb(skb,FREE_READ);
}


static struct atmdev_ops aqd_dev_ops = {
	NULL,		/* no open */
	aqd_close,	/* close */
	NULL,		/* no ioctl */
	NULL,		/* no getsockopt */
	NULL,		/* no setsockopt */
	NULL,		/* no send */
	NULL,		/* no sg_send */
	NULL,		/* no poll */
	NULL,		/* no phy_put */
	NULL,		/* no phy_get */
	NULL		/* no feedback */
};


static struct atm_dev aqd_dev = {
	&aqd_dev_ops,
	NULL,		/* no PHY */
	"aqd",		/* type */
	999,		/* dummy device number */
	NULL,NULL,	/* pretend not to have any VCCs */
	NULL,NULL,	/* no data */
	0,		/* no flags */
	NULL,		/* no local address */
	{ 0 }		/* no ESI, no statistics */
};


int arequipad_attach(struct atm_vcc *vcc)
{
	if (aqd) return -EADDRINUSE;
	aqd = vcc;
	vcc->flags |= ATM_VF_READY | ATM_VF_META;
	    /* allow replies and avoid getting closed if signaling dies */
	vcc->dev = &aqd_dev;
	vcc->aal = ATM_AAL5; /* lie */
	vcc->push = NULL;
	vcc->peek = NULL; /* crash */
	vcc->pop = NULL; /* crash */
	vcc->push_oam = NULL; /* crash */
	/*
	 * Now that we have an arequipad, we should check for stale Arequipa
	 * connections and prune them. @@@
	 */
        return 0;

}
