/*
 * DRET-IPv6	An implementation of the TCP/IP protocol suite for the LINUX
 *		operating system.  INET6 is implemented using the BSD Socket
 *		interface as the means of communication with the user level.
 *
 * Contacts:	
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id: in6.h,v 1.5 1997/03/14 19:51:38 pan Exp $ 
 *	
 * Authors:	
 *	        name	<E_mail>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *		Portions of the software are derived from  various 
 *		networking code publicly available, mainly:
 *			NRL IPv6 code 
 *			NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *			Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */				
#ifndef _LINUX_IN6_H
#define _LINUX_IN6_H

#include <linux/config.h>

#ifdef CONFIG_NET_IPV6_DIM

/* IPv6 specific defines, macros and structures.
   This file is included from in.h */

struct in6_addr {
        union {
                __u8  u6_addr8[16];
                __u16 u6_addr16[8]; 
		__u32 u6_addr32[4];
		/*		__u64 u6_addr64[2];*/
        } u6_addr;
#define s6_addr         u6_addr.u6_addr8
#define s6_addr8        u6_addr.u6_addr8
#define s6_addr16       u6_addr.u6_addr16
#define s6_addr32       u6_addr.u6_addr32
};

struct sockaddr_in6
{
        __u16 sin6_family;
	__u16 sin6_port;
	__u32 sin6_flowinfo;
	struct in6_addr sin6_addr;
};

struct ipv6_mreq {
	/* IPv6 multicast address of group  */
	struct in6_addr ipv6mr_multiaddr;
	
	/* index of source interface */ 
	unsigned int ipv6mr_interface;
};


/*
 * Well defined address
 */
 
extern const struct in6_addr in6addr_any;
#define IN6ADDR_ANY_INIT  {{{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}}}
 
extern const struct in6_addr in6addr_loopback;
#define IN6ADDR_LOOPBACK_INIT  {{{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1}}}

#define		LOOPBACK4_IPV6		__constant_htonl(0x7f000006)


/* 
 * Used by cmsghdr
 */

struct in6_pktinfo {
	unsigned int    ipi6_ifindex; /* interface index */
	struct in6_addr ipi6_addr;    /* IPv6 address */
};


struct opt6_hdr {
   __u8   oh_nexthdr;
   __u8   oh_len;
   __u8   oh_type;
};
/*
 * Constantes definitions
 */
 

/* IPv6 Next Headers values */
enum {
        IPPROTO_HOPOPTS = 0,          /* IPv6 hop-by-hop options */
	IPPROTO_ROUTING = 43,         /* IPv6 routing header */
	IPPROTO_FRAGMENT = 44,        /* IPv6 fragmentation header */
	IPPROTO_ESP = 50,             /* encapsulating security payload */
	IPPROTO_AH = 51,              /* authentication header */
	IPPROTO_NONE = 59,            /* IPv6 no next header */
	IPPROTO_DSTOPTS = 60          /* IPv6 destination options */
};

/*
 * Definitions for flow labels.
 */

#define IPV6_FLOWINFO_FLOWLABEL         __constant_htonl(0x00ffffff)
#define IPV6_FLOWINFO_PRIORITY          __constant_htonl(0x0f000000)
#define IPV6_FLOWINFO_PRIFLOW           __constant_htonl(0x0fffffff)
#define IPV6_FLOWINFO_SRFLAG            __constant_htonl(0x10000000)
#define IPV6_FLOWINFO_VERSION           __constant_htonl(0xf0000000)

#define IPV6_PRIORITY_UNCHARACTERIZED   __constant_htonl(0x00000000)
#define IPV6_PRIORITY_FILLER            __constant_htonl(0x01000000)
#define IPV6_PRIORITY_UNATTENDED        __constant_htonl(0x02000000)
#define IPV6_PRIORITY_RESERVED1         __constant_htonl(0x03000000)
#define IPV6_PRIORITY_BULK              __constant_htonl(0x04000000)
#define IPV6_PRIORITY_RESERVED2         __constant_htonl(0x05000000)
#define IPV6_PRIORITY_INTERACTIVE       __constant_htonl(0x06000000)
#define IPV6_PRIORITY_CONTROL           __constant_htonl(0x07000000)
#define IPV6_PRIORITY_8                 __constant_htonl(0x08000000)
#define IPV6_PRIORITY_9                 __constant_htonl(0x09000000)
#define IPV6_PRIORITY_10                __constant_htonl(0x0a000000)
#define IPV6_PRIORITY_11                __constant_htonl(0x0b000000)
#define IPV6_PRIORITY_12                __constant_htonl(0x0c000000)
#define IPV6_PRIORITY_13                __constant_htonl(0x0d000000)
#define IPV6_PRIORITY_14                __constant_htonl(0x0e000000)
#define IPV6_PRIORITY_15                __constant_htonl(0x0f000000)

#define IPV6_SRFLAG_STRICT              __constant_htonl(0x10000000)
#define IPV6_SRFLAG_LOOSE               __constant_htonl(0x00000000)

#define IPV6_VERSION                    __constant_htonl(0x60000000)

#define IPV6_GET_FLOWLABEL(x)           (ntohl(x) & 0x00ffffff)
#define IPV6_GET_PRIORITY(x)            ((ntohl(x) >> 24) & 0xf)
#define IPV6_GET_VERSION(x)             ((ntohl(x) >> 28) & 0xf)
#define IPV6_SET_FLOWLABEL(x)           (htonl(x) & IPV6_FLOWINFO_FLOWLABEL)
#define IPV6_SET_PRIORITY(x)            (htonl((x & 0xf) << 24))

/* IPv6 constantes */

#define INET_ADDRSTRLEN		16
#define INET6_ADDRSTRLEN	46

#define IPV6_UNICAST_HOPS      IP_TTL
#define IPV6_MULTICAST_IF      IP_MULTICAST_IF
#define IPV6_MULTICAST_LOOP    IP_MULTICAST_LOOP
#define IPV6_MULTICAST_HOPS    IP_MULTICAST_TTL
#define IPV6_DEFAULT_HOP_LIMIT 		64 
#define IPV6_DEFAULT_MULTICAST_HOPS	1
#define IPV6_DEFAULT_MULTICAST_LOOP	1

#define IPV6_ADD_MEMBERSHIP    IP_ADD_MEMBERSHIP
#define IPV6_DROP_MEMBERSHIP   IP_DROP_MEMBERSHIP
#define IPV6_ADDRFORM          22  /* It is the value supposing of IP_ADDRFORM, 
				      but no finding here */
#define IPV6_HDRINCL           IP_HDRINCL
#define IPV6_PKTINFO  23
#define IPV6_HOPLIMIT 24
#define IPV6_NEXTHOP  25
#define IPV6_HOPOPTS  29
   #define PAD0_HOPOPT 0
   #define PADN_HOPOPT 1
   #define JUMBO_HOPOPT 194
#define IPV6_CHECKSUM 26   
#define IPV6_DSTOPTS  27
#define IPV6_SRCRT    28
   #define IPV6_SRCRT_TYPE_0 0


#define IPV6_RECVSRCRT         30
 


/* Constantes for Multicast Address */

#define MADDR6_FLG_WK           0       /* "well-known" multicast address */
#define MADDR6_FLG_TS           1       /* "transient" multicast address */
 
#define MADDR6_SCP_NODE         0x1     /* node-local scope */
#define MADDR6_SCP_LINK         0x2     /* link-local scope */
#define MADDR6_SCP_SITE         0x5     /* site-local scope */
#define MADDR6_SCP_ORG          0x8     /* organization-local scope */
#define MADDR6_SCP_GLO          0xe     /* global scope */
 
#define MADDR6_ALLNODES         1       /* all-nodes groups */
#define MADDR6_ALLROUTERS       2       /* all-routers groups */
#define MADDR6_ALLHOSTS         3       /* all-hosts groups*/ 

/*
 * Macros for IPv6 addresses.
 */
 
#define CLR_ADDR6(a) \
        { \
                (a).s6_addr32[0] = 0; \
                (a).s6_addr32[1] = 0; \
                (a).s6_addr32[2] = 0; \
                (a).s6_addr32[3] = 0; \
        }
 
#define COPY_ADDR6(to, from) \
        { \
                (to).s6_addr32[0] = (from).s6_addr32[0]; \
                (to).s6_addr32[1] = (from).s6_addr32[1]; \
                (to).s6_addr32[2] = (from).s6_addr32[2]; \
                (to).s6_addr32[3] = (from).s6_addr32[3]; \
        }

#define SAME_ADDR6(a, b) \
        (((a).s6_addr32[0] == (b).s6_addr32[0]) && \
         ((a).s6_addr32[1] == (b).s6_addr32[1]) && \
         ((a).s6_addr32[2] == (b).s6_addr32[2]) && \
         ((a).s6_addr32[3] == (b).s6_addr32[3]))
 
#define IS_ANYADDR6(a) \
        (((a).s6_addr32[0] == 0) && \
         ((a).s6_addr32[1] == 0) && \
         ((a).s6_addr32[2] == 0) && \
         ((a).s6_addr32[3] == 0))

#define MULTISOL_ADDR6(a, b) \
        { \
                (a).s6_addr32[0] = __constant_htonl(0xff020000); \
                (a).s6_addr32[1] = 0                           ; \
                (a).s6_addr32[2] = __constant_htonl(0x1)       ; \
                (a).s6_addr32[3] = (b).s6_addr32[3]           ; \
        }

#define IS_LOOPADDR6(a) \
        (((a).s6_addr32[0] == 0) && \
         ((a).s6_addr32[1] == 0) && \
         ((a).s6_addr32[2] == 0) && \
         ((a).s6_addr32[3] == __constant_htonl(0x1)))

#define IS_LOOPSOCKADDR(a) IS_LOOPADDR6((a)->sin6_addr) 

#define IS_IPV4ADDR6(a) \
        (((a).s6_addr32[0] == 0) && \
         ((a).s6_addr32[1] == 0) && \
         ((a).s6_addr32[2] == 0) && \
         ((a).s6_addr32[3] != 0))

#define IS_IPV4SOCKADDR(a) IS_IPV4ADDR6((a)->sin6_addr)

#define IS_LOCALADDR6(a) ((a).s6_addr8[0] == 0xfe)
 
#define IS_LINKLADDR6(a) \
        (IS_LOCALADDR6(a) && ((a).s6_addr8[1] == 0x80))
 
#define IS_SITELADDR6(a) \
        (IS_LOCALADDR6(a) && ((a).s6_addr8[1] == 0xc0))
 
#define IS_MULTIADDR6(a)    ((a).s6_addr8[0] == 0xff)

#define IS_MAPPEDADDR6(a)  ((a).s6_addr32[0] == 0) && \
	                   ((a).s6_addr32[1] == 0) && \
	                   ((a).s6_addr32[2] == __constant_htonl(0x0000FFFF))
			    
#define IS_MULTI_LINK(a)   ((a).s6_addr32[0]== __constant_htonl(0xFF020000)&& \
                             (a).s6_addr32[1]== 0 && \
                             (a).s6_addr32[2]== 0 && \
                             (a).s6_addr32[3]== __constant_htonl(0x00000001))
#define IS_MULTI_HOST(a)   ((a).s6_addr32[0]== __constant_htonl(0xFF010000)&& \
                             (a).s6_addr32[1]== 0 && \
                             (a).s6_addr32[2]== 0 && \
                             (a).s6_addr32[3]== __constant_htonl(0x00000001))
 
#define MADDR6_FLAGS(a)         ((a).s6_addr8[1] >> 4)
 

#define MADDR6_SCOPE(a)         ((a).s6_addr8[1] & 0x0f)

	   
/* 
 * Advanced APIs: Source routing
 */
#ifndef __KERNEL__
extern unsigned short  inet6_srcrt_space(int type, int segments);

extern struct cmsghdr  *inet6_srcrt_init(void *bp, int type);

extern int             inet6_srcrt_add(struct cmsghdr *cmsg,
				      const struct in6_addr *addr, 
				      unsigned int flags);

extern int             inet6_srcrt_lasthop(struct cmsghdr *cmsg,
					  unsigned int flags);

extern int             inet6_srcrt_reverse(const struct cmsghdr *in, 
					  struct cmsghdr *out);

extern int             inet6_srcrt_segments(const struct cmsghdr *cmsg);

extern struct in6_addr *inet6_srcrt_getaddr(struct cmsghdr *cmsg, int index);

extern int             inet6_srcrt_getflags(struct cmsghdr *cmsg, int offset);

#endif

/* FIXME  must be remplaced by macro  Do not use any more*/

#ifdef __KERNEL__    
/* some adress types for get_ipv6_addr_type() */
enum {
	IPV6_ADDR_UNKNOWN = 0,          /* none of the above */
	IPV6_ADDR_UNICAST,
	IPV6_ADDR_MULTICAST,
	IPV6_ADDR_ANYCAST,		/* Anycast address	        */
	IPV6_ADDR_COMPATV4,	
	IPV6_ADDR_LINKLOCAL,	        
	IPV6_ADDR_SITELOCAL,	
	IPV6_ADDR_ANY,			/* Unspecified address (eg ::0) */
	IPV6_ADDR_LOOPBACK              /* Loopback (::1)               */
};

extern inline int get_ipv6_addr_type(struct in6_addr *addr)
{
	__u32 st = addr->s6_addr32[0];

	/* 
	 * UCast Provider Based Address
	 * 0x4/3
	 */

	if ((st & __constant_htonl(0xE0000000)) == 
	    __constant_htonl(0x40000000))
	{
		return IPV6_ADDR_UNICAST;
	}

	if ((st & __constant_htonl(0xFF000000)) == 
	    __constant_htonl(0xFF000000))
	{
		return IPV6_ADDR_MULTICAST;
	}
	
	if ((st & __constant_htonl(0xFFC00000)) == 
	    __constant_htonl(0xFE800000))
	{
		return IPV6_ADDR_LINKLOCAL;
	}

	if ((st & __constant_htonl(0xFFC00000)) == 
	    __constant_htonl(0xFEC00000))
	{
		return IPV6_ADDR_SITELOCAL;
	}

	if ((st |
	    addr->s6_addr32[1] |
	    addr->s6_addr32[2]) == 0)
	{
		if (addr->s6_addr32[3] == 0)
		{
			return IPV6_ADDR_ANY;
		}
		
		if (addr->s6_addr32[3] == __constant_htonl(0x00000001))
		{
			return IPV6_ADDR_LOOPBACK;
		}

		return IPV6_ADDR_COMPATV4;
	}

	return IPV6_ADDR_UNKNOWN;      
}
#endif

#endif /* CONFIG_NET_IPV6_DIM */

#endif /* _LINUX_IN6_H */
