/*
 * DRET-IPv6	An implementation of the TCP/IP protocol suite for the LINUX
 *		operating system.  INET6 is implemented using the BSD Socket
 *		interface as the means of communication with the user level.
 *
 * Contacts:	
 *              INRIA      <Christophe.Diot@sophia.inria.fr>
 *              MASI       <Eric.Horlait@masi.ibp.fr>
 *              This software has been developped with the financial support
 *              of DRET (French Military Research Agency).
 *
 * Version:    $Id: nd.h,v 1.3 1997/03/21 20:38:28 pan Exp $ 
 *	
 * Authors:	
 *	        name	<E_mail>
 *
 *
 * Fixes:
 *
 *
 * Description:
 *
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *              This notice must be copied with the distributed package.
 *
 *		Portions of the software are derived from  various 
 *		networking code publicly available, mainly:
 *			NRL IPv6 code 
 *			NetBSD code from INRIA (Francis.Dupont@inria.fr)
 *			Pedro Roque's Linux version (Roque@di.fc.up.pt)
 *
 */				
#ifndef _LINUX_ND_H
#define _LINUX_ND_H

#include <linux/config.h>

#ifdef CONFIG_NET_IPV6_DIM

#include <asm/atomic.h>
#include <linux/skbuff.h>
#include <linux/dret/in6.h>
#include <linux/dret/icmpv6.h>

#define MAX_HADDR_LEN 16

/*------------------------------------------------------------
   Neighbor discovery uses four separate caches:
      -(1) The destination cache:
         Destination IPv6 addr -> Neighbor used IPv6 addr
      -(2) The neighbor cache:
         Neighbor IPv6 addr -> Neighbor media addr
      -(3) The prefix list
         Prefix recently learned through advertisements.
      -(4) The router list
         Routers IPv6 addresses.
-------------------------------------------------------------*/
extern atomic_t	nd_lock;
extern unsigned nd_bh_mask;

#define DCE_INVALID 0x1

/*-------------------------------------------------------------------
  
              DATA STRUCTURES

--------------------------------------------------------------------*/ 
struct rt_neigh {
       struct rt_neigh       *parent;
       struct rt_neigh       *left;
       struct rt_neigh       *right;

       struct neighbor       *leaf;

       __u16                 rn_bit;     
       __u16                 rn_flags;
};
extern struct rt_neigh nd_root;

struct neighbor {
        struct in6_addr         naddr;       /* neighbor addr */
        __u8                    len;         /* prefix len    */
        __u8                    type;        /* {unicast, multicast} */
                        
        struct device *         dev;         /* device used to reach n */

        __u8                    flags;

        char                    hh_data[MAX_HADDR_LEN];
        char                    *h_dest;  
        struct sk_buff_head     queue; 
        atomic_t                refcnt;
             
        __u8                    state;
        __u8                    probes;
        __u32                   tstamp;
        __u32                   expires;  
        struct neighbor         *next;
        struct rt_neigh         *main_node;
};


 
/*-------------------------------------------------
       Prefix list element
-------------------------------------------------*/
/*
  struct prefix {
   struct pfxf {
#if defined(__BIG_ENDIAN_BITFIELD)
      __u8   AdvOnLinkFlag     :1,
            AdvAutonomousFlag :1,
            Unused            :6;
#elif defined(__LITTLE_ENDIAN_BITFIELD)
      __u8  Unused            :6,
	    AdvAutonomousFlag :1,
	    AdvOnLinkFlag     :1;
#endif
   } flags;
   unsigned long  AdvValidLifetime;
   unsigned long  AdvPreferredLifetime;
};
*/

/*-------------------------------------------------
       Router list element   
--------------------------------------------------*/
struct router {
};

/*------------------------------------------------------------------------

                       MESSAGE HEADERS

---------------------------------------------------------------------------*/
/*--------------------------------------
  Neighbor advertisement, solicitation
---------------------------------------*/
struct n_adv_sol {
   struct icmpv6_hdr hdr;
   union  {
      __u32 reserved;
      struct u_adv {
#if defined(__BIG_ENDIAN_BITFIELD)
         __u32     router:1,
                   sollicited:1,
                   override:1,
                   reserved:29;
#elif defined(__LITTLE_ENDIAN_BITFIELD)
         __u32     reserved1:5,
                   router:1,
                   solicited:1,
                   override:1,
                   reserved:24;
#endif
      } n_adv;
   } u2;
   struct in6_addr target;
};

/*------------------------------------
   Router solicitation
--------------------------------------*/
struct r_sol {
   struct icmpv6_hdr hdr;
   __u32  reserved;
};

/*------------------------------------
   Router Advertisment
-------------------------------------*/
struct r_adv {
   struct icmpv6_hdr hdr;
   __u8 hop_limit;
   struct r_flgs {
#if defined(__BIG_ENDIAN_BITFIELD)
         __u8     managed:1,
                  other:1,
                  reserved:6;
#elif defined(__LITTLE_ENDIAN_BITFIELD)
         __u8      reserved:6,
                   other:1,
                   managed:1;
#endif
      } flags;
   __u16 lifetime;
   __u32 reachable_t;
   __u32 retrans_t;
};

/*--------------------------------------
   tlla/slla options
---------------------------------------*/
struct nd_opt {
   __u8   type;
   __u8   len;
   __u8   lla[MAX_ADDR_LEN];
};



/*---------------------------------------
   pfx opt
-----------------------------------------*/
struct p_flgs {
#if defined(__BIG_ENDIAN_BITFIELD)
         __u8      onlink   : 1,
                   auto1     : 1,
                   reserved : 6;
#elif defined(__LITTLE_ENDIAN_BITFIELD)
         __u8      reserved : 6, 
	           auto1     : 1, 
	           onlink   : 1;
#endif
   };

struct mtu_opt {
   __u8            type;
   __u8            len;
   __u16           reserved;
   __u32           mtu;
};

struct pfx_opt {
   __u8            type;
   __u8            len;
   __u8            plen;

   struct p_flgs  flags;

   __u32           valid_lt;
   __u32           pref_lt;
   __u32           reserved2;
   struct in6_addr prefix;
}; 

struct r_red {
   struct icmpv6_hdr hdr;
   __u32        reserved;
   struct in6_addr tgt;
   struct in6_addr addr;
};

/*-------------------------------------------------------------------------

                       CONSTANTS

---------------------------------------------------------------------------*/
/*------------------------------------------
       Neighbor Cache Entry States (7.3.2.)
---------------------------------------------*/
#define ND_INCOMPLETE  0x11
#define ND_REACHABLE   0x02
#define ND_STALE       0x04
#define ND_DELAY       0x09
#define ND_PROBE       0x21

#define ND_TIMED       0x01

/*------------------------------------------
       Neighbor flags 
--------------------------------------------*/
#define NCF_NOARP              0x01    /* no ARP needed on this device */
#define NDF_NOARP              NCF_NOARP
#define NCF_SUBNET             0x02    /* NC entry for subnet          */
#define NCF_INVALID            0x04
#define NCF_DELAY_EXPIRED      0x08    /* time to move to PROBE        */
#define NCF_ROUTER             0x10    /* neighbour is a router        */
#define NDF_ISROUTER           NCF_ROUTER
#define NCF_HHVALID            0x20    /* Hardware header is valid     */


/*-----------------------------------------------
   ICMP codes for neighbor discovery messages
--------------------------------------------------*/
#define ND_ROUTER_SOLICITATION        133
#define ND_ROUTER_ADVERTISEMENT       134
#define ND_NEIGHBOR_SOLICITATION      135
#define ND_NEIGHBOR_ADVERTISEMENT     136
#define ND_REDIRECT                   137



#define REACHABLE_TIME              30*HZ
#define RETRANSTIMER                 2*HZ


/*------------------------------------------------
 Node constants (10.) 
--------------------------------------------------*/

#define MAX_MULTICAST_SOLICIT       3
#define MAX_UNICAST_SOLICIT         3

#define MAX_ANYCAST_DELAY_TIME      HZ

#define MAX_NEIGHBOR_ADVERTISEMENT  3

#define REACHABLE_TIME              30*HZ
#define RETRANS_TIMER               HZ
#define DELAY_FIRST_PROBE_TIME      5*HZ

#define MIN_RANDOM_FACTOR           .5
#define MAX_RANDOM_FACTOR           1.5


#define MAX_ND_QLEN                 5

/*
#define MAX_RTR_SOLICITATIONS
#define RTR_SOLICITATION_INTERVAL
#define MAX_RTR_SOLICITATION_DELAY
*/
/*-----------------------------------------------
 Router constants -> sys/sbin/ndpd-router/ndpd.h
-------------------------------------------------*/
/*
#define MAX_INITIAL_RTR_ADVERTISEMENTS
#define MAX_INITIAL_RTR_ADVERT_INTERVAL
#define MAX_FINAL_RTR_ADVERTISEMENTS
#define MAX_RA_DELAY_TIME
#define MIN_DELAY_BETWEEN_RAS
*/
/*-----------------------------------------------
  ND Options
------------------------------------------------*/
#define ND_OPT_MAXNUM        5
#define ND_OPT_SLLA          1
#define ND_OPT_TLLA          2
#define ND_OPT_PREFIX_INFO   3
#define ND_OPT_REDIRECT      4
#define ND_OPT_MTU           5

/*----------------------------------------------
  ND prefix flags
------------------------------------------------*/
#define ND_PREF_FLG_L 0x40
#define ND_PREF_FLG_A 0x80


#endif

#endif /* _LINUX_ND_H_ */


