/* atmdev.h - ATM device driver declarations */
 
/* Written 1995,1996 by Werner Almesberger, EPFL LRC */
 

#ifndef LINUX_ATMDEV_H
#define LINUX_ATMDEV_H


#include <linux/config.h>
#include <linux/uio.h>
#include <linux/atmioc.h>
#include <asm/atomic.h>


#define MAX_ATM_ITF	10	/* for now, should be lowered */
#define MAX_ATM_VCC	128	/* for now, should be dynamic */

#define ESI_LEN		6

#define ATM_OC3_PCR	(155520000/270*260/8/53)
			/* OC3 link rate:  155520000 bps
			   SONET overhead: /270*260 (9 section, 1 path)
			   bits per cell:  /8/53
			   max cell rate:  353207.547 cells/sec */

#define ATM_SD(s)	((struct atm_vcc *) ((s)->data))


struct atm_aal_stats {
	long		tx,tx_err;	/* TX okay and errors */
	long		rx,rx_err;	/* RX okay and errors */
	long		rx_drop;	/* RX out of memory */
};


struct atm_dev_stats {
	struct atm_aal_stats aal0;
	struct atm_aal_stats aal34;
	struct atm_aal_stats aal5;
};


#define ATM_GETNAMES	_IOW('a',ATMIOC_ITF+3,struct atm_iobuf)
					/* get interface names (numbers) */
#define ATM_GETTYPE	_IOW('a',ATMIOC_ITF+4,struct atmif_sioc)
					/* get interface type name */
#define ATM_GETESI	_IOW('a',ATMIOC_ITF+5,struct atmif_sioc)
					/* get interface ESI */
#define ATM_GETADDR	_IOW('a',ATMIOC_ITF+6,struct atmif_sioc)
					/* get itf's local ATM addr. list */
#define ATM_RSTADDR	_IOW('a',ATMIOC_ITF+7,struct atmif_sioc)
					/* reset itf's ATM address list */
#define ATM_ADDADDR	_IOW('a',ATMIOC_ITF+8,struct atmif_sioc)
					/* add a local ATM address */
#define ATM_DELADDR	_IOW('a',ATMIOC_ITF+9,struct atmif_sioc)
					/* remove a local ATM address */
#define ATM_GETCIRANGE	_IOW('a',ATMIOC_ITF+10,struct atmif_sioc)
					/* get connection identifier range */
#define ATM_SETCIRANGE	_IOW('a',ATMIOC_ITF+11,struct atmif_sioc)
					/* set connection identifier range */
#define ATM_GETSTAT	_IOW('a',ATMIOC_SARCOM+0,struct atmif_sioc)
					/* get AAL layer statistics */
#define ATM_GETSTATZ	_IOW('a',ATMIOC_SARCOM+1,struct atmif_sioc)
					/* get AAL layer statistics and zero */
#define ATM_SETSC	_IOW('a',ATMIOC_SPECIAL+1,int)
					/* enable or disable single-copy */

/* for ATM_GETTYPE */
#define ATM_ITFTYP_LEN	8	/* maximum length of interface type name */



struct atm_iobuf {
	int length;
	void *buffer;
};

/* for ATM_GETCIRANGE / ATM_SETCIRANGE */

#define ATM_CI_MAX      -1              /* use maximum range of VPI/VCI */
 
struct atm_cirange {
	char	vpi_bits;		/* 1..8, ATM_CI_MAX (-1) for maximum */
	char	vci_bits;		/* 1..16, ATM_CI_MAX (-1) for maximum */
};

/* for ATM_SETSC; actually taken from the ATM_VF number space */

#define ATM_SC_RX	1024		/* enable RX single-copy */
#define ATM_SC_TX	2048		/* enable TX single-copy */

#define ATM_BACKLOG_DEFAULT 32 /* if we get more, we're likely to time out
				  anyway */

#ifdef __KERNEL__

#include <linux/sched.h> /* struct wait_queue */
#include <linux/time.h> /* struct timeval */
#include <linux/net.h>
#include <linux/skbuff.h> /* struct sk_buff */
#include <linux/atm.h>

#ifdef CONFIG_AREQUIPA
#include <net/sock.h> /* for struct sock */
#endif


#define ATM_VF_ADDR	1	/* Address is in use. Set by anybody, cleared
				   by device driver. */
#define ATM_VF_READY	2	/* VC is ready to transfer data. Set by device
				   driver, cleared by anybody. */
#define ATM_VF_PARTIAL	4	/* resources are bound to PVC (partial PVC
				   setup), controlled by socket layer */
#define ATM_VF_BOUND	4	/* local SAP is set, controlled by SVC socket
				   layer */
#define ATM_VF_REGIS	8	/* registered with demon, controlled by SVC
				   socket layer */
#define ATM_VF_RELEASED 16	/* demon has indicated/requested release,
				   controlled by SVC socket layer */
#define ATM_VF_HASQOS	32	/* QOS parameters have been set */
#define ATM_VF_LISTEN	64	/* socket is used for listening */
#define ATM_VF_META	128	/* SVC socket isn't used for normal data
				   traffic and doesn't depend on signaling
				   to be available */
#define ATM_VF_AQREL	256	/* Arequipa VC is being released */
#define ATM_VF_AQDANG	512	/* VC is in Arequipa's dangling list */
#define ATM_VF_SCRX	ATM_SC_RX /* 1024; allow single-copy in the RX dir. */
#define ATM_VF_SCTX	ATM_SC_TX /* 2048; allow single-copy in the TX dir. */


struct atm_vcc {
	unsigned short	flags;		/* VCC flags (ATM_VF_*) */
	unsigned char	family;		/* address family; 0 if unused */
	unsigned char	aal;		/* ATM Adaption Layer */
	short		vpi;		/* VPI and VCI (types must be equal */
					/* with sockaddr) */
	int 		vci;
	unsigned long	aal_options;	/* AAL layer options */
	unsigned long	atm_options;	/* ATM layer options */
	struct atm_dev	*dev;		/* device back pointer */
	struct atm_qos	qos;		/* QOS */
	unsigned long	tx_quota,rx_quota; /* buffer quotas */
	atomic_t	tx_inuse,rx_inuse; /* buffer space in use */
	void (*push)(struct atm_vcc *vcc,struct sk_buff *skb);
	void (*pop)(struct atm_vcc *vcc,struct sk_buff *skb); /* optional */
	struct sk_buff *(*peek)(struct atm_vcc *vcc,unsigned long pdu_size,
	    __u32 (*fetch)(struct atm_vcc *vcc,int i));
					/* super-efficient xfers; note that */
					/* PDU_SIZE may be rounded */
	struct sk_buff *(*alloc_tx)(struct atm_vcc *vcc,unsigned int size);
					/* TX allocation routine - can be */
					/* modified by protocol or by driver.*/
					/* NOTE: this interface will change */
	int (*push_oam)(struct atm_vcc *vcc,void *cell);
	void		*dev_data;	/* per-device data */
	void		*proto_data;	/* per-protocol data */
	struct timeval	timestamp;	/* AAL timestamps */
	struct sk_buff_head recvq;	/* receive queue */
	struct atm_aal_stats *stats;	/* pointer to AAL stats group */
	struct wait_queue *sleep;	/* if socket is busy */
	struct wait_queue *wsleep;	/* if waiting for write buffer space */
	struct atm_vcc	*prev,*next;
	/* SVC part --- may move later */
	short		itf;		/* interface number */
	struct sockaddr_atmsvc local;
	struct sockaddr_atmsvc remote;
	void (*callback)(struct atm_vcc *vcc);
	struct sk_buff_head listenq;
	int		backlog_quota;	/* number of connection requests we */
					/* can still accept */
	int		reply;
	void		*user_back;	/* user backlink - not touched */
#ifdef CONFIG_AREQUIPA
	struct sock	*upper;		/* our "master" */
	struct socket	*sock;		/* back pointer to our own socket */
	struct atm_vcc	*aq_next,*aq_prev; /* for consistency checks */
	unsigned long	generation;	/* generation number */
#endif
};


struct atm_dev_addr {
	struct sockaddr_atmsvc addr;	/* ATM address */
	struct atm_dev_addr *next;	/* next address */
};


struct atm_dev {
	const struct atmdev_ops *ops;	/* device operations; NULL if unused */
	const struct atmphy_ops *phy;	/* PHY operations, may be undefined */
					/* (NULL) */
	const char	*type;		/* device type name */
	int		number;		/* device index */
	struct atm_vcc	*vccs;		/* VCC table (or NULL) */
	struct atm_vcc	*last;		/* last VCC (or undefined) */
	void		*dev_data;	/* per-device data */
	void		*phy_data;	/* private PHY date */
	unsigned long	flags;		/* device flags, TBD */
	struct atm_dev_addr *local;	/* local ATM addresses */
	unsigned char	esi[ESI_LEN];	/* ESI ("MAC" addr) */
	struct atm_cirange ci_range;	/* VPI/VCI range */
	struct atm_dev_stats stats;	/* statistics */
/* */ int sending;
};


/*
 * ioctl, getsockopt, setsockopt, sg_send, and poll are optional and can
 * be set to NULL.
 */

#define ATM_OF_IMMED	1	/* Attempt immediate delivery */
#define ATM_OF_INRATE	2	/* Attempt in-rate delivery */

struct atmdev_ops { /* only send is required */
	int (*open)(struct atm_vcc *vcc,short vpi,int vci);
	void (*close)(struct atm_vcc *vcc);
	int (*ioctl)(struct atm_dev *dev,unsigned int cmd,unsigned long arg);
	int (*getsockopt)(struct atm_vcc *vcc,int level,int optname,
	    char *optval,int *optlen);
	int (*setsockopt)(struct atm_vcc *vcc,int level,int optname,
	    char *optval,int optlen);
	int (*send)(struct atm_vcc *vcc,struct sk_buff *skb);
	int (*sg_send)(struct atm_vcc *vcc,unsigned long start,
	    unsigned long size);
#if 0 /* keep the current hack for now */
	int (*send_iovec)(struct atm_vcc *vcc,struct iovec *iov,int size,
	    void (*discard)(struct atm_vcc *vcc,void *user),void *user);
#endif
	void (*poll)(struct atm_vcc *vcc,int nonblock);
	int (*send_oam)(struct atm_vcc *vcc,void *cell,int flags);
	void (*phy_put)(struct atm_dev *dev,unsigned char value,
	    unsigned long addr);
	unsigned char (*phy_get)(struct atm_dev *dev,unsigned long addr);
	void (*feedback)(struct atm_vcc *vcc,struct sk_buff *skb,
	    unsigned long start,unsigned long dest,int len);
	int (*change_qos)(struct atm_vcc *vcc,struct atm_qos *qos);
	void (*free_rx_skb)(struct atm_vcc *vcc, struct sk_buff *skb);
		/* @@@ temporary hack */
};


struct atmphy_ops {
	int (*start)(struct atm_dev *dev);
	int (*ioctl)(struct atm_dev *dev,unsigned int cmd,unsigned long arg);
	void (*interrupt)(struct atm_dev *dev);
};


struct atm_dev *atm_dev_register(const char *type,const struct atmdev_ops *ops,
    unsigned long flags);
void atm_dev_deregister(struct atm_dev *dev);
int atm_find_ci(struct atm_vcc *vcc,short *vpi,int *vci);

#endif /* __KERNEL__ */

#endif
