#include <stddef.h>

#include "rsvp_daemon.h"
#include "rapi_lib.h"		/* Define flowspec formats */
#include "rsvp_specs.h"		/* Flowspec descriptor format */
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include "rsvp_TCif.h"                /* Adaptation interface */

#include "tc_global.h"
#include "tc_errno.h"

/* external declarations */
static int	udp_socket = -1;
static struct ifreq ifr;

static int	kern_translate_error __P((int kernel_error));
static int	kern_tcclear __P((int));
static int	rsvp2tc_spec __P((FLOWSPEC *, SENDER_TSPEC *,
				    tc_fs_t *, int));
static tc_filt_t *
		rsvp2tc_filter __P((FILTER_SPEC *, Session *, int));
static void	log_tc_filter __P((tc_filt_t *, int));

/*** static char	*fmt_tc_fs __P((tc_fs_t *)); ***/

#ifndef hton16
#define hton16(x)	((u_int16_t)htons((u_int16_t)(x)))
#define HTON16(x)	HTONS(x)
#define hton32(x)	((u_int32_t)htonl((u_int32_t)(x)))
#define HTON32(x)	HTONL(x)
#endif

/* Translate TC kernel error to (error code, error value) pair in
 *	rsvp_errno format.  Index rel to ETCBASE.
 * 
 * (*) in comment means should never happen.
 *
 * Receipt of EBDHDL most likely represents an internal RSVP error, but
 * there is no useful (can report a subcode) error class for this.
 */
int Xlate_kern_err[] = {
	EBDHDL *256+RSVP_Err_TC_ERROR,/* EBDHDL: Bad handle          */
	EEXCLS *256+RSVP_Err_TC_ERROR,/* EEXCLS: Class exists (*)    */
	ENOCLS *256+RSVP_Err_TC_ERROR,/* ENOCLS: Class unknown (*)   */
	ENOTMT *256+RSVP_Err_TC_ERROR,/* ENOTMT: Class has flows (*) */

					/* EDELAY: Delay bound violation */
					/* also occurs if no class for pflow */
	RSVP_Erv_DelayBnd*256+RSVP_Err_ADMISSION,

					/* ENOBWD: Not enough bandwidth */
	RSVP_Erv_Bandwidth*256+RSVP_Err_ADMISSION,

					/* EBADTSPEC: out-of-range RSPEC */
	RSVP_Erv_Crazy_Flowspec*256+RSVP_Err_ADMISSION,

					/* EBADRSPEC: out-of-range TSPEC */
	RSVP_Erv_Crazy_Tspec*256+RSVP_Err_ADMISSION,
 
					/* Kernel doesn't like filter	*/
					/* should perhaps be rsvp ambig filt
					   error instead. */
	EBADFILT *256+RSVP_Err_TC_ERROR
};

#define TC_flags (TCF_E_POLICE | TCF_M_POLICE | TCF_B_POLICE)
	
/************************************************************************
 *
 *	Interface routines to call the TC kernel functions
 * 
 ************************************************************************/

/* Note: these calls use the address of the kern_resv block as a parameter
 *	to carry the interface number.  This block actually contains the
 *	the other parameters too, but we leave the parameters explicit to
 *	make the code exactly parallel the Functional Specification.
 */
void
TC_init(int kernel_socket,int Oif)
	{
	int i;

	/*  Initialize UDP socket used to communicate with the
	 *	kernel traffic control code.
	 */
	udp_socket = kernel_socket;

        /* clear old reservations from the kernel and then make a
	 *	reservation for RSVP packets, on each interface.
	 */
/*        for (i = 0; i < if_num; i++) {*/
                kern_tcclear(Oif);
/*        }*/
}

/*
 * TC_AddFlowspec(): Call the kernel to make reservation for a flow.
 * 	It checks admission control, and returns a handle for the
 *	reservation, or -1 if an error.  It may also set *fwd_specpp
 * 	to point to a modified flowspec to be forwarded.
 */
u_long
TC_AddFlowspec(int OIf, FLOWSPEC *spec, SENDER_TSPEC *stspec, int flags,
							FLOWSPEC **fwd_specpp)
	{
	char		*if_name;
	struct tcreq	 ir;
	struct tcreq	*irp = &ir;

	if (IsDebug(32))
	   printf("Add Flow\n");
	rsvp_errno = 0;
	*fwd_specpp = NULL;
	if_name = if_vec[OIf].if_name;
	if (!if_vec[OIf].if_up) {
		rsvp_errno = Set_Errno(RSVP_Err_TC_ERROR, 0);
		return(TC_OK);  /**** XXX For testing? */
	}

	if (rsvp2tc_spec(spec, stspec, &irp->iq_fs, (flags&TC_flags)) < 0)
		return (TC_ERROR);

	strncpy(irp->iq_name, if_name, IFNAMSIZ);
	irp->iq_function = IF_ADDFLOW;
	
	strcpy( ifr.ifr_name,if_name);
	irp->iq_flags    = TC_CBQ;
	ifr.ifr_data = irp;

	if (ioctl(udp_socket, SIOCSIFTC, &ifr) < 0) {
		rsvp_errno = kern_translate_error(errno);
	/****	log(LOG_ERR, errno, "SIOCIFADDFLOW");	****/
		return (TC_ERROR);
	}
	memcpy(&ir,ifr.ifr_data,sizeof(struct tcreq));
	irp = ifr.ifr_data;
	return (irp->iq_handle);
}

/*
 * TC_DelFlowspec(): This routine deletes flow for specified handle
 */
int
TC_DelFlowspec(int OIf, u_long rhandle) {
	char           *if_name;
	struct tcreq  ir;

	if (IsDebug(32))
	   printf("DEL Flow\n");
	rsvp_errno = 0;
	if_name = if_vec[OIf].if_name;
	if (!if_vec[OIf].if_up) {
		rsvp_errno = Set_Errno(RSVP_Err_TC_ERROR, 0);
		return(TC_ERROR);
	}
	strncpy(ir.iq_name, if_name, IFNAMSIZ);
	ir.iq_handle = rhandle;
	ir.iq_function = IF_DELFLOW;

	strcpy( ifr.ifr_name,if_name);
	ir.iq_flags    = TC_CBQ;
	ifr.ifr_data = &ir;

	if (ioctl(udp_socket, SIOCSIFTC, &ifr) < 0) {
		rsvp_errno = kern_translate_error(errno);
	/***	log(LOG_ERR, errno, "SIOCIFDELFLOW");  ***/
		return (TC_ERROR);
	}
	return (TC_OK);
}

/*
 * TC_AddFilter(): Adds a filter for an existing flow.
 *
 *	Returns fhandle or TC_ERROR.
 */
u_long
TC_AddFilter(int OIf, u_long rhandle, Session *dest, FILTER_SPEC *filtp) {
	char		*if_name;
	struct tcreq   ir;
	tc_filt_t	*ifp = NULL;
	long           fhdl;

/*	printf ("*** TC_Add_filter ***\n");*/
	if (IsDebug(32))
	   printf("Add Filter\n");
	rsvp_errno = 0;
	if_name = if_vec[OIf].if_name;
	if (!if_vec[OIf].if_up) {
		rsvp_errno = Set_Errno(RSVP_Err_TC_ERROR, 0);
		return(TC_ERROR);
	}

	strncpy(ir.iq_name, if_name, IFNAMSIZ);
	ir.iq_handle = rhandle;
	
	ir.iq_function = IF_ADDFILT;
	ifp = rsvp2tc_filter(filtp, dest, 0);
	if (ifp == NULL) {
		return (TC_ERROR);
	}
	ir.iq_function = IF_ADDFILT;
	ir.iq_filt = *ifp;

	strcpy( ifr.ifr_name,if_name);
	ir.iq_flags    = TC_CBQ;
	ifr.ifr_data = &ir;
	
	if (ioctl(udp_socket, SIOCSIFTC, &ifr) < 0) {
		/* Since we constructed kernel filter here, assume for now
		 *	that any error must be internal, between us and
		 *	the kernel.
		 * 
		 * Note that this will happen, because the tc kernel
		 * has stricter rules for filters than RSVP currently does
		 * (since the RSVP rules are currently somewhat broken).
		 */
		rsvp_errno = kern_translate_error(errno);
	/****	log(LOG_ERR, 0, "SIOCIFADDFILT\n");	****/
		return (TC_ERROR);
	}
	if (IsDebug(DEBUG_EVENTS)) {
		log(LOG_DEBUG, 0, "       kernel handle=0x%x\n", rhandle);
		log(LOG_DEBUG, 0, "  TC_FILT:\n");
		log_tc_filter(ifp, 5);
	}
	/* For now, until kernel implements fhandles, use rhandle as
	 * the fhandle.
	 */
	memcpy(&ir,ifr.ifr_data,sizeof(struct tcreq));

	fhdl = ir.iq_fhdl;
	if (IsDebug(32))
	   printf("New Filterhandle= %d\n",fhdl);
	return (rhandle);
}


/*
 * TC_DelFilter(): Deletes existing filter.
 */
int
TC_DelFilter(int OIf, u_long rhandle)
	{
	char           *if_name;
	struct tcreq  ir;
	tc_filt_t	*ifp = NULL;

	if (IsDebug(32))
	   printf("DEL filter\n");
	rsvp_errno = 0;
	if_name = if_vec[OIf].if_name;
	if (!if_vec[OIf].if_up) {
		rsvp_errno = Set_Errno(RSVP_Err_TC_ERROR, 0);
		return(TC_ERROR);
	}

	strncpy(ir.iq_name, if_name, IFNAMSIZ);
	ir.iq_handle = rhandle;
	ir.iq_function = IF_ADDFILT;
	ir.iq_filt.f_pf = PF_UNSPEC; /* a NULL filter: deletes it */

	strcpy( ifr.ifr_name,if_name);
	ir.iq_flags    = TC_CBQ;
	ifr.ifr_data = &ir;
	
	if (ioctl(udp_socket, SIOCSIFTC, &ifr) < 0) {
		/* Since we constructed kernel filter here, assume for now
		 *	that any error must be internal, between us and
		 *	the kernel.
		 * 
		 * Note that this will happen, because the tc kernel
		 * has stricter rules for filters than RSVP currently does
		 * (since the RSVP rules are currently somewhat broken).
		 */
		rsvp_errno = kern_translate_error(errno);
	/****	log(LOG_ERR, 0, "SIOCIFADDFILT\n");	****/
		return (TC_ERROR);
	}
	if (IsDebug(DEBUG_EVENTS)) {
		log(LOG_DEBUG, 0, "       kernel handle=0x%x\n", rhandle);
		log(LOG_DEBUG, 0, "  TC_FILT:\n");
		log_tc_filter(ifp, 5);
	}
	return (TC_OK);
}

/*
 * TC_ModFlowspec(): Modifies a flowspec of a given flow.
 *
 *	It may also set *fwd_specpp to point to a modified flowspec to
 *	be forwarded.
 */
int
TC_ModFlowspec(int OIf, u_long rhandle,
	 		FLOWSPEC *specp, SENDER_TSPEC *stspecp, int flags,
					FLOWSPEC **fwd_specpp)
	{
	char           *if_name;
	struct tcreq	ir;
	struct tcreq	*irp = &ir;

	rsvp_errno = 0;
	*fwd_specpp = NULL;
	if_name = if_vec[OIf].if_name;
	if (!if_vec[OIf].if_up) {
		rsvp_errno = Set_Errno(RSVP_Err_TC_ERROR, 0);
		return(TC_ERROR);
	}

	strncpy(irp->iq_name, if_name, IFNAMSIZ);
	irp->iq_function = IF_MODFLOW;
	irp->iq_handle = rhandle;
	if (rsvp2tc_spec(specp, stspecp, &irp->iq_fs, (flags&TC_flags))<0)
		return (TC_ERROR);

	strcpy( ifr.ifr_name,if_name);
	irp->iq_flags    = TC_CBQ;
	ifr.ifr_data = irp;
	if (ioctl(udp_socket, SIOCSIFTC, &ifr) < 0) {
		rsvp_errno = kern_translate_error(errno);
	/****	log(LOG_ERR, 0, "SIOCIFMODFLOW\n");	****/
		return (TC_ERROR);
	}
	return (TC_OK);
}

/*
 * TC_Advertise(): Update OPWA ADSPEC.
 */
ADSPEC *
TC_Advertise(int OIf, ADSPEC * old_asp, int flags)
	{
	Object_header *copy_object(Object_header *);

	if (!if_vec[OIf].if_up) {
		return(NULL);
	}
	return(copy_adspec(old_asp));
}

static int
kern_translate_error(int kernel_error)
{
    	int e;

    	if (kernel_error >= ETCBASE && kernel_error < ETCMAX)
		e = Xlate_kern_err[errno - ETCBASE];
	else
	    	e = kernel_error << 8 | RSVP_Err_TC_ERROR;
	return (e);
}

/*
 * kern_tcclear(): This routine resets an interface, cleaning up old
 * 	reservations. It is used when RSVP daemon is initialized.
 */
static int
kern_tcclear(int in_if) {
    	char           *if_name;
	struct tcreq  ir;
	int 		rc;

    
    	if_name = if_vec[in_if].if_name;
	if_vec[in_if].if_up = 0;	    
    
	log(LOG_INFO, 0, "TC: Re-init TC on %s %d \n", if_name, sizeof(if_name));
    
	/*
	 * Delete all state for unlocked flows. Puts the scheduler in a
	 * known starting state when RSVP is reset.
	 */
	ir.iq_function = IF_RESET;
	ir.iq_flags    = TC_CBQ;
	/*	strncpy( ifr.ifr_name,if_name, sizeof(if_name));*/
	strncpy( ir.iq_name , if_name, sizeof(if_name));
	
	strcpy( ifr.ifr_name,if_name);
	ifr.ifr_data = &ir;
	rc = ioctl(udp_socket, SIOCSIFTC, &ifr);
	if (rc< 0) {
	    	if (errno == EINVAL||errno == ENXIO) {
		    	log(LOG_INFO, 0, 
				"No TC traffic control on interface %s\n",
			    	if_name);
			return(0);
		} else {
		    	log(LOG_ERR, errno,
				"Error resetting TC interface", 0);
			return (-1);
		}
	}
	/*
	 *	Mark Traffic Control up on interface
	 */
	memcpy(&ir,ifr.ifr_data,sizeof(struct tcreq));
	
	if_vec[in_if].if_up = 1;
	return (1);
}

/*
 *  rsvp2tc_spec(): Create kernel tc flowspec from RSVP flowspec.
 *	'flags' contains F_POLICE bit if merging data so policing should
 *	 be done.  Set rsvp_errno and return -1 if flowspec is illegal.
 */
static int
rsvp2tc_spec(
	FLOWSPEC	*specp,
	SENDER_TSPEC	*stspecp,
	tc_fs_t	*i_fs,
	int		flags)
	{
	IS_serv_hdr_t	*sp, *stp;

	if (Obj_CType(specp) != ctype_FLOWSPEC_Intserv0) {
		/* Unknown CTYPE */
		rsvp_errno = Set_Errno(RSVP_Err_UNKNOWN_CTYPE,
				class_FLOWSPEC*256+Obj_CType(specp));
		return(-1);
	}
	if (Obj_CType(stspecp) != ctype_SENDER_TSPEC) {
		rsvp_errno = Set_Errno(RSVP_Err_UNKNOWN_CTYPE,
				class_SENDER_TSPEC*256+Obj_CType(stspecp));
		return(-1);
	}

	sp = (IS_serv_hdr_t *) &specp->flow_body.spec_u;
	stp = (IS_serv_hdr_t *) &stspecp->stspec_body.tspec_u;

	switch (sp->issh_service) {

	    case GUARANTEED_SERV:
		{ Guar_flowspec_t *s = (Guar_flowspec_t *) sp;
		  gen_Tspec_t *t = (gen_Tspec_t *) stp;

		i_fs->fs_tos = TOS_GUARANTEED;
		if (flags)
			  /* wrong policer, but the right one doesn't exist*/
			i_fs->fs_plc = PLC_TBF2;
		else
			i_fs->fs_plc = PLC_NONE;

		/* Set RSPEC info. */
		i_fs->fs_rspec.rs_g.g_rate = s->Gspec_R;

		/* Set TSPEC info. 
		 */
		if (stspecp) {
			 /* And why would there not be?? */

			 i_fs->fs_tspec.ts_tb.tbf_rate
				= MIN(s->Gspec_r, t->gtspec_r);

			 i_fs->fs_tspec.ts_tb.tbf_bkt
				= MAX(s->Gspec_b, t->gtspec_b);

			 /* Will take MIN of m, MIN of p, and MAX of M, too */
		} else {
			/* No sender tspec (shouldn't happen, but..)
			 * Best we can do for now is use whatever
			 * receiver gave us.
			 */
			i_fs->fs_tspec.ts_tb.tbf_rate = s->Gspec_r;
			i_fs->fs_tspec.ts_tb.tbf_bkt = s->Gspec_b;
		}
		}
		break;

	    case CONTROLLED_LOAD_SERV:
		{ CL_flowspec_t *s = (CL_flowspec_t *) sp;
		  gen_Tspec_t *t = (gen_Tspec_t *) stp;

		i_fs->fs_tos = TOS_CLOAD; /* Obs */
		if (flags)
			i_fs->fs_plc = PLC_TBF2;
		else
			i_fs->fs_plc = PLC_NONE;


		/* No RSPEC info. */
		i_fs->fs_rspec.rs_p.p_dly = 0;

		if (stspecp) {
			 i_fs->fs_tspec.ts_tb.tbf_rate
				= MIN(s->CLspec_r, t->gtspec_r);

			 i_fs->fs_tspec.ts_tb.tbf_bkt
				= MAX(s->CLspec_b, t->gtspec_b);

			 /* Will take MIN of m and MAX of M, too */
		} else {
			/* No sender tspec (shouldn't happen, but..)
			 * Best we can do for now is use whatever
			 * receiver gave us.
			 */
			i_fs->fs_tspec.ts_tb.tbf_rate = s->CLspec_r;
			i_fs->fs_tspec.ts_tb.tbf_bkt = s->CLspec_b;
		}
		}
		break;

	    default:
			/* Unknown service */
		rsvp_errno = Set_Errno(RSVP_Err_ADMISSION,
							 RSVP_Erv_No_Serv);
		return(-1);
	}
		
	/*
	 * Future: set share class info from local mapping function,
	 * RSVP policy credentials, etc.
	 */
	i_fs->fs_handle = -1;

	return(0);	/* OK */
}	

/*
 * rsvp2tc_filter():  Builds kernel TC filter from RSVP filter
 *	spec in static area.  Sets rsvp_errno and return NULL if
 *	error in filter spec, else returns ptr to static area.
 *
 *	IPv4 addresses and ports in the tc filter spec are given in
 *	network byte order.
 */
static tc_filt_t *
rsvp2tc_filter(FilterSpec *filtp, Session *dest, int flags) {

	static tc_filt_t fl;

	switch (Obj_CType(filtp)) {
	case ctype_FILTER_SPEC_ipv4:	  
	  fl.f_pf = PF_INET;
	  fl.f_ipv4_protocol = IPPROTO_UDP; /* RSVP spec bug forces this
					       assumption */
	  fl.f_ipv4_destaddr = dest->d_session->sess4_addr.s_addr;
	  fl.f_ipv4_destport = dest->d_session->sess4_port;
	  fl.f_ipv4_srcaddr = filtp->filt4_srcaddr.s_addr;
	  fl.f_ipv4_srcport = filtp->filt4_srcport;
	  break;
	case ctype_FILTER_SPEC_ipv6:
	  fl.f_pf = PF_INET6;
	  fl.f_ipv6_protocol = IPPROTO_UDP;
	  fl.f_ipv6_destaddr = dest->d_session->sess6_addr;
	  fl.f_ipv6_srcaddr  = filtp->filt6_srcaddr;
	  fl.f_ipv6_vrprflow = htonl ((unsigned long) 0x60000000/* + (unsigned long) dest->d_session->sess6_port*/ );	    
	  break;
	default:
	  printf ("rsvp2tc_filter:Don't know about FILTER_SPEC %d\n",Obj_CType(filtp));
	  assert(0);
	}
		  
	/*
	 *  If the data will be coming in on a tunnel, the filter will
	 *  have to be adjusted so that the filter offset points to the
	 *  beginning of the unencapsulated data.  Thus the filter offset
	 *  must be incremented by the size of the encapsulating ip
	 *  header.
	 *
	 *  Note that there is an implicit assumption that the encapsulating
	 *  header size is fixed length.
	 */
	/*  XXX what to do... This can't possibly be right, since
	 *  rsvp doesn't know about all the different sorts of tunnels
	if (flags & F_MTUNNEL) {
	}
	****/
	return (&fl);
}

/*
 *  log_tc_filter(): Log TC filter spec
 */
static void
log_tc_filter(tc_filt_t *filt, int pr_ofs)
{
	if (filt == NULL) {
		log(LOG_DEBUG, 0, "Trying to parse NULL filter\n");
		return;
	}
	log(LOG_DEBUG, 0, "---------- Parsing TC Filter: -------\n");
	if (filt->f_pf == PF_INET) {
	  log(LOG_DEBUG, 0, "%*s PF %s Proto %s: \n", pr_ofs, "",
	      filt->f_pf == PF_INET ? "IPv4" : "(Unknown!)",
	      filt->f_ipv4_protocol == IPPROTO_UDP ? "UDP" :
	      filt->f_ipv4_protocol == IPPROTO_TCP ? "TCP" : "Unknown");
	  log(LOG_DEBUG, 0, "%*s SrcAddr %d SrcPort %d: \n", pr_ofs, "",
	      filt->f_ipv4_srcaddr, ntoh16(filt->f_ipv4_srcport));
	  log(LOG_DEBUG, 0, "%*s DestAddr %d DestPort %d: \n", pr_ofs, "",
	      filt->f_ipv4_destaddr, ntoh16(filt->f_ipv4_destport));
	} else {
	  log (LOG_DEBUG, 0,"%*s PF %s Proto %s: \n", pr_ofs, "",
	       filt->f_pf == PF_INET6 ? "IPv6" : "(Unknown!)",
	       filt->f_ipv6_protocol == IPPROTO_UDP ? "UDP" :
	       filt->f_ipv6_protocol == IPPROTO_TCP ? "TCP" : "Unknown");
	  log(LOG_DEBUG, 0, "%*s SrcAddr %lx:%lx:%lx:%lx\n", pr_ofs, "",
	      filt->f_ipv6_srcaddr.s6_addr32[0], 
	      filt->f_ipv6_srcaddr.s6_addr32[1],
	      filt->f_ipv6_srcaddr.s6_addr32[2],
	      filt->f_ipv6_srcaddr.s6_addr32[3]);
	   log(LOG_DEBUG, 0, "%*s DestAddr %lx:%lx:%lx:%lx \n", pr_ofs, "",
	       filt->f_ipv6_destaddr.s6_addr32[0],
	       filt->f_ipv6_destaddr.s6_addr32[1],
	       filt->f_ipv6_destaddr.s6_addr32[2],
	       filt->f_ipv6_destaddr.s6_addr32[3]);
	}
}


