/*
 *  @(#) $Id: rsvp_unicast.c,v 4.15 1997/12/18 01:38:36 lindell Exp $
 */

/************************ rsvp_unicast.c  ****************************
 *                                                                   *
 *       System-dependent Unicast route lookup routine               *
 *                                                                   *
 *********************************************************************/

#include "rsvp_daemon.h"
#include <sys/param.h>
#include <sys/socket.h>

static int init_failed = 0;
int locate_LIH(unsigned int, net_addr *);

int
locate_LIH(unsigned int ifidx, net_addr *addr)
{
	int i;
	for (i = 0; i < if_num; i++) {
		if (IsNumAPI(i))
			continue;
		if (ifidx == if_vec[i].if_index) {
			if (net_addr_equal(addr, &(NET_GET_IF_PHY(&(if_vec[i].if_addr)))))
				return i;
		}
	}
	return -1;
}

#if defined(PF_ROUTE)  && !defined(sgi_53) && !defined(Linux)

/*
 * This code will assumes a routing socket and will work on many newer
 * BSD-like machines
 *
 */

/*
 * Copyright 1995 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies, that both the above
 * copyright notice and this permission notice appear in all
 * supporting documentation, and that the name of M.I.T. not be used
 * in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  M.I.T. makes
 * no representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.
 * 
 * THIS SOFTWARE IS PROVIDED BY M.I.T. ``AS IS''.  M.I.T. DISCLAIMS
 * ALL EXPRESS OR IMPLIED WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT
 * SHALL M.I.T. BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <errno.h>
#include <net/if.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/route.h>
#include <netinet/in.h>

/*
 * Round up 'a' to next multiple of 'size'
 */ 
#define ROUNDUP(a, size) (((a) & ((size)-1)) ? (1 + ((a) | ((size)-1))) : (a))
  
/*
 * Step to next socket address structure;
 * if sa_len is 0, assume it is sizeof(u_long).
 */
#define NEXT_SA(ap) ap = (struct sockaddr *) \
    ((caddr_t) ap + (ap->sa_len ? ROUNDUP(ap->sa_len, sizeof (u_long)) : \
                                    sizeof(u_long)))

static int seq;
/*
 * For now, we open a new socket every time we want to get a route, to
 * save some effort in parsing.  Eventually we should keep it open and
 * listen for changes.  For now, just open and close it to make sure that
 * we can.
 */
int
unicast_init(void)
{
	int s;

	s = socket(PF_ROUTE, SOCK_RAW, AF_INET);
	if (s < 0) {
		log(LOG_INFO, 0, "Unicast routing information unavailable\n");
		return 0;
	}
	close(s);
	return 1;
}

#define RTM_BUFLEN 2048		/* XXX should be in header file */
#define WORD_BNDARY(x)  (((x)+sizeof(long)-1) & ~(sizeof(long)-1))
/*
 * Find the kernel's idea of the route to destination DEST (in host order)
 */
int
unicast_route(net_addr *addr)
{
	int s,sz;
	struct sockaddr_in *s4;
	struct sockaddr *sa, *sp, *rti_info[RTAX_MAX];
	struct rt_msghdr *mhp;
	net_addr nadr,*addr2;
	char buf[RTM_BUFLEN];
	struct sockaddr_dl *sdlp;
	char ifname[IFNAMSIZ];
	int one, len;
	int i; 
	unsigned int ifidx = 0;
#ifdef	USE_IPV6
	struct sockaddr_in6 *s6;
#endif	/* USE_IPV6 */


	/* Open socket */
	s = socket(PF_ROUTE, SOCK_RAW, PF_UNSPEC);
	if (s < 0) {
		log(LOG_ERR, errno,
		    "Could not open routing socket");
		return -1;
	}

	one = 1;
	setsockopt(s, SOL_SOCKET, SO_USELOOPBACK, &one, sizeof one);

	/* Set up GET message */
	memset(buf, 0, sizeof buf);

	mhp = (struct rt_msghdr *)buf;
	sp = (struct sockaddr *)
               &buf[WORD_BNDARY(sizeof (struct rt_msghdr))];

	addr2 = net_addr_ip(addr);
	switch (NET_GET_TYPE(addr2)) {
		case NET_ADDR_IPv4:
			sz = sizeof(struct sockaddr_in);
			s4 = (struct sockaddr_in *) sp;
			s4->sin_family = AF_INET;
			s4->sin_addr.s_addr = NET_GET_ADDR_IPv4(addr2).s_addr;
			s4->sin_port = 0;
#ifdef SOCKADDR_LEN
			s4->sin_len = sz;
#endif
			break;
#ifdef	USE_IPV6
		case NET_ADDR_IPv6:
			sz = sizeof(struct sockaddr_in6);
			s6 = (struct sockaddr_in6 *) sp;
			s6->sin6_family = AF_INET6;
			s6->sin6_addr = NET_GET_ADDR_IPv6(addr2);
			s6->sin6_port = 0;
#ifdef SOCKADDR_LEN
			s6->sin6_len = sz;
#endif
			break;
#endif	/* USE_IPV6 */
		default:
			return(-1);
	}

	mhp->rtm_version = RTM_VERSION;
	mhp->rtm_type = RTM_GET;
	/* mhp->rtm_addrs = RTA_DST|RTA_IFP; */
	mhp->rtm_addrs = RTA_DST|RTA_IFP|RTA_IFA;
	mhp->rtm_seq = ++seq;
	mhp->rtm_msglen = ((char *)(sp)) + sz - buf;

	/* Send the message */
	if (write(s, buf, (size_t)mhp->rtm_msglen) < 0) {
		log(LOG_ERR, errno, "writing message on routing socket");
		close(s);
		return -1;
	}

	/* Read the reply */
	do {
		len = read(s, buf, sizeof buf);
	} while (len > 0 && mhp->rtm_seq != seq && mhp->rtm_pid != getpid());

	close(s);

	if (len < 0 || mhp->rtm_errno) {
		if (IsDebug(DEBUG_ROUTE))
			log(LOG_DEBUG, errno, "unicast route lookup failed");
		return -1;
	}

	if (len == 0) {
		log(LOG_ERR, 0, "unicast route lookup is confused");
		return -1;
	}

	sa = (struct sockaddr *) (mhp + 1);
    
    for (i = 0; i < RTAX_MAX; i++) {
        if (mhp->rtm_addrs & (1 << i)) {
            rti_info[i] = sa;
            NEXT_SA(sa);
        } else
            rti_info[i] = NULL;
    }   
 
	if ((sa = rti_info[RTAX_IFP]) != NULL) {
		sdlp = (struct sockaddr_dl *)sa;
		if (sdlp->sdl_family != AF_LINK || sdlp->sdl_nlen == 0) {
			if (IsDebug(DEBUG_ROUTE))
				log(LOG_DEBUG, 0, "unicast route through weird ifp");
			return -1;
		}
		strncpy(ifname, sdlp->sdl_data, sdlp->sdl_nlen);
		if (sdlp->sdl_nlen < IFNAMSIZ)
			ifname[sdlp->sdl_nlen] = '\0';
		ifidx = if_nametoindex(ifname);
	}

    if ((sa = rti_info[RTAX_IFA]) != NULL) {
	switch (sa->sa_family) {
		case AF_INET:
			s4 = (struct sockaddr_in *)sa;
			NET_SET_ADDR_IPv4(&nadr, s4->sin_addr);
			break;
#ifdef	USE_IPV6
		case AF_INET6:
			s6 = (struct sockaddr_in6 *)sa;
			NET_SET_ADDR_IPv6(&nadr, s6->sin6_addr);
			break;
#endif	/* USE_IPV6 */
		default:
			return (-1);
	}
    }
	else 
		return -1;

	return (locate_LIH(ifidx, &nadr));
}
		
# else

static int unicast_route1(u_long);

int
unicast_route(net_addr *addr)
{
	switch (NET_GET_TYPE(addr)) {
		case NET_ADDR_IPv4:
			return(unicast_route1(NET_GET_ADDR_IPv4(addr).s_addr));
		case NET_ADDR_UDP_IPv4:
			return(unicast_route1(NET_GET_ADDR_UDP_IPv4(addr)
				.sin_addr.s_addr));
#ifdef	USE_IPV6
#ifdef	WORKAROUNDS
		case NET_ADDR_IPv6:
			return(local_v6);
		case NET_ADDR_UDP_IPv6:
			return(local_v6);
#endif	/* WORKAROUNDS */
#endif	/* USE_IPV6 */
		default:
			return(-1);
	}
}

/* 
 * ifndef PF_ROUTE - no routing socket. OS-specific code required,
 */

#ifdef SOLARIS
/* Following Solaris code was supplied by Don Hoffman of Sun Microsystems.
 */

/*
 * Copyright (c) Sun Microsystems, Inc.  1994. All rights reserved.
 * 
 * License is granted to copy, to use, and to make and to use derivative works
 * for research and evaluation purposes, provided that Sun Microsystems is
 * acknowledged in all documentation pertaining to any such copy or
 * derivative work. Sun Microsystems grants no other licenses expressed or
 * implied. The Sun Microsystems  trade name should not be used in any
 * advertising without its written permission.
 * 
 * SUN MICROSYSTEMS MERCHANTABILITY OF THIS SOFTWARE OR THE SUITABILITY OF THIS
 * SOFTWARE FOR ANY PARTICULAR PURPOSE.  The software is provided "as is"
 * without express or implied warranty of any kind.
 * 
 * These notices must be retained in any copies of any part of this software.
 */
/* Portions of this code were dereived from Solaris route.c and netstat.c */

#include <fcntl.h>
#include <sys/stropts.h>
#include <net/route.h>
#include <netinet/in.h>
#include <sys/tihdr.h>
#include <sys/tiuser.h>
#include <inet/mib2.h>

/* 
 * Defines
 */
#define MAX_ROUTES	(4 * 512)
#ifndef T_CURRENT
#define T_CURRENT       MI_T_CURRENT
#endif

/* 
 * Structure definitions
 */
typedef struct mib_item_s {
	struct mib_item_s	* next_item;
	long			group;
	long			mib_id;
	long			length;
	char			* valp;
} mib_item_t;

struct xroute {
	char ifname[IFNAMSIZ];
	struct sockaddr_in netmask;
	struct sockaddr_in out_if;
	struct sockaddr_in dst;
	struct sockaddr_in gw;
};

/*
 * Forward function declarations
 */
static mib_item_t 	*mibget (int sd);
static void		mibfree(mib_item_t*);
static int		is_host(mib2_ipRouteEntry_t *rp);
static int		is_gateway(mib2_ipRouteEntry_t *rp);
int putmsg(int, struct strbuf *, struct strbuf *, int);
int getmsg(int, struct strbuf *, struct strbuf *, int *);

/* 
 * Globals
 */
static int 		sd;
static struct xroute	routes[MAX_ROUTES];
static struct xroute 	default_route;
static int 		have_default = 0;
static int 		nhost = 0;
static int 		nroute = 0;


static int
ifname_to_if(char *name) {
	int i;
	for (i = 0; i < if_num; i++)
		if (strncmp(if_vec[i].if_name, name, IFNAMSIZ) == 0)
			return(i);
	log(LOG_ERR, 0, "Couldn't find interface %s\n", name);
#ifdef SUNMOD
	return(0);
#else
	return(-1);
#endif
}

#ifdef SBM_ED
u_long
unicast_route_gw(u_long addr) {
	int i;
	struct sockaddr_in sin;
	if (read_routes() == -1)
		return(-1);

	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = addr;
	i = rtfindx(&sin);
	
	if (i == -1 && have_default) {
		if (IsDebug(DEBUG_ROUTE)) {
			log(LOG_DEBUG, 0, "Using default route ...\n");
			log(LOG_DEBUG, 0, "route out interface %s\n", 
							default_route.ifname);
		}
		return(default_route.gw.sin_addr.s_addr);
	}
	else if (i >= 0)
		return(routes[i].gw.sin_addr.s_addr);
	else return -1;
}
#endif 

static int
inet_netmatch(struct sockaddr_in *sin1, struct sockaddr_in *sin2) {
	return (inet_netof(sin1->sin_addr) == inet_netof(sin2->sin_addr));
}

static int
net_match(struct sockaddr_in *dst, struct xroute *rp) {
	u_long nm = rp->netmask.sin_addr.s_addr;
	if (inet_netmatch(dst, &rp->dst)) {
		u_long ip1 = dst->sin_addr.s_addr;
		u_long ip2 = rp->dst.sin_addr.s_addr;
		if (inet_netmatch(dst, &rp->out_if)) {
			if ((ip1 & nm) == ip2)
				return(1);
			else
				return(0);
		} else
			return(1);
	}
	return(0);
}

/*
 * Find a route to dst as the kernel would.
 */
static int
rtfindx(struct sockaddr_in *dst) {
	int i;

	for (i = 0; i < nhost; i++)
		if (memcmp(&(routes[i].dst), dst, sizeof(struct sockaddr)) == 0)
			return(i);
	for (i = nhost; i < nroute; i++)
		if (net_match(dst, &routes[i]))
			return(i);
	return(-1);
}

static int
read_routes() {
	mib_item_t		*item;
	mib_item_t 		*first_item = nilp(mib_item_t);
	mib2_ipRouteEntry_t	*rp;
	int			ret_code = 1;
	int			doinghost;
	int			i;
	

	/* Get the route list. */
	/* TBD: don't need to do this every time.  Only to a new mibget every
	 *  X seconds? */
	if ((first_item = mibget(sd)) == nilp(mib_item_t)) {
		close(sd);
		ret_code = -1;
		goto leave;
	}
	
	/* Look up the entry */
	have_default = 0;
	nroute = 0;
	nhost = 0;
	
	/* host routes first */
	doinghost = 1;
again:
	for ( item = first_item ; item ; item = item->next_item ) {
		/* skip all the other trash that comes up the mib stream */
		if ((item->group != MIB2_IP) || (item->mib_id != MIB2_IP_21))
			continue;
		
		rp = (mib2_ipRouteEntry_t *)item->valp;
		for(;(u_long)rp < (u_long)(item->valp + item->length); rp++){
			/* skip routes that aren't what we want this pass */
			if(doinghost) {
				if(!is_host(rp))
					continue;
			}
			else {
				if(!is_gateway(rp))
					continue;
			}
			
			/* fill in the blanks */
			memset(&routes[nroute], 0, sizeof(struct xroute));
			routes[nroute].dst.sin_family = AF_INET;
			routes[nroute].dst.sin_addr.s_addr = rp->ipRouteDest;
			routes[nroute].gw.sin_family = AF_INET;
			routes[nroute].gw.sin_addr.s_addr = rp->ipRouteNextHop;
			routes[nroute].netmask.sin_family = AF_INET;
			routes[nroute].netmask.sin_addr.s_addr = 
				rp->ipRouteMask;
			routes[nroute].out_if.sin_family = AF_INET;
			routes[nroute].out_if.sin_addr.s_addr = 
				rp->ipRouteInfo.re_src_addr;
				
			if(rp->ipRouteIfIndex.o_length >= IFNAMSIZ)
				rp->ipRouteIfIndex.o_length = IFNAMSIZ - 1;
			for(i=0; i<rp->ipRouteIfIndex.o_length; i++)
				routes[nroute].ifname[i] = 
					rp->ipRouteIfIndex.o_bytes[i];
			routes[nroute].ifname[i] = '\0';
			
			if (routes[nroute].dst.sin_addr.s_addr == INADDR_ANY) {
				default_route = routes[nroute];
				have_default = 1;
			} else
				nroute++;

		}
	}
	
	/* net routes next */
	if (doinghost) {
		nhost = nroute;
		doinghost = 0;
		goto again;
	}
leave:
	mibfree(first_item);
	first_item = nilp(mib_item_t);
	return ret_code; 
}

static int
is_host(mib2_ipRouteEntry_t *rp)
{
	if (rp->ipRouteMask == (IpAddress)-1) 
		return 1;
	else
		return 0;
}

static int
is_gateway(mib2_ipRouteEntry_t *rp)
{
	if (rp->ipRouteInfo.re_ire_type == IRE_GATEWAY ||
	    rp->ipRouteInfo.re_ire_type == IRE_NET ||
	    rp->ipRouteInfo.re_ire_type == IRE_ROUTE_ASSOC ||
	    rp->ipRouteInfo.re_ire_type == IRE_ROUTE_REDIRECT)
		return 1;
	else
		return 0;
}



static mib_item_t * 
mibget (int sd)
{
	char			buf[512];
	int			flags;
	int			j, getcode;
	struct strbuf		ctlbuf, databuf;
	struct T_optmgmt_req	*tor = (struct T_optmgmt_req *)buf;
	struct T_optmgmt_ack	*toa = (struct T_optmgmt_ack *)buf;
	struct T_error_ack	*tea = (struct T_error_ack *)buf;
	struct opthdr		*req;
	mib_item_t		*first_item = (mib_item_t *)0;
	mib_item_t		*last_item  = (mib_item_t *)0;
	mib_item_t		*temp;

	tor->PRIM_type = T_OPTMGMT_REQ;
	tor->OPT_offset = sizeof (struct T_optmgmt_req);
	tor->OPT_length = sizeof (struct opthdr);
	tor->MGMT_flags = T_CURRENT;
	req = (struct opthdr *)&tor[1];
	req->level = MIB2_IP;		/* any MIB2_xxx value ok here */
	req->name  = 0;
	req->len   = 0;

	ctlbuf.buf = buf;
	ctlbuf.len = tor->OPT_length + tor->OPT_offset;
	flags = 0;
	if (putmsg(sd, &ctlbuf, (struct strbuf *)0, flags) == -1) {
		perror("mibget: putmsg(ctl) failed");
		goto error_exit;
	}
	/*
	 * each reply consists of a ctl part for one fixed structure
	 * or table, as defined in mib2.h.  The format is a T_OPTMGMT_ACK,
	 * containing an opthdr structure.  level/name identify the entry,
	 * len is the size of the data part of the message.
	 */
	req = (struct opthdr *)&toa[1];
	ctlbuf.maxlen = sizeof (buf);
	j = 1;
	while (1) {
		flags = 0;
		getcode = getmsg(sd, &ctlbuf, (struct strbuf *)0, &flags);
		if (getcode == -1) {
			perror("mibget getmsg(ctl) failed");
#ifdef ndef
			if (Dflag) {
				(void) fprintf(stderr,
					"#   level   name    len\n");
				i = 0;
				for (last_item = first_item; last_item;
					last_item = last_item->next_item)
					(void) printf("%d  %4ld   %5ld   %ld\n",
						++i,
						last_item->group,
						last_item->mib_id,
						last_item->length);
			}
#endif
			goto error_exit;
		}
		if (getcode == 0 &&
				ctlbuf.len >= sizeof (struct T_optmgmt_ack) &&
				toa->PRIM_type == T_OPTMGMT_ACK &&
				toa->MGMT_flags == T_SUCCESS &&
				req->len == 0) {
#ifdef ndef
			if (Dflag)
				(void) printf(
		"mibget getmsg() %d returned EOD (level %ld, name %ld)\n",
					j, req->level, req->name);
#endif
			return (first_item);		/* this is EOD msg */
		}

		if (ctlbuf.len >= sizeof (struct T_error_ack) &&
				tea->PRIM_type == T_ERROR_ACK) {
			(void) fprintf(stderr,
	"mibget %d gives T_ERROR_ACK: TLI_error = 0x%lx, UNIX_error = 0x%lx\n",
				j, tea->TLI_error, tea->UNIX_error);
			errno = (tea->TLI_error == TSYSERR)
				? tea->UNIX_error : EPROTO;
			goto error_exit;
		}

		if (getcode != MOREDATA ||
				ctlbuf.len < sizeof (struct T_optmgmt_ack) ||
				toa->PRIM_type != T_OPTMGMT_ACK ||
				toa->MGMT_flags != T_SUCCESS) {
			(void) printf(
	"mibget getmsg(ctl) %d returned %d, ctlbuf.len = %d, PRIM_type = %ld\n",
				j, getcode, ctlbuf.len, toa->PRIM_type);
			if (toa->PRIM_type == T_OPTMGMT_ACK)
				(void) printf(
	"T_OPTMGMT_ACK: MGMT_flags = 0x%lx, req->len = %ld\n",
					toa->MGMT_flags, req->len);
			errno = ENOMSG;
			goto error_exit;
		}

		temp = (mib_item_t *)malloc(sizeof (mib_item_t));
		if (!temp) {
			perror("mibget malloc failed");
			goto error_exit;
		}
		if (last_item)
			last_item->next_item = temp;
		else
			first_item = temp;
		last_item = temp;
		last_item->next_item = (mib_item_t *)0;
		last_item->group = req->level;
		last_item->mib_id = req->name;
		last_item->length = req->len;
		last_item->valp = (char *)malloc((int)req->len);
#ifdef ndef
		if (Dflag)
			(void) printf(
	"msg %d:  group = %4ld   mib_id = %5ld   length = %ld\n",
				j, last_item->group, last_item->mib_id,
				last_item->length);
#endif

		databuf.maxlen = last_item->length;
		databuf.buf    = last_item->valp;
		databuf.len    = 0;
		flags = 0;
		getcode = getmsg(sd, (struct strbuf *)0, &databuf, &flags);
		if (getcode == -1) {
			perror("mibget getmsg(data) failed");
			goto error_exit;
		} else if (getcode != 0) {
			goto error_exit;
		}
		j++;
	}

error_exit:;
	while (first_item) {
		last_item = first_item;
		first_item = first_item->next_item;
		free(last_item);
	}
	return (first_item);
}

static void
mibfree(mib_item_t *first_item)
{
	mib_item_t	*last_item;
	
	while(first_item) {
		last_item = first_item;
		first_item = first_item->next_item;
		free(last_item);
	}
}

int
unicast_init() {
	sd = open("/dev/ip", O_RDWR);
	if (sd == -1) {
		log(LOG_INFO, 0, "Unicast routing information unavailable\n");
		init_failed = 1;
		return 0;
	}
	return 1;
}

int
unicast_route1(u_long addr) {
	int i;
	struct sockaddr_in sin;
	if (init_failed)
		return(-1);
	if (read_routes() == -1)
		return(-1);
		
	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = addr;

	i = rtfindx(&sin);
	if (i == -1 && have_default) {
		if(strlen(default_route.ifname) == 0) {
			/* The interface name is not set on def route.  Look it
			 *  up. 
			 */
			if((i = rtfindx(&default_route.gw)) == -1)
				return(-1);
		}
		if (IsDebug(DEBUG_ROUTE)) {
			log(LOG_DEBUG, 0, "Using default route ...\n");
			log(LOG_DEBUG, 0, "route out interface %s\n", 
							default_route.ifname);
		}
		return(ifname_to_if(default_route.ifname));
	} else if (i == -1) {
		if (IsDebug(DEBUG_ROUTE))
			log(LOG_DEBUG, 0, "No route\n");
		return(-1);
	}
	if(strlen(routes[i].ifname) == 0) {
		/* The interface name is not set on GW routes.  Look it
		 *  up. 
		 */
		if((i = rtfindx(&routes[i].gw)) == -1)
			return(-1);
	}
	if (IsDebug(DEBUG_ROUTE))
		log(LOG_DEBUG, 0, "route out interface %s\n", routes[i].ifname);
	return(ifname_to_if(routes[i].ifname));
}

#else /* end of SOLARIS */

#ifdef sgi_53

/* Do not have support for access to unicast routing table in IRIX 5.3.
 */
unicast_init()
	{
#ifdef HOST_ONLY
	return(0);
#else
	return(1);
#endif
}

int
unicast_route1(u_long dest)
{
	/*  This dummy routine is here to allow pre-routing-socket code to
	 *  compile.  It should never be called.
	 */
	assert(0);
	return(0);
}

#else  /* end sgi_53 */


#ifdef Linux
/*
 *
 * This routine ported by mrpark@palgong.kyungpook.ac.kr
 * Please send e-mail to mrpark@palgong.kyungpook.ac.kr
 * 
*/
#include <linux/socket.h>
#include <linux/inet.h>
#include <net/route.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include "rsvp_daemon.h"

/*
 * This structure contains data shared by many of routes.
 */	


#define RT_BH_REDIRECT		0
#define RT_BH_GARBAGE_COLLECT 	1
#define RT_BH_FREE	 	2

/*
 * Defines
 */
#define MAX_ROUTES	(4*512)
#ifndef T_CURRENT
#define T_CURRENT	MI_T_CURRENT
#endif

/*
 * Structure definitions, it contains /proc/net/rt_cache file
 */
typedef struct mib_item_s {
	struct mib_item_s	*next_item;
	char			ifname[IFNAMSIZ];
	__u32			dst;
	__u32			gateway;
	int			flags;
	int			refcnt;
	int			use;
	int			metric;
	__u32			source;
	__u32			netmask;
	int			mtu;
	long			window;
	int			irtt;
} mib_item_t;

struct xroute {
	char			ifname[IFNAMSIZ];
	struct	sockaddr_in	netmask;
	struct	sockaddr_in	out_if;
	struct	sockaddr_in	dst;
	struct	sockaddr_in	gw;
};

static	mib_item_t	*mibget(FILE *);
static 	void		mibfree(mib_item_t *);
static	int		is_host(mib_item_t	*rp);
static	int		is_gateway(mib_item_t *rp);
static	int		read_routes();
static	int		rtfindx(struct sockaddr_in *);
static 	int		ifname_to_if( char *) ;
static 	int		net_match(struct sockaddr_in * , struct xroute *);
static	int		inet_netmatch(struct sockaddr_in * , struct sockaddr_in *); 
 	/* check whether which net contained of destination addr */ 

/*
 * Global setting
 */
static FILE		*sd;
static struct xroute	routes[MAX_ROUTES];
static struct xroute	default_route;
static int		have_default=0;
static int		nhost = 0;
static int		nroute = 0;

	
int
unicast_init()
{
	sd = fopen("/proc/net/rt_cache","r");
	if(sd==NULL) {
		perror("can't open mib stream");
		return 0;
	}
	return 1;
}

int unicast_route1(u_long addr) {
	int i;
	struct sockaddr_in	sin;

	printf ("unicast_route1:addr=%ux\n",addr);
	if(read_routes() == -1)
	{
		fprintf(stderr,"errors read_routes routine!!\n");
		return (-1);
	}
	memset(&sin, 0, sizeof(sin));
	sin.sin_family= AF_INET;
	sin.sin_addr.s_addr=addr;


	i = rtfindx(&sin);
	if( i== -1 && have_default) {
		fprintf(stderr,"Inside have default routine\n");
		if( IsDebug(DEBUG_ROUTE) ) {
			log(LOG_DEBUG, 0, "Using default route ...\n");
			log(LOG_DEBUG,0,"route out interface %s\n", default_route.ifname);
		}
		return(ifname_to_if(default_route.ifname));
	} else if( i==-1) {
		if( IsDebug(DEBUG_ROUTE)) 
			log(LOG_DEBUG,0,"No Route \n");
		return(-1);
	}
		log(LOG_DEBUG, 0, "route out interface %s\n",routes[i].ifname);
	if(IsDebug(DEBUG_ROUTE))
		log(LOG_DEBUG, 0, "route out interface %s\n",routes[i].ifname);
	return(ifname_to_if(routes[i].ifname));
}

static int
ifname_to_if( char *name)
{
	int i;
	
	for(i=0;i<if_num;i++)
		if(strncmp(if_vec[i].if_name,name,IFNAMSIZ) ==0)
			return	(i);
	log(LOG_ERR,0, "Couldn't find interface %s\n",name);
	return(-1);
}

static int
rtfindx(struct sockaddr_in *dst) {
	int i;
	
	for(i=0;i<nhost; i++)
	{
		if(memcmp(&(routes[i].dst),dst, sizeof(struct sockaddr_in)) ==0 )
			return(i);
	}
	for(i=0;i< nroute; i++)
		if( net_match(dst, &routes[i]))
			return(i);
	return(-1);
}

static int
net_match(struct sockaddr_in * dst, struct xroute *rp) 	/* check whether which net contained of destination addr */ 
{
	u_long nm = rp->netmask.sin_addr.s_addr;

	if( inet_netmatch(dst, &rp->dst) ) {
		u_long ip1 = dst->sin_addr.s_addr;
		u_long ip2 = rp->dst.sin_addr.s_addr;
		if(inet_netmatch(dst, &rp->out_if)) {
			if((ip1&nm) == ip2)
				return(1);
			else
				return(0);
		} else 
			return(1);
	}
	return(0);
}

static int
inet_netmatch(struct sockaddr_in * sin1, struct sockaddr_in *sin2) 
{
	return(inet_netof(sin1->sin_addr) == inet_netof(sin2->sin_addr));
}


static mib_item_t*
mibget(FILE *sd)
{
	char	buf[13][127];
	int	t_refcnt,t_use,t_metric,t_mtu,t_irtt,t_hh,t_arp;
	int	t_flags;
	long 	t_window;
	unsigned long	t_dst,t_gateway,t_source;

	mib_item_t *fibs;
	mib_item_t *fib_first;
	mib_item_t *fib_last;

	fib_first = (mib_item_t *)NULL;
	fib_last = (mib_item_t *)NULL;

	rewind(sd);
	fscanf(sd, "%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n", 
	buf[0],buf[1],buf[2],buf[3],buf[4],buf[5],buf[6],buf[7],buf[8],buf[9], \
	buf[10],buf[11],buf[12]);

	while(feof(sd)==0) {
	fibs=(mib_item_t *)malloc(sizeof(mib_item_t));
		fscanf(sd, "%s\t%08lX\t%08lX\t%02X\t%d\t%d\t%d\t%08lX\t%d\t%ld\t%d\t%d\t%s\n", 
		buf[0],&t_dst,&t_gateway,&t_flags,&t_refcnt,&t_use,&t_metric,&t_source,&t_mtu,&t_window , \
		&t_irtt,&t_hh,buf[1]);

		t_arp = atoi(buf[1]);

		strcpy(fibs->ifname,buf[0]);

		fibs->dst = t_dst;
		fibs->gateway = t_gateway;
		fibs->flags = t_flags;
		fibs->refcnt = t_refcnt;
		fibs->use = t_use;
		fibs->metric = t_metric;
		fibs->source = t_source;
		fibs->mtu = t_mtu;
		fibs->window = t_window;
		fibs->irtt=t_irtt;

		fibs->next_item=(mib_item_t*)NULL;

		if(fib_last != NULL) {
			fib_last->next_item = fibs;
			fib_last = fibs;
		}
		else {
			fib_last = fibs;	
			fib_first = fibs;
		}
	} /* end while eof */

	if(fib_first)
		return(fib_first);
	else
		return((mib_item_t*)NULL);
}


static int
read_routes() 
{
	mib_item_t	*item;
	mib_item_t	*first_item = (mib_item_t *)NULL;
	int		ret_code =1;
	int		doinghost;
	
	first_item = mibget(sd);
	if(first_item == (mib_item_t *)NULL) {
		fclose(sd);
		perror( "Can't access routing table fib!!!");
		ret_code= -1;
		goto leave;
	}
	have_default = nroute = nhost=0;
	doinghost=1;
	item=first_item;

again:
	for(item; item; item=item->next_item) {
		if(doinghost){
			if(!is_host(item))
				continue;
		} else {
			if(!is_gateway(item))
				continue;
		}
		memset(&routes[nroute],0,sizeof(struct xroute));
		routes[nroute].dst.sin_family=AF_INET;
		routes[nroute].dst.sin_addr.s_addr = item->dst;
		routes[nroute].gw.sin_family=AF_INET;
		routes[nroute].gw.sin_addr.s_addr = item->gateway;
		routes[nroute].netmask.sin_family=AF_INET;
		if(routes[nroute].netmask.sin_addr.s_addr == (__u32)NULL)
			routes[nroute].netmask.sin_addr.s_addr = 0XFFFFFFFF;
		routes[nroute].out_if.sin_family=AF_INET;
		routes[nroute].out_if.sin_addr.s_addr=item->source;
		strcpy(routes[nroute].ifname,item->ifname);
	if( routes[nroute].gw.sin_addr.s_addr==INADDR_ANY) {
	fprintf(stderr,"%d\t%08lX\t%08lX\t%08lX\t%08lX\n",nroute,
	(unsigned long)routes[nroute].dst.sin_addr.s_addr,(unsigned long)routes[nroute].gw.sin_addr.s_addr, \
	(unsigned long)routes[nroute].netmask.sin_addr.s_addr,(unsigned long)routes[nroute].out_if.sin_addr.s_addr);
		default_route=routes[nroute];
		have_default=1;
	} else 
		nroute++;
	}
	/*	net routes next */
	if(doinghost) {
		nhost = nroute;
		doinghost=0;
		goto again;
	}
leave:
	mibfree(first_item);
	first_item = (mib_item_t*)NULL;
	return ret_code;
}

static int 
is_host(mib_item_t *rp)
{
	if(rp->flags & RTF_HOST)
	{
		return 1;
	}
	else
		return 0;
}

static int
is_gateway(mib_item_t *rp)
{
	if(rp->flags == RTF_GATEWAY ||
	   rp->flags != RTF_HOST )
/*
	   rp->flags == IRE_ROUTE_ASSOC ||
	   rp->flags == IRE_ROUTE_REDIRECT)
*/
	{
		return 1;
	}
	else
		return 0;
}

static void
mibfree(mib_item_t *first_item)
{
	mib_item_t	*last_item;
	
	while(first_item) {
		last_item=first_item;
		first_item = first_item->next_item;
		free(last_item);
	}
}

#else /* mrpark@palgong.kyungpook.ac.kr	Linux*/

#ifndef Linux

/*	Following code should work for many systems derived from
 *	older BSDs, but it is known to work for Sun OS.
 */

/*
 * Copyright (c) 1983 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */
#include <fcntl.h>
#include <kvm.h>
#include <nlist.h>
#include <stdio.h>

#include <sys/sockio.h>
#include <sys/mbuf.h>

#include <net/if.h>
#include <net/route.h>
#include <netinet/in.h>

#include <netdb.h>

#include <assert.h>

#include <syslog.h>

struct nlist nl[] = {
#define	N_RTHOST	0
	{ "_rthost" },
#define	N_RTNET		1
	{ "_rtnet" },
#define N_RTHASHSIZE	2
	{ "_rthashsize" },
	{""}
};

kvm_t	*kd;
char	usage[] = "[-n] host";

struct xroute {
	char ifname[IFNAMSIZ];
	struct sockaddr_in netmask;
	struct sockaddr_in out_if;
	struct sockaddr_in dst;
	struct sockaddr_in gw;
} routes[256];

int nhost = 0;
int nroute = 0;

struct xroute default_route;
int have_default = 0;

int unicast_fd;

/* forward declarations */
int read_routes();
int net_match(struct sockaddr_in *, struct xroute *);
int rtfindx(struct sockaddr_in *);
int ifname_to_if(char *);
int kread(unsigned long addr, char *buf, unsigned nbytes);
static int inet_netmatch(struct sockaddr_in *, struct sockaddr_in *);

int
unicast_init() {
	if ((kd = kvm_open(NULL, NULL, NULL, O_RDONLY, NULL)) == NULL) {
		log(LOG_INFO, 0, "Unicast routing information unavailable\n");
		init_failed = 1;
		return(0);
	}
	if (kvm_nlist(kd, nl) < 0) {
		log(LOG_ERR, 0, "netstat: bad namelist\n");
		init_failed = 1;
		return(0);
	}
	unicast_fd = socket(AF_INET, SOCK_DGRAM, 0);
	if (unicast_fd < 0) {
		log(LOG_ERR, errno, "unicast socket");
		init_failed = 1;
		return(0);
	}
	return(1);
}

int
unicast_route1(u_long addr) {
	int i;
	struct sockaddr_in sin;
	if (init_failed)
		return(-1);
	if (read_routes() == -1)
		return(-1);
	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = addr;
	i = rtfindx(&sin);
	if (i == -1 && have_default) {
		if (IsDebug(DEBUG_ROUTE)) {
			log(LOG_DEBUG, 0, "Using default route ...\n");
			log(LOG_DEBUG, 0, "route out interface %s\n", 
						default_route.ifname);
		}
		return(ifname_to_if(default_route.ifname));
	} else if (i == -1) {
		if (IsDebug(DEBUG_ROUTE))
			log(LOG_DEBUG, 0, "No route\n");
		return(-1);
	}
	if (IsDebug(DEBUG_ROUTE))
		log(LOG_DEBUG, 0, "route out interface %s\n", routes[i].ifname);
	return(ifname_to_if(routes[i].ifname));
}

int
ifname_to_if(char *name) {
	int i;
	for (i = 0; i < if_num; i++)
		if (strncmp(if_vec[i].if_name, name, IFNAMSIZ) == 0)
			return(i);
	log(LOG_ERR, 0, "Couldn't find interface %s\n", name);
	return(-1);
}
int
read_routes() {
	struct mbuf mbf, *mbfp = &mbf;
	register struct mbuf *m;
	struct mbuf **rhash;
	int i, doinghost = 1;
	int hashsize;
	struct ifreq ifrbuf, *ifr = &ifrbuf;
	int status;

	kread(nl[N_RTHASHSIZE].n_value, (char *)&hashsize, sizeof (hashsize));
	rhash = (struct mbuf **)malloc( hashsize*sizeof (struct mbuf *) );
	kread(nl[N_RTHOST].n_value, (char *)rhash, hashsize*sizeof (struct mbuf *));
	nroute = 0;
again:
	for (i = 0; i < hashsize; i++) {
		if (rhash[i] == 0)
			continue;
		m = rhash[i];
		while (m) {
			struct ifnet ifz, *ifp = &ifz;
			struct rtentry *rt;
			struct ifaddr ifx, *ifxp = &ifx;
			char ifbuf[IFNAMSIZ];

			kread((u_long)m, (char *)mbfp, sizeof(struct mbuf));
			rt = mtod(mbfp, struct rtentry *);
			assert(rt->rt_ifp);
			routes[nroute].dst = *(struct sockaddr_in *)&rt->rt_dst;
			routes[nroute].gw = *(struct sockaddr_in *)&rt->rt_gateway;
			kread((u_long)rt->rt_ifp, (char *)ifp, sizeof(struct ifnet));
			assert(ifp->if_addrlist);
			kread((u_long)ifp->if_addrlist, (char *)ifxp, sizeof(struct ifaddr));
			assert(ifxp->ifa_addr.sa_family == AF_INET);
			routes[nroute].out_if = *(struct sockaddr_in *)&ifx.ifa_addr;
			kread((u_long)ifp->if_name, ifbuf, IFNAMSIZ);
			sprintf(routes[nroute].ifname, "%s%d", ifbuf, 
				ifp->if_unit);
			memcpy(ifr->ifr_name, routes[nroute].ifname, IFNAMSIZ);
			status = ioctl(unicast_fd, SIOCGIFNETMASK, ifr);
			if (status < 0) {
				perror("ioctl");
				return(-1);
			}
			routes[nroute].netmask = *(struct sockaddr_in *)&ifr->ifr_addr;
			if (routes[nroute].dst.sin_addr.s_addr == INADDR_ANY) {
				default_route = routes[nroute];
				have_default = 1;
			} else
				nroute++;
			assert(nroute < 256);
			m = mbfp->m_next;
		}
	}
	if (doinghost) {
		kread(nl[N_RTNET].n_value, (char *)rhash, hashsize*sizeof (struct mbuf *));
		nhost = nroute;
		doinghost = 0;
		goto again;
	}
	free(rhash);
	return(1);
}

int
kread(unsigned long addr, char *buf, unsigned nbytes) {
	return kvm_read(kd, addr, buf, nbytes);
}

/*
 * Find a route to dst as the kernel would.
 */
int
rtfindx(struct sockaddr_in *dst) {
	int i;

	for (i = 0; i < nhost; i++)
		if (memcmp(&routes[i].dst, dst, sizeof(struct sockaddr)) == 0)
			return(i);
	for (i = nhost; i < nroute; i++)
		if (net_match(dst, &routes[i]))
			return(i);
	return(-1);
}

int
net_match(struct sockaddr_in *dst, struct xroute *rp) {
	u_long nm = rp->netmask.sin_addr.s_addr;
	if (inet_netmatch(dst, &rp->dst)) {
		u_long ip1 = dst->sin_addr.s_addr;
		u_long ip2 = rp->dst.sin_addr.s_addr;
		if (inet_netmatch(dst, &rp->out_if)) {
			if ((ip1 & nm) == ip2)
				return(1);
			else
				return(0);
		} else
			return(1);
	}
	return(0);
}

int
inet_netmatch(struct sockaddr_in *sin1, struct sockaddr_in *sin2) {
	return (inet_netof(sin1->sin_addr) == inet_netof(sin2->sin_addr));
}
#endif
#endif /* ! Linux   mrpark@ */
#endif /* ! sgi */
#endif /* ! SOLARIS */
#endif /* ! PF_ROUTE */
