#ifndef	__rsvp_socks_h__
#define	__rsvp_socks_h__

/*
 * @(#) $Id: rsvp_socks.h,v 4.7 1997/12/09 23:53:44 lindell Exp $
 */

/************************ rsvp_socks.h *******************************
 *                                                                   *
 *    IPv4/IPv6 Sockets Interface based on RFC2133 and more          *
 *                                                                   *
 *********************************************************************/

/****************************************************************************

            RSVPD -- ReSerVation Protocol Daemon

                USC Information Sciences Institute
                Marina del Rey, California

		Original Version: Shai Herzog, Nov. 1993.
		Current Version:  Bob Lindell, July 1997.

  Copyright (c) 1997 by the University of Southern California
  All rights reserved.

  Permission to use, copy, modify, and distribute this software and its
  documentation in source and binary forms for any purpose and without
  fee is hereby granted, provided that both the above copyright notice
  and this permission notice appear in all copies, and that any
  documentation, advertising materials, and other materials related to
  such distribution and use acknowledge that the software was developed
  in part by the University of Southern California, Information
  Sciences Institute.  The name of the University may not be used to
  endorse or promote products derived from this software without
  specific prior written permission.

  THE UNIVERSITY OF SOUTHERN CALIFORNIA makes no representations about
  the suitability of this software for any purpose.  THIS SOFTWARE IS
  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  Other copyrights might apply to parts of this software and are so
  noted when applicable.

********************************************************************/

#if	defined(AF_INET6) && defined(IPPROTO_IPV6)
#define	USE_IPV6
#endif	/* defined(AF_INET6) && defined(IPPROTO_IPV6) */
#ifdef USE_IPV6
#include <linux/dret/in6.h>
extern const struct in6_addr in6addr_any; 
#define  IN6_IS_ADDR_MULTICAST(a) ((a)->s6_addr8[0] == 0xff)
#endif


/******************************************************************************
 *
 *	Strictly RFC 2133
 *
 *****************************************************************************/

#ifndef	INET_ADDRSTRLEN
#define	INET_ADDRSTRLEN		sizeof("XXX.XXX.XXX.XXX")
#endif	/* INET_ADDRSTRLEN */

#if	!(defined(__FreeBSD__) && defined(USE_IPV6))

struct if_nameindex {
	unsigned int if_index;
	char *if_name;
};


extern struct if_nameindex * if_nameindex();
extern void if_freenameindex(struct if_nameindex *);
extern unsigned int if_nametoindex(const char *ifname);
extern char * if_indextoname(unsigned int ifindex,char *ifname);

extern const char * inet_ntop2(int af,const void *src,char *dst,size_t size);
extern int inet_pton2(int af,const char *src,void *dst);

extern struct hostent * gethostbyname_2(const char *name,int af);

#else	/* !(defined(__FreeBSD__) && defined(USE_IPV6)) */

#define	ipv6_mreq	oipv6_mreq

#endif	/* !(defined(__FreeBSD__) && defined(USE_IPV6)) */

/******************************************************************************
 *
 *	Extensions to RFC 2133
 *
 *****************************************************************************/

#ifdef	USE_IPV6
#define	MAX_ADDRSTRLEN		INET6_ADDRSTRLEN
#else	/* USE_IPV6 */
#define	MAX_ADDRSTRLEN		INET_ADDRSTRLEN
#endif	/* USE_IPV6 */
#define	MAX_ADDRPORTSTRLEN	(MAX_ADDRSTRLEN + sizeof("/XXXXX"))
#define	MAX_HOSTSTRLEN		MAXHOSTNAMELEN
#define	MAX_HOSTPORTSTRLEN	(MAX_HOSTSTRLEN + sizeof("/XXXXX"))

#define	IN_IS_ADDR_MULTICAST(x)		IN_MULTICAST(ntoh32((x)->s_addr))
#define	IN_ARE_ADDR_EQUAL(x,y)		((x)->s_addr == (y)->s_addr)
#define IN6_ARE_ADDR_EQUAL(a,b)          \
        (((a)->s6_addr32[0] == (b)->s6_addr32[0]) && \
         ((a)->s6_addr32[1] == (b)->s6_addr32[1]) && \
         ((a)->s6_addr32[2] == (b)->s6_addr32[2]) && \
         ((a)->s6_addr32[3] == (b)->s6_addr32[3]))

#define	MAX_INTERFACES		128

extern const struct in_addr inaddr_any;


struct if_attributes {
	struct sockaddr *addr;
	unsigned int prefix;
	int flags;
};

extern struct if_attributes *if_attributes(unsigned int index);
extern unsigned int if_addrtoindex(struct sockaddr *);
extern struct sockaddr *if_indextoaddr(unsigned int index,int af);

/******************************************************************************
 *
 *	Macros and functions for socket structures.
 *
 *****************************************************************************/

#ifdef	__FreeBSD__
#define	NET_SOCKADDR_SIZE_IPv4(x) \
	{ \
		(x)->sin_len = sizeof(struct sockaddr_in); \
	}
#define	NET_SOCKADDR_SIZE_IPv6(x) \
	{ \
		(x)->sin6_len = sizeof(struct sockaddr_in6); \
	}
#else	/* __FreeBSD__ */
#define	NET_SOCKADDR_SIZE_IPv4(x)
#define	NET_SOCKADDR_SIZE_IPv6(x)
#endif	/* __FreeBSD__ */

#define NET_SOCKADDR_IPv4(x,addr)	NET_SOCKADDR_UDP_IPv4(x,addr,hton16(0))
#define	NET_SOCKADDR_UDP_IPv4(x,addr,port) { \
		memset((char *) (x),0,sizeof(struct sockaddr_in)); \
		(x)->sin_family = AF_INET; \
		(x)->sin_addr = addr; \
		(x)->sin_port = port; \
		NET_SOCKADDR_SIZE_IPv4(x); \
	}

#ifdef	USE_IPV6
#define NET_SOCKADDR_IPv6(x,addr)	NET_SOCKADDR_UDP_IPv6(x,addr,hton16(0))
#define	NET_SOCKADDR_UDP_IPv6(x,addr,port) { \
		memset((char *) (x),0,sizeof(struct sockaddr_in6)); \
		(x)->sin6_family = AF_INET6; \
		(x)->sin6_addr = addr; \
		(x)->sin6_port = port; \
		NET_SOCKADDR_SIZE_IPv6(x); \
	}
#endif	/* USE_IPV6 */

extern const char * net_inaddr_print(const char *s,int af);
extern const char * net_inaddr_host(const char *s,int af);
extern const char * net_sockaddr_print(const struct sockaddr *s);
extern const char * net_sockaddr_host(const struct sockaddr *s);

/******************************************************************************
 *
 *	Useful Definitions and Macros
 *
 *****************************************************************************/

#ifndef	TRUE
#define	TRUE (0 == 0)
#endif	/* TRUE */
#ifndef	FALSE
#define	FALSE (0 != 0)
#endif	/* FALSE */

#define	SYS_ERROR		(-1)
#define	SYS_NOERROR		0
#define	FAILED(x)		((x) == SYS_ERROR)

#define	TWO_N(n)	(1 << (n))
#define	NBITS(x)	(8 * sizeof(x))
#define	BSET_FRAC(n)	TWO_N(n % NBITS(char))
#define	BSET_WHOLE(n)	((n) / NBITS(char))
#define BSET_CLR(n,x)	(((char *) (x))[BSET_WHOLE(n)] &= ~BSET_FRAC(n))
#define BSET_SET(n,x)	(((char *) (x))[BSET_WHOLE(n)] |= BSET_FRAC(n))
#define BSET_ISSET(n,x)	((((char *) (x))[BSET_WHOLE(n)] & BSET_FRAC(n)) != 0)

/*
 *	These are generic macros for operations on type fields.
 */

#define	NET_GET_TYPE(x)			((x)->type)
#define	NET_SET_TYPE(x,y)		(NET_GET_TYPE(x) = (y))
#define	NET_EQUAL_TYPE(x,y)		(NET_GET_TYPE(x) == NET_GET_TYPE(y))
#define	NET_UNEQUAL_TYPE(x,y)		(NET_GET_TYPE(x) != NET_GET_TYPE(y))

#define	NET_SET_TYPE_AND_VALUE(a,b,c,d)	{ NET_SET_TYPE(a,b); c(a) = (d); }

/******************************************************************************
 *
 *	Define a network address type.
 *
 *****************************************************************************/

/*
 *	These enumerated type values, and corresponding types, specify
 *	the type of address to be used as arguments to send and recv
 *	operations.
 */

typedef enum {
	NET_IF_UNSPECIFIED = -1,
	NET_ADDR_IPv4 = 0,
	NET_ADDR_UDP_IPv4,
	NET_ADDR_IPv6,
	NET_ADDR_UDP_IPv6,
	NET_ADDR_SIZE				/* last element */
} net_addr_type;

typedef struct {
	net_addr_type type;
	union {
		struct in_addr addr_ipv4;
		struct sockaddr_in addr_udp_ipv4;
#ifdef	USE_IPV6
		struct in6_addr addr_ipv6;
		struct sockaddr_in6 addr_udp_ipv6;
#endif	/* USE_IPV6 */
	} u;
} net_addr;

#define	NET_GET_ADDR(x)			((x)->u)
#define	NET_GET_ADDR_IPv4(x)		((x)->u.addr_ipv4)
#define	NET_GET_ADDR_UDP_IPv4(x)	((x)->u.addr_udp_ipv4)
#define	NET_GET_ADDR_IPv6(x)		((x)->u.addr_ipv6)
#define	NET_GET_ADDR_UDP_IPv6(x)	((x)->u.addr_udp_ipv6)

#define	NET_SET_ADDR_IPv4(x,y) \
	NET_SET_TYPE_AND_VALUE(x,NET_ADDR_IPv4,NET_GET_ADDR_IPv4,y)
#define	NET_SET_ADDR_UDP_IPv4(x,y) \
	NET_SET_TYPE_AND_VALUE(x,NET_ADDR_UDP_IPv4,NET_GET_ADDR_UDP_IPv4,y)
#define	NET_SET_ADDR3_UDP_IPv4(x,y,z) { \
		NET_SET_TYPE(x,NET_ADDR_UDP_IPv4); \
		NET_SOCKADDR_UDP_IPv4(&NET_GET_ADDR_UDP_IPv4(x),y,z); \
	}
#define	NET_SET_ADDR_IPv6(x,y) \
	NET_SET_TYPE_AND_VALUE(x,NET_ADDR_IPv6,NET_GET_ADDR_IPv6,y)
#define	NET_SET_ADDR_UDP_IPv6(x,y) \
	NET_SET_TYPE_AND_VALUE(x,NET_ADDR_UDP_IPv6,NET_GET_ADDR_UDP_IPv6,y)
#define	NET_SET_ADDR3_UDP_IPv6(x,y,z) { \
		NET_SET_TYPE(x,NET_ADDR_UDP_IPv6); \
		NET_SOCKADDR_UDP_IPv6(&NET_GET_ADDR_UDP_IPv6(x),y,z); \
	}

extern int net_addr_equal(const net_addr *,const net_addr *);
extern const char * net_addr_print(const net_addr *);
extern const char * net_addr_host(const net_addr *);
extern net_addr * net_addr_ip(net_addr *);
extern int net_addr_assign(net_addr *,const struct sockaddr *);
extern int net_addr_assign_udp(net_addr *,const struct sockaddr *);
extern int net_addr_ascii(net_addr *addr,const char *str);

/******************************************************************************
 *
 *	Define a network interface type.
 *
 *****************************************************************************/

/*
 *	These enumerated type values, and corresponding types, specify
 *	the type of network interface identifiers that will be used
 *	when interfacing rsvpd with other modules such as routing and
 *	traffic control.
 */

typedef enum {
	NET_IF_UNKNOWN,
	NET_IF_PHY,
	NET_IF_VIF,
	NET_IF_ID,
	NET_IF_STRING
} net_if_type;

typedef struct {
	int num;
	net_addr addr;
} net_if_vif;

typedef struct {
	net_if_type type;
	union {
		int num;
		char *str;
		net_addr inf;
		net_if_vif vif;
	} u;
} net_if;

#define	NET_GET_IF_ID(x)		((x)->u.num)
#define	NET_GET_IF_STRING(x)		((x)->u.str)
#define	NET_GET_IF_PHY(x)		((x)->u.inf)
#define	NET_GET_IF_VIF_ADDR(x)		((x)->u.vif.addr)
#define	NET_GET_IF_VIF_ID(x)		((x)->u.vif.num)

#define	NET_SET_IF_PHY(x,y) \
	NET_SET_TYPE_AND_VALUE(x,NET_IF_PHY,NET_GET_IF_PHY,y)
#define	NET_SET_IF_ID(x,y) \
	NET_SET_TYPE_AND_VALUE(x,NET_IF_ID,NET_GET_IF_ID,y)
#define	NET_SET_IF_STRING(x,y) \
	NET_SET_TYPE_AND_VALUE(x,NET_IF_STRING,NET_GET_IF_STRING,y)
#define	NET_SET_IF_VIF(x,y,z) { \
		NET_SET_TYPE(x,NET_IF_VIF); \
		NET_GET_IF_VIF_ADDR(x) = y; \
		NET_GET_IF_VIF_ID(x) = z; \
	}

extern int net_if_equal(const net_if *,const net_if *);
extern const char * net_if_print(const net_if *);
extern net_addr * net_get_default_interface();

#endif	/* __rsvp_socks_h__ */
