/*
 * @(#) $Id: rsvp_objs.c,v 4.4 1997/11/15 00:36:57 lindell Exp $
 */

/************************ rsvp_objs.c ********************************
 *                                                                   *
 *    Protocol Independent RSVP Object Manipulation                  *
 *                                                                   *
 *********************************************************************/

/****************************************************************************

            RSVPD -- ReSerVation Protocol Daemon

                USC Information Sciences Institute
                Marina del Rey, California

		Original Version: Shai Herzog, Nov. 1993.
		Current Version:  Bob Lindell, July 1997.

  Copyright (c) 1997 by the University of Southern California
  All rights reserved.

  Permission to use, copy, modify, and distribute this software and its
  documentation in source and binary forms for any purpose and without
  fee is hereby granted, provided that both the above copyright notice
  and this permission notice appear in all copies, and that any
  documentation, advertising materials, and other materials related to
  such distribution and use acknowledge that the software was developed
  in part by the University of Southern California, Information
  Sciences Institute.  The name of the University may not be used to
  endorse or promote products derived from this software without
  specific prior written permission.

  THE UNIVERSITY OF SOUTHERN CALIFORNIA makes no representations about
  the suitability of this software for any purpose.  THIS SOFTWARE IS
  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

  Other copyrights might apply to parts of this software and are so
  noted when applicable.

********************************************************************/

#include "rsvp_daemon.h"
#include "rsvp_api.h"

int
hop_if_create(RSVP_HOP *hop,net_if *inf,int lih)
{
	net_addr *addr;

	switch(NET_GET_TYPE(inf)) {
		case NET_IF_PHY:
			addr = &NET_GET_IF_PHY(inf);
			break;
		case NET_IF_VIF:
			addr = &NET_GET_IF_VIF_ADDR(inf);
			break;
		default:
			return(FALSE);
	}
	switch (NET_GET_TYPE(addr)) {
		case NET_ADDR_IPv4:
			Init_Object(hop,RSVP_HOP,ctype_RSVP_HOP_ipv4);
			hop->hop4_addr = NET_GET_ADDR_IPv4(addr);
			hop->hop4_LIH = lih;
			return(TRUE);
#ifdef	USE_IPV6
		case NET_ADDR_IPv6:
			Init_Object(hop,RSVP_HOP,ctype_RSVP_HOP_ipv6);
			hop->hop6_addr = NET_GET_ADDR_IPv6(addr);
			hop->hop6_LIH = lih;
			return(TRUE);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
hop_if_assign(RSVP_HOP *hop,net_if *inf,int lih)
{
	net_addr *addr;

	if (Obj_Class(hop) != class_RSVP_HOP)
		return(FALSE);
	switch(NET_GET_TYPE(inf)) {
		case NET_IF_PHY:
			addr = &NET_GET_IF_PHY(inf);
			break;
		case NET_IF_VIF:
			addr = &NET_GET_IF_VIF_ADDR(inf);
			break;
		default:
			return(FALSE);
	}
	switch(NET_GET_TYPE(addr)) {
		case NET_ADDR_IPv4:
			Obj_CType(hop) = ctype_RSVP_HOP_ipv4;
			hop->hop4_addr = NET_GET_ADDR_IPv4(addr);
			hop->hop4_LIH = lih;
			return(TRUE);
#ifdef	USE_IPV6
		case NET_ADDR_IPv6:
			Obj_CType(hop) = ctype_RSVP_HOP_ipv6;
			hop->hop6_addr = NET_GET_ADDR_IPv6(addr);
			hop->hop6_LIH = lih;
			return(TRUE);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
hop_confirm_assign(RSVP_HOP *hop,CONFIRM *confirm)
{
	if (Obj_Class(hop) != class_RSVP_HOP)
		return(FALSE);
	if (Obj_Class(confirm) != class_CONFIRM)
		return(FALSE);
	switch(Obj_CType(confirm)) {
		case ctype_CONFIRM_ipv4:
			Obj_CType(hop) = ctype_RSVP_HOP_ipv4;
			hop->hop4_addr = confirm->conf4_addr;
			return(TRUE);
#ifdef	USE_IPV6
		case ctype_CONFIRM_ipv6:
			Obj_CType(hop) = ctype_RSVP_HOP_ipv6;
			hop->hop6_addr = confirm->conf6_addr;
			return(TRUE);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
hop_lih(RSVP_HOP *hop)
{
	if (Obj_Class(hop) != class_RSVP_HOP)
		return(-1);
	switch(Obj_CType(hop)) {
		case ctype_RSVP_HOP_ipv4:
			return(hop->hop4_LIH);
#ifdef	USE_IPV6
		case ctype_RSVP_HOP_ipv6:
			return(hop->hop6_LIH);
#endif	/* USE_IPV6 */
		default:
			return(-1);
	}
}

net_addr *
hop_addr(RSVP_HOP *hop)
{
	static net_addr addr;

	if (Obj_Class(hop) != class_RSVP_HOP)
		return(NULL);
	switch(Obj_CType(hop)) {
		case ctype_RSVP_HOP_ipv4:
			NET_SET_ADDR_IPv4(&addr,hop->hop4_addr);
			return(&addr);
#ifdef	USE_IPV6
		case ctype_RSVP_HOP_ipv6:
			NET_SET_ADDR_IPv6(&addr,hop->hop6_addr);
			return(&addr);
#endif	/* USE_IPV6 */
		default:
			return(NULL);
	}
}

int
hop_eq(RSVP_HOP *hop1,RSVP_HOP *hop2)
{
	if (Obj_Class(hop1) != class_RSVP_HOP)
		return(FALSE);
	if (Obj_Class(hop1) != Obj_Class(hop2))
		return(FALSE);
	if (Obj_CType(hop1) != Obj_CType(hop2))
		return(FALSE);
	switch(Obj_CType(hop1)) {
		case ctype_RSVP_HOP_ipv4:
			return(memcmp((char *) &hop1->hop_u.hop_ipv4,
				(char *) &hop2->hop_u.hop_ipv4,
				sizeof(hop1->hop_u.hop_ipv4)) == 0);
#ifdef	USE_IPV6
		case ctype_RSVP_HOP_ipv6:
			return(memcmp((char *) &hop1->hop_u.hop_ipv6,
				(char *) &hop2->hop_u.hop_ipv6,
				sizeof(hop1->hop_u.hop_ipv6)) == 0);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
hop_addr_lt(RSVP_HOP *hop1,RSVP_HOP *hop2)
{
	if (Obj_Class(hop1) != class_RSVP_HOP)
		return(FALSE);
	if (Obj_Class(hop1) != Obj_Class(hop2))
		return(FALSE);
	if (Obj_CType(hop1) != Obj_CType(hop2))
		return(FALSE);
	switch(Obj_CType(hop1)) {
		case ctype_RSVP_HOP_ipv4:
			return(memcmp((char *) &hop1->hop4_addr,
				(char *) &hop2->hop4_addr,
				sizeof(hop1->hop4_addr)) < 0);
#ifdef	USE_IPV6
		case ctype_RSVP_HOP_ipv6:
			return(memcmp((char *) &hop1->hop6_addr,
				(char *) &hop2->hop6_addr,
				sizeof(hop1->hop6_addr)) < 0);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
hop_addr_eq(RSVP_HOP *hop1,RSVP_HOP *hop2)
{
	if (Obj_Class(hop1) != class_RSVP_HOP)
		return(FALSE);
	if (Obj_Class(hop1) != Obj_Class(hop2))
		return(FALSE);
	if (Obj_CType(hop1) != Obj_CType(hop2))
		return(FALSE);
	switch(Obj_CType(hop1)) {
		case ctype_RSVP_HOP_ipv4:
			return(IN_ARE_ADDR_EQUAL(&hop1->hop4_addr,
				&hop2->hop4_addr));
#ifdef	USE_IPV6
		case ctype_RSVP_HOP_ipv6:
			return(IN6_ARE_ADDR_EQUAL(&hop1->hop6_addr,
				&hop2->hop6_addr));
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
hop_in_scope(RSVP_HOP *hop,SCOPE *scope)
{
	char *cp,*end,*addr;
	int size;

	if (Obj_Class(hop) != class_RSVP_HOP)
		return(FALSE);
	if (Obj_Class(scope) != class_SCOPE)
		return(FALSE);
	switch(Obj_CType(hop)) {
		case ctype_RSVP_HOP_ipv4:
				if (Obj_CType(scope) != ctype_SCOPE_list_ipv4)
					return(FALSE);
				size = sizeof(Scope_list_ipv4);
				addr = (char *) &hop->hop4_addr;
				break;
#ifdef	USE_IPV6
		case ctype_RSVP_HOP_ipv6:
				if (Obj_CType(scope) != ctype_SCOPE_list_ipv6)
					return(FALSE);
				size = sizeof(Scope_list_ipv6);
				addr = (char *) &hop->hop6_addr;
				break;
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
	end = (char *) Next_Object(scope);
	for (cp = (char *) Obj_data(scope);cp < end; cp += size)
		if (memcmp(cp,addr,size) == 0)
			return(TRUE);
	return(FALSE);
}

char *
hop_print(RSVP_HOP *hop)
{
	static char buffer[MAX_HOSTPORTSTRLEN + sizeof(" LIH=XXXXXXXXXX")];

	if (Obj_Class(hop) != class_RSVP_HOP)
		return(NULL);
	switch(Obj_CType(hop)) {
		case ctype_RSVP_HOP_ipv4:
			sprintf(buffer,"%s LIH=%d",IsHopAPI(hop) ? "(API)" :
				net_inaddr_host((char *) &hop->hop4_addr,
				AF_INET),hop->hop4_LIH);
			return(buffer);
#ifdef	USE_IPV6
		case ctype_RSVP_HOP_ipv6:
/*		  printf ("hop_print:RSVP_HOP_IPv6\n");*/
			sprintf(buffer,"%s LIH=%d",IsHopAPI(hop) ? "(API)" :
				net_inaddr_host((char *) &hop->hop6_addr,
				AF_INET6),hop->hop6_LIH);
			return(buffer);
#endif	/* USE_IPV6 */
		default:
			return(NULL);
	}
}

int
session_if_eq(SESSION *session,net_if *inf)
{
	net_addr *addr;

	if (Obj_Class(session) != class_SESSION)
		return(FALSE);
	switch(NET_GET_TYPE(inf)) {
		case NET_IF_PHY:
			addr = &NET_GET_IF_PHY(inf);
			break;
		case NET_IF_VIF:
			addr = &NET_GET_IF_VIF_ADDR(inf);
			break;
		default:
			return(FALSE);
	}
	switch(NET_GET_TYPE(addr)) {
		case NET_ADDR_IPv4:
			if ((Obj_CType(session) != ctype_SESSION_ipv4)
					&& (Obj_CType(session)
					!= ctype_SESSION_ipv4GPI))
				return(FALSE);
			return(IN_ARE_ADDR_EQUAL(&session->sess4_addr,
				&NET_GET_ADDR_IPv4(addr)));
#ifdef	USE_IPV6
		case NET_ADDR_IPv6:
			if ((Obj_CType(session) != ctype_SESSION_ipv6)
					&& (Obj_CType(session)
					!= ctype_SESSION_ipv6GPI))
				return(FALSE);
			return(IN6_ARE_ADDR_EQUAL(&session->sess6_addr,
				&NET_GET_ADDR_IPv6(addr)));
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

net_addr *
session_addr(SESSION *session)
{
	static net_addr addr;

	if (Obj_Class(session) != class_SESSION)
		return(NULL);
	switch(Obj_CType(session)) {
		case ctype_SESSION_ipv4:
		case ctype_SESSION_ipv4GPI:
			NET_SET_ADDR_IPv4(&addr,session->sess4_addr);
			return(&addr);
#ifdef	USE_IPV6
		case ctype_SESSION_ipv6:
		case ctype_SESSION_ipv6GPI:
			NET_SET_ADDR_IPv6(&addr,session->sess6_addr);
			return(&addr);
#endif	/* USE_IPV6 */
		default:
			return(NULL);
	}
}

int
session_eq(SESSION *session1,SESSION *session2)
{
	if (Obj_Class(session1) != class_SESSION)
		return(FALSE);
	if (Obj_Class(session1) != Obj_Class(session2))
		return(FALSE);
	if (Obj_CType(session1) != Obj_CType(session2))
		return(FALSE);
	switch(Obj_CType(session1)) {
		case ctype_SESSION_ipv4:
		case ctype_SESSION_ipv4GPI:
			return(memcmp((char *) &session1->sess_u.sess_ipv4,
				(char *) &session2->sess_u.sess_ipv4,
				sizeof(session1->sess_u.sess_ipv4)) == 0);
#ifdef	USE_IPV6
		case ctype_SESSION_ipv6:
		case ctype_SESSION_ipv6GPI:
			return(memcmp((char *) &session1->sess_u.sess_ipv6,
				(char *) &session2->sess_u.sess_ipv6,
				sizeof(session1->sess_u.sess_ipv6)) == 0);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
session_eq_except_flags(SESSION *session1,SESSION *session2)
{
	if (Obj_Class(session1) != class_SESSION)
		return(FALSE);
	if (Obj_Class(session1) != Obj_Class(session2))
		return(FALSE);
	if (Obj_CType(session1) != Obj_CType(session2))
		return(FALSE);
	switch(Obj_CType(session1)) {
		case ctype_SESSION_ipv4:
		case ctype_SESSION_ipv4GPI:
				if (!IN_ARE_ADDR_EQUAL(&session1->sess4_addr,
						&session2->sess4_addr))
					return(FALSE);
				if (session1->sess4_prot !=
						session2->sess4_prot)
					return(FALSE);
				if (session1->sess4_port ==
						session2->sess4_port)
					return(TRUE);
				return(FALSE);
#ifdef	USE_IPV6
		case ctype_SESSION_ipv6:
		case ctype_SESSION_ipv6GPI:
				if (!IN6_ARE_ADDR_EQUAL(&session1->sess6_addr,
						&session2->sess6_addr))
					return(FALSE);
				if (session1->sess6_prot !=
						session2->sess6_prot)
					return(FALSE);
				if (session1->sess6_port ==
						session2->sess6_port)
					return(TRUE);
				return(FALSE);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
session_eq_except_port(SESSION *session1,SESSION *session2)
{
	if (Obj_Class(session1) != class_SESSION)
		return(FALSE);
	if (Obj_Class(session1) != Obj_Class(session2))
		return(FALSE);
	if (Obj_CType(session1) != Obj_CType(session2))
		return(FALSE);
	switch(Obj_CType(session1)) {
		case ctype_SESSION_ipv4:
		case ctype_SESSION_ipv4GPI:
				if (!IN_ARE_ADDR_EQUAL(&session1->sess4_addr,
						&session2->sess4_addr))
					return(FALSE);
				if (session1->sess4_prot !=
						session2->sess4_prot)
					return(FALSE);
				if (session1->sess4_port == 0)
					return(TRUE);
				if (session2->sess4_port == 0)
					return(TRUE);
				return(FALSE);
#ifdef	USE_IPV6
		case ctype_SESSION_ipv6:
		case ctype_SESSION_ipv6GPI:
				if (!IN6_ARE_ADDR_EQUAL(&session1->sess6_addr,
						&session2->sess6_addr))
					return(FALSE);
				if (session1->sess6_prot !=
						session2->sess6_prot)
					return(FALSE);
				if (session1->sess6_port == 0)
					return(TRUE);
				if (session2->sess6_port == 0)
					return(TRUE);
				return(FALSE);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

char *
session_print(SESSION *session)
{
	static char buffer[MAX_HOSTPORTSTRLEN + sizeof("[XXX]")];
/*	char *tmp;*/

/*	printf ("session_print:START! MAX_HOSTPORTSTRLEN=%d\n",MAX_HOSTPORTSTRLEN);*/
	if (Obj_Class(session) != class_SESSION)
		return(NULL);
	switch(Obj_CType(session)) {
		case ctype_SESSION_ipv4:
		case ctype_SESSION_ipv4GPI:
			sprintf(buffer,"%s/%hu[%d]",net_inaddr_host(
				(char *) &session->sess4_addr,AF_INET),
				ntoh16(session->sess4_port),
				session->sess4_prot);
			return(buffer);
#ifdef	USE_IPV6
		case ctype_SESSION_ipv6:
		case ctype_SESSION_ipv6GPI:
/*		  printf ("session_print:ipv6GPI\n");		*/
/*		  printf ("tmp=%s\n",tmp);*/
		  sprintf(buffer,"%s/%hu[%d]",net_inaddr_host((char *) &session->sess6_addr,AF_INET6),
			  ntoh16(session->sess6_port),
			  session->sess6_prot);
/*		  printf ("return (buffer=%s)\n",buffer);*/
		  return(buffer);

#endif	/* USE_IPV6 */
		default:
			return(NULL);
	}
}

int
session_multicast(SESSION *session)
{
	if (Obj_Class(session) != class_SESSION)
		return(FALSE);
	switch(Obj_CType(session)) {
		case ctype_SESSION_ipv4:
		case ctype_SESSION_ipv4GPI:
			return(IN_IS_ADDR_MULTICAST(&session->sess4_addr));
#ifdef	USE_IPV6
		case ctype_SESSION_ipv6:
		case ctype_SESSION_ipv6GPI:
			return(IN6_IS_ADDR_MULTICAST(&session->sess6_addr));
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

u_int16_t
session_get_port(SESSION *session)
{
	if (Obj_Class(session) != class_SESSION)
		return(0);
	switch(Obj_CType(session)) {
		case ctype_SESSION_ipv4:
		case ctype_SESSION_ipv4GPI:
			return(session->sess4_port);
#ifdef	USE_IPV6
		case ctype_SESSION_ipv6:
		case ctype_SESSION_ipv6GPI:
			return(session->sess6_port);
#endif	/* USE_IPV6 */
		default:
			return(0);
	}
}

int
session_set_port(SESSION *session,u_int16_t port)
{
	if (Obj_Class(session) != class_SESSION)
		return(FALSE);
	switch(Obj_CType(session)) {
		case ctype_SESSION_ipv4:
		case ctype_SESSION_ipv4GPI:
			session->sess4_port = port;
			return(TRUE);
#ifdef	USE_IPV6
		case ctype_SESSION_ipv6:
		case ctype_SESSION_ipv6GPI:
			session->sess6_port = port;
			return(TRUE);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

u_char
session_get_flags(SESSION *session)
{
	if (Obj_Class(session) != class_SESSION)
		return(0);
	switch(Obj_CType(session)) {
		case ctype_SESSION_ipv4:
		case ctype_SESSION_ipv4GPI:
			return(session->sess4_flgs);
#ifdef	USE_IPV6
		case ctype_SESSION_ipv6:
		case ctype_SESSION_ipv6GPI:
			return(session->sess6_flgs);
#endif	/* USE_IPV6 */
		default:
			return(0);
	}
}

int
session_set_flags(SESSION *session,u_char flags)
{
	if (Obj_Class(session) != class_SESSION)
		return(FALSE);
	switch(Obj_CType(session)) {
		case ctype_SESSION_ipv4:
		case ctype_SESSION_ipv4GPI:
			session->sess4_flgs = flags;
			return(TRUE);
#ifdef	USE_IPV6
		case ctype_SESSION_ipv6:
		case ctype_SESSION_ipv6GPI:
			session->sess6_flgs = flags;
			return(TRUE);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
session_create(SESSION *session,net_addr *addr,u_char protid,
	u_char flags,int gpi)
{
/*  printf ("session_create:START! NET_GET_TYPE(addr)=%d, gpi=%d\n",NET_GET_TYPE(addr),gpi);*/
	switch (NET_GET_TYPE(addr)) {
		case NET_ADDR_IPv4:
/*		  printf ("session_create:IPv4:\n");*/
			Init_Object(session,SESSION,
				    gpi ? ctype_SESSION_ipv4GPI : ctype_SESSION_ipv4);
			session->sess4_addr = NET_GET_ADDR_IPv4(addr);
			session->sess4_port = hton16(0);
			session->sess4_prot = protid;
			session->sess4_flgs = flags;
			return(TRUE);
		case NET_ADDR_UDP_IPv4:		 
/*		  printf ("session_create:UDP_IPv4:\n");*/
			Init_Object(session,SESSION,gpi ?
				ctype_SESSION_ipv4GPI : ctype_SESSION_ipv4);
			session->sess4_addr = NET_GET_ADDR_UDP_IPv4(addr)
						.sin_addr;
			session->sess4_port = NET_GET_ADDR_UDP_IPv4(addr)
						.sin_port;
			session->sess4_prot = protid;
			session->sess4_flgs = flags;
			return(TRUE);
#ifdef	USE_IPV6
		case NET_ADDR_IPv6:
/*		  printf ("session_create:IPv6:\n");*/
			Init_Object(session,SESSION,gpi ?
				ctype_SESSION_ipv6GPI : ctype_SESSION_ipv6);
			session->sess6_addr = NET_GET_ADDR_IPv6(addr);
			session->sess6_port = hton16(0);
			session->sess6_prot = protid;
			session->sess6_flgs = flags;
			return(TRUE);
		case NET_ADDR_UDP_IPv6:
/*		  printf ("session_create:UDP_IPv6:\n");*/
			Init_Object(session,SESSION,gpi ?
				ctype_SESSION_ipv6GPI : ctype_SESSION_ipv6);
/*			printf ("a:");*/
			session->sess6_addr = NET_GET_ADDR_UDP_IPv6(addr)
						.sin6_addr;
/*			printf ("b:");*/
			session->sess6_port = NET_GET_ADDR_UDP_IPv6(addr)
						.sin6_port;
/*			printf ("c:");*/
			session->sess6_prot = protid;
/*			printf ("d:");*/
			session->sess6_flgs = flags;
/*			printf ("\n");*/
			return(TRUE);
#endif	/* USE_IPV6 */
		default:
		  printf ("WARNING: unknown ADDR_TYPE encountered !\n");
		  return(FALSE);
	}
}

int
confirm_if_create(CONFIRM *confirm,net_if *inf)
{
	net_addr *addr;

	switch(NET_GET_TYPE(inf)) {
		case NET_IF_PHY:
			addr = &NET_GET_IF_PHY(inf);
			break;
		case NET_IF_VIF:
			addr = &NET_GET_IF_VIF_ADDR(inf);
			break;
		default:
			return(FALSE);
	}
	switch (NET_GET_TYPE(addr)) {
		case NET_ADDR_IPv4:
			Init_Object(confirm,CONFIRM,ctype_CONFIRM_ipv4);
			confirm->conf4_addr = NET_GET_ADDR_IPv4(addr);
			return(TRUE);
#ifdef	USE_IPV6
		case NET_ADDR_IPv6:
			Init_Object(confirm,CONFIRM,ctype_CONFIRM_ipv6);
			confirm->conf6_addr = NET_GET_ADDR_IPv6(addr);
			return(TRUE);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

u_char
errorspec_get_errcode(ERROR_SPEC *errorspec)
{
	if (Obj_Class(errorspec) != class_ERROR_SPEC)
		return(0);
	switch(Obj_CType(errorspec)) {
		case ctype_ERROR_SPEC_ipv4:
			return(errorspec->errspec4_code);
#ifdef	USE_IPV6
		case ctype_ERROR_SPEC_ipv6:
			return(errorspec->errspec6_code);
#endif	/* USE_IPV6 */
		default:
			return(0);
	}
}

int
errorspec_set_errcode(ERROR_SPEC *errorspec,u_char code)
{
	if (Obj_Class(errorspec) != class_ERROR_SPEC)
		return(FALSE);
	switch(Obj_CType(errorspec)) {
		case ctype_ERROR_SPEC_ipv4:
			errorspec->errspec4_code = code;
			return(TRUE);
#ifdef	USE_IPV6
		case ctype_ERROR_SPEC_ipv6:
			errorspec->errspec6_code = code;
			return(TRUE);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

u_char
errorspec_get_errflags(ERROR_SPEC *errorspec)
{
	if (Obj_Class(errorspec) != class_ERROR_SPEC)
		return(0);
	switch(Obj_CType(errorspec)) {
		case ctype_ERROR_SPEC_ipv4:
			return(errorspec->errspec4_flags);
#ifdef	USE_IPV6
		case ctype_ERROR_SPEC_ipv6:
			return(errorspec->errspec6_flags);
#endif	/* USE_IPV6 */
		default:
			return(0);
	}
}

int
errorspec_set_errflags(ERROR_SPEC *errorspec,u_char flags)
{
	if (Obj_Class(errorspec) != class_ERROR_SPEC)
		return(FALSE);
	switch(Obj_CType(errorspec)) {
		case ctype_ERROR_SPEC_ipv4:
			errorspec->errspec4_flags = flags;
			return(TRUE);
#ifdef	USE_IPV6
		case ctype_ERROR_SPEC_ipv6:
			errorspec->errspec6_flags = flags;
			return(TRUE);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

ERROR_SPEC *
errorspec_create(net_if *inf,u_char code,u_int16_t value,u_char flags)
{
	net_addr *addr;
	static ERROR_SPEC es;

	switch(NET_GET_TYPE(inf)) {
		case NET_IF_PHY:
			addr = &NET_GET_IF_PHY(inf);
			break;
		case NET_IF_VIF:
			addr = &NET_GET_IF_VIF_ADDR(inf);
			break;
		default:
			return(FALSE);
	}
	switch(NET_GET_TYPE(addr)) {
		case NET_ADDR_IPv4:
			Init_Object(&es,ERROR_SPEC,ctype_ERROR_SPEC_ipv4);
			es.errspec4_enode = NET_GET_ADDR_IPv4(addr);
			es.errspec4_code = code;
			es.errspec4_value = value;
			es.errspec4_flags = flags;
			return(&es);
#ifdef	USE_IPV6
		case NET_ADDR_IPv6:
			Init_Object(&es,ERROR_SPEC,ctype_ERROR_SPEC_ipv6);
			es.errspec6_enode = NET_GET_ADDR_IPv6(addr);
			es.errspec6_code = code;
			es.errspec6_value = value;
			es.errspec6_flags = flags;
			return(&es);
#endif	/* USE_IPV6 */
		default:
			return(NULL);
	}
}

int
filterspec_eq(FILTER_SPEC *filterspec1,FILTER_SPEC *filterspec2)
{
	if ((Obj_Class(filterspec1) != class_FILTER_SPEC)
			&& (Obj_Class(filterspec1) != class_SENDER_TEMPLATE))
		return(FALSE);
	if ((Obj_Class(filterspec2) != class_FILTER_SPEC)
			&& (Obj_Class(filterspec2) != class_SENDER_TEMPLATE))
		return(FALSE);
	if (Obj_CType(filterspec1) != Obj_CType(filterspec2))
		return(FALSE);
	switch(Obj_CType(filterspec1)) {
		case ctype_FILTER_SPEC_ipv4:
			return(IN_ARE_ADDR_EQUAL(&filterspec1->filt4_srcaddr,
					&filterspec2->filt4_srcaddr));
		case ctype_FILTER_SPEC_ipv4GPI:
			return(IN_ARE_ADDR_EQUAL(
					&filterspec1->filtgpi4_srcaddr,
					&filterspec2->filtgpi4_srcaddr));
#ifdef	USE_IPV6
		case ctype_FILTER_SPEC_ipv6:
			return(IN6_ARE_ADDR_EQUAL(&filterspec1->filt6_srcaddr,
					&filterspec2->filt6_srcaddr));
		case ctype_FILTER_SPEC_ipv6GPI:
			return(IN6_ARE_ADDR_EQUAL(
					&filterspec1->filtgpi6_srcaddr,
					&filterspec2->filtgpi6_srcaddr));
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
filterspec_lt(FILTER_SPEC *filterspec1,FILTER_SPEC *filterspec2)
{
	if ((Obj_Class(filterspec1) != class_FILTER_SPEC)
			&& (Obj_Class(filterspec1) != class_SENDER_TEMPLATE))
		return(FALSE);
	if ((Obj_Class(filterspec2) != class_FILTER_SPEC)
			&& (Obj_Class(filterspec2) != class_SENDER_TEMPLATE))
		return(FALSE);
	if (Obj_CType(filterspec1) != Obj_CType(filterspec2))
		return(FALSE);
	switch(Obj_CType(filterspec1)) {
		case ctype_FILTER_SPEC_ipv4:
			return(memcmp((char *) &filterspec1->filt4_srcaddr,
				(char *) &filterspec2->filt4_srcaddr,
				sizeof(filterspec1->filt4_srcaddr)) < 0);
		case ctype_FILTER_SPEC_ipv4GPI:
			return(memcmp((char *) &filterspec1->filtgpi4_srcaddr,
				(char *) &filterspec2->filtgpi4_srcaddr,
				sizeof(filterspec1->filtgpi4_srcaddr)) < 0);
#ifdef	USE_IPV6
		case ctype_FILTER_SPEC_ipv6:
			return(memcmp((char *) &filterspec1->filt6_srcaddr,
				(char *) &filterspec2->filt6_srcaddr,
				sizeof(filterspec1->filt6_srcaddr)) < 0);
		case ctype_FILTER_SPEC_ipv6GPI:
			return(memcmp((char *) &filterspec1->filtgpi6_srcaddr,
				(char *) &filterspec2->filtgpi6_srcaddr,
				sizeof(filterspec1->filtgpi6_srcaddr)) < 0);
#endif	/* USE_IPV6 */
		default:
			return(FALSE);
	}
}

int
filterspec_in_scope(FILTER_SPEC *filterspec,SCOPE *scope)
{
	char *cp,*end,*addr;
	int size;

	if ((Obj_Class(filterspec) != class_FILTER_SPEC)
			&& (Obj_Class(filterspec) != class_SENDER_TEMPLATE))
		return(-1);
	if (Obj_Class(scope) != class_SCOPE)
		return(-1);
	switch(Obj_CType(filterspec)) {
		case ctype_FILTER_SPEC_ipv4:
				if (Obj_CType(scope) != ctype_SCOPE_list_ipv4)
					return(-1);
				size = sizeof(Scope_list_ipv4);
				addr = (char *) &filterspec->filt4_srcaddr;
				break;
		case ctype_FILTER_SPEC_ipv4GPI:
				if (Obj_CType(scope) != ctype_SCOPE_list_ipv4)
					return(-1);
				size = sizeof(Scope_list_ipv4);
				addr = (char *) &filterspec->filtgpi4_srcaddr;
				break;
#ifdef	USE_IPV6
		case ctype_FILTER_SPEC_ipv6:
				if (Obj_CType(scope) != ctype_SCOPE_list_ipv6)
					return(-1);
				size = sizeof(Scope_list_ipv6);
				addr = (char *) &filterspec->filt6_srcaddr;
				break;
		case ctype_FILTER_SPEC_ipv6GPI:
				if (Obj_CType(scope) != ctype_SCOPE_list_ipv6)
					return(-1);
				size = sizeof(Scope_list_ipv6);
				addr = (char *) &filterspec->filtgpi6_srcaddr;
				break;
#endif	/* USE_IPV6 */
		default:
			return(-1);
	}
	end = (char *) Next_Object(scope);
	for (cp = (char *) Obj_data(scope);cp < end; cp += size)
		if (memcmp(cp,addr,size) == 0)
			return((cp - (char *) Obj_data(scope)) / size);
	return(-1);
}

net_addr *
filterspec_addr(FILTER_SPEC *filterspec)
{
	static net_addr addr;

	if ((Obj_Class(filterspec) != class_FILTER_SPEC)
			&& (Obj_Class(filterspec) != class_SENDER_TEMPLATE))
		return(NULL);
	switch(Obj_CType(filterspec)) {
		case ctype_FILTER_SPEC_ipv4:
			NET_SET_ADDR_IPv4(&addr,filterspec->filt4_srcaddr);
			return(&addr);
		case ctype_FILTER_SPEC_ipv4GPI:
			NET_SET_ADDR_IPv4(&addr,filterspec->filtgpi4_srcaddr);
			return(&addr);
#ifdef	USE_IPV6
		case ctype_FILTER_SPEC_ipv6:
			NET_SET_ADDR_IPv6(&addr,filterspec->filt6_srcaddr);
			return(&addr);
		case ctype_FILTER_SPEC_ipv6GPI:
			NET_SET_ADDR_IPv6(&addr,filterspec->filtgpi6_srcaddr);
			return(&addr);
#endif	/* USE_IPV6 */
		default:
			return(NULL);
	}
}

u_int32_t
filterspec_port(FILTER_SPEC *filterspec)
{
	if ((Obj_Class(filterspec) != class_FILTER_SPEC)
			&& (Obj_Class(filterspec) != class_SENDER_TEMPLATE))
		return(0);
	switch(Obj_CType(filterspec)) {
		case ctype_FILTER_SPEC_ipv4:
			return(filterspec->filt4_srcport);
		case ctype_FILTER_SPEC_ipv4GPI:
			return(filterspec->filtgpi4_srcgpi);
#ifdef	USE_IPV6
		case ctype_FILTER_SPEC_ipv6:
			return(filterspec->filt6_srcport);
		case ctype_FILTER_SPEC_ipv6GPI:
			return(filterspec->filtgpi6_srcgpi);
#endif	/* USE_IPV6 */
		default:
			return(0);
	}
}
