/*
 * librfc1884: Function libary for conversions
 *              defined in RFC 1884
 *
 * Version:		$Id: librfc1884.c,v 0.16 2001/03/13 $
 * 
 * Author:		Peter Bieringer <pb@bieringer.de>
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <netdb.h>
#include "include/modifiedfromlinuxkernel/include/net/ipv6-split.h"
#include "ipv6calc.h"


/* function decompress a given IPv6 address (reverse RFC 1884)
 *  
 * in : *addrstring = IPv6 address
 * out: *resultstring = result
 * ret: ==0: ok, !=0: error
 *
 * Based on code in from 'ircd'
 */
int compaddr_to_uncompaddr(char *addrstring, char *resultstring) {
	int retval = 1;
	char cnt, *cp, *op, *strp;

#ifdef DEBUG_librfc1884
	fprintf(stderr, "librfc1884/compaddr_to_uncompaddr: got input: %s\n", addrstring);
#endif

	strp = strstr(addrstring, "::");
	if (strp) {
#ifdef DEBUG_librfc1884
		fprintf(stderr, "librfc1884/compaddr_to_uncompaddr: found '::' in IPv6 address\n");
#endif

		/* check for additional "::" occurance - not allowed! */
		if (strstr(strp+1, "::")) {
			sprintf(resultstring, "%s", "More than 2 colons in address are not allowed!");
			retval = 1;
			return (retval);
		};
		
		cnt = 0;
		cp = addrstring;
		op = resultstring;
	   	while (*cp) {
			if (*cp == ':')	cnt += 1;
			if (*cp++ == '.') {
				cnt += 1;
				break;
			};
	    };
		cp = addrstring;
		while (*cp) {
			*op++ = *cp++;
			if (*(cp-1) == ':' && *cp == ':') {
				if ((cp-1) == addrstring) {
					op--;
					*op++ = '0';
					*op++ = ':';
				};
		   		*op++ = '0';
				while (cnt++ < 7) {
					*op++ = ':';
					*op++ = '0';
				};
			};
		};
		if (*(op-1)==':') *op++ = '0';
		*op = '\0';

#ifdef DEBUG_librfc1884
		fprintf(stderr, "librfc1884/compaddr_to_uncompaddr: result: %s\n", resultstring);
#endif
	} else {
		strcpy(resultstring, addrstring);
#ifdef DEBUG_librfc1884
		fprintf(stderr, "librfc1884/compaddr_to_uncompaddr: address is not in compressed format\n");
#endif
	};

	retval = 0;
	return (retval);
};


/* function IPv6addrstruct to compressed format (RFC 1884)
 *
 *  compress the biggest '0' block, leading has precedence
 *
 * in : *addrstring = IPv6 address
 * out: *resultstring = result
 * ret: ==0: ok, !=0: error
 */
int ipv6addrstruct_to_compaddr(ipv6calc_ipv6addr *ipv6addrp, char *resultstring) {
	char tempstring[NI_MAXHOST], temp2string[NI_MAXHOST];
	int retval = 1, result;
	int zstart = -1, zend = -1, tstart = -1, tend = -1, i;

	if ( ipv6addrp->scope & IPV6_ADDR_COMPATv4 ) {
		/* compatv4 address */
#ifdef DEBUG_librfc1884
		fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: IPV6_ADDR_COMPATv4 type - fast conversion\n");
#endif
		result = sprintf(tempstring, "::%u.%u.%u.%u", ipv6addrp->addr6[6] >> 8, ipv6addrp->addr6[6] & 0xff, ipv6addrp->addr6[7] >> 8, ipv6addrp->addr6[7] & 0xff);
		retval = 0;
	} else if ( ipv6addrp->scope & IPV6_ADDR_MAPPED ) {
		/* mapped address */
#ifdef DEBUG_librfc1884
		fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: IPV6_ADDR_MAPPED type - fast conversion\n");
#endif
		result = sprintf(tempstring, "::%x:%u.%u.%u.%u", ipv6addrp->addr6[5], ipv6addrp->addr6[6] >> 8, ipv6addrp->addr6[6] & 0xff, ipv6addrp->addr6[7] >> 8, ipv6addrp->addr6[7] & 0xff);
		retval = 0;
	} else if ( (ipv6addrp->addr6[0] == 0) && (ipv6addrp->addr6[1] == 0) && (ipv6addrp->addr6[2] == 0) && (ipv6addrp->addr6[3] == 0) && (ipv6addrp->addr6[4] == 0) && (ipv6addrp->addr6[5] == 0) && (ipv6addrp->addr6[6] == 0) && (ipv6addrp->addr6[7] == 0) ) {
		/* unspecified address */
#ifdef DEBUG_librfc1884
		fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: unspecified address - fast conversion\n");
#endif
		result = sprintf(tempstring, "::");
		retval = 0;
	} else if ( (ipv6addrp->addr6[0] == 0) && (ipv6addrp->addr6[1] == 0) && (ipv6addrp->addr6[2] == 0) && (ipv6addrp->addr6[3] == 0) && (ipv6addrp->addr6[4] == 0) && (ipv6addrp->addr6[5] == 0) && (ipv6addrp->addr6[6] == 0) && (ipv6addrp->addr6[7] == 1) ) {
		/* loopback address */
#ifdef DEBUG_librfc1884
		fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: loopback - fast conversion\n");
#endif
		result = sprintf(tempstring, "::1");
		retval = 0;
	} else {
#ifdef DEBUG_librfc1884
		fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: normal address, detect '0' blocks now\n");
#endif
		for ( i = 0; i <= 7; i++ ) {
			if ( ipv6addrp->addr6[i] == 0 ) {
				/* found a '0' */
#ifdef DEBUG_librfc1884
				fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: Found '0' in word '%d'\n", i);
#endif
				if ( tstart == -1 ) {
					/* possible starting '0' block */ 
#ifdef DEBUG_librfc1884
					fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: Found a possible '0' starting block at '%d'\n", i);
#endif
					tstart = i;				
				};
			} else {
				/* end of a '0' block */
#ifdef DEBUG_librfc1884
				fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: Found non '0' in word '%d'\n", i);
#endif
				if ( tstart != -1 ) {
					tend = i - 1;
					if ( ( tend - tstart ) > 0 ) {
						/* ok, a block with 2 or more '0' */
#ifdef DEBUG_librfc1884
						fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: Found a '0' block from '%d' to '%d' with length '%d'\n", tstart, tend, tend - tstart + 1);
#endif
						if ( zstart < 0 ) {
							/* no other block before, init */
							zstart = tstart;
							zend = tend;
#ifdef DEBUG_librfc1884
							fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: First found '0' block from '%d' to '%d' with length '%d'\n", zstart, zend, zend - zstart + 1);
#endif
						} else if ( ( zend - zstart ) < ( tend - tstart ) ) {
							/* ok, bigger block found */
							zstart = tstart;
							zend = tend;
#ifdef DEBUG_librfc1884
							fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: Found bigger '0' block from '%d' to '%d' with length '%d'\n", zstart, zend, zend - zstart + 1);
#endif
						} else {
#ifdef DEBUG_librfc1884
							fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: This '0' block is not bigger than the last one - skip\n");
#endif

						};
					};
					tstart = -1;
					tend = -1;
				};
			};
		};
		/* cleanup */
		if ( tstart >= 0 ) {
			tend = 7;
			/* trailing '0' block */
			if ( ( tend - tstart ) > 0 ) {
				/* ok, a block with 2 or more '0' */
				if ( zstart < 0 ) {
					/* no other block before, init */
					zstart = tstart;
					zend = tend;
				} else if ( ( zend - zstart ) < ( tend - tstart ) ) {
					/* ok, bigger block found */
					zstart = tstart;
					zend = tend;
				};
			};
		};

#ifdef DEBUG_librfc1884
		if ( zstart != -1 ) {
			fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: biggest '0' block is from word '%d' to '%d'\n", zstart, zend);
		} else {
			fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: no '0' block found\n");
		};
#endif

		/* create string */
		sprintf(tempstring, "%s", "");
		for ( i = 0; i <= 7; i++ ) {
			if ( i == zstart ) {
#ifdef DEBUG_librfc1884
				fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: start of '0' at '%d'\n", i);
#endif
				sprintf(temp2string, "%s:", tempstring);
			} else if ( i == 0 ) {
#ifdef DEBUG_librfc1884
				fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: normal start value at '%d' (%x)\n", i, ipv6addrp->addr6[i]);
#endif
				sprintf(temp2string, "%x", ipv6addrp->addr6[i]);
			} else if ( ( i > zend ) || ( i < zstart ) ) {
				sprintf(temp2string, "%s:%x", tempstring, ipv6addrp->addr6[i]);
			} else if ( ( i == 7 ) && ( zend == i )) {
				sprintf(temp2string, "%s:", tempstring);
			};
			sprintf(tempstring, "%s", temp2string);
		};
		
#ifdef DEBUG_librfc1884
		fprintf(stderr, "librfc1884/ipv6addrstruct_to_compaddr: new method: '%s'\n", tempstring);
#endif
		retval = 0;
	};

	         
	if ( ( retval == 0 ) && ( ipv6addrp->flag_prefixuse == 1 ) ) {
		sprintf(resultstring, "%s/%u", tempstring, ipv6addrp->prefixlength);
	} else {
		sprintf(resultstring, "%s", tempstring);
	};
	
	return (retval);
};

