/*
 * libipv6calc: Function libary for conversions
 *
 * Version:		$Id: libipv6calc.c,v 0.16 2001/03/13 $
 * 
 * Author:		Peter Bieringer <pb@bieringer.de>
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <netdb.h>
#if defined(__NetBSD__)
/* work around incompatibilities between Linux and KAME */
#define s6_addr32              __u6_addr.__u6_addr32
#define in6_u                  __u6_addr
#define u6_addr32              __u6_addr32
#include <netinet/in.h>
#else
#ifndef _NETINET_IN_H
#include <linux/ipv6.h>
#endif
#endif 
#include "include/modifiedfromlinuxkernel/include/net/ipv6-split.h"
#include "include/modifiedfromlinuxkernel/net/ipv6/addrconf-split.c"
#include "ipv6calc.h"

/* function stores an IPv6 address string into a structure
 *
 * in : *addrstring = IPv6 address
 * out: *resultstring = error message
 * out: ipv6addr = IPv6 address structure
 * ret: ==0: ok, !=0: error
 */
int addr_to_ipv6addrstruct(char *addrstring, char *resultstring, ipv6calc_ipv6addr *ipv6addrp) {
	int retval = 1, result, i, cpoints = 0, ccolons = 0;
	char *addronlystring, *cp, tempstring[NI_MAXHOST];
	struct in6_addr inet6addr;
	int scope = 0, expecteditems = 0;
	int32_t temp[8];
	int compat[4];

	sprintf(resultstring, "%s", ""); /* clear result string */

#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/addr_to_ipv6addrstruct: got input %s\n", addrstring);
#endif
	
    /* save prefix length first, if available */
	ipv6addrp->flag_prefixuse = 0; /* reset flag first */
    addronlystring = strtok (addrstring, "/");
	cp = strtok (NULL, "/");
    if ( cp != NULL ) {
		i = atol(cp);
		if (i < 0 || i > 128 ) {
			sprintf(resultstring, "Illegal prefix length: '%s'", cp);
			retval = 1;
			return (retval);
		};
		ipv6addrp->flag_prefixuse = 1;
		ipv6addrp->prefixlength = i;
#ifdef DEBUG_libipv6calc
		fprintf(stderr, "libipv6calc/addr_to_ipv6addrstruct: prefix length %d\n", ipv6addrp->prefixlength);
#endif
	};

	/* uncompress string, if necessary */
	if (strstr(addronlystring, "::")) {
		result = compaddr_to_uncompaddr(addronlystring, tempstring);
		if ( result != 0 ) {
			sprintf(resultstring, "%s", tempstring);
			retval = 1;
			return (retval);
		};
	} else {
		sprintf(tempstring, "%s", addronlystring);
	};

	/* count ":", must be 6 (compat) or 7 (other) */
	for (i = 0; i < strlen(tempstring); i++) {
		if (tempstring[i] == ':') {
			ccolons++;
		};
		if (tempstring[i] == '.') {
			cpoints++;
		};
	};
	if ( ! ( ( ( ccolons == 7 ) && ( cpoints == 0 ) ) ||  ( ( ccolons == 6 ) && ( cpoints == 3 ) ) ) ) {
		if (strstr(addronlystring, "::")) {
			sprintf(resultstring, "Error, given address expanded to '%s' is not valid!", tempstring);
		} else {
			sprintf(resultstring, "Error, given address '%s' is not valid!", addronlystring);
		};
		retval = 1;
		return (retval);
	};

	/* clear variables */
	for ( i = 0; i <= 7; i++ ) {
		compat[i] = 0;
	};
	for ( i = 0; i <= 7 ; i++ ) {
		ipv6addrp->addr6[i] = 0;
	};


	/* scan address into array */
	if ( ccolons == 6 ) {
		/* compatv4/mapped format */
		expecteditems = 10;
		result = sscanf(tempstring, "%x:%x:%x:%x:%x:%x:%d.%d.%d.%d", &temp[0], &temp[1], &temp[2], &temp[3], &temp[4], &temp[5], &compat[0], &compat[1], &compat[2], &compat[3]);
		/* check compat */
		for ( i = 0; i <= 3; i++ ) {
			if ( ( compat[i] < 0 ) || ( compat[i] > 255 ) )	{
				sprintf(resultstring, "Error, given compatv4/mapped address '%s' is not valid (%d)!", addronlystring, compat[i]);
				retval = 1;
				return (retval);
			};
		};
		temp[6] = ( compat[0] << 8 ) | compat[1];
		temp[7] = ( compat[2] << 8 ) | compat[3];
		scope = IPV6_ADDR_COMPATv4;
	} else {
		/* normal format */
		expecteditems = 8;
		result = sscanf(tempstring, "%x:%x:%x:%x:%x:%x:%x:%x", &temp[0], &temp[1], &temp[2], &temp[3], &temp[4], &temp[5], &temp[6], &temp[7]);
	};
	
#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/addr_to_ipv6addrstruct: reading into array, got items: %d\n", result);
#endif

	if ( result != expecteditems ) {
		sprintf(resultstring, "Error splitting address %s, got %d items instead of %d!", addronlystring, result, expecteditems);
		retval = 1;
		return (retval);
	};

	/* check address words range */
	for ( i = 0; i <= 7; i++ ) {
		if ( ( temp[i] < 0x0 ) || ( temp[i] > 0xffff ) )	{
			sprintf(resultstring, "Error, given address '%s' is not valid on position %d (%x)!", addronlystring, i, temp[i]);
			retval = 1;
			return (retval);
		};
	};
	
	/* copy into structure */
	for ( i = 0; i <= 7; i++ ) {
		ipv6addrp->addr6[i] = temp[i];
	};

	/* get type of address */
	for ( i = 0; i < 4; i++ ) {
		inet6addr.s6_addr32[i] = (ipv6addrp->addr6[i <<1] <<16) | ipv6addrp->addr6[(i <<1 )+1];
	};
	
#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/addr_to_ipv6addrstruct: In structure %08x %08x %08x %08x\n", inet6addr.s6_addr32[0], inet6addr.s6_addr32[1], inet6addr.s6_addr32[2], inet6addr.s6_addr32[3]);
#endif
	
	result = ipv6_addr_type(&inet6addr); 
#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/addr_to_ipv6addrstruct: Got scope %02x\n", result);
#endif
	ipv6addrp->scope = result;

	if ( scope != 0 ) {
		/* test, whether compatv4/mapped is really one */
		if ( ! ( ( ipv6addrp->scope & IPV6_ADDR_COMPATv4 ) || ( ipv6addrp->scope & IPV6_ADDR_MAPPED ) ) ) {
			sprintf(resultstring, "Error, given address '%s' is not valid compatv4/mapped one!", addronlystring);
			retval = 1;
			return (retval);
		};
	};

#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/addr_to_ipv6addrstruct: First word is: %x, address info value: %x\n", inet6addr.s6_addr32[0], result);
#endif
	
	retval = 0;
	return (retval);
};


/* function stores the ipv6addr structure in an uncompressed IPv6 format string
 *
 * in:  ipv6addr = IPv6 address structure
 * out: *resultstring = IPv6 address (modified)
 * ret: ==0: ok, !=0: error
 */
int ipv6addrstruct_to_uncompaddr(ipv6calc_ipv6addr *ipv6addrp, char *resultstring) {
	int retval = 1, result;

	/* print array */
	if (ipv6addrp->flag_prefixuse == 1) {
		if ( ( ipv6addrp->scope & IPV6_ADDR_COMPATv4 ) || ( ipv6addrp->scope & IPV6_ADDR_MAPPED ) ) {
			result = sprintf(resultstring, "%x:%x:%x:%x:%x:%x:%u.%u.%u.%u/%u", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6] >> 8, ipv6addrp->addr6[6] & 0xff, ipv6addrp->addr6[7] >> 8, ipv6addrp->addr6[7] & 0xff, ipv6addrp->prefixlength);
		} else {
			result = sprintf(resultstring, "%x:%x:%x:%x:%x:%x:%x:%x/%u", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6], ipv6addrp->addr6[7], ipv6addrp->prefixlength);
		};
	} else {
		if ( ( ipv6addrp->scope & IPV6_ADDR_COMPATv4 ) || ( ipv6addrp->scope & IPV6_ADDR_MAPPED ) ) {
			result = sprintf(resultstring, "%x:%x:%x:%x:%x:%x:%u.%u.%u.%u", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6] >> 8, ipv6addrp->addr6[6] & 0xff, ipv6addrp->addr6[7] >> 8, ipv6addrp->addr6[7] & 0xff);
		} else {
			result = sprintf(resultstring, "%x:%x:%x:%x:%x:%x:%x:%x", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6], ipv6addrp->addr6[7]);
		};
	};

#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/ipv6addrstruct_to_uncompaddr: result string: %s\n", resultstring);
#endif
	return (retval);
};


/* function stores the ipv6addr structure in an full uncompressed IPv6 format string
 *
 * in:  ipv6addr = IPv6 address structure
 * out: *resultstring = IPv6 address (modified)
 * ret: ==0: ok, !=0: error
 */
int ipv6addrstruct_to_fulluncompaddr(ipv6calc_ipv6addr *ipv6addrp, char *resultstring) {
	int retval = 1, result;

	/* print array */
	if (ipv6addrp->flag_prefixuse == 1) {
		if ( ( ipv6addrp->scope & IPV6_ADDR_COMPATv4 ) || ( ipv6addrp->scope & IPV6_ADDR_MAPPED ) ) {
			result = sprintf(resultstring, "%04x:%04x:%04x:%04x:%04x:%04x:%u.%u.%u.%u/%u", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6] >> 8, ipv6addrp->addr6[6] & 0xff, ipv6addrp->addr6[7] >> 8, ipv6addrp->addr6[7] & 0xff, ipv6addrp->prefixlength);
		} else {
			result = sprintf(resultstring, "%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x/%u", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6], ipv6addrp->addr6[7], ipv6addrp->prefixlength);
		};
	} else {
		if ( ( ipv6addrp->scope & IPV6_ADDR_COMPATv4 ) || ( ipv6addrp->scope & IPV6_ADDR_MAPPED ) ) {
			result = sprintf(resultstring, "%04x:%04x:%04x:%04x:%04x:%04x:%u.%u.%u.%u", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6] >> 8, ipv6addrp->addr6[6] & 0xff, ipv6addrp->addr6[7] >> 8, ipv6addrp->addr6[7] & 0xff);
		} else {
			result = sprintf(resultstring, "%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6], ipv6addrp->addr6[7]);
		};
	};

#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr2uncomp: result string: %s\n", resultstring);
#endif
	return (retval);
}


/* mask prefix bits (set suffix bits to 0)
 * 
 * in:  structure via reference
 * out: modified structure
 */
void ipv6addrstruct_maskprefix(ipv6calc_ipv6addr *ipv6addrp) {
	int nbit, nword;
	unsigned int mask, newword;
#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr_maskprefix: called\n");
#endif
   
	if (ipv6addrp->flag_prefixuse != 1) {
		/* hmm, no prefix specified. skip */
		return;
	};

	for (nbit = 127; nbit >= 0; nbit--) {
		if (nbit >= ipv6addrp->prefixlength) {
			/* set bit to zero */
			
			/* calculate word (16 bit) - matches with addr6p[]*/
			nword = (nbit & 0x70) >> 4;
				 
			/* calculate mask */
			mask = 0x8000 >> ((nbit & 0x0f));
			newword = ipv6addrp->addr6[nword] & (~ mask );
			
#ifdef DEBUG_libipv6calc
			fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr_maskprefix: bit: %d = nword: %d, mask: %04x, word: %04x newword: %04x\n", nbit, nword, mask, ipv6addrp->addr6[nword], newword);
#endif
			ipv6addrp->addr6[nword] = newword;
		};
	};
}


/* mask suffix bits (set prefix bits to 0) 
 *
 * in:  structure via reference
 * out: modified structure
 */
void ipv6addrstruct_masksuffix(ipv6calc_ipv6addr *ipv6addrp) {
	int nbit, nword;
	unsigned int mask, newword;
#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr_masksuffix: called\n");
#endif
   
	if (ipv6addrp->flag_prefixuse != 1) {
		/* hmm, no prefix specified. skip */
		return;
	};

	for (nbit = 127; nbit >= 0; nbit--) {
		if (nbit < ipv6addrp->prefixlength) {
			/* set bit to zero */
			
			/* calculate word (16 bit) - matches with addr6p[]*/
			nword = (nbit & 0x70) >> 4;
				 
			/* calculate mask */
			mask = 0x8000 >> ((nbit & 0x0f));
			newword = ipv6addrp->addr6[nword] & (~ mask );
			
#ifdef DEBUG_libipv6calc
			fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr_masksuffix: bit: %d = nword: %d, mask: %04x, word: %04x newword: %04x\n", nbit, nword, mask, ipv6addrp->addr6[nword], newword);
#endif
			ipv6addrp->addr6[nword] = newword;
		};
	};
};


