/*
 * libipv6calc: Function libary for conversions
 *
 * Version:		$Id: addr2if_inet6.c,v 0.10 2001/03/03 $
 * 
 * Author:		Peter Bieringer <pb@bieringer.de>
 *
 * some hints taken from ifconfig.c (net-tools)
 * 
 * Credits to:
 *	net-tools authors 
 *	ircd authors
 *
 *  Arkadiusz Miskiewicz <misiek@pld.ORG.PL>
 *  Jun-ichiro itojun Hagino <itojun@iijlab.net>
 *  Horacio J. Pea <horape@compendium.com.ar>
 * 
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#ifndef _NETINET_IN_H
# include <linux/ipv6.h>
#endif
#include "include/modifiedfromkernel/include/net/ipv6-split.h"
#include "include/modifiedfromkernel/net/ipv6/addrconf-split.c"
#include "ipv6calc.h"


/* function stores an IPv6 address string into a structure
 *
 * in : *addrstring = IPv6 address
 * out: *resultstring = error message
 * out: ipv6addr = IPv6 address structure
 * ret: ==0: ok, !=0: error
 */
int addr2ipv6calc_ipv6addr(char *addrstring, char *resultstring, ipv6calc_ipv6addr *ipv6addrp) {
	int retval = 1, result, i;
	char *addronlystring, *cp, tempstring[NI_MAXHOST];
	struct in6_addr inet6addr;

	sprintf(resultstring, "%s", ""); /* clear result string */

#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/uncomp2ipv6calc_ipv6addr: got input %s\n", addrstring);
#endif
	
    /* save prefix length first, if available */
	ipv6addrp->flag_prefixuse = 0; /* reset flag first */
    addronlystring = strtok (addrstring, "/");
	cp = strtok (NULL, "/");
    if ( cp != NULL ) {
		i = atol(cp);
		if (i < 0 || i > 128 ) {
			sprintf(resultstring, "illegal prefix length: '%s'", cp);
			retval = 1;
			return (retval);
		}
		ipv6addrp->flag_prefixuse = 1;
		ipv6addrp->prefixlength = i;
#ifdef DEBUG_libipv6calc
		fprintf(stderr, "libipv6calc/uncomp2ipv6calc_ipv6addr: prefix length %d\n", ipv6addrp->prefixlength);
#endif
	}

	/* uncompress string, if necessary */
	result = compaddr2uncompaddr(addronlystring, tempstring);

	/* scan address into array */
	result = sscanf(tempstring, "%x:%x:%x:%x:%x:%x:%x:%x", &ipv6addrp->addr6[0], &ipv6addrp->addr6[1], &ipv6addrp->addr6[2], &ipv6addrp->addr6[3], &ipv6addrp->addr6[4], &ipv6addrp->addr6[5], &ipv6addrp->addr6[6], &ipv6addrp->addr6[7]);

#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/uncomp2ipv6calc_ipv6addr: reading into array, got items: %d\n", result);
#endif

	if ( result != 8 ) {
		sprintf(resultstring, "error splitting address %s, got only %d items!", addronlystring, result);
		retval = 1;
		return (retval);
	}

	/* get type of address */
	for ( i = 0; i < 4; i++ ) {
		inet6addr.s6_addr32[i] = (ipv6addrp->addr6[i <<1] <<16) | ipv6addrp->addr6[(i <<1 )+1];
	}
	
#ifdef DEBUG_ipv6calc_ipv6addr
	fprintf(stderr, "ipv6calc_ipv6addr: In structure %08x %08x %08x %08x\n", inet6addr.s6_addr32[0], inet6addr.s6_addr32[1], inet6addr.s6_addr32[2], inet6addr.s6_addr32[3]);
#endif
	
	result = ipv6_addr_type(&inet6addr); 
	ipv6addrp->scope = result & IPV6_ADDR_SCOPE_MASK;

#ifdef DEBUG_ipv6calc_ipv6addr
	fprintf(stderr, "addr2if_inet6: First word is: %x, address info value: %x\n", inet6addr.s6_addr32[0], result);
#endif
	
	retval = 0;
	return (retval);
}


/* function stores the ipv6addr structure in an uncompressed IPv6 format string
 *
 * in:  ipv6addr = IPv6 address structure
 * out: *resultstring = IPv6 address (modified)
 * ret: ==0: ok, !=0: error
 */
int ipv6calc_ipv6addr2uncompaddr(ipv6calc_ipv6addr *ipv6addrp, char *resultstring) {
	int retval = 1, result;

	/* print array */
	if (ipv6addrp->flag_prefixuse == 1) {
		result = sprintf(resultstring, "%x:%x:%x:%x:%x:%x:%x:%x/%d", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6], ipv6addrp->addr6[7], ipv6addrp->prefixlength);
	} else {
		result = sprintf(resultstring, "%x:%x:%x:%x:%x:%x:%x:%x", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6], ipv6addrp->addr6[7]);
	}

#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr2uncomp: result string: %s\n", resultstring);
#endif
	return (retval);
};


/* function stores the ipv6addr structure in an full uncompressed IPv6 format string
 *
 * in:  ipv6addr = IPv6 address structure
 * out: *resultstring = IPv6 address (modified)
 * ret: ==0: ok, !=0: error
 */
int ipv6calc_ipv6addr2fulluncompaddr(ipv6calc_ipv6addr *ipv6addrp, char *resultstring) {
	int retval = 1, result;

	/* print array */
	if (ipv6addrp->flag_prefixuse == 1) {
		result = sprintf(resultstring, "%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x/%d", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6], ipv6addrp->addr6[7], ipv6addrp->prefixlength);
	} else {
		result = sprintf(resultstring, "%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6], ipv6addrp->addr6[7]);
	}

#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr2uncomp: result string: %s\n", resultstring);
#endif
	return (retval);
}


/* mask prefix bits (set suffix bits to 0)
 * 
 * in:  structure via reference
 * out: modified structure
 */
void ipv6calc_ipv6addr_maskprefix(ipv6calc_ipv6addr *ipv6addrp) {
	int nbit, nword;
	unsigned int mask, newword;
#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr_maskprefix: called\n");
#endif
   
	if (ipv6addrp->flag_prefixuse != 1) {
		/* hmm, no prefix specified. skip */
		return;
	}

	for (nbit = 127; nbit >= 0; nbit--) {
		if (nbit >= ipv6addrp->prefixlength) {
			/* set bit to zero */
			
			/* calculate word (16 bit) - matches with addr6p[]*/
			nword = (nbit & 0x70) >> 4;
				 
			/* calculate mask */
			mask = 0x8000 >> ((nbit & 0x0f));
			newword = ipv6addrp->addr6[nword] & (~ mask );
			
#ifdef DEBUG_libipv6calc
			fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr_maskprefix: bit: %d = nword: %d, mask: %04x, word: %04x newword: %04x\n", nbit, nword, mask, ipv6addrp->addr6[nword], newword);
#endif
			ipv6addrp->addr6[nword] = newword;
		}
	}
}


/* mask suffix bits (set prefix bits to 0) 
 *
 * in:  structure via reference
 * out: modified structure
 */
void ipv6calc_ipv6addr_masksuffix(ipv6calc_ipv6addr *ipv6addrp) {
	int nbit, nword;
	unsigned int mask, newword;
#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr_masksuffix: called\n");
#endif
   
	if (ipv6addrp->flag_prefixuse != 1) {
		/* hmm, no prefix specified. skip */
		return;
	}

	for (nbit = 127; nbit >= 0; nbit--) {
		if (nbit < ipv6addrp->prefixlength) {
			/* set bit to zero */
			
			/* calculate word (16 bit) - matches with addr6p[]*/
			nword = (nbit & 0x70) >> 4;
				 
			/* calculate mask */
			mask = 0x8000 >> ((nbit & 0x0f));
			newword = ipv6addrp->addr6[nword] & (~ mask );
			
#ifdef DEBUG_libipv6calc
			fprintf(stderr, "libipv6calc/ipv6calc_ipv6addr_masksuffix: bit: %d = nword: %d, mask: %04x, word: %04x newword: %04x\n", nbit, nword, mask, ipv6addrp->addr6[nword], newword);
#endif
			ipv6addrp->addr6[nword] = newword;
		};
	};
};


/* function decompress a given IPv6 address
 *  
 * examples:
 * in : *addrstring = IPv6 address
 * out: *resultstring = result
 * ret: ==0: ok, !=0: error
 */
int compaddr2uncompaddr(char *addrstring, char *resultstring) {
	int retval = 1;
	char cnt, *cp, *op;

#ifdef DEBUG_libipv6calc
	fprintf(stderr, "libipv6calc/compaddr2uncompaddr: got input: %s\n", addrstring);
#endif

	if (strstr(addrstring, "::")) {
#ifdef DEBUG_libipv6calc
		fprintf(stderr, "libipv6calc/compaddr2uncompaddr: found '::' in IPv6 address\n");
#endif
		cnt = 0;
		cp = addrstring;
		op = resultstring;
	   	while (*cp) {
			if (*cp == ':')	cnt += 1;
			if (*cp++ == '.') {
				cnt += 1;
				break;
			};
	    };
		cp = addrstring;
		while (*cp) {
			*op++ = *cp++;
			if (*(cp-1) == ':' && *cp == ':') {
				if ((cp-1) == addrstring) {
					op--;
					*op++ = '0';
					*op++ = ':';
				};
		   		*op++ = '0';
				while (cnt++ < 7) {
					*op++ = ':';
					*op++ = '0';
				}
			}
		}
		if (*(op-1)==':') *op++ = '0';
		*op = '\0';

#ifdef DEBUG_libipv6calc
		fprintf(stderr, "libipv6calc/compaddr2uncompaddr: result: %s\n", resultstring);
#endif
	} else {
		strcpy(resultstring, addrstring);
#ifdef DEBUG_libipv6calc
		fprintf(stderr, "libipv6calc/compaddr2uncompaddr: address is not in compressed format\n");
#endif
	};

	retval = 0;
	return (retval);
};


/* function IPv6addr to compressed format
 * examples:
 *
 * in : *addrstring = IPv6 address
 * out: *resultstring = result
 * ret: ==0: ok, !=0: error
 */
int ipv6calc_ipv6addr2compaddr(ipv6calc_ipv6addr *ipv6addrp, char *resultstring) {
	struct addrinfo hints, *aitop;
	char tempstring[NI_MAXHOST];
	int retval = 1, result;

	result = sprintf(tempstring, "%x:%x:%x:%x:%x:%x:%x:%x", ipv6addrp->addr6[0], ipv6addrp->addr6[1], ipv6addrp->addr6[2], ipv6addrp->addr6[3], ipv6addrp->addr6[4], ipv6addrp->addr6[5], ipv6addrp->addr6[6], ipv6addrp->addr6[7]);

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = AF_INET6;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags = AI_NUMERICHOST;  
	result  = getaddrinfo(tempstring, NULL, &hints, &aitop);
	if (result) {
		sprintf(resultstring, "%s: %s", tempstring, gai_strerror(result));
		retval = 1;
		return (retval);
	}   
	result = getnameinfo(aitop->ai_addr, aitop->ai_addrlen, tempstring, sizeof(tempstring), NULL, 0, NI_NUMERICHOST);
	if (result) {
		sprintf(resultstring, "%s", "unknown");
		retval = 1; 
	} else {
		if ( ipv6addrp->flag_prefixuse == 1 ) {
			sprintf(resultstring, "%s/%d", tempstring, ipv6addrp->prefixlength);
		} else {
			sprintf(resultstring, "%s", tempstring);
		};
		retval = 0;
	};
	freeaddrinfo(aitop);
	         
	return (retval);
};

