/*
 * addr2ip6_int: Function to format a given address to reverse ip6.int format
 *
 * Version:		$Id: addr2ip6_int.c,v 0.05 2001/02/27 $
 * 
 * Author:		Peter Bieringer <pb@bieringer.de>
 *
 * intention from the Perl program "ip6_int" written by Keith Owens <kaos@ocs.com.au>
 * some hints taken from ifconfig.c (net-tools)
 * 
 * Credits to:
 *  Keith Owens <kaos@ocs.com.au>
 *	net-tools authors
 */

#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "ipv6calc.h"

void printhelp_addr2ip6_int() {
	fprintf(stderr, " %s --addr2ip6_int  ipv6addr[/prefixlength]\n", PROGRAM_NAME);
	fprintf(stderr, "  Converts given IPv6 address to a dot separated reverse nibble format for use with DNS\n");
	fprintf(stderr, "   e.g. 3ffe:400:100:f101::1\n    -> 1.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.1.0.1.f.0.0.1.0.0.0.4.0.e.f.f.3.ip6.int\n");
	fprintf(stderr, "   e.g. 3ffe:400:100:f101::1/64 -> 1.0.1.f.0.0.1.0.0.0.4.0.e.f.f.3.ip6.int\n\n");
}  

/* function formats an given IPv6 address to the reverse nibble format used by DNS zone files
 * examples:
 *  3ffe:400:100:f101::1
 *   -> 1.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.1.0.1.f.0.0.1.0.0.0.4.0.e.f.f.3.ip6.int
 *  3ffe:400:100:f101::1/64 -> 1.0.1.f.0.0.1.0.0.0.4.0.e.f.f.3.ip6.int
 *
 * in : *addrstring = IPv6 address
 * out: *resultstring = result
 * ret: ==0: ok, !=0: error
 */
int addr2ip6_int(char *addrstring, char *resultstring) {
	int retval = 1, result;
	char tempstring[NI_MAXHOST], *addronlystring, *cp;
	int prefixlength = 0, flag_prefixlength = 0;
	unsigned int addr6p[8], nibble;
	int nword, nbit, nnibble;

#ifdef DEBUG_addr2ip6_int
	fprintf(stderr, "addr2ip6_int: Got input %s\n", addrstring);
#endif
	
	/* first do an uncompressing */
	result = addr2uncompaddr(addrstring, tempstring);
	
#ifdef DEBUG_addr2ip6_int
	fprintf(stderr, "addr2ip6_int: Got temp result from uncompress: %s Result: %d\n", tempstring, result);
#endif

	if ( result != 0 ) {
		sprintf(resultstring, "%s", tempstring);
		retval = 1;
		return (retval);
	}

    /* save prefix length first, if available */
    addronlystring = strtok (tempstring, "/");
	cp = strtok (NULL, "/");
    if ( cp != NULL ) {
		prefixlength = atol(cp);
		if (prefixlength < 0 || prefixlength > 128 ) {
			sprintf(resultstring, "illegal prefix length");
			retval = 1;
			return (retval);
		}
		flag_prefixlength = 1;
	}

	/* scan address into array */
	result = sscanf(addronlystring, "%x:%x:%x:%x:%x:%x:%x:%x", &addr6p[0], &addr6p[1], &addr6p[2], &addr6p[3], &addr6p[4], &addr6p[5], &addr6p[6], &addr6p[7]);

#ifdef DEBUG_addr2ip6_int
	fprintf(stderr, "addr2ip6_int: Reading into array, got items: %d\n", result);
#endif

	if ( result != 8 ) {
		sprintf(resultstring, "error splitting address %s, got only %d items!", addronlystring, result);
		retval = 1;
		return (retval);
	}

	/* print out nibble format */
	/* 127 is lowest bit, 0 is highest bit */
	sprintf(resultstring, "%s", "");
	for (nbit = 127; nbit >= 0; nbit = nbit - 4) {
		if ( flag_prefixlength == 1 ) {
			/* must test for prefix length match */
			if ( (nbit + 1) > prefixlength ) {
				/* skip nibble */
				continue;
		   	}
		};
		
		/* calculate word (16 bit) - matches with addr6p[]*/
		nword = (nbit & 0x70) >> 4;
		
		/* calculate nibble */
		nnibble = (nbit & 0x0C) >> 2;

		/* extract nibble */
		nibble = ( addr6p[nword] & ( 0xf << (4 * (3 - nnibble)) ) ) >> ( 4* (3 - nnibble));
		
#ifdef DEBUG_addr2if_inet6
		fprintf(stderr, "addr2ip6_int: bit: %d = nword: %d, nnibble: %d, word: %04x, value: %x\n", nbit, nword, nnibble, addr6p[nword], nibble);
#endif
		sprintf(resultstring, "%s%x.", resultstring, nibble);
	}
	sprintf(resultstring, "%sip6.int", resultstring);
		

#ifdef DEBUG_addr2if_inet6
	fprintf(stderr, "addr2ip_int: Print out: %s\n", resultstring);
#endif

	retval = 0;
	return (retval);
}

