/*
 * addr2if_inet6: Function to format a given address to a /proc/net/if_inet6 format
 *
 * Version:		$Id: addr2if_inet6.c,v 0.06 2001/02/27 $
 * 
 * Author:		Peter Bieringer <pb@bieringer.de>
 *
 * some hints taken from ifconfig.c (net-tools)
 * 
 * Credits to:
 *	net-tools authors 
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <netdb.h>
#ifndef _NETINET_IN_H
# include <linux/ipv6.h>
#endif
#include "include/modifiedfromkernel/include/net/ipv6-split.h"
#include "include/modifiedfromkernel/net/ipv6/addrconf-split.c"
#include "ipv6calc.h"

void printhelp_addr2if_inet6() {
	fprintf(stderr, " %s --addr2if_inet6 ipv6addr[/prefixlength]\n", PROGRAM_NAME);
	fprintf(stderr, "  Converts given IPv6 address to similar format shown in /proc/net/if_inet6:\n    addr scope [prefixlength]\n");
	fprintf(stderr, "   e.g. 3ffe:400:100:f101::1 -> 3ffe04000100f1010000000000000001 00\n");
	fprintf(stderr, "   e.g. 3ffe:400:100:f101::1/64 -> 3ffe04000100f1010000000000000001 00 40\n\n");
} 

/* function formats an given IPv6 address to /proc/net/if_inet6 format
 * examples: addr scope [prefixlength]
 *  3ffe:400:100:f101::1 -> 3ffe04000100f1010000000000000001 00
 *  3ffe:400:100:f101::1/64 -> 3ffe04000100f1010000000000000001 00 40
 *
 * in : *addrstring = IPv6 address
 * out: *resultstring = result
 * ret: ==0: ok, !=0: error
 */
int addr2if_inet6(char *addrstring, char *resultstring) {
	int retval = 1, result, i;
	char tempstring[NI_MAXHOST], *addronlystring, *cp;
	int prefixlength = 0, flag_prefixlength = 0;
	struct in6_addr inet6addr;
	unsigned int addr6p[8], scope;

#ifdef DEBUG_addr2if_inet6
	fprintf(stderr, "addr2if_inet6: Got input %s\n", addrstring);
#endif
	
	/* first do an uncompressing */
	result = addr2uncompaddr(addrstring, tempstring);
	
#ifdef DEBUG_addr2if_inet6
	fprintf(stderr, "addr2if_inet6: Got temp result from uncompress: %s Result: %d\n", tempstring, result);
#endif

	if ( result != 0 ) {
		sprintf(resultstring, "%s", tempstring);
		retval = 1;
		return (retval);
	}

    /* save prefix length first, if available */
    addronlystring = strtok (tempstring, "/");
	cp = strtok (NULL, "/");
    if ( cp != NULL ) {
		prefixlength = atol(cp);
		if (prefixlength < 0 || prefixlength > 128 ) {
			sprintf(resultstring, "illegal prefix length");
			retval = 1;
			return (retval);
		}
		flag_prefixlength = 1;
	}

	/* scan address into array */
	result = sscanf(addronlystring, "%x:%x:%x:%x:%x:%x:%x:%x", &addr6p[0], &addr6p[1], &addr6p[2], &addr6p[3], &addr6p[4], &addr6p[5], &addr6p[6], &addr6p[7]);

#ifdef DEBUG_addr2if_inet6
	fprintf(stderr, "addr2if_inet6: Reading into array, got items: %d\n", result);
#endif

	if ( result != 8 ) {
		sprintf(resultstring, "error splitting address %s, got only %d items!", addronlystring, result);
		retval = 1;
		return (retval);
	}

	/* get type of address */
	for ( i = 0; i < 4; i++ ) {
		inet6addr.s6_addr32[i] = (addr6p[i <<1] <<16) | addr6p[(i <<1 )+1];
	}
	
#ifdef DEBUG_addr2if_inet6
	fprintf(stderr, "In structure: %08x %08x %08x %08x\n", inet6addr.s6_addr32[0], inet6addr.s6_addr32[1], inet6addr.s6_addr32[2], inet6addr.s6_addr32[3]);
#endif
	
	result = ipv6_addr_type(&inet6addr); 
	scope = result & IPV6_ADDR_SCOPE_MASK;

#ifdef DEBUG_addr2if_inet6
	fprintf(stderr, "addr2if_inet6: First word is: %x, address info value: %x\n", inet6addr.s6_addr32[0], result);
#endif
	

	/* print out array */	
	if ( flag_prefixlength == 1 ) {
	    sprintf(resultstring, "%04x%04x%04x%04x%04x%04x%04x%04x %02x %02x", addr6p[0], addr6p[1], addr6p[2], addr6p[3], addr6p[4], addr6p[5], addr6p[6], addr6p[7], scope, prefixlength);
	} else {
	    sprintf(resultstring, "%04x%04x%04x%04x%04x%04x%04x%04x %02x", addr6p[0], addr6p[1], addr6p[2], addr6p[3], addr6p[4], addr6p[5], addr6p[6], addr6p[7], scope);
	}

#ifdef DEBUG_addr2if_inet6
	fprintf(stderr, "addr2if_inet6: Print out: %s\n", resultstring);
#endif

	retval = 0;
	return (retval);
}

