/*
 * addr2fulluncompaddr: Support a function to format a given address to a full uncompressed one
 *                      (anti RFC 1884)
 *
 * Version:		$Id: addr2fulluncompaddr.c,v 0.05 2001/02/27 $
 * 
 * Author:		Peter Bieringer <pb@bieringer.de>
 *
 * some hints taken from ifconfig.c (net-tools)
 * 
 * Credits to:
 *	net-tools authors 
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <netdb.h>
#include "ipv6calc.h"

void printhelp_addr2fulluncompaddr() {
	fprintf(stderr, " %s --addr2fulluncompaddr ipv6addr[/prefixlength]\n", PROGRAM_NAME);
	fprintf(stderr, "  Converts given IPv6 address to a full uncompressed one\n");
	fprintf(stderr, "   e.g. 3ffe:400:100:f101::1 -> 3ffe:0400:0100:f101:0000:0000:0000:0001\n\n");
};

/* function formats an given IPv6 address to a full uncompressed format
 * examples:
 *  3ffe:400:100:f101::1 -> 3ffe:0400:0100:f101:0000:0000:0000:0001
 *
 * in : *addrstring = IPv6 address
 * out: *resultstring = result
 * ret: ==0: ok, !=0: error
 */
int addr2fulluncompaddr(char *addrstring, char *resultstring) {
	int retval = 1, result;
	char tempstring[NI_MAXHOST], *addronlystring, *cp;
	int prefixlength = 0, flag_prefixlength = 0;
	unsigned int addr6p[8];

#ifdef DEBUG_addr2fulluncompaddr
	fprintf(stderr, "addr2fulluncompaddr: Got input %s\n", addrstring);
#endif
	
	/* first do an uncompressing */
	result = addr2uncompaddr(addrstring, tempstring);
	
#ifdef DEBUG_addr2fulluncompaddr
	fprintf(stderr, "addr2fulluncompaddr: Got temp result from uncompress: %s Result: %d\n", tempstring, result);
#endif

	if ( result != 0 ) {
		sprintf(resultstring, "%s", tempstring);
		retval = 1;
		return (retval);
	}

    /* save prefix length first, if available */
    addronlystring = strtok (tempstring, "/");
	cp = strtok (NULL, "/");
    if ( cp != NULL ) {
		prefixlength = atol(cp);
		if (prefixlength < 0 || prefixlength > 128 ) {
			sprintf(resultstring, "illegal prefix length");
			retval = 1;
			return (retval);
		}
		flag_prefixlength = 1;
	}

	/* scan address into array */
	result = sscanf(addronlystring, "%x:%x:%x:%x:%x:%x:%x:%x", &addr6p[0], &addr6p[1], &addr6p[2], &addr6p[3], &addr6p[4], &addr6p[5], &addr6p[6], &addr6p[7]);

#ifdef DEBUG_addr2fulluncompaddr
	fprintf(stderr, "addr2fulluncompaddr: Reading into array, got items: %d\n", result);
#endif

	if ( result != 8 ) {
		sprintf(resultstring, "error splitting address %s, got only %d items!", addronlystring, result);
		retval = 1;
		return (retval);
	}

	/* print out array */	
	if ( flag_prefixlength == 1 ) {
	    sprintf(resultstring, "%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x/%d", addr6p[0], addr6p[1], addr6p[2], addr6p[3], addr6p[4], addr6p[5], addr6p[6], addr6p[7], prefixlength);
	} else {
	    sprintf(resultstring, "%04x:%04x:%04x:%04x:%04x:%04x:%04x:%04x", addr6p[0], addr6p[1], addr6p[2], addr6p[3], addr6p[4], addr6p[5], addr6p[6], addr6p[7]);
	}

#ifdef DEBUG_addr2fulluncompaddr
	fprintf(stderr, "addr2fulluncompaddr: Print out: %s\n", resultstring);
#endif

	retval = 0;
	return (retval);
}

