/*
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>,
 *                   Hannu Savolainen 1993-1996,
 *                   Rob Hooft
 *                   
 *  Routines for control of AdLib FM cards (OPL2/OPL3/OPL4 chips)
 *
 *  Most if code is ported from OSS/Lite.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "opl3.h"

/*
 *	There is 18 possible 2 OP voices
 *	(9 in the left and 9 in the right).
 *	The first OP is the modulator and 2nd is the carrier.
 *
 *	The first three voices in the both sides may be connected
 *	with another voice to a 4 OP voice. For example voice 0
 *	can be connected with voice 3. The operators of voice 3 are
 *	used as operators 3 and 4 of the new 4 OP voice.
 *	In this case the 2 OP voice number 0 is the 'first half' and
 *	voice 3 is the second.
 */

#if 0

#define OPL3_USE_LEFT	0
#define OPL3_USE_RIGHT	1

static struct snd_opl3_physical_voice_info snd_opl3_pv_map[18] =
{
/*       No Mode Side		OP1	OP2	OP3   OP4	*/
/*	---------------------------------------------------	*/
	{ 0,  2, OPL3_USE_LEFT,	{0x00,	0x03,	0x08, 0x0b}},
	{ 1,  2, OPL3_USE_LEFT,	{0x01,	0x04,	0x09, 0x0c}},
	{ 2,  2, OPL3_USE_LEFT,	{0x02,	0x05,	0x0a, 0x0d}},

	{ 3,  2, OPL3_USE_LEFT,	{0x08,	0x0b,	0x00, 0x00}},
	{ 4,  2, OPL3_USE_LEFT,	{0x09,	0x0c,	0x00, 0x00}},
	{ 5,  2, OPL3_USE_LEFT,	{0x0a,	0x0d,	0x00, 0x00}},

	{ 6,  2, OPL3_USE_LEFT,	{0x10,	0x13,	0x00, 0x00}}, /* Used by percussive voices */
	{ 7,  2, OPL3_USE_LEFT,	{0x11,	0x14,	0x00, 0x00}}, /* if the percussive mode */
	{ 8,  2, OPL3_USE_LEFT,	{0x12,	0x15,	0x00, 0x00}}, /* is selected */

	{ 0,  2, OPL3_USE_RIGHT,{0x00,	0x03,	0x08, 0x0b}},
	{ 1,  2, OPL3_USE_RIGHT,{0x01,	0x04,	0x09, 0x0c}},
	{ 2,  2, OPL3_USE_RIGHT,{0x02,	0x05,	0x0a, 0x0d}},

	{ 3,  2, OPL3_USE_RIGHT,{0x08,	0x0b,	0x00, 0x00}},
	{ 4,  2, OPL3_USE_RIGHT,{0x09,	0x0c,	0x00, 0x00}},
	{ 5,  2, OPL3_USE_RIGHT,{0x0a,	0x0d,	0x00, 0x00}},

	{ 6,  2, OPL3_USE_RIGHT,{0x10,	0x13,	0x00, 0x00}},
	{ 7,  2, OPL3_USE_RIGHT,{0x11,	0x14,	0x00, 0x00}},
	{ 8,  2, OPL3_USE_RIGHT,{0x12,	0x15,	0x00, 0x00}}
};

#endif

/*
 *
 */

static void snd_opl3_command( opl3_t *opl3, unsigned int cmd, unsigned char val )
{
  unsigned long flags;
  unsigned short port;

  /*
   * The original 2-OP synth requires a quite long delay after writing to a
   * register. The OPL-3 survives with just two INBs
   */

  port = cmd & OPL3_RIGHT ? opl3 -> r_port : opl3 -> l_port;

  snd_spin_lock( opl3, reg, &flags );
  
  outb( (unsigned char)cmd, port );

  if ( opl3 -> hardware == OPL3_HW_OPL2 ) {
    snd_delay( 1 );
  } else {
    inb( port );
    inb( port );
  }
  
  outb( (unsigned char)(val & 0xff), port + 1 );
      
  if ( opl3 -> hardware == OPL3_HW_OPL2 ) {
    snd_delay( 3 );
  } else {
    inb( port );
    inb( port );
  }
  
  snd_spin_unlock( opl3, reg, &flags );
}

/*
 *
 */
 
static int snd_opl3_detect( opl3_t *opl3 )
{
  /*
   * This function returns 1 if the FM chip is present at the given I/O port
   * The detection algorithm plays with the timer built in the FM chip and
   * looks for a change in the status register.
   *
   * Note! The timers of the FM chip are not connected to AdLib (and compatible)
   * boards.
   *
   * Note2! The chip is initialized if detected.
   */

  unsigned char stat1, stat2, signature;

  if ( opl3 -> hardware == OPL3_HW_OPL3_SV ) return 0;
  /* Reset timers 1 and 2 */
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER_CONTROL, OPL3_TIMER1_MASK | OPL3_TIMER2_MASK );
  /* Reset the IRQ of the FM chip */
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER_CONTROL, OPL3_IRQ_RESET );
  signature = stat1 = inb( opl3 -> l_port );	/* Status register */
  if ( (stat1 & 0xe0) != 0x00 ) return -ENODEV;	/* Should be 0x00 */
  /* Set timer1 to 0xff */
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER1, 0xff );
  /* Unmask and start timer 1 */
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER_CONTROL, OPL3_TIMER2_MASK | OPL3_TIMER1_START );
  /* Now we have to delay at least 80us */
  snd_delay( 50 );
  /* Read status after timers have expired */
  stat2 = inb( opl3 -> l_port );
  /* Stop the timers */
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER_CONTROL, OPL3_TIMER1_MASK | OPL3_TIMER2_MASK );
  /* Reset the IRQ of the FM chip */
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER_CONTROL, OPL3_IRQ_RESET );
  if ( (stat2 & 0xe0) != 0xc0 ) return -ENODEV;	/* There is no YM3812 */

  /* There is a FM chip in this address. Detect the type (OPL2 to OPL4) */
  if ( opl3 -> hardware != OPL3_HW_AUTO ) return 0;

  if ( signature == 0x06 ) {			/* OPL2 */
    opl3 -> hardware = OPL3_HW_OPL2;
  } else {
    unsigned char tmp;

    opl3 -> hardware = OPL3_HW_OPL3;
    /*
     * Detect availability of OPL4 (_experimental_). Works probably
     * only after a cold boot. In addition the OPL4 port
     * of the chip may not be connected to the PC bus at all.
     */
    snd_opl3_command( opl3, OPL3_RIGHT | OPL3_REG_MODE, 0x00 );
    snd_opl3_command( opl3, OPL3_RIGHT | OPL3_REG_MODE, OPL3_MODE2_ENABLE | OPL3_MODE3_ENABLE );
    if ( (tmp = inb( opl3 -> l_port )) == 0x02 ) {	/* Have a OPL4 */
      opl3 -> hardware = OPL3_HW_OPL4;
    }
    snd_opl3_command( opl3, OPL3_RIGHT | OPL3_REG_MODE, 0x00 );
  }  

  return -ENODEV;
}

/*
 *  AdLib timers
 */

/*
 *  Timer 1 - 80us
 */

static unsigned int snd_opl3_timer1_resolution( snd_timer_t *timer )
{
  return 80000;
}

static void snd_opl3_timer1_start( snd_timer_t *timer )
{
  unsigned long flags;
  unsigned char tmp;
  unsigned int ticks;
  opl3_t *opl3;

  opl3 = (opl3_t *)timer -> private_data;
  snd_spin_lock( opl3, timer, &flags );
  ticks = timer -> cticks;
  if ( ticks > 256 ) ticks = 256;
  timer -> cticks -= ticks;
  tmp = (opl3 -> timer_enable | OPL3_TIMER1_START) & ~OPL3_TIMER1_MASK;
  opl3 -> timer_enable = tmp;
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER1, 256 - ticks ); /* timer 1 count */
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER_CONTROL, tmp );  /* enable timer 1 IRQ */
  timer -> flags |= SND_TIMER_FLG_RUNNING;
  snd_spin_unlock( opl3, timer, &flags );
}

static void snd_opl3_timer1_stop( snd_timer_t *timer )
{
  unsigned long flags;
  unsigned char tmp;
  opl3_t *opl3;

  opl3 = (opl3_t *)timer -> private_data;
  snd_spin_lock( opl3, timer, &flags );
  timer -> flags &= ~SND_TIMER_FLG_RUNNING;
  tmp = (opl3 -> timer_enable | OPL3_TIMER1_MASK) & ~OPL3_TIMER1_START;
  opl3 -> timer_enable = tmp;
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER_CONTROL, tmp ); /* disable timer #1 */
  snd_spin_unlock( opl3, timer, &flags );
}

/*
 *  Timer 2 - 320us
 */

static unsigned int snd_opl3_timer2_resolution( snd_timer_t *timer )
{
  return 320000;
}

static void snd_opl3_timer2_start( snd_timer_t *timer )
{
  unsigned long flags;
  unsigned char tmp;
  unsigned int ticks;
  opl3_t *opl3;

  opl3 = (opl3_t *)timer -> private_data;
  snd_spin_lock( opl3, timer, &flags );
  ticks = timer -> cticks;
  if ( ticks > 256 ) ticks = 256;
  timer -> cticks -= ticks;
  tmp = (opl3 -> timer_enable | OPL3_TIMER2_START) & ~OPL3_TIMER2_MASK;
  opl3 -> timer_enable = tmp;
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER2, 256 - ticks ); /* timer 1 count */
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER_CONTROL, tmp );  /* enable timer 1 IRQ */
  timer -> flags |= SND_TIMER_FLG_RUNNING;
  snd_spin_unlock( opl3, timer, &flags );
}

static void snd_opl3_timer2_stop( snd_timer_t *timer )
{
  unsigned long flags;
  unsigned char tmp;
  opl3_t *opl3;

  opl3 = (opl3_t *)timer -> private_data;
  snd_spin_lock( opl3, timer, &flags );
  timer -> flags &= ~SND_TIMER_FLG_RUNNING;
  tmp = (opl3 -> timer_enable | OPL3_TIMER2_MASK) & ~OPL3_TIMER2_START;
  opl3 -> timer_enable = tmp;
  snd_opl3_command( opl3, OPL3_LEFT | OPL3_REG_TIMER_CONTROL, tmp ); /* disable timer #1 */
  snd_spin_unlock( opl3, timer, &flags );
}

/*
 *
 */

static struct snd_stru_timer_hardware snd_opl3_timer1 = {
  80000,			/* resolution in us */
  1,                            /* low ticks */
  256,				/* high ticks */
  NULL,                         /* open */
  NULL,                         /* close */
  snd_opl3_timer1_resolution,	/* resolution */
  snd_opl3_timer1_start,	/* start */
  snd_opl3_timer1_stop,		/* stop */
  NULL,				/* continue */
};
 
static struct snd_stru_timer_hardware snd_opl3_timer2 = {
  320000,			/* resolution in us */
  1,                            /* low ticks */
  256,				/* high ticks */
  NULL,                         /* open */
  NULL,                         /* close */
  snd_opl3_timer2_resolution,	/* resolution */
  snd_opl3_timer2_start,	/* start */
  snd_opl3_timer2_stop,		/* stop */
  NULL,				/* continue */
};
 
static void snd_opl3_timers_init( snd_card_t *card, opl3_t *opl3 )
{
  snd_timer_t *timer;

  timer = snd_timer_new_device( card, "AdLib timer" );
  if ( timer ) {
    strcpy( timer -> name, "AdLib timer #1" );
    timer -> private_data = opl3;
    memcpy( &timer -> hw, &snd_opl3_timer1, sizeof( snd_opl3_timer1 ) );
    if ( snd_timer_register( timer ) < 0 ) {
      snd_timer_free( timer );
      timer = NULL;
    }
  }
  opl3 -> timer1 = timer;

  timer = snd_timer_new_device( card, "GF1 timer" );
  if ( timer ) {
    strcpy( timer -> name, "AdLib timer #2" );
    timer -> private_data = opl3;
    memcpy( &timer -> hw, &snd_opl3_timer2, sizeof( snd_opl3_timer2 ) );
    if ( snd_timer_register( timer ) < 0 ) {
      snd_timer_free( timer );
      timer = NULL;
    }
  }
  opl3 -> timer2 = timer;
}

static void snd_opl3_timers_done( opl3_t *opl3 )
{
  if ( opl3 -> timer1 ) {
    snd_timer_unregister( opl3 -> timer1 );
    opl3 -> timer1 = NULL;
  }
  if ( opl3 -> timer2 ) {
    snd_timer_unregister( opl3 -> timer2 );
    opl3 -> timer2 = NULL;
  }
}


/*
 *
 */

void snd_opl3_interrupt( snd_synth_t *synth )
{
  register unsigned char status;
  opl3_t *opl3;
  snd_timer_t *timer;
  
  opl3 = (opl3_t *)synth -> private_data;
  status = inb( opl3 -> l_port );
#if 0
  snd_printk( "AdLib IRQ status = 0x%x\n", status );
#endif
  if ( !(status & 0x80) ) return;

  if ( status & 0x40 ) {
    timer = opl3 -> timer1;
    snd_opl3_timer1_stop( timer );
    if ( !timer -> cticks ) {
      if ( timer && timer -> callback )
        timer -> callback( timer, timer -> callback_data );
    } else {
      snd_opl3_timer1_start( timer );
    }
  }
  if ( status & 0x20 ) {
    timer = opl3 -> timer2;
    snd_opl3_timer2_stop( timer );
    if ( !timer -> cticks ) {
      if ( timer && timer -> callback )
        timer -> callback( timer, timer -> callback_data );
    } else {
      snd_opl3_timer2_start( timer );
    }
  }  
}

/*
 *
 */

static void snd_opl3_free( void *private_data )
{
  opl3_t *opl3;

  opl3 = (opl3_t *)private_data;
  if ( opl3 -> timers )
    snd_opl3_timers_done( opl3 );
  snd_free( private_data, sizeof( opl3_t ) );
}

snd_synth_t *snd_opl3_new_device( snd_card_t *card, unsigned short l_port, unsigned short r_port, unsigned short hardware, int timers )
{
  opl3_t *opl3;
  snd_synth_t *synth;

  if ( ( synth = snd_synth_new_device( card, "OPL2/OPL3" ) ) == NULL )
    return NULL;
  opl3 = snd_calloc( sizeof( opl3_t ) );
  if ( !opl3 ) {
    snd_synth_free( synth );
    return NULL;
  }
  opl3 -> hardware = hardware;
  opl3 -> l_port = l_port;
  opl3 -> r_port = r_port;
  opl3 -> timers = timers != 0;
  strcpy( synth -> name, synth -> id );
  synth -> private_data = opl3;
  synth -> private_free = snd_opl3_free;
  snd_spin_prepare( opl3, reg );
  snd_spin_prepare( opl3, timer );
  if ( snd_opl3_detect( opl3 ) < 0 ) {
    snd_synth_free( synth );
    snd_printd( "opl3: OPL2/3 chip not detected at 0x%x/0x%x\n", opl3 -> l_port, opl3 -> r_port );
    return synth = NULL;
  }
  switch ( opl3 -> hardware & OPL3_HW_MASK ) {
    case OPL3_HW_OPL2:
      strcpy( synth -> name, "OPL2 FM" );
      break;
    case OPL3_HW_OPL3:
      strcpy( synth -> name, "OPL3 FM" );
      break;
    case OPL3_HW_OPL4:
      strcpy( synth -> name, "OPL4 FM" );
      break;
  }
  if ( opl3 -> timers )
    snd_opl3_timers_init( card, opl3 );
  return synth;
}

/*
 *  INIT part
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_opl3_export;
#endif

int init_module( void )
{
#ifndef LINUX_2_1
  if ( register_symtab( &snd_symbol_table_opl3_export ) < 0 )
    return -ENOMEM;
#endif
  return 0;
}

void cleanup_module( void )
{
}
