/*
 *  Digital Audio (PCM) abstract layer / Mixing devices
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#include "driver.h"
#include "info.h"
#include "pmix.h"

int snd_mixer_ioctl_card( snd_card_t *card, struct file *file, unsigned int cmd, unsigned long arg );

/*
 *
 */
 
static void snd_pmix_oss_compute_buffers( snd_pmix_t *pmix )
{
  unsigned long flags;
  snd_pmix_data_t *pdata = pmix -> pdata;
  unsigned int min_fragment, align = 4, rate, block, blocks;
  int bps;

  if ( !(pmix -> flags & SND_PMIX_FLG_BUFFERS) ) return;
  snd_spin_lock( pmix, playback, &flags );
  pmix -> used_size = pmix -> size;
  min_fragment = pdata -> fsize;
  rate = pmix -> rate;
  if ( rate > 7000 || rate < 9000 ) { 		/* 8000Hz */
    switch ( pdata -> rate ) {
      case 44100:
        rate = 7350;
        align = 6;
        min_fragment = pdata -> fsize / 6;
        break;
      case 22050:
        rate = 7350;
        align = 3;
        min_fragment = pdata -> fsize / 3;
        break;
      default:
        break;
    }
    if ( pdata -> channels == 2 )
      min_fragment >>= 1;
  } else {
    while ( rate < pdata -> rate ) {
      min_fragment >>= 1;
      rate <<= 1;
    }
    switch ( pdata -> channels ) {
      case 1:
        if ( pmix -> channels == 2 ) min_fragment <<= 1;
        break;
      case 2:
        if ( pmix -> channels == 1 ) min_fragment >>= 1;
        break;
    }
  }
  pmix -> flags &= SND_PMIX_FLG_BUFFERS;
  if ( pmix -> requested_block_size > 0 ) {
    block = min_fragment;
    while ( block <= pmix -> requested_block_size ) block <<= 1;
    block >>= 1;
    if ( block < min_fragment ) block = min_fragment;
    blocks = pmix -> used_size / block;
    if ( blocks > pmix -> requested_blocks )
      blocks = pmix -> requested_blocks;
  } else {
    bps = pmix -> rate * pmix -> channels;
    switch ( pmix -> format ) {
      case SND_PCM_SFMT_U16_LE:
      case SND_PCM_SFMT_U16_BE:
        bps <<= 1;
        break;
      case SND_PCM_SFMT_IMA_ADPCM:
        bps >>= 2;
        break;
     }
     bps >>= 2;
     if ( bps <= 0 ) bps = 16;
     block = min_fragment;
     while ( block <= bps ) block <<= 1;
     block >>= 1;
     block -= block % align;
     blocks = pmix -> used_size / block;
  }
  if ( blocks < 2 ) {
    snd_printk( "snd_pmix_oss_compute_buffers: oops ?\n" );
  }
  if ( blocks > 128 ) blocks = 128;
  pmix -> used_size = blocks * block;
  pmix -> blocks = blocks;
  pmix -> bsize = block;
  pmix -> flags |= SND_PMIX_FLG_NEUTRAL;
  pmix -> flags &= SND_PMIX_FLG_BUFFERS;
  snd_spin_unlock( pmix, playback, &flags );
}

static void snd_pmix_oss_fill_with_neutral( snd_pmix_t *pmix )
{
  if ( !(pmix -> flags & SND_PMIX_FLG_NEUTRAL) ) return;
  memset( pmix -> buffer, pmix -> neutral_byte, pmix -> size );
  pmix -> flags &= ~SND_PMIX_FLG_NEUTRAL;
}

static int snd_pmix_oss_reset( snd_pmix_t *pmix )
{
  unsigned long flags;

  snd_spin_lock( pmix, playback, &flags );
  pmix -> used = pmix -> head = pmix -> tail = pmix -> frag_size = 0;
  snd_spin_unlock( pmix, playback, &flags );
  return 0;
}

static int snd_pmix_oss_sync( snd_pmix_t *pmix )
{
  unsigned long flags;

  while ( pmix -> used > 0 ) {
    snd_spin_lock( pmix, sleep, &flags );
    pmix -> flags |= SND_PMIX_FLG_SLEEP;
    snd_sleep( pmix, sleep, HZ * 10 );
    pmix -> flags &= ~SND_PMIX_FLG_SLEEP;
    snd_spin_unlock( pmix, sleep, &flags );
    if ( snd_sleep_abort( pmix, sleep ) ) {
      pmix -> flags |= SND_PMIX_FLG_ABORT;
      return -EAGAIN;
    }
    if ( pmix -> used >= pmix -> blocks && snd_timeout( pmix, sleep ) ) {
      snd_printd( "pmix: timeout, new block discarded\n" );
      return -EIO;
    }
  }
  return 0;
}

static int snd_pmix_oss_rate( snd_pmix_t *pmix, long *result )
{
  if ( *result >= 0 ) {
    if ( *result >= 7000 && *result < 9000 ) {
      if ( pmix -> pdata -> rate == 11025 ) {
        *result = 11025;
      }
    } else {
      if ( *result > (44100 + 22050) / 2 ) *result = 44100; else
      if ( *result > (22050 + 11025) / 2 ) *result = 22050; else
                                           *result = 11025;
    }
    if ( *result != pmix -> rate ) {
      snd_pmix_oss_sync( pmix );
      pmix -> rate = *result;
      pmix -> flags |= SND_PMIX_FLG_NEUTRAL | SND_PMIX_FLG_BUFFERS;
    }
  } else {
    *result = pmix -> rate;
  }
  return 0;
}

static int snd_pmix_oss_channels( snd_pmix_t *pmix, long *result )
{
  if ( *result >= 0 ) {
    if ( *result > 1 ) *result = 2; else
                       *result = 1;
    if ( *result != pmix -> channels ) {
      snd_pmix_oss_sync( pmix );
      pmix -> channels = *result;
      pmix -> flags |= SND_PMIX_FLG_NEUTRAL | SND_PMIX_FLG_BUFFERS;
    }
  } else {
    *result = pmix -> channels;
  }
  return 0;
}

static int snd_pmix_oss_get_block_size( snd_pmix_t *pmix, long *result )
{
  *result = pmix -> bsize;
  return 0;
}

static int snd_pmix_oss_format( snd_pmix_t *pmix, long *result )
{
  if ( *result >= 0 ) {
    switch ( *result ) {
      case SND_PCM_SFMT_MU_LAW:
      case SND_PCM_SFMT_U8:
      case SND_PCM_SFMT_S16_LE:
        if ( *result != pmix -> format ) {
          snd_pmix_oss_sync( pmix );
          pmix -> format = *result;
          pmix -> flags |= SND_PMIX_FLG_NEUTRAL | SND_PMIX_FLG_BUFFERS;
        }
      default:
        if ( pmix -> format != SND_PCM_SFMT_U8 ) {
          snd_pmix_oss_sync( pmix );
          pmix -> format = SND_PCM_SFMT_U8;
          pmix -> flags |= SND_PMIX_FLG_NEUTRAL | SND_PMIX_FLG_BUFFERS;
        }
        *result = pmix -> format;
        return -EINVAL;
    }
  }
  *result = pmix -> format;
  return 0;
}

static int snd_pmix_oss_set_fragment( snd_pmix_t *pmix, long *result )
{
  unsigned int size, count, bytes, tmp;

  size = 0x10000;
  count = (*result >> 16) & 0xffff;
  bytes = *result & 0xffff;
  if ( !bytes ) {
    size = pmix -> requested_block_size;
    if ( !size ) size = 0x10000;
    for ( tmp = size; tmp > 0; tmp >>= 1 ) bytes++;
  } else {
    tmp = 31;
    while ( !(pmix -> pdata -> fsize >> tmp) & 1 ) tmp--;
    if ( bytes < tmp ) bytes = tmp;
    if ( bytes > 17 ) bytes = 17;
    size = 1 << bytes;
  }
  if ( !count ) {
    count = 128;
  } else {
    if ( count < 2 ) count = 2;
    if ( count > 128 ) count = 128;
  }

  if ( pmix -> requested_block_size != size ||
       pmix -> requested_blocks != count ) {
    snd_pmix_oss_sync( pmix );
    pmix -> requested_block_size = size;
    pmix -> requested_blocks = count;
    pmix -> flags |= SND_PMIX_FLG_NEUTRAL | SND_PMIX_FLG_BUFFERS;
  }

  *result = (count << 16) | bytes;
  return 0;
}

static int snd_pmix_oss_get_formats( snd_pmix_t *pmix, long *result )
{
  *result = SND_PCM_FMT_MU_LAW | SND_PCM_FMT_U8 | SND_PCM_SFMT_S16_LE;
  return 0;
}

static int snd_pmix_oss_nonblock( snd_pmix_t *pmix, struct file *file )
{
  file -> f_flags |= O_NONBLOCK;
  return 0;
}

static int snd_pmix_oss_get_caps( snd_pmix_t *pmix, long *result )
{
  *result = SND_PCM_CAP_TRIGGER | SND_PCM_CAP_BATCH | 0x0001;
  return 0;
}

static int snd_pmix_oss_trigger( snd_pmix_t *pmix, long *result )
{
  if ( *result >= 0 ) {
    if ( *result & SND_PCM_ENABLE_PLAYBACK )
      pmix -> flags |= SND_PMIX_FLG_TRIGGER2;
     else
      pmix -> flags &= ~SND_PMIX_FLG_TRIGGER2;
  } else {
    *result = 0;
    if ( pmix -> flags & SND_PMIX_FLG_TRIGGER2 )
      *result = SND_PCM_ENABLE_PLAYBACK;
  }
  return 0;
}

/*
 *  File operations
 */
 
static int snd_pmix_oss_open( void *private_data, snd_info_entry_t *entry, unsigned short mode, void **file_private_data )
{
  snd_pmix_data_t *pdata = (snd_pmix_data_t *)private_data;
  snd_pmix_t *pmix;
  int err;
  
  MOD_INC_USE_COUNT;
  pmix = snd_calloc( sizeof( snd_pmix_t ) );
  if ( !pmix ) return -ENOMEM;
  printk( "o1 - pdata -> ssize = %i\n", pdata -> ssize );
  pmix -> buffer = snd_malloc( pmix -> size = pdata -> ssize );
  if ( !pmix -> buffer ) {
    snd_free( pmix, sizeof( snd_pmix_t ) );
    MOD_DEC_USE_COUNT;
    return -ENOMEM;
  }
  printk( "o2\n" );
  if ( (err = snd_pmix_open_device( (snd_pmix_data_t *)private_data, pmix )) < 0 ) {
    snd_free( pmix -> buffer, pmix -> size );
    snd_free( pmix, sizeof( snd_pmix_t ) );
    MOD_DEC_USE_COUNT;
    return err;
  }
  *file_private_data = pmix;
  pmix -> rate = 8000;
  pmix -> format = SND_PCM_SFMT_U16_LE;
  pmix -> channels = 1;
  pmix -> flags = SND_PMIX_FLG_TRIGGER2;
  printk( "oss open ok..\n" );
  return 0;
}

static int snd_pmix_oss_release( void *private_data, snd_info_entry_t *entry, unsigned short mode, void *file_private_data )
{
  snd_pmix_t *pmix;
  int err;
  
  pmix = (snd_pmix_t *)file_private_data;
  if ( !pmix ) return -EINVAL;
  snd_pmix_oss_sync( pmix );
  if ( (err = snd_pmix_release_device( (snd_pmix_data_t *)private_data, pmix )) < 0 )
    return err;
  snd_free( pmix -> buffer, pmix -> size );
  MOD_DEC_USE_COUNT;
  printk( "oss close ok..\n" );
  return 0;
}

static long snd_pmix_oss_read( void *private_data, void *file_private_data, struct file *file, char *buf, long count )
{
  return -ENXIO;	/* not supported */
}

static long snd_pmix_oss_write( void *private_data, void *file_private_data, struct file *file, const char *buf, long count )
{
  long result;
  unsigned long flags;
  snd_pmix_t *pmix;
  unsigned int tmp;
  
  if ( (file -> f_flags & O_ACCMODE) == O_WRONLY ) return -EIO;

  if ( count <= 0 || !buf ) return 0;
  if ( verify_area( VERIFY_READ, buf, count ) ) return -EFAULT;
   
  pmix = (snd_pmix_t *)file_private_data;
  pmix -> flags &= ~SND_PMIX_FLG_ABORT;
  if ( pmix -> flags & SND_PMIX_FLG_MMAP ) return -EIO;	/* go to hell... */
  if ( pmix -> flags & SND_PMIX_FLG_BUFFERS )
    snd_pmix_oss_compute_buffers( pmix );
  if ( pmix -> flags & SND_PMIX_FLG_NEUTRAL )
    snd_pmix_oss_fill_with_neutral( pmix );
    
  result = 0;

  /*
   * OK. This is patch for .au files which begin with .snd header...
   * This is a little bit hard - it's application work to do conversions...
   */
  if ( pmix -> format == SND_PCM_SFMT_MU_LAW && !pmix -> processed_bytes && !pmix -> used && count > 31 ) {
    unsigned char buffer[ 8 ];

    copy_from_user( buffer, buf, 8 );
    if ( !memcmp( buffer, ".snd", 4 ) ) {
      tmp = (buffer[4]<<24)|(buffer[5]<<16)|(buffer[6]<<8)|(buffer[7]);
      if ( tmp > count ) tmp = count;
      if ( tmp < 128 ) {
        buf += tmp;
        count -= tmp;
        result += tmp;
      }
    }
  }

  while ( count > 0 ) {
    while ( pmix -> used >= pmix -> blocks ) {
      snd_spin_lock( pmix, playback, &flags );
      pmix -> flags |= SND_PMIX_FLG_TRIGGER1;
      if ( !(pmix -> flags & SND_PMIX_FLG_TRIGGER2) ||
           (file -> f_flags & O_NONBLOCK) ) {
        snd_spin_unlock( pmix, playback, &flags );
        return result;
      }
      snd_spin_unlock( pmix, playback, &flags );
      snd_spin_lock( pmix, sleep, &flags );
      pmix -> flags |= SND_PMIX_FLG_SLEEP;
      snd_sleep( pmix, sleep, HZ * 10 );
      pmix -> flags &= ~SND_PMIX_FLG_SLEEP;
      snd_spin_unlock( pmix, sleep, &flags );
      if ( snd_sleep_abort( pmix, sleep ) ) {
        pmix -> flags |= SND_PMIX_FLG_ABORT;
        return result;
      }
      if ( pmix -> used >= pmix -> blocks && snd_timeout( pmix, sleep ) ) {
        snd_printd( "pmix: timeout, new block discarded\n" );
        return -EIO;
      }
    }
    tmp = pmix -> bsize - pmix -> frag_size;
    if ( tmp > count ) tmp = count;   /* correction */
    copy_from_user( pmix -> buffer + ( pmix -> head * pmix -> bsize ) + pmix -> frag_size, buf, tmp );
    count -= tmp;
    result += tmp;
    buf += tmp;
    snd_spin_lock( pmix, playback, &flags );
    pmix -> frag_size += tmp;
    if ( pmix -> frag_size >= pmix -> bsize ) {
      pmix -> used++;
      pmix -> head++;
      pmix -> head %= pmix -> blocks;
      pmix -> frag_size = 0;
    }
    snd_spin_unlock( pmix, playback, &flags );
  }

  if ( pmix -> used ) {
    snd_spin_lock( pmix, playback, &flags );
    pmix -> flags |= SND_PMIX_FLG_TRIGGER1;
    snd_spin_unlock( pmix, playback, &flags );
  }

  return result;
}

#ifdef SND_POLL
unsigned int snd_pmix_oss_poll( void *private_data, void *file_private_data, struct file *file, poll_table *wait )
{
  unsigned long flags;
  snd_pmix_t *pmix;
  unsigned int mask;

  if ( (file -> f_flags & O_ACCMODE) == O_WRONLY ) return 0;

  pmix = (snd_pmix_t *)file_private_data;
  if ( pmix -> flags & SND_PMIX_FLG_BUFFERS )
    snd_pmix_oss_compute_buffers( pmix );
  snd_spin_lock( pmix, sleep, &flags );
  pmix -> flags |= SND_PMIX_FLG_SLEEP;
  snd_poll_wait( file, pmix, sleep, wait );
  snd_spin_unlock( pmix, sleep, &flags );

  mask = 0;
  if ( pmix -> flags & SND_PMIX_FLG_MMAP ) {
    if ( pmix -> interrupts ) mask |= POLLOUT | POLLIN;
  } else {
    if ( pmix -> used < pmix -> blocks ) mask |= POLLOUT | POLLIN;
  }

  return mask;
}
#else
int snd_pmix_oss_select( void *private_data, void *file_private_data, struct file *file, int sel_type, select_table *wait )
{
  unsigned long flags;
  snd_pmix_t *pmix;

  if ( (file -> f_flags & O_ACCMODE) == O_WRONLY ) return 0;

  pmix = (snd_pmix_t *)file_private_data;
  switch ( sel_type ) {
    case SEL_IN:
      break;
    case SEL_OUT:
      if ( pmix -> flags & SND_PMIX_FLG_BUFFERS )
        snd_pmix_oss_compute_buffers( pmix );
      snd_spin_lock( pmix, sleep, &flags );
      if ( pmix -> flags & SND_PMIX_FLG_MMAP ? !pmix -> interrupts : pmix -> used >= pmix -> blocks ) {
        pmix -> flags |= SND_PMIX_FLG_SLEEP;
        snd_select_wait( pmix, sleep, wait );
        snd_spin_unlock( pmix, sleep, &flags );
        return 0;
      }
      snd_spin_unlock( pmix, sleep, &flags );
      break;
    case SEL_EX:
      break;
  }
  return 0;
}
#endif

int snd_pmix_oss_ioctl( void *private_data, void *file_private_data, struct file *file, unsigned int cmd, unsigned long arg )
{
  long result;
  snd_pmix_t *pmix = (snd_pmix_t *)file_private_data;
  int err;

  if ( (file -> f_flags & O_ACCMODE) == O_WRONLY ) return -EIO;
  if ( ( ( cmd >> 8 ) & 0xff ) == 'M' ) /* mixer ioctl - for OSS (grrr) compatibility */
    return snd_mixer_ioctl_card( pmix -> pdata -> card, file, cmd, arg );
  if ( ( ( cmd >> 8 ) & 0xff ) == 'P' ) return -EIO;
  switch ( cmd ) {
    case SND_PCM_IOCTL_OSS_RESET:
      return snd_pmix_oss_reset( pmix );
    case SND_PCM_IOCTL_OSS_SYNC:
      return snd_pmix_oss_sync( pmix );
    case SND_PCM_IOCTL_OSS_RATE:
      if ( (result = snd_ioctl_in( (long *)arg )) < 0 ) return -EFAULT;
      if ( (err = snd_pmix_oss_rate( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_GETRATE:
      result = -1;
      if ( (err = snd_pmix_oss_rate( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_STEREO:
      if ( (result = snd_ioctl_in( (long *)arg )) < 0 ) return -EFAULT;
      result = result > 0 ? 2 : 1;
      if ( (err = snd_pmix_oss_channels( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_GETBLKSIZE:
      result = -1;
      if ( (err = snd_pmix_oss_get_block_size( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_FORMAT:
      if ( (result = snd_ioctl_in( (long *)arg )) < 0 ) return -EFAULT;
      if ( (err = snd_pmix_oss_format( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_GETFORMAT:
      result = -1;
      if ( (err = snd_pmix_oss_format( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_CHANNELS:
      if ( (result = snd_ioctl_in( (long *)arg )) < 0 ) return -EFAULT;
      if ( (err = snd_pmix_oss_channels( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_GETCHANNELS:
      result = -1;
      if ( (err = snd_pmix_oss_channels( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_FILTER:
    case SND_PCM_IOCTL_OSS_GETFILTER:
      return -EIO;
    case SND_PCM_IOCTL_OSS_POST:	/* wrong implementation */
      return snd_pmix_oss_sync( pmix );
    case SND_PCM_IOCTL_OSS_SUBDIVIDE:	/* obsolete, not implemented */
      return 0;
    case SND_PCM_IOCTL_OSS_SETFRAGMENT:
      if ( (result = snd_ioctl_in( (long *)arg )) < 0 ) return -EFAULT;
      if ( (err = snd_pmix_oss_set_fragment( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_GETFORMATS:
      if ( (err = snd_pmix_oss_get_formats( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_NONBLOCK:
      return snd_pmix_oss_nonblock( pmix, file );
    case SND_PCM_IOCTL_OSS_GETCAPS:
      if ( (err = snd_pmix_oss_get_caps( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_GETTRIGGER:
      result = -1;
      if ( (err = snd_pmix_oss_trigger( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_SETTRIGGER:
      if ( (result = snd_ioctl_in( (long *)arg )) < 0 ) return -EFAULT;
      if ( (err = snd_pmix_oss_trigger( pmix, &result )) < 0 ) return err;
      return snd_ioctl_out( (long *)arg, result );
    case SND_PCM_IOCTL_OSS_DUPLEX:
      return -EIO;
  }
  return -ENXIO;
}

/*
 *
 */

struct snd_info_entry_data snd_pmix_oss_fops = {
  snd_pmix_oss_open,
  snd_pmix_oss_release,
  snd_pmix_oss_read,
  snd_pmix_oss_write,
  NULL,
#ifdef SND_POLL
  snd_pmix_oss_poll,
#else
  snd_pmix_oss_select,
#endif
  snd_pmix_oss_ioctl,
  NULL
};
