/*
 *  Driver for Yamaha OPL3-SA[2,3] soundcards
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "cs4231.h"
#include "mpu401.h"
#include "opl3.h"
#include "initval.h"

int snd_index[ SND_CARDS ] = SND_DEFAULT_IDX;	/* Index 1-MAX */
char *snd_id[ SND_CARDS ] = SND_DEFAULT_STR;	/* ID for this card */
int snd_port[ SND_CARDS ] = SND_DEFAULT_PORT;	/* 0xf86,0x370,0x100 */
int snd_wss_port[ SND_CARDS ] = SND_DEFAULT_PORT; /* 0x530,0xe80,0xf40,0x604 */
int snd_fm_port[ SND_CARDS ] = SND_DEFAULT_PORT; /* 0x388 */
int snd_midi_port[ SND_CARDS ] = SND_DEFAULT_PORT; /* 0x330,0x300 */
int snd_irq[ SND_CARDS ] = SND_DEFAULT_IRQ;	/* 0,1,3,5,9,11,12,15 */
int snd_dma1[ SND_CARDS ] = SND_DEFAULT_DMA;	/* 1,3,5,6,7 */
int snd_dma2[ SND_CARDS ] = SND_DEFAULT_DMA;	/* 1,3,5,6,7 */
int snd_dma1_size[ SND_CARDS ] = SND_DEFAULT_DMA_SIZE; /* 8,16,32,64,128 */
int snd_dma2_size[ SND_CARDS ] = SND_DEFAULT_DMA_SIZE; /* 8,16,32,64,128 */
#ifdef MODULE_PARM
MODULE_PARM( snd_index, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_index, "Index value for OPL3-SA soundcard." );
MODULE_PARM( snd_id, "1-" __MODULE_STRING(SND_CARDS) "s" );
MODULE_PARM_DESC( snd_id, "ID string for OPL3-SA soundcard." );
MODULE_PARM( snd_port, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_port, "Port # for OPL3-SA driver." );
MODULE_PARM( snd_wss_port, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_wss_port, "WSS port # for OPL3-SA driver." );
MODULE_PARM( snd_fm_port, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_fm_port, "FM port # for OPL3-SA driver." );
MODULE_PARM( snd_midi_port, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_midi_port, "MIDI port # for OPL3-SA driver." );
MODULE_PARM( snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_irq, "IRQ # for OPL3-SA driver." );
MODULE_PARM( snd_dma1, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_dma1, "DMA1 # for OPL3-SA driver." );
MODULE_PARM( snd_dma2, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_dma2, "DMA2 # for OPL3-SA driver." );
MODULE_PARM( snd_dma1_size, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_dma1_size, "DMA1 size in kB for OPL3-SA driver." );
MODULE_PARM( snd_dma2_size, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_dma2_size, "DMA2 size in kB for OPL3-SA driver." );
#endif

struct snd_opl3sa {
  int version;		/* 2 or 3 */
  int irqnum;
  int dma1num;
  int dma2num;
  unsigned short port;
  snd_card_t *card;
  snd_pcm_t *pcm;
  snd_kmixer_t *mixer;
  snd_rawmidi_t *rawmidi;
  snd_synth_t *synth;
  unsigned short pcm_status_reg;
  snd_spin_define( reg );
};

static struct snd_opl3sa *snd_opl3sa_cards[ SND_CARDS ] = SND_DEFAULT_PTR;

static void snd_opl3sa_use_inc( snd_card_t *card )
{
  MOD_INC_USE_COUNT;
}

static void snd_opl3sa_use_dec( snd_card_t *card )
{
  MOD_DEC_USE_COUNT;
}

static unsigned char snd_opl3sa_read( unsigned short port, unsigned char reg )
{
  unsigned long flags;
  unsigned char result;
  
  snd_cli( &flags );
#if 0
  outb( 0x1d, port );	/* password */
  printk( "read [0x%x] = 0x%x\n", port, inb( port ) );
#endif
  outb( reg, port );	/* register */
  result = inb( port + 1 );
  snd_sti( &flags );
#if 0
  printk( "read [0x%x] = 0x%x [0x%x]\n", port, result, inb( port ) );
#endif
  return result;
}

static void snd_opl3sa_write( unsigned short port, unsigned char reg, unsigned char value )
{
  unsigned long flags;
  
  snd_cli( &flags );
#if 0
  outb( 0x1d, port );	/* password */
#endif
  outb( reg, port );	/* register */
  outb( value, port + 1 );
  snd_sti( &flags );
}

static int snd_opl3sa_detect( struct snd_opl3sa *oplcard,
                              unsigned short port,
                              unsigned short wss_port,
                              unsigned short fm_port,
                              unsigned short midi_port )
{
  snd_card_t *card;
  unsigned char tmp, tmp1;
  char str[2];

  card = oplcard -> card;
  if ( snd_register_ioport( card, port, 2, "OPL3-SA control" ) < 0 )
    return -EBUSY;
  if ( snd_register_ioport( card, wss_port, 8, "OPL3-SA WSS" ) < 0 )
    goto __nodev;
  if ( fm_port >= 0x340 && fm_port < 0x400 )
    if ( snd_register_ioport( card, fm_port, 8, "OPL3-SA AdLib FM" ) < 0 )
      goto __nodev;
  if ( midi_port >= 0x300 && midi_port < 0x340 )
    if ( snd_register_ioport( card, midi_port, 2, "OPL3-SA MPU-401" ) < 0 )
      goto __nodev;
#if 0
  snd_printk( "REG 0A = 0x%x\n", snd_opl3sa_read( port, 0x0a ) );
#endif
  oplcard -> port = port;
  oplcard -> version = 0;
  tmp = snd_opl3sa_read( port, 0x0a );
  switch ( tmp & 0x07 ) {
    case 0x01: 
      oplcard -> version = 2;
      break;
    case 0x02:
    case 0x03:			/* YM715B ?? */
    case 0x04:			/* YM719 - OPL-SA4? */
      oplcard -> version = 3;
      break;
    default:
      snd_printd( "OPL3-SA [0x%x] detect (0) = 0x%x\n", port, tmp );
      goto __nodev;
  }
  str[0] = oplcard -> version + '0';
  str[1] = 0;
  strcat( card -> shortname, str );
  snd_opl3sa_write( port, 0x0a, tmp | 7 );
  if ( (tmp1 = snd_opl3sa_read( port, 0x0a )) != tmp ) {
    snd_printd( "OPL3-SA [0x%x] detect (1) = 0x%x (0x%x)\n", port, tmp, tmp1 );
    goto __nodev;
  }
  snd_opl3sa_write( port, 0x01, 0x00 );	/* Power Management - default */
  snd_opl3sa_write( port, 0x02, 0x00 ); /* System Control - default */
  snd_opl3sa_write( port, 0x03, 0x0d );	/* Interrupt Channel Configuration - IRQ A = OPL3 + MPU + WSS */
  if ( oplcard -> dma2num == SND_DMA_DISABLE ) {
    snd_opl3sa_write( port, 0x06, 0x03 ); /* DMA Configuration - DMA A = WSS-R + WSS-P */
  } else {
    snd_opl3sa_write( port, 0x06, 0x21 ); /* DMA Configuration - DMA B = WSS-R, DMA A = WSS-P */
  }
  snd_opl3sa_write( port, 0x0a, 0x80 | (tmp & 7) ); /* Miscellaneous - default */
  if ( oplcard -> version > 2 ) {
    snd_opl3sa_write( port, 0x12, 0x00 ); /* Digital Block Partial Power Down - default */
    snd_opl3sa_write( port, 0x13, 0x00 ); /* Analog Block Partial Power Down - default */
  }
  return 0;

  __nodev:
  snd_unregister_ioports( card );
  return -ENODEV;
}

static void snd_opl3sa_interrupt( int irq, void *dev_id, struct pt_regs *regs )
{
  register unsigned short status;
  unsigned long flags;
  struct snd_opl3sa *oplcard = (struct snd_opl3sa *)dev_id;

  if ( !oplcard ) return;

  snd_spin_lock( oplcard, reg, &flags );
  outb( 0x04, oplcard -> port );	/* register - Interrupt IRQ-A status */
  status = inb( oplcard -> port + 1 );
  snd_spin_unlock( oplcard, reg, &flags );

  if ( oplcard -> synth && (status & 0x20) )
    snd_opl3_interrupt( oplcard -> synth );

  if ( oplcard -> rawmidi && (status & 0x10) )
    snd_mpu401_uart_interrupt( oplcard -> rawmidi );

  if ( oplcard -> pcm && (status & 0x07) )  /* TI,CI,PI */
    snd_cs4231_interrupt( oplcard -> pcm, status );
}

static int snd_opl3sa_resources( int dev, struct snd_opl3sa *oplcard, snd_card_t *card )
{
  static int possible_irqs[] = { -1 };	/* must be specified by user */
  static int possible_dmas[] = { -1 };	/* must be specified by user */

  if ( (oplcard -> irqnum = snd_register_interrupt( card, "OPL3-SA", snd_irq[ dev ], SND_IRQ_TYPE_ISA, snd_opl3sa_interrupt, oplcard, possible_irqs )) < 0 ) {
    return oplcard -> irqnum;
  }
  if ( (oplcard -> dma1num = snd_register_dma_channel( card, "OPL3-SA playback", snd_dma1[ dev ], SND_DMA_TYPE_ISA, snd_dma1_size[ dev ], possible_dmas )) < 0 ) {
    return oplcard -> dma1num;
  }
  if ( snd_dma2[ dev ] >= 0 ) {
    if ( (oplcard -> dma2num = snd_register_dma_channel( card, "OPL3-SA record", snd_dma2[ dev ], SND_DMA_TYPE_ISA, snd_dma2_size[ dev ], possible_dmas )) < 0 ) {
      return oplcard -> dma2num;
    }
  } else {
    oplcard -> dma2num = SND_DMA_DISABLE;
  }
  return 0;
}

static void snd_opl3sa_set_mute( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, unsigned int mute )
{
  unsigned long flags;
  struct snd_opl3sa *oplcard;
  unsigned char regl, regr, mask, tmp;

  oplcard = (struct snd_opl3sa *)channel -> private_data;
  regl = (unsigned char)(channel -> hw.private_value >> 24);
  regr = (unsigned char)(channel -> hw.private_value >> 16);
  mask = (unsigned char)channel -> hw.private_value;
  snd_spin_lock( oplcard, reg, &flags );
  tmp = snd_opl3sa_read( oplcard -> port, regl ) & ~mask;
  tmp |= mute & SND_MIX_MUTE_LEFT ? mask : 0;
  snd_opl3sa_write( oplcard -> port, regl, tmp );
  if ( channel -> hw.stereo ) {
    tmp = snd_opl3sa_read( oplcard -> port, regr ) & ~mask;
    tmp |= mute & SND_MIX_MUTE_RIGHT ? mask : 0;
    snd_opl3sa_write( oplcard -> port, regr, tmp );
  }
  snd_spin_unlock( oplcard, reg, &flags );
}

static void snd_opl3sa_set_volume_level( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int left, int right )
{
  unsigned long flags;
  struct snd_opl3sa *oplcard;
  unsigned char regl, regr, mask, tmp;

  oplcard = (struct snd_opl3sa *)channel -> private_data;
  regl = (unsigned char)(channel -> hw.private_value >> 24);
  regr = (unsigned char)(channel -> hw.private_value >> 16);
  mask = (unsigned char)channel -> hw.private_value;
  if ( channel -> hw.private_value & 0x2000 ) {
    left = channel -> hw.max - left;
    right = channel -> hw.max - right;
  }
  snd_spin_lock( oplcard, reg, &flags );
  tmp = snd_opl3sa_read( oplcard -> port, regl ) & mask;
  snd_opl3sa_write( oplcard -> port, regl, tmp | left );
  if ( channel -> hw.stereo ) {
    tmp = snd_opl3sa_read( oplcard -> port, regr ) & mask;
    snd_opl3sa_write( oplcard -> port, regr, tmp | right );
  }
  snd_spin_unlock( oplcard, reg, &flags );
}

static void snd_opl3sa_set_3d_volume_level( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int left, int right )
{
  unsigned long flags;
  struct snd_opl3sa *oplcard;
  unsigned char reg;

  oplcard = (struct snd_opl3sa *)channel -> private_data;
  reg = (unsigned char)channel -> hw.private_value;
  snd_spin_lock( oplcard, reg, &flags );
  snd_opl3sa_write( oplcard -> port, reg, right << 4 | left );
  snd_spin_unlock( oplcard, reg, &flags );
}

#define OPL3SA_PRIVATE( left, right, shift, mute ) ((left << 24)|(right << 16)|(shift<<8)|mute)

static int snd_opl3sa_mixer( struct snd_opl3sa *oplcard, snd_kmixer_t *mixer )
{
  static struct snd_stru_mixer_channel_hw master_mix = {
    SND_MIXER_PRI_MASTER,       /* priority */
    SND_MIXER_PRI_PARENT,       /* parent priority */
    SND_MIXER_ID_MASTER,        /* device name */
    SND_MIXER_OSS_VOLUME,       /* OSS device # */
    1, 1, 1, 0, 0,              /* mute/stereo/record/digital/input */
    0, 15,                      /* min, max value */
    -3000, 0, 200,              /* min, max, step - dB */
    OPL3SA_PRIVATE( 0x07, 0x08, 0, 0x80 ) | 0x2000,  /* private value */
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,                       /* record source */
    snd_opl3sa_set_mute,        /* set mute */
    snd_opl3sa_set_volume_level, /* set volume level */
  };
  static struct snd_stru_mixer_channel_hw wide_mix = {
    SND_MIXER_PRI_3D,           /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_3D,            /* device name */
    SND_MIXER_OSS_UNKNOWN,      /* OSS device # */
    0, 1, 0, 0, 0,              /* mute/stereo/record/digital/input */
    0, 7,                       /* min, max value */
    0, 1050, 150,               /* min, max, step - dB */
    0x14,			/* private value */
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,                       /* record source */
    NULL,			/* set mute */
    snd_opl3sa_set_3d_volume_level, /* set volume level */
  };
  static struct snd_stru_mixer_channel_hw bass_mix = {
    SND_MIXER_PRI_BASS,         /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_BASS,          /* device name */
    SND_MIXER_OSS_BASS,         /* OSS device # */
    0, 1, 0, 0, 0,              /* mute/stereo/record/digital/input */
    0, 7,                       /* min, max value */
    0, 1050, 150,               /* min, max, step - dB */
    0x15,			/* private value */
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,                       /* record source */
    NULL,			/* set mute */
    snd_opl3sa_set_3d_volume_level, /* set volume level */
  };
  static struct snd_stru_mixer_channel_hw treble_mix = {
    SND_MIXER_PRI_TREBLE,       /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_TREBLE,        /* device name */
    SND_MIXER_OSS_TREBLE,       /* OSS device # */
    0, 1, 0, 0, 0,              /* mute/stereo/record/digital/input */
    0, 7,                       /* min, max value */
    0, 1050, 150,               /* min, max, step - dB */
    0x16,			/* private value */
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,                       /* record source */
    NULL,			/* set mute */
    snd_opl3sa_set_3d_volume_level, /* set volume level */
  };
  snd_kmixer_channel_t *channel, *channel1, *channel2, *channel3;

  /* reassign AUXA to CD */
  channel1 = snd_mixer_find_channel( mixer, SND_MIXER_PRI_AUXA );
  channel1 -> hw.priority = SND_MIXER_PRI_CD;
  channel1 -> hw.ossdev = SND_MIXER_OSS_CD;
  strcpy( channel1 -> hw.name, SND_MIXER_ID_CD );
  snd_mixer_reorder_channel( mixer, channel1 );
  /* reassign AUXB to SYNTHESIZER */
  channel2 = snd_mixer_find_channel( mixer, SND_MIXER_PRI_AUXB );
  channel2 -> hw.priority = SND_MIXER_PRI_SYNTHESIZER;
  channel2 -> hw.ossdev = SND_MIXER_OSS_SYNTH;
  strcpy( channel2 -> hw.name, SND_MIXER_ID_SYNTHESIZER );
  snd_mixer_reorder_channel( mixer, channel2 );
  /* change MIC channel */
  channel3 = snd_mixer_find_channel( mixer, SND_MIXER_PRI_MIC );
  channel3 -> hw.max = 31;
  channel3 -> hw.min_dB = -3450;
  channel3 -> hw.max_dB = 1200;
  channel3 -> hw.step_dB = 150;
  channel3 -> hw.private_value = OPL3SA_PRIVATE( 0x09, 0x00, 0, 0x80 );
  channel3 -> hw.set_mute = snd_opl3sa_set_mute;
  channel3 -> hw.set_volume_level = snd_opl3sa_set_volume_level;
  /* add Master volume control */
  channel = snd_mixer_new_channel( mixer, &master_mix );
  if ( !channel ) return -ENOMEM;
  channel -> private_data = oplcard;
  /* add 3D Wide, Bass and Treble for OPL3-SA3 */
  if ( oplcard -> version > 2 ) {
    snd_mixer_new_channel( mixer, &wide_mix );
    snd_mixer_new_channel( mixer, &bass_mix );
    snd_mixer_new_channel( mixer, &treble_mix );
  }
  return 0;
}

static int snd_opl3sa_probe( int dev, struct snd_opl3sa *oplcard )
{
  snd_card_t *card;
  snd_pcm_t *pcm;
  snd_kmixer_t *mixer = NULL;
  snd_rawmidi_t *rmidi = NULL;
  snd_synth_t *synth = NULL;
    
  if ( snd_port[ dev ] == SND_AUTO_PORT ||
       snd_wss_port[ dev ] == SND_AUTO_PORT ||
       snd_fm_port[ dev ] == SND_AUTO_PORT ||
       snd_midi_port[ dev ] == SND_AUTO_PORT ) {
    snd_printk( "probing for Yamaha OPL3-SA isn't supported\n" );
    snd_printk( "port = 0x%x, wss_port = 0x%x, fm_port = 0x%x, midi_port = 0x%x\n",
    		snd_port[ dev ],
    		snd_wss_port[ dev ],
    		snd_fm_port[ dev ],
    		snd_midi_port[ dev ] );
    return -EINVAL;
  }
  card = snd_card_new( snd_index[ dev ], snd_id[ dev ],
                       snd_opl3sa_use_inc, snd_opl3sa_use_dec );
  if ( !card ) return -ENOMEM;
  card -> type = SND_CARD_TYPE_OPL3_SA;
  strcpy( card -> abbreviation, "OPL3SA" );
  strcpy( card -> shortname, "Yamaha OPL3-SA" );
  if ( snd_opl3sa_resources( dev, oplcard, card ) < 0 ) {
    snd_card_free( card );
    return -EBUSY;
  }
  oplcard -> card = card;
  if ( snd_opl3sa_detect( oplcard, snd_port[ dev ], snd_wss_port[ dev ],
                                   snd_fm_port[ dev ], snd_midi_port[ dev ] ) ) {
    snd_card_free( card );
    oplcard -> card = NULL;
    return -ENODEV;
  }
  oplcard -> port = snd_port[ dev ];
  if ( snd_card_register( card ) ) {
    snd_card_free( card );
    return -ENOMEM;
  }
  oplcard -> pcm_status_reg = snd_wss_port[ dev ] + 4 + 2;
  pcm = snd_cs4231_new_device( card,
                               snd_wss_port[ dev ] + 4,
                               oplcard -> irqnum,
                               oplcard -> dma1num,
                               oplcard -> dma2num == SND_DMA_DISABLE ? oplcard -> dma1num : oplcard -> dma2num,
                               CS4231_HW_OPL3SA );
  if ( !pcm ) {
    snd_printd( "Oops, WSS not detected at 0x%x\n", snd_wss_port[ dev ] + 4 );
    goto __nodev;
  }
  mixer = snd_cs4231_new_mixer( pcm );
  if ( !mixer ) goto __nodev;
  if ( snd_opl3sa_mixer( oplcard, mixer ) < 0 ) goto __nodev;
  if ( snd_fm_port[ dev ] >= 0x340 && snd_fm_port[ dev ] < 0x400 ) {
    synth = snd_opl3_new_device( card, snd_fm_port[ dev ], snd_fm_port[ dev ] + 2, OPL3_HW_OPL3, 1 );
    if ( !synth ) goto __nodev;
  }
  if ( snd_midi_port[ dev ] >= 0x300 && snd_midi_port[ dev ] < 0x340 ) {
    rmidi = snd_mpu401_uart_new_device( card, MPU401_HW_OPL3SA, snd_midi_port[ dev ], oplcard -> irqnum );
    if ( !rmidi ) goto __nodev;
  }
  if ( synth )
    if ( snd_synth_register( synth ) < 0 ) goto __nodev;
  if ( rmidi ) {
    if ( snd_rawmidi_register( rmidi, 0 ) < 0 ) {
      if ( synth ) snd_synth_unregister( synth ); synth = NULL;
      goto __nodev;
    }
  }
  if ( snd_pcm_register( pcm, 0 ) < 0 ) {
    if ( synth ) snd_synth_unregister( synth ); synth = NULL;
    if ( rmidi ) snd_rawmidi_unregister( rmidi ); rmidi = NULL;
    goto __nodev;
  }
  if ( snd_mixer_register( mixer, 0 ) < 0 ) {
    if ( synth ) snd_synth_unregister( synth ); synth = NULL;
    if ( rmidi ) snd_rawmidi_unregister( rmidi ); rmidi = NULL;
    snd_pcm_unregister( pcm ); pcm = NULL;
    goto __nodev;
  }
  snd_enable_irq( card, oplcard -> irqnum );
  sprintf( card -> longname, "%s at 0x%x, irq %i, dma %i",
    card -> shortname,
    oplcard -> port,
    card -> irqs[ oplcard -> irqnum ] -> irq, 
    card -> dmas[ oplcard -> dma1num ] -> dma );
  if ( oplcard -> dma2num != SND_DMA_DISABLE )
    sprintf( card -> longname + strlen( card -> longname ), "&%i",
      card -> dmas[ oplcard -> dma2num ] -> dma );

  if ( !snd_card_register( card ) ) {
    oplcard -> pcm = pcm;
    oplcard -> mixer = mixer;
    oplcard -> rawmidi = rmidi;
    oplcard -> synth = synth;
    return 0;
  }
  snd_mixer_unregister( mixer ); mixer = NULL;
  snd_pcm_unregister( pcm ); pcm = NULL;
  
  __nodev:
  if ( rmidi ) snd_rawmidi_free( rmidi );
  if ( mixer ) snd_mixer_free( mixer );
  if ( pcm ) snd_pcm_free( pcm );
  oplcard -> card = NULL;
  snd_card_free( card );
  return -ENXIO;
}

int init_module( void )
{
  int dev, cards;
  struct snd_opl3sa *oplcard;

#ifndef LINUX_2_1
  register_symtab( NULL );
#endif
  for ( dev = cards = 0; dev < SND_CARDS && snd_port[ dev ] > 0; dev++ ) {
    oplcard = (struct snd_opl3sa *)snd_malloc( sizeof( struct snd_opl3sa ) );
    if ( !oplcard ) continue;
    snd_spin_prepare( oplcard, reg );
    memset( oplcard, 0, sizeof( struct snd_opl3sa ) );
    if ( snd_opl3sa_probe( dev, oplcard ) < 0 ) {
      snd_free( oplcard, sizeof( struct snd_opl3sa ) );
      if ( snd_port[ dev ] == SND_AUTO_PORT ) break;
      snd_printk( "Yamaha OPL3-SA soundcard #%i not found at 0x%x or device busy\n", dev + 1, snd_port[ dev ] );
      continue;
    }
    snd_opl3sa_cards[ dev ] = oplcard;
    cards++;
  }
  if ( !cards ) { 
    snd_printk( "Yamaha OPL3-SA soundcard not found or device busy\n" );
    return -ENODEV;
  }
  return 0;
}

void cleanup_module( void )
{
  int idx;
  struct snd_opl3sa *oplcard;
  snd_pcm_t *pcm;

  for ( idx = 0; idx < SND_CARDS; idx++ ) {
    oplcard = snd_opl3sa_cards[ idx ];
    if ( oplcard ) {
      snd_card_unregister( oplcard -> card );
      if ( oplcard -> synth )
        snd_synth_unregister( oplcard -> synth );
      if ( oplcard -> rawmidi )
        snd_rawmidi_unregister( oplcard -> rawmidi );
      if ( oplcard -> mixer )
        snd_mixer_unregister( oplcard -> mixer );
      if ( oplcard -> pcm ) {
        pcm = oplcard -> pcm;
        oplcard -> pcm = NULL;	/* turn off interrupts */
        snd_pcm_unregister( pcm );
      }
      snd_card_free( oplcard -> card );
      snd_free( oplcard, sizeof( struct snd_opl3sa ) );
    }
  }
}
