/*
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>,
 *                   Thomas Sailer <sailer@ife.ee.ethz.ch>
 *  Routines for control of S3 SonicVibes (86c617) chip
 *
 *  BUGS:
 *    Looks like 86c617 rev 3 doesn't supports DDMA buffers above 16MB?
 *
 *  TODO:
 *    Loopback control...
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
  
#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "sonicvibes.h"

/*
 *  common I/O routines
 */

static inline void snd_sonicvibes_setdmaa( sonicvibes_t *sonic, unsigned int addr, unsigned int count )
{
  count--;
  outl( addr, sonic -> dmaa_port + SV_DMA_ADDR0 );
  outl( count, sonic -> dmaa_port + SV_DMA_COUNT0 );
  outb( 0x18, sonic -> dmaa_port + SV_DMA_MODE );
#if 0
  printk( "program dmaa: addr = 0x%x, paddr = 0x%x\n", addr, inl( sonic -> dmaa_port + SV_DMA_ADDR0 ) );
#endif
}

static inline void snd_sonicvibes_setdmac( sonicvibes_t *sonic, unsigned int addr, unsigned int count )
{
  /* note: dmac is working in word mode!!! */
  count >>= 1;
  count--;
  outl( addr, sonic -> dmac_port + SV_DMA_ADDR0 );
  outl( count, sonic -> dmac_port + SV_DMA_COUNT0 );
  outb( 0x14, sonic -> dmac_port + SV_DMA_MODE );
#if 0
  printk( "program dmac: addr = 0x%x, paddr = 0x%x\n", addr, inl( sonic -> dmac_port + SV_DMA_ADDR0 ) );
#endif
}

static inline unsigned int snd_sonicvibes_getdmaa( sonicvibes_t *sonic )
{
  return (inl( sonic -> dmaa_port + SV_DMA_COUNT0) & 0xffffff) + 1;
}

static inline unsigned int snd_sonicvibes_getdmac( sonicvibes_t *sonic )
{
  /* note: dmac is working in word mode!!! */
  return ((inl( sonic -> dmac_port + SV_DMA_COUNT0) & 0xffffff) + 1) << 1;
}

static void snd_sonicvibes_out1( sonicvibes_t *sonic, unsigned char reg, unsigned char value )
{
  outb( reg, SV_REG( sonic, INDEX ) );
  snd_delay( 1 );
  outb( value, SV_REG( sonic, DATA ) );
  snd_delay( 1 );
}

static void snd_sonicvibes_out( sonicvibes_t *sonic, unsigned char reg, unsigned char value )
{
  unsigned long flags;

  snd_spin_lock( sonic, reg, &flags );
  outb( reg, SV_REG( sonic, INDEX ) );
  snd_delay( 1 );
  outb( value, SV_REG( sonic, DATA ) );
  snd_delay( 1 );
  snd_spin_unlock( sonic, reg, &flags );
}

#if 0
static unsigned char snd_sonicvibes_in1( sonicvibes_t *sonic, unsigned char reg )
{
  unsigned char value;

  outb( reg, SV_REG( sonic, INDEX ) );
  snd_delay( 1 );
  value = inb( SV_REG( sonic, DATA ) );
  snd_delay( 1 );
  return value;
}
#endif

static unsigned char snd_sonicvibes_in( sonicvibes_t *sonic, unsigned char reg )
{
  unsigned long flags;
  unsigned char value;

  snd_spin_lock( sonic, reg, &flags );
  outb( reg, SV_REG( sonic, INDEX ) );
  snd_delay( 1 );
  value = inb( SV_REG( sonic, DATA ) );
  snd_delay( 1 );
  snd_spin_unlock( sonic, reg, &flags );
  return value;
}

static void snd_sonicvibes_outm1( sonicvibes_t *sonic, unsigned char reg, unsigned char mask, unsigned char value )
{
  unsigned char prev;

  outb( reg, SV_REG( sonic, INDEX ) );
  snd_delay( 1 );
  prev = inb( SV_REG( sonic, DATA ) );
  snd_delay( 1 );
  outb( (prev & mask) | value, SV_REG( sonic, DATA ) );
  snd_delay( 1 );
#if 0
  printk( "outm: 0x%x, prev = 0x%x, mask = 0x%x, value = 0x%x, final = 0x%x\n", reg, prev, mask, value, (prev & mask) | value );
#endif
}

#ifdef SNDCFG_DEBUG
void snd_sonicvibes_debug( sonicvibes_t *sonic )
{
  printk( "SV REGS:          INDEX = 0x%02x  ", inb( SV_REG( sonic, INDEX ) ) );
  printk( "                 STATUS = 0x%02x\n", inb( SV_REG( sonic, STATUS ) ) );
  printk( "  0x00: left input      = 0x%02x  ", snd_sonicvibes_in( sonic, 0x00 ) );
  printk( "  0x20: synth rate low  = 0x%02x\n", snd_sonicvibes_in( sonic, 0x20 ) );
  printk( "  0x01: right input     = 0x%02x  ", snd_sonicvibes_in( sonic, 0x01 ) );
  printk( "  0x21: synth rate high = 0x%02x\n", snd_sonicvibes_in( sonic, 0x21 ) );
  printk( "  0x02: left AUX1       = 0x%02x  ", snd_sonicvibes_in( sonic, 0x02 ) );
  printk( "  0x22: ADC clock       = 0x%02x\n", snd_sonicvibes_in( sonic, 0x22 ) );
  printk( "  0x03: right AUX1      = 0x%02x  ", snd_sonicvibes_in( sonic, 0x03 ) );
  printk( "  0x23: ADC alt rate    = 0x%02x\n", snd_sonicvibes_in( sonic, 0x23 ) );
  printk( "  0x04: left CD         = 0x%02x  ", snd_sonicvibes_in( sonic, 0x04 ) );
  printk( "  0x24: ADC pll M       = 0x%02x\n", snd_sonicvibes_in( sonic, 0x24 ) );
  printk( "  0x05: right CD        = 0x%02x  ", snd_sonicvibes_in( sonic, 0x05 ) );
  printk( "  0x25: ADC pll N       = 0x%02x\n", snd_sonicvibes_in( sonic, 0x25 ) );
  printk( "  0x06: left line       = 0x%02x  ", snd_sonicvibes_in( sonic, 0x06 ) );
  printk( "  0x26: Synth pll M     = 0x%02x\n", snd_sonicvibes_in( sonic, 0x26 ) );
  printk( "  0x07: right line      = 0x%02x  ", snd_sonicvibes_in( sonic, 0x07 ) );
  printk( "  0x27: Synth pll N     = 0x%02x\n", snd_sonicvibes_in( sonic, 0x27 ) );
  printk( "  0x08: MIC             = 0x%02x  ", snd_sonicvibes_in( sonic, 0x08 ) );
  printk( "  0x28: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x28 ) );
  printk( "  0x09: Game port       = 0x%02x  ", snd_sonicvibes_in( sonic, 0x09 ) );
  printk( "  0x29: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x29 ) );
  printk( "  0x0a: left synth      = 0x%02x  ", snd_sonicvibes_in( sonic, 0x0a ) );
  printk( "  0x2a: MPU401          = 0x%02x\n", snd_sonicvibes_in( sonic, 0x2a ) );
  printk( "  0x0b: right synth     = 0x%02x  ", snd_sonicvibes_in( sonic, 0x0b ) );
  printk( "  0x2b: drive ctrl      = 0x%02x\n", snd_sonicvibes_in( sonic, 0x2b ) );
  printk( "  0x0c: left AUX2       = 0x%02x  ", snd_sonicvibes_in( sonic, 0x0c ) );
  printk( "  0x2c: SRS space       = 0x%02x\n", snd_sonicvibes_in( sonic, 0x2c ) );
  printk( "  0x0d: right AUX2      = 0x%02x  ", snd_sonicvibes_in( sonic, 0x0d ) );
  printk( "  0x2d: SRS center      = 0x%02x\n", snd_sonicvibes_in( sonic, 0x2d ) );
  printk( "  0x0e: left analog     = 0x%02x  ", snd_sonicvibes_in( sonic, 0x0e ) );
  printk( "  0x2e: wave source     = 0x%02x\n", snd_sonicvibes_in( sonic, 0x2e ) );
  printk( "  0x0f: right analog    = 0x%02x  ", snd_sonicvibes_in( sonic, 0x0f ) );
  printk( "  0x2f: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x2f ) );
  printk( "  0x10: left PCM        = 0x%02x  ", snd_sonicvibes_in( sonic, 0x10 ) );
  printk( "  0x30: analog power    = 0x%02x\n", snd_sonicvibes_in( sonic, 0x30 ) );
  printk( "  0x11: right PCM       = 0x%02x  ", snd_sonicvibes_in( sonic, 0x11 ) );
  printk( "  0x31: analog power    = 0x%02x\n", snd_sonicvibes_in( sonic, 0x31 ) );
  printk( "  0x12: DMA data format = 0x%02x  ", snd_sonicvibes_in( sonic, 0x12 ) );
  printk( "  0x32: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x32 ) );
  printk( "  0x13: P/C enable      = 0x%02x  ", snd_sonicvibes_in( sonic, 0x13 ) );
  printk( "  0x33: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x33 ) );
  printk( "  0x14: U/D button      = 0x%02x  ", snd_sonicvibes_in( sonic, 0x14 ) );
  printk( "  0x34: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x34 ) );
  printk( "  0x15: revision        = 0x%02x  ", snd_sonicvibes_in( sonic, 0x15 ) );
  printk( "  0x35: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x35 ) );
  printk( "  0x16: ADC output ctrl = 0x%02x  ", snd_sonicvibes_in( sonic, 0x16 ) );
  printk( "  0x36: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x36 ) );
  printk( "  0x17: ---             = 0x%02x  ", snd_sonicvibes_in( sonic, 0x17 ) );
  printk( "  0x37: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x37 ) );
  printk( "  0x18: DMA A upper cnt = 0x%02x  ", snd_sonicvibes_in( sonic, 0x18 ) );
  printk( "  0x38: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x38 ) );
  printk( "  0x19: DMA A lower cnt = 0x%02x  ", snd_sonicvibes_in( sonic, 0x19 ) );
  printk( "  0x39: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x39 ) );
  printk( "  0x1a: ---             = 0x%02x  ", snd_sonicvibes_in( sonic, 0x1a ) );
  printk( "  0x3a: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x3a ) );
  printk( "  0x1b: ---             = 0x%02x  ", snd_sonicvibes_in( sonic, 0x1b ) );
  printk( "  0x3b: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x3b ) );
  printk( "  0x1c: DMA C upper cnt = 0x%02x  ", snd_sonicvibes_in( sonic, 0x1c ) );
  printk( "  0x3c: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x3c ) );
  printk( "  0x1d: DMA C upper cnt = 0x%02x  ", snd_sonicvibes_in( sonic, 0x1d ) );
  printk( "  0x3d: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x3d ) );
  printk( "  0x1e: PCM rate low    = 0x%02x  ", snd_sonicvibes_in( sonic, 0x1e ) );
  printk( "  0x3e: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x3e ) );
  printk( "  0x1f: PCM rate high   = 0x%02x  ", snd_sonicvibes_in( sonic, 0x1f ) );
  printk( "  0x3f: ---             = 0x%02x\n", snd_sonicvibes_in( sonic, 0x3f ) );
}
#endif

static void snd_sonicvibes_setfmt( sonicvibes_t *sonic, unsigned char mask, unsigned char value )
{
  unsigned long flags;

  snd_spin_lock( sonic, reg, &flags );
  outb( SV_MCE | SV_IREG_DMA_DATA_FMT, SV_REG( sonic, INDEX ) );
  if ( mask ) {
    sonic -> format = inb( SV_REG( sonic, DATA ) );
    snd_delay( 1 );
  }
  sonic -> format = (sonic -> format & mask) | value;
  outb( sonic -> format, SV_REG( sonic, DATA ) );
  snd_delay( 1 );
  outb( 0, SV_REG( sonic, INDEX ) );
  snd_delay( 1 );
  snd_spin_unlock( sonic, reg, &flags );
}

static unsigned int snd_sonicvibes_setpll( sonicvibes_t *sonic, unsigned char reg, unsigned int rate, int set )
{
  unsigned long flags;
  unsigned char r, m = 0, n = 0;
  unsigned int xm, xn, xr, xd, metric = ~0U;

  if (rate < 625000/SV_ADCMULT)
    rate = 625000/SV_ADCMULT;
  if (rate > 150000000/SV_ADCMULT)
    rate = 150000000/SV_ADCMULT;
  /* slight violation of specs, needed for continuous sampling rates */
  for (r = 0; rate < 75000000/SV_ADCMULT; r += 0x20, rate <<= 1);
  for (xn = 3; xn < 33; xn++)		/* 35 */
    for (xm = 3; xm < 257; xm++) {
      xr = ((SV_REFFREQUENCY/SV_ADCMULT) * xm) / xn;
      xd = abs((signed)(xr - rate));
      if (xd < metric) {
        metric = xd;
        m = xm - 2;
        n = xn - 2;
      }
    }
#if 0
  printk( "metric = %i, xm = %i, xn = %i\n", metric, xm, xn );
  printk( "pll set - reg = 0x%x, m = 0x%x, r = 0x%x, n = 0x%x\n", reg, m, r, n );
#endif
  if ( set ) {
    snd_spin_lock( sonic, reg, &flags );
    snd_sonicvibes_out1( sonic, reg, m );
    snd_sonicvibes_out1( sonic, reg + 1, r | n );
    snd_spin_unlock( sonic, reg, &flags );
  }
  return (SV_REFFREQUENCY/SV_ADCMULT * (m + 2) / (n + 2)) >> ((r >> 5) & 7);
}

static unsigned int snd_sonicvibes_set_dac_rate( sonicvibes_t *sonic, unsigned int rate, int set )
{
  unsigned div;
  unsigned long flags;

  if (rate > 48000)
    rate = 48000;
  if (rate < 4000)
    rate = 4000;
  div = (rate * 65536 + SV_FULLRATE/2) / SV_FULLRATE;
  if (div > 65535)
    div = 65535;
  if ( set ) {
    snd_spin_lock( sonic, reg, &flags );
    snd_sonicvibes_out1( sonic, SV_IREG_PCM_RATE_HIGH, div >> 8 );
    snd_sonicvibes_out1( sonic, SV_IREG_PCM_RATE_LOW, div );
    snd_spin_unlock( sonic, reg, &flags );
  }
  return (div * SV_FULLRATE + 32768) / 65536;
}

static unsigned int snd_sonicvibes_set_adc_rate( sonicvibes_t *sonic, unsigned int rate, int set )
{
  unsigned long flags;
  unsigned rate1, rate2, div;

  if (rate > 48000)
    rate = 48000;
  if (rate < 4000)
    rate = 4000;
  rate1 = snd_sonicvibes_setpll( sonic, SV_IREG_ADC_PLL, rate, set );
  div = (48000 + rate/2) / rate;
  if (div > 8)
    div = 8;
  rate2 = (48000 + div/2) / div;
  if ( set ) {
    snd_spin_lock( sonic, reg, &flags );
    snd_sonicvibes_out1( sonic, SV_IREG_ADC_ALT_RATE, (div-1) << 4 );
    if (abs((signed)(rate-rate2)) <= abs((signed)(rate-rate1))) {
      snd_sonicvibes_out1( sonic, SV_IREG_ADC_CLOCK, 0x10 );
      rate1 = rate2;
    } else {
      snd_sonicvibes_out1( sonic, SV_IREG_ADC_CLOCK, 0x00 );
    }
    snd_spin_unlock( sonic, reg, &flags );
  } else {
    if (abs((signed)(rate-rate2)) <= abs((signed)(rate-rate1)))
      rate1 = rate2;
  }
  return rate1;
}

static void snd_sonicvibes_trigger( sonicvibes_t *sonic, int what, int up )
{
  unsigned long flags;

  snd_spin_lock( sonic, reg, &flags );
  if ( up ) {
    if ( !(sonic -> enable & what) ) {
      sonic -> enable |= what;
      snd_sonicvibes_out1( sonic, SV_IREG_PC_ENABLE, sonic -> enable );
    }
  } else {
    if ( sonic -> enable & what ) {
      sonic -> enable &= ~what;
      snd_sonicvibes_out1( sonic, SV_IREG_PC_ENABLE, sonic -> enable );
    }
  }
  snd_spin_unlock( sonic, reg, &flags );
}

/*
 *  PCM part
 */

static void snd_sonicvibes_playback_compute_rate( snd_pcm1_t *pcm1 )
{
  sonicvibes_t *sonic = (sonicvibes_t *)pcm1 -> private_data;

  pcm1 -> playback.real_rate = snd_sonicvibes_set_dac_rate( sonic, pcm1 -> playback.rate, 0 );
}

static void snd_sonicvibes_record_compute_rate( snd_pcm1_t *pcm1 )
{
  sonicvibes_t *sonic = (sonicvibes_t *)pcm1 -> private_data;

  pcm1 -> record.real_rate = snd_sonicvibes_set_adc_rate( sonic, pcm1 -> record.rate, 0 );
}

static void snd_sonicvibes_playback_trigger( snd_pcm1_t *pcm1, int up )
{
  snd_sonicvibes_trigger( (sonicvibes_t *)pcm1 -> private_data, 1, up );
}

static void snd_sonicvibes_record_trigger( snd_pcm1_t *pcm1, int up )
{
  snd_sonicvibes_trigger( (sonicvibes_t *)pcm1 -> private_data, 2, up );
}

static void snd_sonicvibes_playback_prepare( snd_pcm1_t *pcm1,
					     unsigned char *buffer,
					     unsigned int size,
					     unsigned int offset,
					     unsigned int count )
{
  unsigned long flags;
  sonicvibes_t *sonic = (sonicvibes_t *)pcm1 -> private_data;
  unsigned char fmt = 0;

  count--;
  if ( pcm1 -> playback.voices > 1 ) fmt |= 1;
  if ( pcm1 -> playback.mode & SND_PCM1_MODE_16 ) fmt |= 2;
  snd_sonicvibes_setfmt( sonic, ~3, fmt );
  snd_sonicvibes_set_dac_rate( sonic, pcm1 -> playback.real_rate, 1 );
  snd_spin_lock( sonic, reg, &flags );
  snd_sonicvibes_setdmaa( sonic, virt_to_bus( buffer ), size );
  snd_sonicvibes_out1( sonic, SV_IREG_DMA_A_UPPER, count >> 8 );
  snd_sonicvibes_out1( sonic, SV_IREG_DMA_A_LOWER, count );
  snd_spin_unlock( sonic, reg, &flags );
}

static void snd_sonicvibes_record_prepare( snd_pcm1_t *pcm1,
                                           unsigned char *buffer,
                                           unsigned int size,
                                           unsigned int offset,
                                           unsigned int count )
{
  unsigned long flags;
  sonicvibes_t *sonic = (sonicvibes_t *)pcm1 -> private_data;
  unsigned char fmt = 0;

  count >>= 1;
  count--;
  if ( pcm1 -> record.voices > 1 ) fmt |= 0x10;
  if ( pcm1 -> record.mode & SND_PCM1_MODE_16 ) fmt |= 0x20;
  snd_sonicvibes_setfmt( sonic, ~0x30, fmt );
  snd_sonicvibes_set_adc_rate( sonic, pcm1 -> record.real_rate, 1 );
  snd_spin_lock( sonic, reg, &flags );
  snd_sonicvibes_setdmac( sonic, virt_to_bus( buffer ), size );
  snd_sonicvibes_out1( sonic, SV_IREG_DMA_C_UPPER, count >> 8 );
  snd_sonicvibes_out1( sonic, SV_IREG_DMA_C_LOWER, count );
  snd_spin_unlock( sonic, reg, &flags );
}

static int snd_sonicvibes_playback_open( snd_pcm1_t *pcm1 )
{
  sonicvibes_t *sonic = (sonicvibes_t *)pcm1 -> private_data;
  int err;

  if ( (err = snd_pcm1_dma_alloc( pcm1, SND_PCM1_PLAYBACK, sonic -> dma1num, "S3 SonicVibes (playback)" )) < 0 )
    return err;
  sonic -> mode |= SV_MODE_PLAY;
  return 0;
}

static int snd_sonicvibes_record_open( snd_pcm1_t *pcm1 )
{
  sonicvibes_t *sonic = (sonicvibes_t *)pcm1 -> private_data;
  int err;

  if ( (err = snd_pcm1_dma_alloc( pcm1, SND_PCM1_RECORD, sonic -> dma2num, "S3 SonicVibes (record)" )) < 0 )
    return err;
  sonic -> mode |= SV_MODE_RECORD;
  return 0;
}

static void snd_sonicvibes_playback_close( snd_pcm1_t *pcm1 )
{
  sonicvibes_t *sonic = (sonicvibes_t *)pcm1 -> private_data;

  snd_pcm1_dma_free( pcm1, SND_PCM1_PLAYBACK, sonic -> dma1num );
  sonic -> mode &= ~SV_MODE_PLAY;
}

static void snd_sonicvibes_record_close( snd_pcm1_t *pcm1 )
{
  sonicvibes_t *sonic = (sonicvibes_t *)pcm1 -> private_data;

  snd_pcm1_dma_free( pcm1, SND_PCM1_RECORD, sonic -> dma2num );
  sonic -> mode &= ~SV_MODE_RECORD;
}

static unsigned int snd_sonicvibes_playback_pointer( snd_pcm1_t *pcm1, unsigned int used_size )
{
  sonicvibes_t *sonic = (sonicvibes_t *)pcm1 -> private_data;

  if ( !(sonic -> enable & 1) ) return 0;
  return used_size - snd_sonicvibes_getdmaa( sonic );
}

static unsigned int snd_sonicvibes_record_pointer( snd_pcm1_t *pcm1, unsigned int used_size )
{
  sonicvibes_t *sonic = (sonicvibes_t *)pcm1 -> private_data;

  if ( !(sonic -> enable & 2) ) return 0;
  return used_size - snd_sonicvibes_getdmac( sonic );
}

static struct snd_stru_pcm1_hardware snd_sonicvibes_playback = {
  NULL,                         /* private data */
  NULL,                         /* private_free */
  SND_PCM1_HW_AUTODMA,          /* flags */
  SND_PCM_FMT_MU_LAW | SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE, /* formats */
  0,                            /* align value */
  6,                            /* minimal fragment */
  4000,                         /* min. rate */
  48000,                        /* max. rate */
  2,                            /* max. voices */
  snd_sonicvibes_playback_open,
  snd_sonicvibes_playback_close,
  snd_sonicvibes_playback_compute_rate,
  snd_sonicvibes_playback_prepare,
  snd_sonicvibes_playback_trigger,
  snd_sonicvibes_playback_pointer,
  snd_pcm1_playback_dma_ulaw,
  snd_pcm1_dma_move,
  snd_pcm1_playback_dma_neutral
};
                                      
static struct snd_stru_pcm1_hardware snd_sonicvibes_record = {
  NULL,                         /* private data */
  NULL,                         /* private_free */
  SND_PCM1_HW_AUTODMA,          /* flags */
  SND_PCM_FMT_MU_LAW | SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE, /* formats */
  0,                            /* align value */
  6,                            /* minimal fragment */
  4000,                         /* min. rate */
  48000,                        /* max. rate */
  2,                            /* max. voices */
  snd_sonicvibes_record_open,
  snd_sonicvibes_record_close,
  snd_sonicvibes_record_compute_rate,
  snd_sonicvibes_record_prepare,
  snd_sonicvibes_record_trigger,
  snd_sonicvibes_record_pointer,
  snd_pcm1_record_dma_ulaw,
  snd_pcm1_dma_move,
  NULL
};

static void snd_sonicvibes_pcm_free( void *private_data )
{
  sonicvibes_t *sonic = (sonicvibes_t *)private_data;
  sonic -> pcm = NULL;
}

snd_pcm_t *snd_sonicvibes_pcm( sonicvibes_t *sonic )
{
  snd_pcm_t *pcm;
  snd_pcm1_t *pcm1;

  pcm = snd_pcm1_new_device( sonic -> card, "s3_86c617" );
  if ( !pcm ) return NULL;
  pcm1 = (snd_pcm1_t *)pcm -> private_data;
  memcpy( &pcm1 -> playback.hw, &snd_sonicvibes_playback, sizeof( snd_sonicvibes_playback ) );
  memcpy( &pcm1 -> record.hw, &snd_sonicvibes_record, sizeof( snd_sonicvibes_record ) );
  pcm1 -> private_data = sonic;
  pcm1 -> private_free = snd_sonicvibes_pcm_free;
  pcm -> info_flags = SND_PCM_INFO_CODEC | SND_PCM_INFO_MMAP |
                      SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_RECORD | SND_PCM_INFO_DUPLEX;
  strcpy( pcm -> name, "S3 SonicVibes" );
  return sonic -> pcm = pcm;
}
                                      
/*
 *  Mixer part
 */

#define SV_MIXS (sizeof(snd_sonicvibes_mixs)/sizeof(struct snd_stru_mixer_channel_hw))
#define SV_PRIVATE( left, right, recsrc ) ((left<<24)|(right<<16)|recsrc)
#define SV_MINVERT 0x2000

static void snd_sonicvibes_recsrc( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int enable )
{
  unsigned long flags;
  sonicvibes_t *sonic;
  
  sonic = (sonicvibes_t *)mixer -> private_data;
  snd_spin_lock( sonic, reg, &flags );
  snd_sonicvibes_outm1( sonic, SV_IREG_LEFT_ADC, 0x1f, channel -> hw.private_value & 0xe0 );
  snd_sonicvibes_outm1( sonic, SV_IREG_RIGHT_ADC, 0x1f, channel -> hw.private_value & 0xe0 );
  snd_spin_unlock( sonic, reg, &flags );
#if 0
  printk( "recsrc: '%s': left = 0x%x, right = 0x%x\n", channel -> hw.name, snd_sonicvibes_in( sonic, SV_IREG_LEFT_ADC ), snd_sonicvibes_in( sonic, SV_IREG_RIGHT_ADC ) );
#endif
}

static void snd_sonicvibes_set_mute( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, unsigned int mute )
{
  unsigned long flags;
  sonicvibes_t *sonic;
  unsigned char regl, regr;
  
  regl = (unsigned char)(channel -> hw.private_value >> 24);
  regr = (unsigned char)(channel -> hw.private_value >> 16);
  sonic = (sonicvibes_t *)mixer -> private_data;
  snd_spin_lock( sonic, reg, &flags );
  snd_sonicvibes_outm1( sonic, regl, 0x7f, (mute & SND_MIX_MUTE_LEFT) ? 0x80 : 0 );
  if ( channel -> hw.stereo )
    snd_sonicvibes_outm1( sonic, regr, 0x7f, (mute & SND_MIX_MUTE_RIGHT) ? 0x80 : 0 );
  snd_spin_unlock( sonic, reg, &flags );
#if 0
  printk( "set_mute: '%s': 0x%x/0x%x: left = 0x%x, right = 0x%x\n", channel -> hw.name, regl, regr, snd_sonicvibes_in( sonic, regl ), channel -> hw.stereo ? snd_sonicvibes_in( sonic, regr ) : 0 );
#endif
}

static void snd_sonicvibes_volume_level( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int left, int right )
{
  unsigned long flags;
  sonicvibes_t *sonic;
  unsigned char regl, regr;
  
  regl = (unsigned char)(channel -> hw.private_value >> 24);
  regr = (unsigned char)(channel -> hw.private_value >> 16);
  sonic = (sonicvibes_t *)mixer -> private_data;
  if ( !(channel -> hw.private_value & SV_MINVERT) ) {
    left = (channel -> hw.max - left);
    right = (channel -> hw.max - right);
  }
  snd_spin_lock( sonic, reg, &flags );
  snd_sonicvibes_outm1( sonic, regl, ~channel -> hw.max, left );
  if ( channel -> hw.stereo )
    snd_sonicvibes_outm1( sonic, regr, ~channel -> hw.max, right );
  snd_spin_unlock( sonic, reg, &flags );
#if 0
  printk( "set_volume: '%s': left = 0x%x, right = 0x%x\n", channel -> hw.name, snd_sonicvibes_in( sonic, regl ), channel -> hw.stereo ? snd_sonicvibes_in( sonic, regr ) : 0 );
#endif
}

static struct snd_stru_mixer_channel_hw snd_sonicvibes_mixs[] = {
  {
    SND_MIXER_PRI_GAIN,         /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_GAIN,          /* device name */
    SND_MIXER_OSS_IMIX,         /* OSS device # */
    0, 1, 0, 0, 1,              /* mute/stereo/record/digital/input */
    0, 15,                      /* min, max value */
    0, 2250, 150,               /* min, max, step - dB */
    SV_PRIVATE( SV_IREG_LEFT_ADC, SV_IREG_RIGHT_ADC, SV_RECSRC_RESERVED ) | SV_MINVERT,
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,                       /* record source */
    NULL,                       /* set mute */
    snd_sonicvibes_volume_level,/* set volume level */
  },
  {
    SND_MIXER_PRI_AUXA,         /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_AUXA,          /* device name */
    SND_MIXER_OSS_LINE1,        /* OSS device # */
    1, 1, 1, 0, 1,              /* mute/stereo/record/digital/input */
    0, 31,                      /* min, max value */
    -3450, 1200, 150,           /* min, max, step - dB */
    SV_PRIVATE( SV_IREG_LEFT_AUX1, SV_IREG_RIGHT_AUX1, SV_RECSRC_AUX1 ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    snd_sonicvibes_recsrc,      /* record source */
    snd_sonicvibes_set_mute,    /* set mute */
    snd_sonicvibes_volume_level,/* set volume level */
  },
  {
    SND_MIXER_PRI_CD,           /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_CD,            /* device name */
    SND_MIXER_OSS_CD,           /* OSS device # */
    1, 1, 1, 0, 1,              /* mute/stereo/record/digital/input */
    0, 31,                      /* min, max value */
    -3450, 1200, 150,           /* min, max, step - dB */
    SV_PRIVATE( SV_IREG_LEFT_CD, SV_IREG_RIGHT_CD, SV_RECSRC_CD ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    snd_sonicvibes_recsrc,      /* record source */
    snd_sonicvibes_set_mute,    /* set mute */
    snd_sonicvibes_volume_level,/* set volume level */
  },
  {
    SND_MIXER_PRI_LINE,         /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_LINE,          /* device name */
    SND_MIXER_OSS_LINE,         /* OSS device # */
    1, 1, 1, 0, 1,              /* mute/stereo/record/digital/input */
    0, 31,                      /* min, max value */
    -3450, 1200, 150,           /* min, max, step - dB */
    SV_PRIVATE( SV_IREG_LEFT_LINE, SV_IREG_RIGHT_LINE, SV_RECSRC_LINE ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    snd_sonicvibes_recsrc,      /* record source */
    snd_sonicvibes_set_mute,    /* set mute */
    snd_sonicvibes_volume_level,/* set volume level */
  },
  {
    SND_MIXER_PRI_MIC,          /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_MIC,           /* device name */
    SND_MIXER_OSS_MIC,          /* OSS device # */
    1, 0, 1, 0, 1,              /* mute/stereo/record/digital/input */
    0, 15,                      /* min, max value */
    -1050, 1200, 150,           /* min, max, step - dB */
    SV_PRIVATE( SV_IREG_MIC, 0, SV_RECSRC_MIC ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    snd_sonicvibes_recsrc,      /* record source */
    snd_sonicvibes_set_mute,    /* set mute */
    snd_sonicvibes_volume_level,/* set volume level */
  },
  {
    SND_MIXER_PRI_SYNTHESIZER,  /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_SYNTHESIZER,   /* device name */
    SND_MIXER_OSS_SYNTH,        /* OSS device # */
    1, 1, 0, 0, 1,              /* mute/stereo/record/digital/input */
    0, 31,                      /* min, max value */
    -3450, 1200, 150,           /* min, max, step - dB */
    SV_PRIVATE( SV_IREG_LEFT_SYNTH, SV_IREG_RIGHT_SYNTH, 0x80 ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,                       /* record source */
    snd_sonicvibes_set_mute,    /* set mute */
    snd_sonicvibes_volume_level,/* set volume level */
  },
  {
    SND_MIXER_PRI_AUXB,         /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_AUXB,          /* device name */
    SND_MIXER_OSS_LINE2,        /* OSS device # */
    1, 1, 1, 0, 1,              /* mute/stereo/record/digital/input */
    0, 31,                      /* min, max value */
    -3450, 1200, 150,           /* min, max, step - dB */
    SV_PRIVATE( SV_IREG_LEFT_AUX2, SV_IREG_RIGHT_AUX2, SV_RECSRC_AUX2 ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    snd_sonicvibes_recsrc,      /* record source */
    snd_sonicvibes_set_mute,    /* set mute */
    snd_sonicvibes_volume_level,/* set volume level */
  },
  {
    SND_MIXER_PRI_MASTER,       /* priority */
    SND_MIXER_PRI_PARENT,       /* parent priority */
    SND_MIXER_ID_MASTER,        /* device name */
    SND_MIXER_OSS_VOLUME,       /* OSS device # */
    1, 1, 1, 0, 1,              /* mute/stereo/record/digital/input */
    0, 31,                      /* min, max value */
    -4650, 0, 150,              /* min, max, step - dB */
    SV_PRIVATE( SV_IREG_LEFT_ANALOG, SV_IREG_RIGHT_ANALOG, SV_RECSRC_OUT ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    snd_sonicvibes_recsrc,      /* record source */
    snd_sonicvibes_set_mute,    /* set mute */
    snd_sonicvibes_volume_level,/* set volume level */
  },
  {
    SND_MIXER_PRI_PCM,          /* priority */
    SND_MIXER_PRI_MASTER,       /* parent priority */
    SND_MIXER_ID_PCM,           /* device name */
    SND_MIXER_OSS_PCM,          /* OSS device # */
    1, 1, 1, 0, 1,              /* mute/stereo/record/digital/input */
    0, 63,                      /* min, max value */
    -9450, 1200, 150,           /* min, max, step - dB */
    SV_PRIVATE( SV_IREG_LEFT_PCM, SV_IREG_RIGHT_PCM, SV_RECSRC_DAC ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    snd_sonicvibes_recsrc,      /* record source */
    snd_sonicvibes_set_mute,    /* set mute */
    snd_sonicvibes_volume_level,/* set volume level */
  }
};

static void snd_sonicvibes_mixer_free( void *data )
{
  ((sonicvibes_t *)data) -> mixer = NULL;
}

snd_kmixer_t *snd_sonicvibes_mixer( sonicvibes_t *sonic )
{
  int idx;
  snd_kmixer_t *mixer;
  snd_kmixer_channel_t *channel;

  if ( !sonic ) return NULL;
  mixer = snd_mixer_new( sonic -> card, "s3_86c617" );
  if ( !mixer ) return NULL;
  strcpy( mixer -> name, "S3 SonicVibes" );
  for ( idx = 0; idx < SV_MIXS; idx++ ) {
    channel = snd_mixer_new_channel( mixer, &snd_sonicvibes_mixs[ idx ] );
    if ( !channel ) {
      snd_mixer_free( mixer );
      return NULL;
    }
  }
  mixer -> hw.caps = SND_MIXER_INFO_CAP_EXCL_RECORD;
  mixer -> private_data = sonic;
  mixer -> private_free = snd_sonicvibes_mixer_free;
  return sonic -> mixer = mixer;
}

/*
 *
 */
 
sonicvibes_t *snd_sonicvibes_create( snd_card_t *card,
                                     struct snd_pci_dev *pci,
                                     int dma1num,
				     int dma2num,
				     int irqnum,
                                     int reverb,
                                     int mge )
{
  sonicvibes_t *sonic;
  
  sonic = (sonicvibes_t *)snd_calloc( sizeof( sonicvibes_t ) );
  if ( !sonic ) return NULL;
  snd_spin_prepare( sonic, reg );
  sonic -> card = card;
  sonic -> pci = pci;
  sonic -> dma1num = dma1num;
  sonic -> dma2num = dma2num;
  sonic -> irqnum = irqnum;
  sonic -> sb_port = pci -> base_address[ 0 ] & ~PCI_BASE_ADDRESS_SPACE;
  sonic -> enh_port = pci -> base_address[ 1 ] & ~PCI_BASE_ADDRESS_SPACE;
  sonic -> synth_port = pci -> base_address[ 2 ] & ~PCI_BASE_ADDRESS_SPACE;
  sonic -> midi_port = pci -> base_address[ 3 ] & ~PCI_BASE_ADDRESS_SPACE;
  sonic -> game_port = pci -> base_address[ 4 ] & ~PCI_BASE_ADDRESS_SPACE;
  snd_pci_read_config_dword( pci, 0x40, &sonic -> dmaa_port );
  snd_pci_read_config_dword( pci, 0x48, &sonic -> dmac_port );
  sonic -> dmaa_port &= ~0x0f;
  sonic -> dmac_port &= ~0x0f;
  snd_pci_write_config_dword( pci, 0x40, sonic -> dmaa_port | 9 );	/* enable + enhanced */
  snd_pci_write_config_dword( pci, 0x48, sonic -> dmac_port | 9 );	/* enable */
  /* ok.. initialize S3 SonicVibes chip */
  outb( SV_RESET, SV_REG( sonic, CONTROL ) );	/* reset chip */
  snd_delay( 10 );
  outb( 0, SV_REG( sonic, CONTROL ) );		/* release reset */
  snd_delay( 10 );
  outb( SV_ENHANCED | SV_INTA | (reverb ? SV_REVERB : 0), SV_REG( sonic, CONTROL ) );
  inb( SV_REG( sonic, STATUS ) );		/* clear IRQs */
  snd_sonicvibes_out( sonic, SV_IREG_DRIVE_CTRL, 0 );	/* drive current 16mA */
  snd_sonicvibes_out( sonic, SV_IREG_PC_ENABLE, sonic -> enable = 0 ); /* disable playback & capture */
  outb( sonic -> irqmask = ~(SV_DMAA_MASK|SV_DMAC_MASK), SV_REG( sonic, IRQMASK ) );
  inb( SV_REG( sonic, STATUS ) );		/* clear IRQs */
  snd_sonicvibes_out( sonic, SV_IREG_ADC_CLOCK, 0 ); /* use PLL as clock source */
  snd_sonicvibes_out( sonic, SV_IREG_ANALOG_POWER, 0 ); /* power up analog parts */
  snd_sonicvibes_out( sonic, SV_IREG_DIGITAL_POWER, 0 ); /* power up digital parts */
  snd_sonicvibes_setpll( sonic, SV_IREG_ADC_PLL, 8000, 1 );
  snd_sonicvibes_out( sonic, SV_IREG_SRS_SPACE, 0x80 ); /* SRS off */
  snd_sonicvibes_out( sonic, SV_IREG_PCM_RATE_LOW, (8000 * 65536 / SV_FULLRATE) & 0xff );
  snd_sonicvibes_out( sonic, SV_IREG_PCM_RATE_HIGH, ((8000 * 65536 / SV_FULLRATE) >> 8) & 0xff );
  snd_sonicvibes_out( sonic, SV_IREG_ADC_OUTPUT_CTRL, 0 );
  snd_sonicvibes_out( sonic, SV_IREG_LEFT_ADC, mge ? 0x10 : 0 );
#if 0
  snd_sonicvibes_debug( sonic );
#endif
  sonic -> revision = snd_sonicvibes_in( sonic, SV_IREG_REVISION );
  return sonic;
}

void snd_sonicvibes_free( sonicvibes_t *sonic )
{
  snd_pci_write_config_dword( sonic -> pci, 0x40, sonic -> dmaa_port );
  snd_pci_write_config_dword( sonic -> pci, 0x48, sonic -> dmac_port );
  snd_free( sonic, sizeof( sonicvibes_t ) );
}

void snd_sonicvibes_interrupt( sonicvibes_t *sonic )
{
  snd_pcm1_t *pcm1;
  unsigned char status;
  
  status = inb( SV_REG( sonic, STATUS ) );
  if ( !(status & (SV_DMAA_IRQ|SV_DMAC_IRQ|SV_MIDI_IRQ)) ) return;
  if ( sonic -> pcm ) {
    if ( status & SV_DMAA_IRQ ) {
      pcm1 = (snd_pcm1_t *)sonic -> pcm -> private_data;
      if ( pcm1 && pcm1 -> playback.ack )
        pcm1 -> playback.ack( pcm1 );
    }
    if ( status & SV_DMAC_IRQ ) {
      pcm1 = (snd_pcm1_t *)sonic -> pcm -> private_data;
      if ( pcm1 && pcm1 -> record.ack )
        pcm1 -> record.ack( pcm1 );
    }
  }
}

/*
 *  INIT part
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_s3_86c617_export;
#endif 

int init_module( void )
{
#ifndef LINUX_2_1
  if ( register_symtab( &snd_symbol_table_s3_86c617_export ) < 0 ) 
    return -ENOMEM;
#endif
  return 0;
}

void cleanup_module( void )
{
}
