/*
 *   ALSA sequencer Queue handling
 *   Copyright (c) 1998 by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"

#include "seq_memory.h"
#include "seq_queue.h"
#include "seq_clientmgr.h"
#include "seq_fifo.h"
#include "seq_timer.h"
#include "seq_info.h"


/* root for the queues */
static queue_t queuetab[SND_SEQ_MAX_QUEUES];


/* FIXME: should add list of active queues (for optimisation of timer), sort of 'run queue' */



/* return pointer to queue structure for specified id */
queue_t *queueptr(int queueid)
{
	if (queueid < 0 || queueid >= SND_SEQ_MAX_QUEUES) {
		snd_printk("Seq: oops. Trying to get pointer to queue %d\n", queueid);
		return NULL;
	}
	return &queuetab[queueid];
}



/* setup queues */
void snd_seq_queue_init(void)
{
	int c;
	queue_t *q;

	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		q = queueptr(c);

		snd_seq_prioq_init(&q->tickq);
		snd_seq_prioq_init(&q->timeq);
		snd_seq_timer_init(&q->timer);
	}
}



/* -------------------------------------------------------- */

snd_spin_define_static(chkqueues_lock);


/* check queues and dispatch events */
void snd_seq_check_queues(void)
{
	static int blocked;
	unsigned long flags;
	int c;
	int dequeue = SND_SEQ_MAX_DEQUEUE;
	snd_seq_event_cell_t *cell;

	/* make this function non-reentrant */
	snd_spin_lock_static(chkqueues_lock, &flags);
	if (blocked) {
		snd_spin_unlock_static(chkqueues_lock, &flags);
		snd_printk("Seq: check_queues: other thread already active\n");
		return;		/* other thread is already checking queues */
	}
	blocked = 1;
	snd_spin_unlock_static(chkqueues_lock, &flags);


	/* NOTE: this is not efficient for large number of queues! */
	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		queue_t *q = queueptr(c);

		if (q == NULL)
			continue;

		/* Process tick queue... */

		/* limit the number of elements dequeued per pass to save the machine from lockups */
		while (dequeue > 0) {
			cell = snd_seq_prioq_cell_peek(&q->tickq);
			if (cell == NULL)
				break;
			if (snd_seq_compare_tick_time(&q->timer.cur_tick, &cell->event.time.tick)) {
				cell = snd_seq_prioq_cell_out(&q->tickq);
				if (cell != NULL)
					snd_seq_dispatch_event(cell);
				dequeue--;
			} else {
				/* event remains in the queue */
				break;
			}
		}


		/* Process time queue... */

		/* limit the number of elements dequeued per pass to save the machine from lockups */
		while (dequeue > 0) {
			cell = snd_seq_prioq_cell_peek(&q->timeq);
			if (cell == NULL)
				break;
			if (snd_seq_compare_real_time(&q->timer.cur_time, &cell->event.time.real)) {
				cell = snd_seq_prioq_cell_out(&q->timeq);
				if (cell != NULL)
					snd_seq_dispatch_event(cell);
				dequeue--;
			} else {
				/* event remains in the queue */
				break;
			}
		}
	}

	/* free lock */
	blocked = 0;
}



/* enqueue a event to singe queue */
static void snd_seq_enqueue_single_event(snd_seq_event_cell_t * cell)
{
	int dest;
	queue_t *q;

	if (cell == NULL) {
		snd_printk("oops: snd_seq_enqueue_single_event() called with NULL cell\n");
		return;
	}
	dest = cell->event.dest.queue;	/* dest queue */
	q = queueptr(dest);

	if (q == NULL) {
		snd_printk("oops: snd_seq_enqueue_single_event() called with NULL queue\n");
		return;
	}
	/* enqueue event in the real-time or midi queue */
	switch (cell->event.flags & SND_SEQ_TIME_STAMP_MASK) {
		case SND_SEQ_TIME_STAMP_TICK:
			snd_seq_prioq_cell_in(&q->tickq, cell);
			break;

		case SND_SEQ_TIME_STAMP_REAL:
			snd_seq_prioq_cell_in(&q->timeq, cell);
			break;
	}

	/* trigger dispatching */
	snd_seq_check_queues();

	return;
}



/* enqueue a event received from one the clients */
void snd_seq_enqueue_event(snd_seq_event_cell_t * cell)
{
	int dest;

	if (cell == NULL) {
		snd_printk("oops: snd_seq_enqueue_event() called with NULL cell\n");
		return;
	}
	dest = cell->event.dest.queue;	/* dest queue */

	/* handle relative time stamps, convert them into absolute */
	if ((cell->event.flags & SND_SEQ_TIME_MODE_MASK) == SND_SEQ_TIME_MODE_REL) {
		queue_t *q;

		q = queueptr(dest);
		if (q == NULL) {
			snd_printk("oops: snd_seq_enqueue_event() called with NULL queue\n");
			return;
		}
		switch (cell->event.flags & SND_SEQ_TIME_STAMP_MASK) {
			case SND_SEQ_TIME_STAMP_TICK:
				cell->event.time.tick += q->timer.cur_tick;
				cell->event.flags &= ~SND_SEQ_TIME_STAMP_MASK;
				cell->event.flags |= SND_SEQ_TIME_STAMP_TICK;
				break;

			case SND_SEQ_TIME_STAMP_REAL:
				snd_seq_inc_real_time(&cell->event.time.real, &q->timer.cur_time);
				cell->event.flags &= ~SND_SEQ_TIME_STAMP_MASK;
				cell->event.flags |= SND_SEQ_TIME_STAMP_REAL;
				break;
		}
	}
	if ((dest >= 0) && (dest < 255)) {
		snd_seq_enqueue_single_event(cell);
	}
	if (dest == 255) {
		snd_printk("Seq: broadcasts to all queues are not supported yet (dest = %d)\n", dest);

		/* release the original cell */
		snd_seq_cell_free(cell);
	}
}


/* access to queue's timer */
void snd_seq_queue_timer_stop(int queueid)
{
	queue_t *q;

	q = queueptr(queueid);
	if (q)
		snd_seq_timer_stop(&q->timer);
}

void snd_seq_queue_timer_start(int queueid)
{
	queue_t *q;

	q = queueptr(queueid);
	if (q)
		snd_seq_timer_start(&q->timer);
}

void snd_seq_queue_timer_continue(int queueid)
{
	queue_t *q;

	q = queueptr(queueid);
	if (q)
		snd_seq_timer_continue(&q->timer);
}

void snd_seq_queue_timer_set_tempo(int queueid, int tempo)
{
	queue_t *q;

	q = queueptr(queueid);
	if (q)
		snd_seq_timer_set_tempo(&q->timer, tempo);
}

void snd_seq_queue_timer_set_ppq(int queueid, int ppq)
{
	queue_t *q;

	q = queueptr(queueid);
	if (q)
		snd_seq_timer_set_ppq(&q->timer, ppq);
}



/* exported to seq_info.c */
void snd_seq_info_queues_read(snd_info_buffer_t * buffer, void *private_data)
{
	int c;
	int bpm;
	queue_t *q;
	timer_t *tmr;

	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		q = queueptr(c);
		if (q == NULL)
			continue;
		tmr = &q->timer;

		if (tmr->tempo)
			bpm = (int) ((60.0E6 / (double) tmr->tempo) + 0.5);
		else
			bpm = 0;

		snd_iprintf(buffer, "queue %d:\n", c);
		snd_iprintf(buffer, "owned by client    : %d [not implemented]\n", -1);
		snd_iprintf(buffer, "queued time events : %d\n", snd_seq_prioq_avail(&q->timeq));
		snd_iprintf(buffer, "queued tick events : %d\n", snd_seq_prioq_avail(&q->timeq));
		snd_iprintf(buffer, "timer state        : %s\n", tmr->running ? "Running" : "Stopped");
		snd_iprintf(buffer, "timer PPQ          : %d\n", tmr->ppq);
		snd_iprintf(buffer, "current tempo      : %d\n", tmr->tempo);
		snd_iprintf(buffer, "current BPM        : %d\n", bpm);
		snd_iprintf(buffer, "current time       : %d.%09d s\n", tmr->cur_time.sec, tmr->cur_time.nsec);
		snd_iprintf(buffer, "current tick       : %d\n", tmr->cur_tick);
		snd_iprintf(buffer, "\n");
	}
}
