/*
 *  ALSA sequencer Client Manager
 *  Copyright (c) 1998 by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"
#include "minors.h"

#include "seq.h"
#include "seq_clientmgr.h"
#include "seq_memory.h"
#include "seq_queue.h"
#include "seq_timer.h"
#include "seq_info.h"


/* Client Manager

 * this module handles the connections of userland and kernel clients
 * 
 */


snd_mutex_define_static(register);

static client_t clienttab[SND_SEQ_MAX_CLIENTS];

static usage_t client_usage = {0, 0};


/* return pointer to client structure for specified id */
static client_t *clientptr(int clientid)
{
	if (clientid < 0 || clientid >= SND_SEQ_MAX_CLIENTS) {
		snd_printk("Seq: oops. Trying to get pointer to client %d\n", clientid);
		return NULL;
	}
	return &clienttab[clientid];
}



static void usage_alloc(usage_t * res, int num)
{
	res->cur += num;
	if (res->cur > res->peak)
		res->peak = res->cur;
}

static void usage_free(usage_t * res, int num)
{
	res->cur -= num;
}



/* initialise data structures */
void init_data(void)
{
	int c;
	client_t *client;

	/* zap out the client table */
	for (c = 0; c < SND_SEQ_MAX_CLIENTS; c++) {
		client = clientptr(c);
		client->type = NO_CLIENT;
	}
}


static int seq_create_user_client(void)
{
	int c;
	client_t *client;

	/* find free slot in the client table */
	c = 0;
	while (c < SND_SEQ_MAX_CLIENTS) {
		client = clientptr(c);
		if (client->type == NO_CLIENT) {
			user_client_t *new_client = (user_client_t *) snd_malloc(sizeof(user_client_t));

			if (new_client == NULL)
				return -ENOMEM;

			/* init client data */

			snd_mutex_down_static(register);
			memset(new_client, 0, sizeof(user_client_t));
			new_client->index = c;

			snd_seq_fifo_init(&new_client->outqueue);
			snd_spin_prepare(new_client, input_sleep);

			new_client->file = NULL;
			strcpy(new_client->name, "");
			new_client->ports = NULL;

			client->type = USER_CLIENT;
			client->ptr.user = new_client;
			usage_alloc(&client_usage, 1);
			snd_mutex_up_static(register);

			return c;
		}
		c++;
	}
	return -ENODEV;		/* no free slot found, return failure code */
}



static void seq_free_user_client(int c)
{
	client_t *client = clientptr(c);

	snd_mutex_down_static(register);

	if (client->type == USER_CLIENT) {
		/* release data from outqueue */	/* FIXME: should also release external data */
		fifo_t *q = &client->ptr.user->outqueue;

		while (snd_seq_fifo_avail(q))
			snd_seq_cell_free(snd_seq_fifo_cell_out(q));

		snd_free(client->ptr.user, sizeof(user_client_t));
		client->type = NO_CLIENT;
		usage_free(&client_usage, 1);

	} else {
		snd_printk("Seq: Trying to free unused user client %d, type = %d\n", c, client->type);
	}
	snd_mutex_up_static(register);
}


static int seq_create_kernel_client(void)
{
	int c;
	client_t *client;

	/* find free slot in the client table */
	c = 0;
	while (c < SND_SEQ_MAX_CLIENTS) {
		client = clientptr(c);
		if (client->type == NO_CLIENT) {
			kernel_client_t *new_client = (kernel_client_t *) snd_malloc(sizeof(kernel_client_t));

			if (new_client == NULL)
				return -ENOMEM;

			/* init client data */
			snd_mutex_down_static(register);
			memset(new_client, 0, sizeof(kernel_client_t));

			snd_seq_fifo_init(&new_client->outqueue);
			strcpy(new_client->name, "");
			new_client->input = NULL;
			new_client->private_data = NULL;
			new_client->ports = NULL;

			client->type = KERNEL_CLIENT;
			client->ptr.kernel = new_client;
			usage_alloc(&client_usage, 1);

			snd_mutex_up_static(register);

			return c;
		}
		c++;
	}
	return -ENODEV;		/* no free slot found, return failure code */
}



static void seq_free_kernel_client(int c)
{
	client_t *client = clientptr(c);

	snd_mutex_down_static(register);

	if (client->type == KERNEL_CLIENT) {
		snd_free(client->ptr.kernel, sizeof(kernel_client_t));	/* FIXME: should also drain out queue & release external data */
		client->type = NO_CLIENT;
		usage_free(&client_usage, 1);
	} else {
		snd_printk("Seq: Trying to free unused kernel client %d, type = %d\n", c, client->type);
	}

	snd_mutex_up_static(register);
}


static void seq_free_client(int c)
{
	client_t *client = clientptr(c);

	switch (client->type) {
		case NO_CLIENT:
			snd_printk("Seq: Trying to free unused client %d\n", c);
			break;

		case USER_CLIENT:
			seq_free_user_client(c);
			break;

		case KERNEL_CLIENT:
			seq_free_kernel_client(c);
			break;

		default:
			snd_printk("Seq: Trying to free client %d with undefined type = %d\n", c, client->type);
	}
}



/* -------------------------------------------------------- */

static int snd_seq_open(unsigned short minor, int cardnum, int device, struct file *file)
{
	int c;			/* client id */
	client_t *client;

	c = seq_create_user_client();
	if (c < 0)
		return c;	/* failure code */

	/* store pointer to client */
	client = clientptr(c);
	(user_client_t *) file->private_data = client->ptr.user;

	/* fill client data */
	client->ptr.user->file = file;
	sprintf(client->ptr.user->name, "Client-%d", c);

	snd_printk("seq registering user-land client %d\n", c);

	MOD_INC_USE_COUNT;

	return 0;
}


static int snd_seq_release(unsigned short minor, int cardnum, int device, struct file *file)
{
	user_client_t *clientp = (user_client_t *) file->private_data;
	int c = clientp->index;

	MOD_DEC_USE_COUNT;

	snd_printk("seq releasing user-land client %d\n", c);

	clientp = NULL;
	seq_free_client(c);

	return 0;
}


/* handle client read() */
static long snd_seq_read(struct file *file, char *buf, long count)
{
	user_client_t *clientp = (user_client_t *) file->private_data;
	long result = 0;
	unsigned long flags;

	/*snd_printk( ">> snd_seq_read \n>> client=%d, file*=%p, buf*=%p, count=%ld\n",
	   clientp->index,file,buf,count); */

	if (verify_area(VERIFY_WRITE, buf, count))
		return -EFAULT;

	/* check client structures are in place */
	if (clientp == NULL)
		return -EIO;

	/* while we have room for at least one event */
	while (count >= sizeof(snd_seq_event_t)) {
		while (snd_seq_fifo_avail(&clientp->outqueue) == 0) {
			/* no data available in outqueue, block */
			if (file->f_flags & O_NONBLOCK)
				return result;
			snd_spin_lock(clientp, input_sleep, &flags);
			clientp->outqueue.flags |= SND_WK_SLEEP;
			snd_sleep(&clientp->outqueue, input, 10 * HZ);
			clientp->outqueue.flags &= ~SND_WK_SLEEP;
			snd_spin_unlock(clientp, input_sleep, &flags);
			if (snd_sleep_abort(&clientp->outqueue, input))
				return result;
		}
		while (snd_seq_fifo_avail(&clientp->outqueue)) {
			/* data available in queue */

			snd_seq_event_cell_t *cell;
               		int len;

			/* check if buffer is big enough for the new event */
			cell = snd_seq_fifo_cell_peek(&clientp->outqueue);
			if (cell == NULL) {
				snd_printk("Whoops, snd_seq_read got a NULL from the fifo\n");
				return result;
			}
			switch (cell->event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
				case SND_SEQ_EVENT_LENGTH_VARIABLE:
					/* handle variable length data */
					len = sizeof(snd_seq_event_t) + cell->event.data.ext.len;
					break;
				case SND_SEQ_EVENT_LENGTH_FIXED:
				default:
					/* normal event, no special action needed */
					len = sizeof(snd_seq_event_t);
			}
			if (count < len)
				return result;	/* not enough room for this new event */

			/* get event */
			cell = snd_seq_fifo_cell_out(&clientp->outqueue);
			if (cell == NULL) {
				snd_printk("Whoops, snd_seq_read got a NULL from the fifo\n");
				return result;
			}
			switch (cell->event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
				case SND_SEQ_EVENT_LENGTH_VARIABLE:
					/* handle variable length data */
					len = sizeof(snd_seq_event_t) + cell->event.data.ext.len;
					copy_to_user(buf, &cell->event, sizeof(snd_seq_event_t));
					if (cell->event.data.ext.ptr != NULL) {
						copy_to_user(buf+sizeof(snd_seq_event_t), cell->event.data.ext.ptr, cell->event.data.ext.len);
						snd_seq_ext_free(cell->event.data.ext.ptr,cell->event.data.ext.len);
					}
					break;
					
				case SND_SEQ_EVENT_LENGTH_FIXED:
				default:
					/* normal event, no special action needed */
					len = sizeof(snd_seq_event_t);
					copy_to_user(buf, &cell->event, sizeof(snd_seq_event_t));					
			}
			snd_seq_cell_free(cell);
			
			result += len;
			count -= len;
			buf += len;
		}
	}

	return result;
}


/* handle write() */
static long snd_seq_write(struct file *file, const char *buf, long count)
{
	user_client_t *clientp = (user_client_t *) file->private_data;
	snd_seq_event_t *ev = (snd_seq_event_t *) buf;
	int rd = 0;
	snd_seq_event_cell_t *cell;

	//snd_printk( ">> snd_seq_write \n>> client=%d, file*=%p, buf*=%p, count=%ld\n", clientp->index,file,buf,count);

	/* check if the data is accessible */
	if (verify_area(VERIFY_READ, buf, count))
		return -EFAULT;

	/* check client structures are in place */
	if (clientp == NULL)
		return -EIO;

	/* only process whole event */
	while (count >= sizeof(snd_seq_event_t)) {

		/* only start processing if we have enough free cells (>low-water) */
		if (snd_seq_unused_cells() < 100)	/* FIXME: should be configurable */
			break;

		/* we got one event, copy it into our own data structure and */
		/* feed to the prioQ */
		cell = snd_seq_event_dup_from_user(ev);
		if (cell) {
			cell->event.source.queue = 0;
			cell->event.source.client = clientp->index;	/* fill in client index */
			
			switch (cell->event.flags & SND_SEQ_EVENT_LENGTH_MASK) {
				case SND_SEQ_EVENT_LENGTH_VARIABLE: {
					/* handle variable length data */
					int msg_len = cell->event.data.ext.len;
					unsigned char *org_msg = (char *)ev + sizeof(snd_seq_event_t);
					unsigned char *new_msg;
					
					cell->event.data.ext.ptr =  new_msg = snd_seq_ext_malloc(msg_len);
					if (new_msg != NULL) {
						copy_from_user(new_msg, org_msg, msg_len);
					} else {
						snd_printk("failed to allocate %d bytes for variable length event\n",cell->event.data.ext.len);
					}
					
					count -= msg_len;
					ev += msg_len;
					rd += msg_len;
					}
					break;

				case SND_SEQ_EVENT_LENGTH_FIXED:				
				default:
					/* normal event, no special action needed */
			}
			
			snd_seq_enqueue_event(cell);
		} else {
			/* failure, let client know what we have read */
			return rd;
		}

		/* up to next event */
		count -= sizeof(snd_seq_event_t);
		ev += sizeof(snd_seq_event_t);
		rd += sizeof(snd_seq_event_t);
	}
	return rd;
}


#ifdef SND_POLL
static unsigned int snd_seq_poll(struct file *file, poll_table * wait)
{
	user_client_t *clientp = (user_client_t *) file->private_data;
	unsigned int mask = 0;
	unsigned long flags;

	/* check client structures are in place */
	if (clientp == NULL)
		return -EIO;

	/* should do this only if device is opened for read */
	{
		snd_spin_lock(clientp, input_sleep, &flags);
		clientp->outqueue.flags |= SND_WK_SLEEP;
		snd_poll_wait(file, &clientp->outqueue, input, wait);
		snd_spin_unlock(clientp, input_sleep, &flags);
	}

	/* check if data is available in the outqueue */
	if (snd_seq_fifo_avail(&clientp->outqueue) >= 1)
		mask |= POLLIN | POLLRDNORM;

	/* should have at least 100 free cells in our pool */
	/* FIXME: This low-water mark should be configurable */
	if (snd_seq_unused_cells() > 100)
		mask |= POLLOUT | POLLWRNORM;

	return mask;
}
#else
static int snd_seq_select(struct file *file, int sel_type, select_table * wait)
{
	user_client_t *clientp = (user_client_t *) file->private_data;
	unsigned int mask;
	unsigned long flags;

	switch (sel_type) {
    	 case SEL_IN:	/* check if data is available in the outqueue */

    	 	/* FIXME: should check for r/w mode */
              	snd_spin_lock( client_p, input_sleep, &flags );
                if ( !(snd_seq_fifo_avail(&clientp->outqueue) >= 1) ) {
                   	clientp->outqueue.flags |= SND_WK_SLEEP;
                     	snd_select_wait( &clientp->outqueue, input, wait );
                    	snd_spin_unlock( client_p, input_sleep, &flags );
                    	return 0;
                }
                clientp->outqueue.flags &= ~SND_WK_SLEEP;
                snd_spin_unlock( client_p, input_sleep, &flags );
                return 1;
                
	 case SEL_OUT:
		if (snd_seq_unused_cells() > 100)
			return 1;
		return 0;
		
	 case SEL_EX:
	 	break;
	}

	return 0;
}
#endif


/*-----------------------------------------------------*/


/* dispatch a single event to the client */
static void snd_seq_dispatch_single_event(snd_seq_event_cell_t * cell)
{
	client_t *dest_client;
	int dest = cell->event.dest.client;	/* dest client */

	dest_client = clientptr(dest);
	if (dest_client == NULL) {
		snd_printk("snd_seq_dispatch_single_event called with dest_client==NULL\n");
		return;
	}
	switch (dest_client->type) {
		case USER_CLIENT:
			//snd_printk("seq: event to user client %d\n",dest);

			/* write in client's outgoing fifo */
			snd_seq_fifo_cell_in(&dest_client->ptr.user->outqueue, cell);
			break;

		case KERNEL_CLIENT:
			//snd_printk("seq: event to kernel client %d\n",dest);

			if (dest_client->ptr.kernel->input != NULL) {
				int result;
				
				/* drain queue if old events are present */
				while (snd_seq_fifo_avail(&dest_client->ptr.kernel->outqueue)) {
					snd_seq_event_cell_t *queued_cell;
					queued_cell = snd_seq_fifo_cell_peek(&dest_client->ptr.kernel->outqueue);
					
					result = dest_client->ptr.kernel->input(&cell->event, dest_client->ptr.kernel->private_data);
					if (result) {
						/* event successfully processed, it can mow be removed */
						snd_seq_fifo_cell_out(&dest_client->ptr.kernel->outqueue);
						snd_seq_cell_free(cell);
					} else {
						snd_printk("seq: kernel client %d refuses to process queued event\n",dest);
						break;
					}	
				}

				/* directly pass event to client's routine */
				result = dest_client->ptr.kernel->input(&cell->event, dest_client->ptr.kernel->private_data);
				if (result) {
						/* event was successfully processed, it can mow be removed */
						snd_seq_cell_free(cell);
					} else {
						/* client refused the event (shouldn't happen!), queue it */
						snd_printk("seq: kernel client %d refuses to process event\n",dest);
						snd_seq_fifo_cell_in(&dest_client->ptr.kernel->outqueue, cell);
					}		
			} else {
				/* no input routine registered, get rid of the event */
				snd_printk("seq: kernel client %d has no input processor, event discarded\n",dest);
				snd_seq_cell_free(cell);
			}
			break;

		default:
			//snd_printk("seq: dispatch_single_event failed, no client with id=%d registered\n", dest);

			/* client not found, discard this event... */
			snd_seq_cell_free(cell);
			break;
	}
}


/*
 * dispatch event to the client 
 *
 * at this moment we can only send the event to either all clients 
 * (broadcast) or to a specific client.
 * in a later stadium multicast groups will be added 
 *
 */
void snd_seq_dispatch_event(snd_seq_event_cell_t * cell)
{
	client_t *dest_client;
	int dest;

	if (!cell)
		return;		/* something screwed up */

	dest = cell->event.dest.client;		/* dest client */

	//snd_printk( "Seq: dispatch event to %d\n",dest);


	/* send to specific client */
	if ((dest >= 0) && (dest <= 191)) {
		snd_seq_dispatch_single_event(cell);
		return;
	}
	/* send to a multicast group */
	if ((dest > 191) && (dest < 255)) {
		snd_printk("Seq: multicast groups not supported yet (dest = %d)\n", dest);
		/* Idea: multicast groups: 
		   a multicast message is send to a list of destinations:
		   per list item map: client, port, channel
		 */

		/* release the original cell */
		snd_seq_cell_free(cell);
		return;
	}
	/* handle broadcasts */
	if (dest == 255) {
		/* send the event to all clients */
		for (dest = 0; dest < SND_SEQ_MAX_CLIENTS; dest++) {
			dest_client = clientptr(dest);
			if (dest_client->type != NO_CLIENT) {
				snd_seq_event_cell_t *new_cell;

				/* send duplicates to all the clients */
				new_cell = snd_seq_cell_dup(cell);
				if (new_cell) {
					new_cell->event.dest.client = dest;
					snd_seq_dispatch_single_event(new_cell);
				}
			}
		}
		/* release the original cell */
		snd_seq_cell_free(cell);
		return;
	}
}


/*-----------------------------------------------------*/



/* fill client_info structure with info on the requested client (passed in client field) */
static int snd_seq_client_get_info(snd_seq_client_info_t *client_info)
{
	int c;
	client_t *client;
			
	/* requested client number */
	c = client_info->client;

	if (c <= 0 || c > SND_SEQ_MAX_CLIENTS)
		return -EINVAL;	/* out of range */

	client = clientptr(c);

	/* fill the info fields */	
	client_info->type = client->type;
	switch (client_info->type) {
		case USER_CLIENT:
			strcpy(client_info->name, client->ptr.user->name);
			break;

		case KERNEL_CLIENT:
			strcpy(client_info->name, client->ptr.kernel->name);
			break;

		default:
			strcpy(client_info->name, "???");
	}
	return 0;
}


/* CLIENT_INFO ioctl() */
static int snd_seq_ioctl_get_client_info(int c, snd_seq_client_info_t * _client_info)
{
	snd_seq_client_info_t client_info;
	client_t *client;
	int result;

	if (c < 0 || c >= SND_SEQ_MAX_CLIENTS)
		return -EINVAL;	/* out of range */

	client = clientptr(c);
	switch (client->type) {
		case USER_CLIENT:
			if (verify_area(VERIFY_READ, _client_info, sizeof(client_info)))
				return -EFAULT;
			copy_from_user(&client_info, _client_info, sizeof(client_info));
			
			result = snd_seq_client_get_info(&client_info);
			if (result < 0)
				return result;				
				
			if (verify_area(VERIFY_WRITE, _client_info, sizeof(client_info)))
				return -EFAULT;				
			copy_to_user(_client_info, &client_info, sizeof(client_info));
			break;

		case KERNEL_CLIENT:
			result = snd_seq_client_get_info(&client_info);
			if (result < 0)
				return result;
			break;				

		default:
			return -ENXIO;
	}

	return 0;
}

/* SET_NAME ioctl() */
static int snd_seq_ioctl_set_client_name(int c, char *name)
{
	client_t *client;

	if (c < 0 || c >= SND_SEQ_MAX_CLIENTS)
		return -EINVAL;	/* out of range */

	client = clientptr(c);
	switch (client->type) {
		case USER_CLIENT:
			if (verify_area(VERIFY_READ, name, SND_SEQ_MAX_CLIENT_NAME))
				return -EFAULT;
			copy_from_user(client->ptr.user->name, name, SND_SEQ_MAX_CLIENT_NAME);
			return 0;

		case KERNEL_CLIENT:
			strncpy(client->ptr.kernel->name, name, SND_SEQ_MAX_CLIENT_NAME);
			return 0;

		default:
			return -ENXIO;
	}

	return -ENXIO;
}


/* CREATE_PORT ioctl() */
static int snd_seq_ioctl_create_port(int c, snd_seq_port_info_t * info)
{
	client_t *client;

	if (c < 0 || c >= SND_SEQ_MAX_CLIENTS)
		return -EINVAL;	/* out of range */

	client = clientptr(c);
	switch (client->type) {
		case USER_CLIENT:
			{
				int p;
				client_port_t *port;
				snd_seq_port_info_t _info;

				p = snd_seq_create_port(&client->ptr.user->ports);
				snd_printk("seq registered port %d (user)\n", p);

				if ((info != NULL) && (p >= 0)) {
					/* set passed parameters */
					if (verify_area(VERIFY_READ, info, sizeof(snd_seq_port_info_t)))
						return -EFAULT;
					copy_from_user(&_info, info, sizeof(snd_seq_port_info_t));

					_info.client = c;
					_info.port = p;
					port = snd_seq_get_port_ptr(&client->ptr.user->ports, _info.port);
					if (port != NULL)
						snd_seq_set_port_info(port, &_info);

					if (verify_area(VERIFY_WRITE, info, sizeof(snd_seq_port_info_t)))
						return -EFAULT;
					copy_to_user(info, &_info, sizeof(snd_seq_port_info_t));
				}
				return p;
			}

		case KERNEL_CLIENT:
			{
				int p;
				client_port_t *port;

				p = snd_seq_create_port(&client->ptr.kernel->ports);
				snd_printk("seq registered port %d (kernel)\n", p);
				if ((info != NULL) && (p >= 0)) {
					/* set passed parameters */
					info->client = c;
					info->port = p;
					port = snd_seq_get_port_ptr(&client->ptr.kernel->ports, info->port);
					if (port != NULL)
						snd_seq_set_port_info(port, info);
				}
				return p;
			}

		default:
			return -ENXIO;
	}
	return -ENXIO;
}




/* GET_PORT_INFO ioctl() (on any client) */
static int snd_seq_ioctl_get_port_info(int c, snd_seq_port_info_t * info)
{
	client_t *client;

	if (c < 0 || c >= SND_SEQ_MAX_CLIENTS)
		return -EINVAL;	/* out of range */

	client = clientptr(c);
	switch (client->type) {
		case USER_CLIENT:
			{
				client_port_t *port;
				int req_cl;
				int req_port;
				snd_seq_port_info_t _info;

				if (verify_area(VERIFY_READ, info, sizeof(snd_seq_port_info_t)))
					return -EFAULT;
				copy_from_user(&_info, info, sizeof(snd_seq_port_info_t));

				req_cl = _info.client;
				req_port = _info.port;

				/* lookup requested client */
				if (req_cl < 0 || req_cl >= SND_SEQ_MAX_CLIENTS)
					return -EINVAL;		/* out of range */
				client = clientptr(c);
				if (client == NULL)
					return -ENXIO;

				switch (client->type) {
					case USER_CLIENT:
						port = snd_seq_get_port_ptr(&client->ptr.user->ports, req_port);
						break;
					case KERNEL_CLIENT:
						port = snd_seq_get_port_ptr(&client->ptr.kernel->ports, req_port);
						break;
					default:
						return -ENXIO;
				}
				if (port == NULL)
					return -ENXIO;

				/* get port info */
				snd_seq_get_port_info(port, &_info);

				if (verify_area(VERIFY_WRITE, info, sizeof(snd_seq_port_info_t)))
					return -EFAULT;
				copy_to_user(info, &_info, sizeof(snd_seq_port_info_t));
				return 0;
			}

		case KERNEL_CLIENT:
			{
				client_port_t *port;
				int req_cl;
				int req_port;

				req_cl = info->client;
				req_port = info->port;

				/* lookup requested client */
				if (req_cl < 0 || req_cl >= SND_SEQ_MAX_CLIENTS)
					return -EINVAL;		/* out of range */
				client = clientptr(c);
				if (client == NULL)
					return -ENXIO;

				switch (client->type) {
					case USER_CLIENT:
						port = snd_seq_get_port_ptr(&client->ptr.user->ports, req_port);
						break;
					case KERNEL_CLIENT:
						port = snd_seq_get_port_ptr(&client->ptr.kernel->ports, req_port);
						break;
					default:
						return -ENXIO;
				}
				if (port == NULL)
					return -ENXIO;

				/* get port info */
				snd_seq_get_port_info(port, info);
				return 0;
			}


		default:
			return -ENXIO;
	}
	return -ENXIO;
}




/* SET_PORT_INFO ioctl() (only ports on this/own client) */
static int snd_seq_ioctl_set_port_info(int c, snd_seq_port_info_t * info)
{
	client_t *client;

	if (c < 0 || c >= SND_SEQ_MAX_CLIENTS)
		return -EINVAL;	/* out of range */

	client = clientptr(c);
	switch (client->type) {
		case USER_CLIENT:
			{
				client_port_t *port;
				snd_seq_port_info_t _info;

				if (verify_area(VERIFY_READ, info, sizeof(snd_seq_port_info_t)))
					return -EFAULT;
				copy_from_user(&_info, info, sizeof(snd_seq_port_info_t));

				_info.client = c;	/* only set our own ports ! */
				port = snd_seq_get_port_ptr(&client->ptr.user->ports, _info.port);
				if (port != NULL)
					snd_seq_set_port_info(port, &_info);

				if (verify_area(VERIFY_WRITE, info, sizeof(snd_seq_port_info_t)))
					return -EFAULT;
				copy_to_user(info, &_info, sizeof(snd_seq_port_info_t));
				return 0;
			}

		case KERNEL_CLIENT:
			{
				client_port_t *port;

				port = snd_seq_get_port_ptr(&client->ptr.kernel->ports, info->port);
				if (port == NULL)
					return -ENXIO;

				/* set port info */
				info->client = c;	/* only set our own ports ! */
				snd_seq_set_port_info(port, info);
				return 0;
			}

		default:
			return -ENXIO;
	}
	return -ENXIO;
}

/* fill queue_info structure */
static int snd_seq_get_queue_info(snd_seq_queue_info_t * info)
{
	timer_t *tmr = &queueptr(info->queue)->timer;

	if (tmr == NULL)
		return -ENXIO;
		
	info->tick = tmr->cur_tick;
	info->time.nsec = tmr->cur_time.nsec;
	info->time.sec = tmr->cur_time.sec;

	info->running = tmr->running;
	info->tempo = tmr->tempo;
	info->ppq = tmr->ppq;
	info->flags = 0;	/* not yet used... */
	
	return 0;	/* goes always ok... */
}


/* set timer information. parameters that are -1 left unchanged */
/* FIXME: only for owner client of this timer?? 
	--> depends how security for this queue is set  
	- timer is standard accessible for all clients, until one client
	  has 'locked' the timer, from then only that client can change
	  timer fields. by setting the owner field (back) to 255 the timer
	  becomes unlocked
*/
static int snd_seq_set_queue_info(snd_seq_queue_info_t * info)
{
	if (info->tempo >= 0)
		snd_seq_queue_timer_set_tempo(info->queue, info->tempo);
	
	if (info->ppq >= 0)
		snd_seq_queue_timer_set_ppq(info->queue, info->ppq);

	/* return updated queue_info struct */	
	return snd_seq_get_queue_info(info);
}

/* SET_QUEUE_INFO ioctl() */
static int snd_seq_ioctl_set_queue_info(int c, snd_seq_queue_info_t * info)
{
	client_t *client;

	if (c < 0 || c >= SND_SEQ_MAX_CLIENTS)
		return -EINVAL;	/* out of range */

	client = clientptr(c);
	switch (client->type) {
		case USER_CLIENT:
			{
				snd_seq_queue_info_t _info;
				int result;

				if (verify_area(VERIFY_READ, info, sizeof(snd_seq_queue_info_t)))
					return -EFAULT;
				copy_from_user(&_info, info, sizeof(snd_seq_queue_info_t));

				result = snd_seq_set_queue_info(&_info);
				if (result < 0)
					return result;

				if (verify_area(VERIFY_WRITE, info, sizeof(snd_seq_queue_info_t)))
					return -EFAULT;
				copy_to_user(info, &_info, sizeof(snd_seq_queue_info_t));
				return 0;
			}

		case KERNEL_CLIENT:
				return snd_seq_set_queue_info(info);

		default:
			return -ENXIO;
	}
	return -ENXIO;
}


/* GET_QUEUE_INFO ioctl() */
static int snd_seq_ioctl_get_queue_info(int c, snd_seq_queue_info_t * info)
{
	client_t *client;

	if (c < 0 || c >= SND_SEQ_MAX_CLIENTS)
		return -EINVAL;	/* out of range */

	client = clientptr(c);
	switch (client->type) {
		case USER_CLIENT:
			{
				snd_seq_queue_info_t _info;
				int result;

				if (verify_area(VERIFY_READ, info, sizeof(snd_seq_queue_info_t)))
					return -EFAULT;
				copy_from_user(&_info, info, sizeof(snd_seq_queue_info_t));

				result = snd_seq_get_queue_info(&_info);
				if (result < 0)
					return result;

				if (verify_area(VERIFY_WRITE, info, sizeof(snd_seq_queue_info_t)))
					return -EFAULT;
				copy_to_user(info, &_info, sizeof(snd_seq_queue_info_t));
				return 0;
			}

		case KERNEL_CLIENT:
				return snd_seq_get_queue_info(info);

		default:
			return -ENXIO;
	}
	return -ENXIO;
}





static int snd_seq_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	user_client_t *clientp = (user_client_t *) file->private_data;
	int client = clientp->index;

	/*snd_printk(">> seq ioctl - cmd = 0x%x\n", cmd);*/

	switch (cmd) {

		case SND_SEQ_IOCTL_VERSION:
			/* return sequencer version number */
			return snd_ioctl_out((long *) arg, SND_SEQ_VERSION);

		case SND_SEQ_IOCTL_CLIENT_ID:
			/* return the id of this client */
			return snd_ioctl_out((long *) arg, clientp->index);

		case SND_SEQ_IOCTL_SET_CLIENT_NAME:
			/* set name for this client */
			return snd_seq_ioctl_set_client_name(client, (char *) arg);

		case SND_SEQ_IOCTL_GET_CLIENT_INFO:
			/* return info on specified client */
			return snd_seq_ioctl_get_client_info(client, (snd_seq_client_info_t *) arg);

		case SND_SEQ_IOCTL_CREATE_PORT:
			/* create a port for this client */
			return snd_seq_ioctl_create_port(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_GET_PORT_INFO:
			/* get info for specified port */
			return snd_seq_ioctl_get_port_info(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_SET_PORT_INFO:
			/* set info for specified port */
			return snd_seq_ioctl_set_port_info(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_INFO:
			/* get info for specified queue */
			return snd_seq_ioctl_get_queue_info(client, (snd_seq_queue_info_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_INFO:
			/* set info for specified queue */
			return snd_seq_ioctl_set_queue_info(client, (snd_seq_queue_info_t *) arg);

		default:
			snd_printk("seq unknown ioctl() 0x%x (type='%c', number=0x%2x)\n",
				   cmd, _IOC_TYPE(cmd), _IOC_NR(cmd));
	}
	return -ENXIO;
}



/* -------------------------------------------------------- */


/* exported to kernel modules */
int snd_seq_register_kernel_client(snd_seq_client_callback_t * callback, void *private_data)
{
	int c;			/* client id */
	client_t *client;

	c = seq_create_kernel_client();
	if (c < 0)
		return c;	/* failure code */

	/* fill client data */
	client = clientptr(c);
	client->ptr.kernel->input = callback->input;
	client->ptr.kernel->private_data = private_data;
	sprintf(client->ptr.kernel->name, "Client-%d", c);

	snd_printk("seq registering kernel client %d\n", c);

	/* return client number to caller */
	return c;
}

/* exported to kernel modules */
int snd_seq_unregister_kernel_client(int client)
{
	snd_printk("seq releasing kernel client %d\n", client);

	seq_free_client(client);
	return 0;
}


/* exported, called by kernel clients to enqueue events */
int snd_seq_kernel_client_enqueue(int client, snd_seq_event_t * ev)
{
	snd_seq_event_cell_t *cell;

	/* we got one event, copy it into our own data structure and */
	/* feed it to the prioQ */
	cell = snd_seq_event_dup(ev);
	if (cell) {
		cell->event.source.queue = 0;
		cell->event.source.client = client;	/* fill in client index */
		snd_seq_enqueue_event(cell);
		return 1;	/* success */
	} else {
		return -1;
	}
}


/* 
 * exported, called by kernel clients to dispatch events directly to other
 * clients, bypassing the queues. No processing of time stamps is done.
 *
 */
int snd_seq_kernel_client_dispatch(int client, snd_seq_event_t * ev)
{
	snd_seq_event_cell_t *cell;

	/* we got one event, copy it into our own data structure and */
	/* send it directly to the client */
	cell = snd_seq_event_dup(ev);
	if (cell) {
		snd_seq_dispatch_event(cell);
		return 1;	/* success */
	} else {
		return -1;
	}
}


/* exported, called by kernel clients to perform same functions as with
 * userland ioctl() 
 *
 */
int snd_seq_kernel_client_ctl(int client, unsigned int cmd, void *arg)
{
	snd_printk(">> seq kernel client ctl - cmd = 0x%x\n", cmd);

	switch (cmd) {
		case SND_SEQ_IOCTL_SET_CLIENT_NAME:
			/* set name for this client */
			return snd_seq_ioctl_set_client_name(client, (char *) arg);

		case SND_SEQ_IOCTL_CREATE_PORT:
			/* create a port for this client */
			return snd_seq_ioctl_create_port(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_GET_PORT_INFO:
			/* get info for specified port */
			return snd_seq_ioctl_get_port_info(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_SET_PORT_INFO:
			/* set info for specified port */
			return snd_seq_ioctl_set_port_info(client, (snd_seq_port_info_t *) arg);

		case SND_SEQ_IOCTL_GET_QUEUE_INFO:
			/* get info for specified queue */
			return snd_seq_ioctl_get_queue_info(client, (snd_seq_queue_info_t *) arg);

		case SND_SEQ_IOCTL_SET_QUEUE_INFO:
			/* set info for specified queue */
			return snd_seq_ioctl_set_queue_info(client, (snd_seq_queue_info_t *) arg);

		default:
			snd_printk("seq unknown ctl() 0x%x (type='%c', number=0x%2x)\n",
				   cmd, _IOC_TYPE(cmd), _IOC_NR(cmd));
	}
	return -ENXIO;
}


/*---------------------------------------------------------------------------*/

/*
 *  /proc interface
 */


static void snd_seq_info_dump_ports(snd_info_buffer_t * buffer, client_port_t * ports)
{
	client_port_t *p = ports;

	while (p) {
#if SHOW_CAPABILITY
		snd_iprintf(buffer, "  Port %3d : [%s] (%s,%s,%s,%s)\n", 
			p->port, p->name, 
			(p->capability & SND_SEQ_PORT_CAP_MIDI_IN  ? "MIDI_in"  : ""),
			(p->capability & SND_SEQ_PORT_CAP_MIDI_OUT ? "MIDI_out" : ""),
			(p->capability & SND_SEQ_PORT_CAP_SYNC_IN  ? "Sync_in"  : ""),
			(p->capability & SND_SEQ_PORT_CAP_SYNC_OUT ? "Sync_out" : ""));
#else
		snd_iprintf(buffer, "  Port %3d : [%s]\n", 
			p->port, p->name);
#endif
		p = p->next;
	}
}


/* exported to seq_info.c */
void snd_seq_info_clients_read(snd_info_buffer_t * buffer, void *private_data)
{
	int c;
	client_t *client;

	snd_iprintf(buffer, "Client info.\n");
	snd_iprintf(buffer, "  cur  clients : %d\n", client_usage.cur);
	snd_iprintf(buffer, "  peak clients : %d\n", client_usage.peak);
	snd_iprintf(buffer, "  max  clients : %d\n", SND_SEQ_MAX_CLIENTS);
	snd_iprintf(buffer,"\n");

	/* list the client table */
	for (c = 0; c < SND_SEQ_MAX_CLIENTS; c++) {
		client = clientptr(c);
		switch (client->type) {
			case NO_CLIENT:
				/*snd_iprintf(buffer, "Client %3d : Free\n", c);*/
				break;

			case USER_CLIENT:
				snd_iprintf(buffer, "Client %3d : User   [%s]\n", c, client->ptr.user->name);
				snd_seq_info_dump_ports(buffer, client->ptr.user->ports);
				break;

			case KERNEL_CLIENT:
				snd_iprintf(buffer, "Client %3d : Kernel [%s]\n", c, client->ptr.kernel->name);
				snd_seq_info_dump_ports(buffer, client->ptr.kernel->ports);
				break;
		}
	}
}

/*---------------------------------------------------------------------------*/


/*
 *  REGISTRATION PART
 */

static snd_minor_t snd_seq_reg =
{
	"sequencer",
	NULL,			/* unregister */
	NULL,			/* lseek */
	snd_seq_read,		/* read */
	snd_seq_write,		/* write */
	snd_seq_open,		/* open */
	snd_seq_release,	/* release */
#ifdef SND_POLL
	snd_seq_poll,		/* poll */
#else
	snd_seq_select,		/* select */
#endif
	snd_seq_ioctl,		/* ioctl */
	NULL			/* mmap */
};


/* register sequencer device */
int snd_sequencer_device_init(void)
{
	int err;

	snd_mutex_down_static(register);

	if ((err = snd_register_minor(SND_MINOR_SEQUENCER, &snd_seq_reg)) < 0) {
		snd_mutex_up_static(register);

		return err;
	}
	snd_mutex_up_static(register);
	return 0;
}


  /* unregister sequencer device */
void snd_sequencer_device_done(void)
{
	snd_unregister_minor(SND_MINOR_SEQUENCER);
}
