/*
 *  Synthesizer abstract layer
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "synth.h"
#include "info.h"

snd_synth_t *snd_synth_devices[ SND_SYNTH_DEVICES ];

snd_mutex_define_static( register );
snd_mutex_define_static( open );

/*
 *
 */

/*
 *
 */

snd_synth_t *snd_synth_new_device( snd_card_t *card, char *id )
{
  snd_synth_t *synth;

  synth = (snd_synth_t *)snd_malloc( sizeof( snd_synth_t ) );
  if ( !synth ) return NULL;
  memset( synth, 0, sizeof( snd_synth_t ) );
  synth -> card = card;
  if ( id ) {
    strncpy( synth -> id, id, sizeof( synth -> id ) - 1 );
  }
  return synth;
}

int snd_synth_free( snd_synth_t *synth )
{
  if ( !synth ) return -EINVAL;
  if ( synth -> private_free )
    synth -> private_free( synth -> private_data );
  snd_free( synth, sizeof( snd_synth_t ) );
  return 0;  
}

int snd_synth_register( snd_synth_t *synth )
{
  int idx;

  if ( !synth ) return -EINVAL;
  snd_mutex_down_static( register );
  for ( idx = 0; idx < SND_SYNTH_DEVICES; idx++ ) {
    if ( snd_synth_devices[ idx ] ) continue;
    snd_synth_devices[ idx ] = synth;
    break;
  }
  if ( idx >= SND_SYNTH_DEVICES ) {
    snd_mutex_up_static( register );
    return -ENOMEM;
  }
  if ( idx < SND_CARDS )
    snd_oss_info_register( SND_OSS_INFO_DEV_SYNTH, idx, synth -> name );  
  snd_mutex_up_static( register );
  return 0;
}

int snd_synth_unregister( snd_synth_t *synth )
{
  int idx;

  if ( !synth ) return -EINVAL;
  snd_mutex_down_static( register );
  for ( idx = 0; idx < SND_SYNTH_DEVICES; idx++ ) {
    if ( snd_synth_devices[ idx ] == synth ) {
      snd_synth_devices[ idx ] = NULL;
      if ( idx < SND_CARDS )
        snd_oss_info_unregister( SND_OSS_INFO_DEV_SYNTH, idx );  
      snd_mutex_up_static( register );
      return snd_synth_free( synth );
    }
  }
  snd_mutex_up_static( register );
  return -EINVAL;
}

/*
 *  Info interface
 */

static void snd_synth_proc_read( snd_info_buffer_t *buffer, void *private_data )
{
  int idx;
  snd_synth_t *synth;

  snd_mutex_down_static( register );
  snd_mutex_down_static( open );
  for ( idx = 0; idx < SND_SYNTH_DEVICES; idx++ ) {
    synth = snd_synth_devices[ idx ];
    if ( synth == NULL ) continue;
    snd_iprintf( buffer, "%02i-", idx );
    if ( synth -> card )
      snd_iprintf( buffer, "%02i", synth -> card -> number );
     else
      snd_iprintf( buffer, "XX" );
    snd_iprintf( buffer, ": %s\n", synth -> name );
  }
  snd_mutex_up_static( open );
  snd_mutex_up_static( register );
}

/*
 *  ENTRY functions
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_synth_export;
#endif

static snd_info_entry_t *snd_synth_proc_entry = NULL;

int init_module( void )
{
  snd_info_entry_t *entry;

#ifndef LINUX_2_1
  if ( register_symtab( &snd_symbol_table_synth_export ) < 0 )
    return -ENOMEM;
#endif
  memset( snd_synth_devices, 0, sizeof( snd_synth_devices ) );
  if ( (entry = snd_info_create_entry( NULL, "synth" )) != NULL ) {
    entry -> t.text.read_size = SND_SYNTH_DEVICES * 128;
    entry -> t.text.read = snd_synth_proc_read;
    if ( snd_info_register( entry ) < 0 ) {
      snd_info_free_entry( entry );
      entry = NULL;
    }
  }
  snd_synth_proc_entry = entry;
  return 0;
}

void cleanup_module( void )
{
  if ( snd_synth_proc_entry ) {
    snd_info_unregister( snd_synth_proc_entry );
    snd_synth_proc_entry = NULL;
  }
}
