/*
 *   ALSA sequencer Priority Queue
 *   Copyright (c) 1998 by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"
#include "seq_timer.h"
#include "seq_prioq.h"
#include "seq_timer.h"


/* Implementation is a simple linked list for now...

   This priority queue orders the events on timestamp. For events with an
   equeal timestamp the queue behaves as a FIFO. 

   *
   *           +-------+
   *  Head --> | first |
   *           +-------+
   *                 |ptr_l
   *           +-----v-+
   *           |       |
   *           +-------+
   *                 |
   *           +-----v-+
   *           |       |
   *           +-------+
   *                 |
   *           +-----v-+
   *  Tail --> | last  |
   *           +-------+
   *

 */

void snd_seq_prioq_init(prioq_t * f)
{
	unsigned long flags;

	if (f == NULL) {
		snd_printk("oops: snd_seq_prioq_init() called with NULL prioq\n");
		return;
	}
	snd_spin_prepare(f, lock);
	snd_spin_lock(f, lock, &flags);
	f->head = NULL;
	f->tail = NULL;
	f->cells = 0;
	snd_spin_unlock(f, lock, &flags);
}


/* compare timestamp between events */
/* return 1 if a >= b; 0 */
static inline int compare_timestamp(snd_seq_event_t * a, snd_seq_event_t * b)
{
	if ((a->flags & SND_SEQ_TIME_STAMP_MASK) == SND_SEQ_TIME_STAMP_REAL) {
		/* compare ticks */
		return (snd_seq_compare_tick_time(&a->time.tick, &b->time.tick));
	} else {
		/* compare real time */
		return (snd_seq_compare_real_time(&a->time.real, &b->time.real));
	}
}

/* enqueue cell to prioq */
void snd_seq_prioq_cell_in(prioq_t * f, snd_seq_event_cell_t * cell)
{
	snd_seq_event_cell_t *cur, *prev;
	unsigned long flags;

	if (f == NULL) {
		snd_printk("oops: snd_seq_prioq_cell_in() called with NULL prioq\n");
		return;
	}
	if (cell == NULL) {
		snd_printk("oops: snd_seq_prioq_cell_in() called with NULL cell\n");
		return;
	}
	snd_spin_lock(f, lock, &flags);

	/* check if this element needs to inserted at the end (ie. ordered 
	   data is inserted) This will be very likeley if a sequencer 
	   application or midi file player is feeding us (sequential) data */
	if (f->tail) {
		if (compare_timestamp(&cell->event, &f->tail->event)) {
			/* add new cell to tail of the fifo */
			f->tail->ptr_l = cell;
			f->tail = cell;
			cell->ptr_l = NULL;
			f->cells++;
			snd_spin_unlock(f, lock, &flags);
			return;
		}
	}
	/* traverse list of elements to find the place where the new cell is
	   to be inserted... Note that this is a order n process ! */

	prev = NULL;		/* previous cell */
	cur = f->head;		/* cursor */

	while (cur != NULL) {
		/* compare timestamps */
		if (!compare_timestamp(&cell->event, &cur->event)) {
			/* new cell has earlier schedule time, */
			/* insert it before cursor */

			if (prev != NULL)
				prev->ptr_l = cell;
			cell->ptr_l = cur;

			/* this is the first cell, set head to it */
			if (f->head == cur)
				f->head = cell;

			f->cells++;
			snd_spin_unlock(f, lock, &flags);
			return;
		} else {
			/* new cell has equal or larger schedule time, */
			/* move cursor to next cell */
			prev = cur;
			cur = cur->ptr_l;
		}
	}

	/* reached end of the list */
	if (prev != NULL)
		prev->ptr_l = cell;
	cell->ptr_l = NULL;
	f->tail = cell;
	if (f->head == NULL)
		f->head = cell;
	f->cells++;
	snd_spin_unlock(f, lock, &flags);
}

/* dequeue cell from prioq */
snd_seq_event_cell_t *snd_seq_prioq_cell_out(prioq_t * f)
{
	snd_seq_event_cell_t *cell;
	unsigned long flags;

	if (f == NULL) {
		snd_printk("oops: snd_seq_prioq_cell_in() called with NULL prioq\n");
		return NULL;
	}
	snd_spin_lock(f, lock, &flags);

	if (f->head != NULL) {
		cell = f->head;
		f->head = cell->ptr_l;

		/* reset tail if this was the last element */
		if (f->tail == cell)
			f->tail = NULL;

		cell->ptr_l = NULL;
		f->cells--;
	} else {
		/* empty queue */
		snd_spin_unlock(f, lock, &flags);
		return NULL;
	}

	snd_spin_unlock(f, lock, &flags);
	return cell;
}

/* return number of events available in prioq */
int snd_seq_prioq_avail(prioq_t * f)
{
	if (f == NULL) {
		snd_printk("oops: snd_seq_prioq_cell_in() called with NULL prioq\n");
		return 0;
	}
	return f->cells;
}


/* peek at cell at the head of the prioq */
snd_seq_event_cell_t *snd_seq_prioq_cell_peek(prioq_t * f)
{
	if (f == NULL) {
		snd_printk("oops: snd_seq_prioq_cell_in() called with NULL prioq\n");
		return NULL;
	}
	return f->head;
}
