/*
 *   ALSA sequencer Queue handling
 *   Copyright (c) by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"

#include "seq_memory.h"
#include "seq_queue.h"
#include "seq_clientmgr.h"
#include "seq_fifo.h"
#include "seq_timer.h"


/* root for the queues */
static queue_t queuetab[SND_SEQ_MAX_QUEUES];


/* list of active queues (for optimisation of timer) */

/* test only... */
extern snd_seq_real_time_t loc_time;




/* return pointer to queue structure for specified id */
static queue_t *queueptr(int queueid)
{
	if (queueid < 0 || queueid >= SND_SEQ_MAX_QUEUES) {
		snd_printk("Seq: oops. Trying to get pointer to queue %d\n", queueid);
		return NULL;
	}
	return &queuetab[queueid];
}



/* setup queues */
void snd_seq_queue_init(void)
{
	int c;
	queue_t *q;

	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		q = queueptr(c);

		snd_seq_prioq_init(&q->tickq);
		snd_seq_prioq_init(&q->timeq);
	}

	/* setup defaults */
	q->ppq = 384;

	q->cur_tick = 0;
	q->cur_time.sec = 0;
	q->cur_time.nsec = 0;
}


/* -------------------------------------------------------- */

snd_spin_define_static(chkqueues_lock);


/* check queues and dispatch events */
void snd_seq_check_queues(void)
{
	static int blocked;
	unsigned long flags;
	int c;
	int dequeue = SND_SEQ_MAX_DEQUEUE;
	queue_t *q;
	snd_seq_event_cell_t *cell;

	/* make this function non-reentrant */
	snd_spin_lock_static(chkqueues_lock, &flags);
	if (blocked) {
		snd_spin_unlock_static(chkqueues_lock, &flags);
		snd_printk("Seq: check_queues: other thread already active\n");
		return;		/* other thread is already checking queues */
	}
	blocked = 1;
	snd_spin_unlock_static(chkqueues_lock, &flags);


	/* NOTE: this is not efficient for large number of queues! */
	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		q = queueptr(c);
		if (q == NULL)
			continue;

#if 0
		/* dummy... directly echo it to the addressed client */
		while (snd_seq_prioq_avail(&q->tickq)) {
			cell = snd_seq_prioq_cell_out(&q->tickq);
			if (cell != NULL)
				snd_seq_dispatch_event(cell);
		}
#endif


#if 0
		while (snd_seq_prioq_avail(&q->timeq)) {
			cell = snd_seq_prioq_cell_out(&q->timeq);
			if (cell != NULL)
				snd_seq_dispatch_event(cell);
		}
#endif


		/* limit the number of elements dequeued per pass to save the machine from lockups */
		while (dequeue > 0) {
			cell = snd_seq_prioq_cell_peek(&q->timeq);
			if (cell == NULL)
				break;
			if (snd_seq_compare_real_time(&loc_time, &cell->event.time.real)) {
				cell = snd_seq_prioq_cell_out(&q->timeq);
				if (cell != NULL)
					snd_seq_dispatch_event(cell);
				dequeue--;
			} else {
				/* event remains in the queue */
				break;
			}
		}
	}

	/* free lock */
	blocked = 0;
}



/* enqueue a event to singe queue */
static void snd_seq_enqueue_single_event(snd_seq_event_cell_t * cell)
{
	int dest;
	queue_t *q;

	if (cell == NULL) {
		snd_printk("oops: snd_seq_enqueue_single_event() called with NULL cell\n");
		return;
	}
	dest = cell->event.dest.queue;	/* dest queue */
	q = queueptr(dest);

	if (q == NULL) {
		snd_printk("oops: snd_seq_enqueue_single_event() called with NULL queue\n");
		return;
	}
	/* enqueue event in the real-time or midi queue */
	switch (cell->event.flags & SND_SEQ_TIME_STAMP_MASK) {
		case SND_SEQ_TIME_STAMP_TICK:
			snd_seq_prioq_cell_in(&q->tickq, cell);
			break;

		case SND_SEQ_TIME_STAMP_REAL:
			snd_seq_prioq_cell_in(&q->timeq, cell);
			break;
	}

	/* trigger dispatching */
	snd_seq_check_queues();

	return;
}



/* enqueue a event received from one the clients */
void snd_seq_enqueue_event(snd_seq_event_cell_t * cell)
{
	int dest;

	if (cell == NULL) {
		snd_printk("oops: snd_seq_enqueue_event() called with NULL cell\n");
		return;
	}
	dest = cell->event.dest.queue;	/* dest queue */

	/* handle relative time stamps */
	if ((cell->event.flags & SND_SEQ_TIME_MODE_MASK) == SND_SEQ_TIME_MODE_REL) {
		switch (cell->event.flags & SND_SEQ_TIME_STAMP_MASK) {
			case SND_SEQ_TIME_STAMP_TICK:
				snd_printk("relative song stamps not yet supported \n");
				break;
			case SND_SEQ_TIME_STAMP_REAL:
				snd_seq_inc_real_time(&cell->event.time.real, &loc_time);
				break;
		}
	}
	if ((dest >= 0) && (dest < 255)) {
		snd_seq_enqueue_single_event(cell);
	}
	if (dest == 255) {
		snd_printk("Seq: broadcasts to all queues are not supported yet (dest = %d)\n", dest);

		/* release the original cell */
		snd_seq_cell_free(cell);
	}
}
