/*
 *   ALSA sequencer Priority Queue
 *   Copyright (c) by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stddef.h>
#include <stdlib.h>

/*#include "driver.h" */
#include "seq_prioq.h"




/* Implementation is a simple linked list for now...

   This priority queue orders the events on timestamp. For events with an
   equeal timestamp the queue behaves as a FIFO. 


   +-------+
   Head --> | first |
   +-------+
   |ptr_l
   +-----v-+
   |       |
   +-------+
   |
   +-----v-+
   |       |
   +-------+
   |
   +-----v-+
   Tail --> | last  |
   +-------+


 */

void snd_seq_prioq_init(prioq_t * f)
{
	f->head = NULL;
	f->tail = NULL;
	f->cells = 0;
}


/* enqueue cell to prioq */
void snd_seq_prioq_cell_in(prioq_t * f, snd_seq_event_cell_t * cell)
{
	snd_seq_event_cell_t *cur, *prev;

	/* traverse list of element to find the place where the new cell is
	   to be inserted... Note that this is a order n process ! */

	prev = NULL;		/* previous cell */
	cur = f->head;		/* cursor */

	while (cur != NULL) {
		/* compare timestamps */
		if (cell->event.time.tick < cur->event.time.tick) {
			/* new cell has earlier schedule time, */
			/* insert it before cursor */

			if (prev != NULL)
				prev->ptr_l = cell;
			cell->ptr_l = cur;

			/* set head this is the first cell */
			if (f->head == cur)
				f->head = cell;

			f->cells++;
			return;
		} else {
			/* new cell has equal or larger schedule time, */
			/* move cursor to next cell */
			prev = cur;
			cur = cur->ptr_l;
		}
	}

	/* reached end of the list */
	if (prev != NULL)
		prev->ptr_l = cell;
	cell->ptr_l = NULL;
	f->tail = cell;
	if (f->head == NULL)
		f->head = cell;
	f->cells++;
}

/* dequeue cell from prioq */
snd_seq_event_cell_t *snd_seq_prioq_cell_out(prioq_t * f)
{
	snd_seq_event_cell_t *cell;

	if (f->head != NULL) {
		cell = f->head;
		f->head = cell->ptr_l;

		/* reset tail if this was the last element */
		if (f->tail == cell)
			f->tail = NULL;

		cell->ptr_l = NULL;
		f->cells--;
	} else {
		/* empty queue */
		return NULL;
	}

	return cell;
}

/* return number of events available in prioq */
int snd_seq_prioq_avail(prioq_t * f)
{
	return f->cells;
}


/* peek at cell at the head of the prioq */
snd_seq_event_cell_t *snd_seq_prioq_cell_peek(prioq_t * f)
{
	return f->head;
}
