/*
 *  ALSA sequencer Memory Manager
 *  Copyright (c) by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include "driver.h"
#include "minors.h"
#include "info.h"
#include "control.h"

#include "seq_memory.h"


/* design note: the pool is a contigious block of memory, if we dynamicly
   want to add additional cells to the pool be better store this in another
   pool as we need to know the base address of the pool when releasing
   memory. */

typedef struct {
	snd_seq_event_cell_t *ptr;	/* pointer to start of pool */
	snd_seq_event_cell_t *free;	/* pointer to the head of the free list */

	int total_elements;
	int available;

	/* statistics */
	int min_available;

	  snd_spin_define(lock);
} pool_t;

/* pool of free event cells */
static pool_t *free_pool;

/* counter for tracking memory leaks for 'external' data */
static int ext_alloc;

/* release this cell */
void snd_seq_cell_free(snd_seq_event_cell_t * cell)
{
	unsigned long flags;

	snd_spin_lock(free_pool, lock, &flags);
	if (cell != NULL) {
		if (free_pool->free != NULL) {
			/* normal situation */
			cell->ptr_l = free_pool->free;	/* chain in old element */
			cell->ptr_r = NULL;
			free_pool->free = cell;
		} else {
			/* first element */
			free_pool->free = cell;
			cell->ptr_l = NULL;
			cell->ptr_r = NULL;
		}
		free_pool->available++;
	}
	snd_spin_unlock(free_pool, lock, &flags);

#if 0
	/* debug... */
	snd_printk("Seq: memory total=%d avail=%d\n",
		   free_pool->total_elements,
		   free_pool->available);
#endif
}


/* return pointer to cell. NULL on failure */
snd_seq_event_cell_t *snd_seq_cell_alloc(void)
{
	snd_seq_event_cell_t *cell;
	unsigned long flags;

	snd_spin_lock(free_pool, lock, &flags);
	if (free_pool->free != NULL) {
		cell = free_pool->free;
		free_pool->free = cell->ptr_l;
		free_pool->available--;
		if (free_pool->available < free_pool->min_available)
			free_pool->min_available = free_pool->available;

		/* clear cell pointers */
		cell->ptr_l = NULL;
		cell->ptr_r = NULL;
	} else {
		/* no element available... */
		snd_printk("Seq: cell_alloc failed: no cells available\n");
		cell = NULL;
	}
	snd_spin_lock(free_pool, lock, &flags);

#if 0
	/* debug... */
	snd_printk("Seq: memory total=%d avail=%d\n",
		   free_pool->total_elements,
		   free_pool->available);
#endif
	return cell;
}


/* duplicate event, NULL on failure */
extern snd_seq_event_cell_t *snd_seq_event_dup(snd_seq_event_t * event)
{
	snd_seq_event_cell_t *new_cell;

	if (event == NULL)
		return NULL;
	
	new_cell = snd_seq_cell_alloc();
	snd_printk( "new_cell = 0x%x\n", (int)new_cell );
	if (new_cell) {
		snd_printk( "new_cell = (1)\n" );
		memcpy(&new_cell->event, event, sizeof(snd_seq_event_t));
		snd_printk( "new cell = (2)\n" );
	}
	return new_cell;
}

extern snd_seq_event_cell_t *snd_seq_event_dup_from_user(snd_seq_event_t * event)
{
	snd_seq_event_cell_t *new_cell;

	if (event == NULL)
		return NULL;

	new_cell = snd_seq_cell_alloc();
	if (new_cell) {
		copy_from_user(&new_cell->event, event, sizeof(snd_seq_event_t));
	}
	return new_cell;
}


/* duplicate event cell, NULL on failure */
extern snd_seq_event_cell_t *snd_seq_cell_dup(snd_seq_event_cell_t * cell)
{
	snd_seq_event_cell_t *new_cell;

	if (cell == NULL)
		return 0;

	new_cell = snd_seq_cell_alloc();
	if (new_cell) {
		memcpy(&new_cell->event, &cell->event, sizeof(snd_seq_event_t));
	}
	return new_cell;
}


/* return number of unused (free) cells */
int snd_seq_unused_cells(void)
{
	return (free_pool->available);
}


/* return total number of allocated cells */
int snd_seq_total_cells(void)
{
	return (free_pool->total_elements);
}



/* init memory, allocate room specified number of events */
void snd_sequencer_memory_init(int events)
{
	int cell;
	snd_seq_event_cell_t *cellptr;
	unsigned long flags;

	snd_spin_prepare(free_pool, lock);

	/* create pool block */
	free_pool = snd_malloc(sizeof(pool_t));
	if (!free_pool) {
		snd_printk("Seq: malloc failed for free_pool\n");
		return;
	}
	snd_spin_lock(free_pool, lock, &flags);
	free_pool->ptr = NULL;
	free_pool->free = NULL;
	free_pool->total_elements = 0;
	free_pool->available = 0;
	snd_spin_unlock(free_pool, lock, &flags);


	/* alloc memory */
	free_pool->ptr = snd_malloc(sizeof(snd_seq_event_cell_t) * events);
	if (free_pool->ptr) {
		memset(free_pool->ptr, 0, sizeof(snd_seq_event_cell_t) * events);
		free_pool->total_elements = events;

		snd_printk("Seq: memory init, %d events, base at 0x%p\n", events, free_pool->ptr);
		snd_printk("Seq: size of element = %d\n", sizeof(snd_seq_event_cell_t));

		/* add the new cell's to the free cell list by calling the free()
		   function for each one */
		for (cell = 0; cell < events; cell++) {
			cellptr = &free_pool->ptr[cell];
			snd_seq_cell_free(cellptr);
		}
	} else {
		snd_printk("Seq: malloc for sequencer events failed\n");
	}

	/* init statistics */
	free_pool->min_available = free_pool->available;
}


/* release event memory */
void snd_sequencer_memory_done(void)
{
	if (free_pool) {
		if (free_pool->ptr != NULL) {
			snd_printk("Seq: memory done, total=%d avail=%d min_avail=%d\n",
				   free_pool->total_elements,
				   free_pool->available,
				   free_pool->min_available);

			snd_free(free_pool->ptr, sizeof(snd_seq_event_cell_t) * free_pool->total_elements);
			free_pool->ptr = NULL;
			free_pool->free = NULL;
			free_pool->total_elements = 0;
			free_pool->available = 0;
		}
		snd_free(free_pool, sizeof(pool_t));
	}
	
	if (ext_alloc > 0) {
		snd_printk("seq: memory leak alert, still %d blocks of external data allocated\n", ext_alloc);
	}
}


/* wrapper for allocating and freeing 'external' data (eg. sysex, meta
   events etc.) for now it is just passed to the snd_malloc and snd_free
   calls, but in a later stadium a different allocated could be used. */
   
void *snd_seq_ext_malloc(unsigned long size)
{
	ext_alloc++;
	
	return snd_malloc(size);
}

void snd_seq_ext_free(void *obj, unsigned long size)
{
	ext_alloc--;
	if (ext_alloc < 0) {
		snd_printk("seq: whoops, more ext data free()s than malloc()...\n");
	}
	
	snd_free(obj,size);
}
